/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>

#include "gettext.h"
#include "cdw_config.h"
#include "cdw_mkudffs_options.h"
#include "cdw_fs.h"
#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_utils.h"
#include "cdw_form.h"
#include "cdw_logging.h"
#include "cdw_file_picker.h"
#include "cdw_file_manager.h"
#include "cdw_main_window.h"
#include "canonicalize.h"






/*
mkudffs 1.0.0b2 for UDF FS 1.0.0-cvs, 2002/02/09
Usage:
	mkudffs [options] device [blocks-count]
Switches:
	--help               - don't include
	--blocksize=, -b     - first page, dropdown
	--udfrev=, -r        - first page, dropdown
	--lvid=              - main page of wizard, input field
	--vid=               - second page, input field
	--vsid=              - second page, input field
	--fsid=              - second page, input field
	--strategy=          - first page, dropdown
	--spartable=         - first page, dropdown
	--packetlen=         - ???
	--media-type=        - first page, dropdown
	--space=             - first page, dropdown
	--ad=                - first page, dropdown
	--noefe              - first page, checkbox
	--u8                 - first page, common dropdown
	--u16                - first page, common dropdown
	--utf8               - first page, common dropdown
	--bridge             - ???
	--closed             - ???
*/





/* First page. */
enum {
	f_blocksize_l = 0,
	f_blocksize_dd,
	f_udfrev_l,
	f_udfrev_dd,
	f_strategy_l,
	f_strategy_dd,
	f_spartable_l,
	f_spartable_dd,
	f_media_type_l,
	f_media_type_dd,
	f_space_l,
	f_space_dd,
	f_ad_l,
	f_ad_dd,
	f_noefe_l,
	f_noefe_cb,
	f_encoding_l,
	f_encoding_dd,
	f_mkudffs_other_l,
	f_mkudffs_other_i,
	f_mount_point_l,
	f_mount_point_i,
	f_mount_point_b,
	f_rsync_options_t,
	f_rsync_options_dl,
	f_rsync_options_i
};


static struct {

	cdw_udf_t *udf;

	cdw_form_t *cdw_form_p1;
	cdw_form_t *cdw_form_p2;

} options_window;



static FIELD *mkudffs_fields_p1[CDW_MKUDFFS_UDF_MAIN_OPTIONS_N_FIELDS + 1]; /* +1 for last field == NULL */


static int cdw_mkudffs_options_on_mount_point(void *dummy1, void *dummy2);


static CDW_DROPDOWN *cdw_mkudffs_blocksize_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_mkudffs_udfrev_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_mkudffs_strategy_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_mkudffs_spartable_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_mkudffs_media_type_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_mkudffs_space_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_mkudffs_ad_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_mkudffs_encoding_dropdown(WINDOW *window, int begin_y, int begin_x, int width);

static cdw_form_dropdown_maker_t dropdown_makers[] = {
	cdw_mkudffs_blocksize_dropdown,
	cdw_mkudffs_udfrev_dropdown,
	cdw_mkudffs_strategy_dropdown,
	cdw_mkudffs_spartable_dropdown,
	cdw_mkudffs_media_type_dropdown,
	cdw_mkudffs_space_dropdown,
	cdw_mkudffs_ad_dropdown,
	cdw_mkudffs_encoding_dropdown
};


static char *cdw_mkudffs_symlinks_text(int n_cols);

static cdw_form_text_maker_t text_makers[] = {
	cdw_mkudffs_symlinks_text
};



/**
   \brief Create fields that are shown on "XXX" page in options

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_GEN_ERROR if one of fields was created incorrectly
*/
cdw_rv_t cdw_mkudffs_options_form_p1(cdw_form_t *cdw_form, void *cdw_udf, int first_col, __attribute__((unused)) int second_col, __attribute__((unused)) int width_wide, int width_narrow)
{
	options_window.udf = (cdw_udf_t *) cdw_udf;
	options_window.cdw_form_p1 = cdw_form;
	options_window.cdw_form_p1->fields = mkudffs_fields_p1;
	cdw_udf_t *udf = options_window.udf;

	second_col += 3;
	int third_col = first_col + width_wide - 4;
	int width_narrow1 = width_narrow + 3;
	int width_narrow2 = (int) (width_narrow / 1.5);

	cdw_assert (CDW_MKUDFFS_OTHER_OPTIONS_LEN_MAX == 200, "ERROR: maximal length of string has changed. Update the text message in code below.\n");
	cdw_assert (CDW_RSYNC_OPTIONS_LEN_MAX == 600, "ERROR: maximal length of string changed. Update the text message in code below.\n");

	cdw_form_descr_t descr[] = {
		/* type                      begin_y   begin_x      n_cols    n_lines   field enum            data1                                       data2 */
		{ CDW_WIDGET_ID_STATIC_LABEL,      1,  first_col,   width_narrow1,  1,  f_blocksize_l,        _("Block size:"),                               0 },
		{ CDW_WIDGET_ID_DROPDOWN,          1,  second_col,  width_narrow2,  1,  f_blocksize_dd,       dropdown_makers,                                0 },

		{ CDW_WIDGET_ID_STATIC_LABEL,      2,  first_col,   width_narrow1,  1,  f_udfrev_l,           _("UDF revision:"),                             0 },
		{ CDW_WIDGET_ID_DROPDOWN,          2,  second_col,  width_narrow2,  1,  f_udfrev_dd,          dropdown_makers,                                1 },

		{ CDW_WIDGET_ID_STATIC_LABEL,      3,  first_col,   width_narrow1,  1,  f_strategy_l,         _("Allocation strategy:"),                      0 },
		{ CDW_WIDGET_ID_DROPDOWN,          3,  second_col,  width_narrow2,  1,  f_strategy_dd,        dropdown_makers,                                2 },

		{ CDW_WIDGET_ID_STATIC_LABEL,      4,  first_col,   width_narrow1,  1,  f_spartable_l,        _("Sparing tables:"),                           0 },
		{ CDW_WIDGET_ID_DROPDOWN,          4,  second_col,  width_narrow2,  1,  f_spartable_dd,       dropdown_makers,                                3 },

		{ CDW_WIDGET_ID_STATIC_LABEL,      5,  first_col,   width_narrow1,  1,  f_media_type_l,       _("Media type:"),                               0 },
		{ CDW_WIDGET_ID_DROPDOWN,          5,  second_col,  width_narrow2,  1,  f_media_type_dd,      dropdown_makers,                                4 },

		{ CDW_WIDGET_ID_STATIC_LABEL,      6,  first_col,   width_narrow1,  1,  f_space_l,            _("Space:"),                                    0 },
		{ CDW_WIDGET_ID_DROPDOWN,          6,  second_col,  width_narrow2,  1,  f_space_dd,           dropdown_makers,                                5 },

		{ CDW_WIDGET_ID_STATIC_LABEL,      7,  first_col,   width_narrow1,  1,  f_ad_l,               _("Allocation descriptors:"),                   0 },
		{ CDW_WIDGET_ID_DROPDOWN,          7,  second_col,  width_narrow2,  1,  f_ad_dd,              dropdown_makers,                                6 },

		{ CDW_WIDGET_ID_STATIC_LABEL,      8,  first_col,   width_narrow1,  1,  f_noefe_l,            _("Don't use Extended File Entries:"),          0 },
		{ CDW_WIDGET_ID_CHECKBOX,          8,  second_col + 1,  1,          1,  f_noefe_cb,           (void *) NULL,                 udf->mkudffs_noefe },

		{ CDW_WIDGET_ID_STATIC_LABEL,      9,  first_col,   width_narrow1,  1,  f_encoding_l,         _("Encoding of file names:"),                   0 },
		{ CDW_WIDGET_ID_DROPDOWN,          9,  second_col,  width_narrow2,  1,  f_encoding_dd,        dropdown_makers,                                7 },

		{ CDW_WIDGET_ID_DYNAMIC_LABEL,    10,  first_col,   width_wide,     1,  f_mkudffs_other_l,    _("Other mkudffs options (200 chars max):"),    0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,  11,  first_col,   width_wide,     1,  f_mkudffs_other_i,    udf->mkudffs_other_options,   CDW_MKUDFFS_OTHER_OPTIONS_LEN_MAX },

		/* This isn't really mkudffs option. It is used by
		   "mount", which is a step in a recipe using mkudffs
		   to create UDF image. */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,    13,  first_col,   width_wide,      1,  f_mount_point_l,     _("Existing mount directory (mount point):"),   0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,  14,  first_col,   width_wide - 5,  1,  f_mount_point_i,     udf->mount_point,                               0 },
		/* 2TRANS: do not translate */
		{ CDW_WIDGET_ID_BUTTON,           14,  third_col,   1,               1,  f_mount_point_b,     _(">"),                         CDW_COLORS_DIALOG },

		{ CDW_WIDGET_ID_TEXT,             15,  first_col,   width_wide,      2,  f_rsync_options_t,   text_makers,                                    0 },
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,    17,  first_col,   width_wide,      1,  f_rsync_options_dl,  _("rsync options string (600 chars max):"),     0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,  18,  first_col,   width_wide,      1,  f_rsync_options_i,   udf->rsync_options,     CDW_RSYNC_OPTIONS_LEN_MAX },

		/* guard */
		{ -1,                              0,  0,           0,               0,  0,                   (void *) NULL,                                   0 }};


	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		cdw_form_bind_input_and_label(cdw_form, f_mkudffs_other_i, f_mkudffs_other_l);
		cdw_form_bind_input_and_label(cdw_form, f_rsync_options_i, f_rsync_options_dl);
		cdw_form_bind_input_and_label(cdw_form, f_mount_point_i, f_mount_point_l);
		cdw_form_set_widget_callback(cdw_form, f_mount_point_b, (cdw_form_widget_function_t) cdw_mkudffs_options_on_mount_point);

		return CDW_OK;
	}
}





cdw_rv_t cdw_mkudffs_options_validate_p1(cdw_udf_t *udf, int *fi)
{
	cdw_rv_t crv = cdw_string_security_parser(udf->mkudffs_other_options, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_mkudffs_other_i;
		return CDW_NO;
	}



	crv = cdw_string_security_parser(udf->mount_point, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_mount_point_i;
		return CDW_NO;
	}

	int rv = cdw_fs_check_existing_path(udf->mount_point, W_OK, CDW_FS_DIR);
	if (rv != 0) {
		cdw_sdm ("INFO: \"%s\": errno: %s\n", udf->mount_point, strerror(rv));
		/* 2TRANS: this is title of dialog window. */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window. "Mount point" as an entry in /etc/fstab. */
				   _("Mount point is not an existing, writable directory."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		*fi = f_mount_point_i;
		return CDW_NO;
	}

	/* The function does only very, very simple check, not looking
	   for symlinks deep in structure of selected files. For now
	   it will have to be enough. */
	crv = cdw_selected_files_search(udf->mount_point, CAN_EXISTING);
	if (crv == CDW_OK || crv == CDW_ERROR) {
		/* 2TRANS: this is title of dialog window. */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window. "Mount point" as an entry in /etc/fstab. */
				   _("Mount point directory is on list of selected files."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		*fi = f_mount_point_i;
		return CDW_NO;
	}



	crv = cdw_string_security_parser(udf->rsync_options, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_rsync_options_i;
		return CDW_NO;
	}

	if (!strlen(udf->rsync_options)) {
		/* 2TRANS: this is title of dialog window. */
		crv = cdw_buttons_dialog(_("Warning"),
					 /* 2TRANS: this is message in dialog window. */
					 _("No options for rsync are set.\nImage file created with empty rsync options may be invalid or incomplete. Press OK to continue, or press Cancel to go back to Options."),
					 CDW_BUTTONS_OK_CANCEL, CDW_COLORS_WARNING);

		if (crv == CDW_OK) {
			; /* Pass */
		} else { /* CDW_NO */
			*fi = f_rsync_options_i;
			return CDW_NO;
		}
	}


	/*
	  man mkudffs(1):
	  --spartable= spartable-number

                  Specify the number of sparing tables to write to
                  disc. Valid numbers are 1 - 4.  This option implies
                  --media-type=cdrw.
	*/
	if (udf->mkudffs_spartable != CDW_MKUDFFS_SPARTABLE_UNSPECIFIED) {
		if (udf->mkudffs_media_type == CDW_MKUDFFS_MEDIA_TYPE_UNSPECIFIED
		    || udf->mkudffs_media_type == CDW_MKUDFFS_MEDIA_TYPE_CDRW) {

			; /* Pass. */
		} else {
			/* 2TRANS: this is title of dialog window. */
			cdw_buttons_dialog(_("Error"),
					   /* 2TRANS: this is message in dialog window, informing about dependencies between options. */
					   _("Selecting specific value of \"Sparing table\" requires either not specifying \"Media type\" or setting it to \"CD-RW\"."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

			*fi = f_media_type_dd;
			return CDW_NO;
		}
	}

	return CDW_OK;
}





cdw_rv_t cdw_mkudffs_options_save_p1(cdw_udf_t *udf, cdw_form_t *cdw_form)
{
	CDW_DROPDOWN *dd = NULL;

	dd = cdw_form_get_dropdown(cdw_form, f_blocksize_dd);
	udf->mkudffs_blocksize = cdw_dropdown_get_current_item_id(dd);

	dd = cdw_form_get_dropdown(cdw_form, f_udfrev_dd);
	udf->mkudffs_udfrev = cdw_dropdown_get_current_item_id(dd);

	dd = cdw_form_get_dropdown(cdw_form, f_strategy_dd);
	udf->mkudffs_strategy = cdw_dropdown_get_current_item_id(dd);

	dd = cdw_form_get_dropdown(cdw_form, f_spartable_dd);
	udf->mkudffs_spartable = cdw_dropdown_get_current_item_id(dd);

	dd = cdw_form_get_dropdown(cdw_form, f_media_type_dd);
	udf->mkudffs_media_type = cdw_dropdown_get_current_item_id(dd);

	dd = cdw_form_get_dropdown(cdw_form, f_space_dd);
	udf->mkudffs_space = cdw_dropdown_get_current_item_id(dd);

	dd = cdw_form_get_dropdown(cdw_form, f_ad_dd);
	udf->mkudffs_ad = cdw_dropdown_get_current_item_id(dd);

	udf->mkudffs_noefe = cdw_form_get_checkbox_state(cdw_form, f_noefe_cb);

	dd = cdw_form_get_dropdown(cdw_form, f_encoding_dd);
	udf->mkudffs_encoding = cdw_dropdown_get_current_item_id(dd);

	const char *s = cdw_form_get_string(cdw_form, f_mkudffs_other_i);
	strncpy(udf->mkudffs_other_options, s, CDW_MKUDFFS_OTHER_OPTIONS_LEN_MAX);
	        udf->mkudffs_other_options[CDW_MKUDFFS_OTHER_OPTIONS_LEN_MAX] = '\0';

	s = cdw_form_get_string(cdw_form, f_rsync_options_i);
	strncpy(udf->rsync_options, s, CDW_RSYNC_OPTIONS_LEN_MAX);
	        udf->rsync_options[CDW_RSYNC_OPTIONS_LEN_MAX] = '\0';

	s = cdw_form_get_string(cdw_form, f_mount_point_i);
	cdw_string_set(&(udf->mount_point), s);

	return CDW_OK;
}










static cdw_id_clabel_t block_size_items[CDW_MKUDFFS_BLOCKSIZE_MAX] = {
	{ CDW_MKUDFFS_BLOCKSIZE_UNSPECIFIED,  gettext_noop("Unspecified") },
	{ CDW_MKUDFFS_BLOCKSIZE_1024,         gettext_noop("1024")        },
	{ CDW_MKUDFFS_BLOCKSIZE_2048,         gettext_noop("2048")        },
	{ CDW_MKUDFFS_BLOCKSIZE_4096,         gettext_noop("4096")        }
};

const char *cdw_mkudffs_option_blocksize[CDW_MKUDFFS_BLOCKSIZE_MAX] = {
	"",
	"1024",
	"2048",
	"4096"
};


CDW_DROPDOWN *cdw_mkudffs_blocksize_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, CDW_MKUDFFS_BLOCKSIZE_MAX, block_size_items);

	cdw_dropdown_set_current_item_by_id(dd, options_window.udf->mkudffs_blocksize);

	return dd;
}





static cdw_id_clabel_t udfrev_items[CDW_MKUDFFS_UDFREV_MAX] = {
	{ CDW_MKUDFFS_UDFREV_UNSPECIFIED,  gettext_noop("Unspecified") },
	{ CDW_MKUDFFS_UDFREV_0x0201,       gettext_noop("2.01")        },
	{ CDW_MKUDFFS_UDFREV_0x0200,       gettext_noop("2.00")        },
	{ CDW_MKUDFFS_UDFREV_0x0150,       gettext_noop("1.50")        },
	{ CDW_MKUDFFS_UDFREV_0x0102,       gettext_noop("1.02")        }
};

const char *cdw_mkudffs_option_udfrev[CDW_MKUDFFS_UDFREV_MAX] = {
	"",
	"0x0201",
	"0x0200",
	"0x0150",
	"0x0102"
};

CDW_DROPDOWN *cdw_mkudffs_udfrev_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, CDW_MKUDFFS_UDFREV_MAX, udfrev_items);

	cdw_dropdown_set_current_item_by_id(dd, options_window.udf->mkudffs_udfrev);

	return dd;
}





static cdw_id_clabel_t strategy_items[CDW_MKUDFFS_STRATEGY_MAX] = {
	{ CDW_MKUDFFS_STRATEGY_UNSPECIFIED,  gettext_noop("Unspecified") },
	{ CDW_MKUDFFS_STRATEGY_4,            gettext_noop("4")           },
	{ CDW_MKUDFFS_STRATEGY_4096,         gettext_noop("4096")        }
};

const char *cdw_mkudffs_option_strategy[CDW_MKUDFFS_STRATEGY_MAX] = {
	"",
	"4",
	"4096"
};

CDW_DROPDOWN *cdw_mkudffs_strategy_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, CDW_MKUDFFS_STRATEGY_MAX, strategy_items);

	cdw_dropdown_set_current_item_by_id(dd, options_window.udf->mkudffs_strategy);

	return dd;
}





static cdw_id_clabel_t spartable_items[CDW_MKUDFFS_SPARTABLE_MAX] = {
	{ CDW_MKUDFFS_SPARTABLE_UNSPECIFIED,  gettext_noop("Unspecified") },
	{ CDW_MKUDFFS_SPARTABLE_1,            gettext_noop("1")           },
	{ CDW_MKUDFFS_SPARTABLE_2,            gettext_noop("2")           },
	{ CDW_MKUDFFS_SPARTABLE_3,            gettext_noop("3")           },
	{ CDW_MKUDFFS_SPARTABLE_4,            gettext_noop("4")           }
};

const char *cdw_mkudffs_option_spartable[CDW_MKUDFFS_SPARTABLE_MAX] = {
	"",
	"1",
	"2",
	"3",
	"4"
};

CDW_DROPDOWN *cdw_mkudffs_spartable_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, CDW_MKUDFFS_SPARTABLE_MAX, spartable_items);

	cdw_dropdown_set_current_item_by_id(dd, options_window.udf->mkudffs_spartable);

	return dd;
}





static cdw_id_clabel_t media_type_items[CDW_MKUDFFS_MEDIA_TYPE_MAX] = {
	{ CDW_MKUDFFS_MEDIA_TYPE_UNSPECIFIED,  gettext_noop("Unspecified") },
	{ CDW_MKUDFFS_MEDIA_TYPE_HD,           gettext_noop("Hard Disc")   },
	{ CDW_MKUDFFS_MEDIA_TYPE_DVD,          gettext_noop("DVD")         },
	{ CDW_MKUDFFS_MEDIA_TYPE_DVDRAM,       gettext_noop("DVD-RAM")     },
	{ CDW_MKUDFFS_MEDIA_TYPE_WORM,         gettext_noop("WORM")        },
	{ CDW_MKUDFFS_MEDIA_TYPE_MO,           gettext_noop("MO")          },
	{ CDW_MKUDFFS_MEDIA_TYPE_CDRW,         gettext_noop("CD-RW")       }
};

const char *cdw_mkudffs_option_media_type[CDW_MKUDFFS_MEDIA_TYPE_MAX] = {
	"",
	"hd",
	"dvd",
	"dvdram",
	"worm",
	"mo",
	"cdrw"
};

CDW_DROPDOWN *cdw_mkudffs_media_type_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, CDW_MKUDFFS_MEDIA_TYPE_MAX, media_type_items);

	cdw_dropdown_set_current_item_by_id(dd, options_window.udf->mkudffs_media_type);

	return dd;
}







static cdw_id_clabel_t space_items[CDW_MKUDFFS_SPACE_MAX] = {
	{ CDW_MKUDFFS_SPACE_UNSPECIFIED,    gettext_noop("Unspecified")  },
	{ CDW_MKUDFFS_SPACE_FREEDBITMAP,    gettext_noop("freedbitmap")  },
	{ CDW_MKUDFFS_SPACE_FREEDTABLE,     gettext_noop("freedtable")   },
	{ CDW_MKUDFFS_SPACE_UNALLOCBITMAP,  gettext_noop("unallocbitmap")},
	{ CDW_MKUDFFS_SPACE_UNALLOCTABLE,   gettext_noop("unalloctable") }
};

const char *cdw_mkudffs_option_space[CDW_MKUDFFS_SPACE_MAX] = {
	"",
	"freedbitmap",
	"freedtable",
	"unallocbitmap",
	"unalloctable"
};

CDW_DROPDOWN *cdw_mkudffs_space_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, CDW_MKUDFFS_SPACE_MAX, space_items);

	cdw_dropdown_set_current_item_by_id(dd, options_window.udf->mkudffs_space);

	return dd;
}





static cdw_id_clabel_t ad_items[CDW_MKUDFFS_AD_MAX] = {
	{ CDW_MKUDFFS_AD_UNSPECIFIED,    gettext_noop("Unspecified") },
	{ CDW_MKUDFFS_AD_INICB,          gettext_noop("In ICB")      },
	{ CDW_MKUDFFS_AD_SHORT,          gettext_noop("Short")       },
	{ CDW_MKUDFFS_AD_LONG,           gettext_noop("Long")        }
};

const char *cdw_mkudffs_option_ad[CDW_MKUDFFS_AD_MAX] = {
	"",
	"inicb",
	"short",
	"long"
};

CDW_DROPDOWN *cdw_mkudffs_ad_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, CDW_MKUDFFS_AD_MAX, ad_items);

	cdw_dropdown_set_current_item_by_id(dd, options_window.udf->mkudffs_ad);

	return dd;
}







static cdw_id_clabel_t encoding_items[CDW_MKUDFFS_ENCODING_MAX] = {
	{ CDW_MKUDFFS_ENCODING_UNSPECIFIED,    gettext_noop("Unspecified")    },
	{ CDW_MKUDFFS_ENCODING_U8,             gettext_noop("8-bit Unicode")  },
	{ CDW_MKUDFFS_ENCODING_U16,            gettext_noop("16-bit Unicode") },
	{ CDW_MKUDFFS_ENCODING_UTF8,           gettext_noop("UTF-8")          }
};

const char *cdw_mkudffs_option_encoding[CDW_MKUDFFS_ENCODING_MAX] = {
	"",
	"u8",
	"u16",
	"utf8"
};

CDW_DROPDOWN *cdw_mkudffs_encoding_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, CDW_MKUDFFS_ENCODING_MAX, encoding_items);

	cdw_dropdown_set_current_item_by_id(dd, options_window.udf->mkudffs_encoding);

	return dd;
}





char *cdw_mkudffs_symlinks_text(int n_cols)
{
	/* 2TRANS: this is a message displayed in options panel. It is
	   about rsync's options for handling symbolic links. "vvv" is
	   a kind of arrow pointing downwards to said field. */
	return cdw_string_wrap(_("You should specify options for handling of symlinks in field below vvv"), (size_t) n_cols, CDW_ALIGN_LEFT);
}




/* *** Second page. *** */





enum {
	f_lvid_l = 0, /* Logical Volume Identifier. */
	f_lvid_i,
	f_vsid_l,     /* Volume Set Identifier. */
	f_vsid_i,
	f_fsid_l,     /* File Set Identifier. */
	f_fsid_i
};




static FIELD *mkudffs_fields_p2[CDW_MKUDFFS_UDF_META_OPTIONS_N_FIELDS + 1]; /* +1 for last field == NULL */




/**
   \brief Create fields that are shown on "XXX" page in options

   \return CDW_OK when all fields were created (probably) without problems
   \return CDW_GEN_ERROR if one of fields was created incorrectly
*/
cdw_rv_t cdw_mkudffs_options_form_p2(cdw_form_t *cdw_form, void *cdw_udf, int first_col, __attribute__((unused)) int second_col, int width_wide, __attribute__((unused)) int width_narrow)
{
	options_window.udf = (cdw_udf_t *) cdw_udf;
	options_window.cdw_form_p2 = cdw_form;
	options_window.cdw_form_p2->fields = mkudffs_fields_p2;
	cdw_udf_t *udf = options_window.udf;

	int w_lvid = CDW_UDF_LVID_LEN_MAX < width_wide ? CDW_UDF_LVID_LEN_MAX : width_wide;
	int w_fsid = CDW_UDF_FSID_LEN_MAX < width_wide ? CDW_UDF_FSID_LEN_MAX : width_wide;

	cdw_assert (CDW_UDF_LVID_LEN_MAX == 126, "ERROR: value of option has changed. Update the text message in code below.\n");
	cdw_assert (CDW_UDF_VSID_LEN_MAX == 126, "ERROR: value of option has changed. Update the text message in code below.\n");
	cdw_assert (CDW_UDF_FSID_LEN_MAX == 30, "ERROR: value of option has changed. Update the text message in code below.\n");
	cdw_form_descr_t descr[] = {
		/* type                      begin_y   begin_x     n_cols  n_lines  field enum        data1                                         data2 */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     2,  first_col,  width_wide,  1,  f_lvid_l,         _("Logical Volume Identifier (\"lvid\", 126 chars max):"),   0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   3,  first_col,  w_lvid,      1,  f_lvid_i,         udf->mkudffs_lvid,             CDW_UDF_LVID_LEN_MAX },

		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     4,  first_col,  width_wide,  1,  f_vsid_l,         _("Volume Set Identifier (126 chars max):"),      0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   5,  first_col,  width_wide,  1,  f_vsid_i,         udf->mkudffs_vsid,             CDW_UDF_VSID_LEN_MAX },

		{ CDW_WIDGET_ID_DYNAMIC_LABEL,     6,  first_col,  width_wide,  1,  f_fsid_l,         _("File Set Identifier (30 chars max):"),         0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,   7,  first_col,  w_fsid,      1,  f_fsid_i,         udf->mkudffs_fsid,             CDW_UDF_FSID_LEN_MAX },

		/* guard */
		{ -1,                              0,  0,          0,           0,  0,                (void *) NULL,                                               0 }};


	cdw_rv_t crv = cdw_form_description_to_fields(descr, cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		cdw_form_bind_input_and_label(cdw_form, f_lvid_i, f_lvid_l);
		cdw_form_bind_input_and_label(cdw_form, f_vsid_i, f_vsid_l);
		cdw_form_bind_input_and_label(cdw_form, f_fsid_i, f_fsid_l);

		return CDW_OK;
	}
}





cdw_rv_t cdw_mkudffs_options_validate_p2(cdw_udf_t *udf, int *fi)
{
	cdw_rv_t crv = cdw_string_security_parser(udf->mkudffs_lvid, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_lvid_i;
		return CDW_NO;
	}

	crv = cdw_string_security_parser(udf->mkudffs_vsid, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_vsid_i;
		return CDW_NO;
		}

	crv = cdw_string_security_parser(udf->mkudffs_fsid, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_fsid_i;
		return CDW_NO;
	}

	return CDW_OK;
}





cdw_rv_t cdw_mkudffs_options_save_p2(cdw_udf_t *udf, cdw_form_t *cdw_form)
{
	const char *s = NULL;

	s = cdw_form_get_string(cdw_form, f_lvid_i);
	strncpy(udf->mkudffs_lvid, s, CDW_UDF_LVID_LEN_MAX);
	        udf->mkudffs_lvid[CDW_UDF_LVID_LEN_MAX] = '\0';

	s = cdw_form_get_string(cdw_form, f_vsid_i);
	strncpy(udf->mkudffs_vsid, s, CDW_UDF_VSID_LEN_MAX);
	        udf->mkudffs_vsid[CDW_UDF_VSID_LEN_MAX] = '\0';

	s = cdw_form_get_string(cdw_form, f_fsid_i);
	strncpy(udf->mkudffs_fsid, s, CDW_UDF_FSID_LEN_MAX);
	        udf->mkudffs_fsid[CDW_UDF_FSID_LEN_MAX] = '\0';


	return CDW_OK;
}





int cdw_mkudffs_options_on_mount_point(__attribute__((unused)) void *dummy1, __attribute__((unused)) void *dummy2)
{
	/* 2TRANS: this is title of dialog window */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Path to mount point"),
					     /* 2TRANS: this is message in dialog window;
					     below it there is an input field where user can
					     enter path to ISO9660 or UDF image file */
					     _("Please enter FULL path to mount point (mount directory):"),
					     &(options_window.udf->mount_point),
					     CDW_FS_DIR, R_OK | W_OK, CDW_FS_EXISTING);

	cdw_main_window_wrefresh();

	redrawwin(options_window.cdw_form_p1->subwindow);
	wrefresh(options_window.cdw_form_p1->subwindow);
	redrawwin(options_window.cdw_form_p1->window);
	wrefresh(options_window.cdw_form_p1->window);


	char *fp = options_window.udf->mount_point;
	if (crv == CDW_OK) {
		int rv = set_field_buffer(options_window.cdw_form_p1->fields[f_mount_point_i], 0, fp);
		if (rv != E_OK) {
			cdw_vdm ("ERROR: failed to set field buffer with string = \"%s\"\n", fp);
		} else {
			form_driver(options_window.cdw_form_p1->form, REQ_END_LINE);
		}
		cdw_vdm ("INFO: file picker returns \"%s\"\n", fp);
		return 1;
	} else if (crv == CDW_CANCEL) {
		cdw_vdm ("INFO: pressed escape in file picker, fullpath is \"%s\"\n", fp);
		return 0;
	} else {
		cdw_vdm ("INFO: file picker returns CDW_ERROR, fullpath is \"%s\"\n", fp);
		return 0;
	}
}
