/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * ratop - top() style application for argus data.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#define ARGUS_CURSES	1

#include <stdlib.h>

#include <signal.h>
#include <ctype.h>

#ifdef ARGUS_CURSES
#include <curses.h>
#if !defined(CYGWIN) && !defined(__NetBSD__)
#include <term.h>
#else
#include <termios.h>
#endif
#endif

#include <argus_client.h>
#include <ratop.h>

#include <math.h>

extern int ArgusSrcUserDataLen;
extern int ArgusDstUserDataLen;

struct ArgusServiceRecord ArgusThisSrv;

void RaProcessSrvRecord (struct ArgusServiceRecord *);


#ifdef ARGUS_CURSES
 
WINDOW *RaWindow = NULL;
WINDOW *RaHeaderWindow = NULL;
WINDOW *RaAvailableWindow = NULL;
WINDOW *RaFailedWindow = NULL;
WINDOW *RaRecordWindow = NULL;

#define FAILEDWINSIZE    15
#define HEADERWINSIZE    5
 
int RaRecordWindowNum = 0;
int RaCursesInit      = 0;
int RaServerMode      = 0;
int RaWindowStatus    = 1;
int RaWindowModified  = 1;
int RaWindowImmediate = 1;
int RaWindowLines     = 0;
int RaWindowStartLine = 0;
  
extern struct timeval RaClientTimeout;

struct timeval RaUpdateTime = {0, 0};
struct timeval RaUpdateTimeInterval = {0, 500000};

int RaInitCurses (void);

#include <sys/ioctl.h>

#define MAXFLOWMODELS		31

int RaThisFlowModelIndex = MAXFLOWMODELS - 1;

char *RaSrvStatsFlowModelDesc [MAXFLOWMODELS] = {
"Total Aggregate Traffic",
"Protocol Aggregate Traffic",
"Services Aggregate Traffic",
"Src Class A Aggregate Traffic",
"Src Class A Protocol Aggregate Traffic",
"Src Class A Service Aggregate Traffic",
"Src Class Aggregate Traffic",
"Src Class Protocol Aggregate Traffic",
"Src Class Service Aggregate Traffic",
"Source Aggregate Traffic",
"Source Protocol Aggregate Traffic",
"Source Service Aggregate Traffic",
"Dest Class A Aggregate Traffic",
"Dest Class A Protocol Aggregate Traffic",
"Dest Class A Service Aggregate Traffic",
"Dest Class Aggregate Traffic",
"Dest Class Protocol Aggregate Traffic",
"Dest Class Service Aggregate Traffic",
"Destination Aggregate Traffic",
"Destination Protocol Aggregate Traffic",
"Destination Service Aggregate Traffic",
"Class Matrix Aggregate Traffic",
"Class Matrix Protocol Traffic",
"Class Matrix Service Traffic",
"Source Class Destination Aggregate Traffic",
"Source Class Destination Protocol Aggregate Traffic",
"Source Class Destination Services Aggregate Traffic",
"Flow Aggregate Traffic",
"Microflow Protocol Aggregate Traffic",
"Microflow Services Aggregate Traffic",
"Microflow Traffic",
};

char *RaSrvStatsFlowModelFile [] = {
"RAGATOR_PRESERVE_FIELDS=yes",
"RAGATOR_REPORT_AGGREGATION=yes",
"Flow  100 ip     *               *             *        *        *    200  120   180",
"Flow  101 ip     *               *             *        *        *    201  120   180",
"Flow  102 ip     *               *             *        *        *    202  120   180",
"Flow  103 ip     *               *             *        *        *    203  120   180",
"Flow  104 ip     *               *             *        *        *    204  120   180",
"Flow  105 ip     *               *             *        *        *    205  120   180",
"Flow  106 ip     *               *             *        *        *    206  120   180",
"Flow  107 ip     *               *             *        *        *    207  120   180",
"Flow  108 ip     *               *             *        *        *    208  120   180",
"Flow  109 ip     *               *             *        *        *    209  120   180",
"Flow  110 ip     *               *             *        *        *    210  120   180",
"Flow  111 ip     *               *             *        *        *    211  120   180",
"Flow  112 ip     *               *             *        *        *    212  120   180",
"Flow  113 ip     *               *             *        *        *    213  120   180",
"Flow  114 ip     *               *             *        *        *    214  120   180",
"Flow  115 ip     *               *             *        *        *    215  120   180",
"Flow  116 ip     *               *             *        *        *    216  120   180",
"Flow  117 ip     *               *             *        *        *    217  120   180",
"Flow  118 ip     *               *             *        *        *    218  120   180",
"Flow  119 ip     *               *             *        *        *    219  120   180",
"Flow  120 ip     *               *             *        *        *    220  120   180",
"Flow  121 ip     *               *             *        *        *    221  120   180",
"Flow  122 ip     *               *             *        *        *    222  120   180",
"Flow  123 ip     *               *             *        *        *    223  120   180",
"Flow  124 ip     *               *             *        *        *    224  120   180",
"Flow  125 ip     *               *             *        *        *    225  120   180",
"Flow  126 ip     *               *             *        *        *    226  120   180",
"Flow  127 ip     *               *             *        *        *    227  120   180",
"Flow  128 ip     *               *             *        *        *    228  120   180",
"Flow  129 ip     *               *             *        *        *    229  120   180",
"Flow  130 ip     *               *             *        *        *    230  120   180",
"Model 200 ip  0.0.0.0         0.0.0.0          no       no       no",
"Model 201 ip  0.0.0.0         0.0.0.0          yes      no       no",
"Model 202 ip  0.0.0.0         0.0.0.0          yes      no       yes",
"Model 203 ip  255.0.0.0       0.0.0.0          no       no       no",
"Model 204 ip  255.0.0.0       0.0.0.0          yes      no       no",
"Model 205 ip  255.0.0.0       0.0.0.0          yes      no       yes",
"Model 206 ip  class           0.0.0.0          no       no       no",
"Model 207 ip  class           0.0.0.0          yes      no       no",
"Model 208 ip  class           0.0.0.0          yes      no       yes",
"Model 209 ip  255.255.255.255 0.0.0.0          no       no       no",
"Model 210 ip  255.255.255.255 0.0.0.0          yes      no       no",
"Model 211 ip  255.255.255.255 0.0.0.0          yes      no       yes",
"Model 212 ip  0.0.0.0         255.0.0.0        no       no       no",
"Model 213 ip  0.0.0.0         255.0.0.0        yes      no       no",
"Model 214 ip  0.0.0.0         255.0.0.0        yes      no       yes",
"Model 215 ip  0.0.0.0         class            no       no       no",
"Model 216 ip  0.0.0.0         class            yes      no       no",
"Model 217 ip  0.0.0.0         class            yes      no       yes",
"Model 218 ip  0.0.0.0         255.255.255.255  no       no       no",
"Model 219 ip  0.0.0.0         255.255.255.255  yes      no       no",
"Model 220 ip  0.0.0.0         255.255.255.255  yes      no       yes",
"Model 221 ip  class           class            no       no       no",
"Model 222 ip  class           class            yes      no       no",
"Model 223 ip  class           class            yes      no       yes",
"Model 224 ip  class           255.255.255.255  no       no       no",
"Model 225 ip  class           255.255.255.255  yes      no       no",
"Model 226 ip  class           255.255.255.255  yes      no       yes",
"Model 227 ip  255.255.255.255 255.255.255.255  no       no       no",
"Model 228 ip  255.255.255.255 255.255.255.255  yes      no       no",
"Model 229 ip  255.255.255.255 255.255.255.255  yes      no       yes",
"Model 230 ip  255.255.255.255 255.255.255.255  yes      yes      yes",
NULL,
};

struct RaFlowModelStruct *RaFlowModel = NULL;

int 
RaInitCurses ()
{
   struct termios newtty;

   RaCursesInit++;

   if (tcgetattr(0, &RaOrigTty) == -1)
      ArgusLog(LOG_ERR, "tcgetattr() failed %s\n", strerror(errno));

   newtty = RaOrigTty;
   newtty.c_lflag &= ~ICANON;
   newtty.c_lflag &= ~ECHO;
   newtty.c_cc[VMIN] = 1;
   newtty.c_cc[VTIME] = 0;

   if (tcsetattr(0, TCSAFLUSH, &newtty) == -1)
       ArgusLog(LOG_ERR, "cannot put tty into raw mode\n");

   RaWindow = initscr();

   noraw();
   cbreak();
   nodelay(RaWindow, TRUE);
   noecho();
   nonl();
   intrflush(stdscr, FALSE);
   keypad(stdscr, TRUE);

   clearok(RaWindow, TRUE);
   werase(RaWindow);
   wrefresh(RaWindow);
 
   RaHeaderWindow = newwin (HEADERWINSIZE, COLS, 0, 0);
   RaWindowLines  = LINES - (HEADERWINSIZE + 1);
   RaWindowStartLine = 0;
   Nflag = RaWindowLines - 3;
   RaAvailableWindow = newwin (RaWindowLines, COLS, HEADERWINSIZE, 0);
   RaFailedWindow = RaAvailableWindow;
   idlok (RaHeaderWindow, TRUE);
   idlok (RaAvailableWindow, TRUE);
   notimeout(RaHeaderWindow, TRUE);
   refresh();
   return (1);
}
#endif

void RaTopLoop(void);
extern struct timeval ArgusGlobalTime;
struct timeval RaTopStartTime = {0, 0};

int RaInitialized = 0;


void
ArgusClientInit ()
{
   struct ArgusModeStruct *mode = NULL;

   if (!(RaInitialized)) {
      RaWriteOut = 0;
      RaCumulativeMerge = 1;

      if ((mode = ArgusModeList) != NULL)
         while (mode) {
            if (isdigit((int)*mode->mode)) {
               RaThisFlowModelIndex = atoi (mode->mode);

               if (RaThisFlowModelIndex > (MAXFLOWMODELS - 1))
                  RaThisFlowModelIndex = MAXFLOWMODELS - 1;
            }
            mode = mode->nxt;
         }


      if ((RaFlowModel = RaReadFlowModelFile (RaSrvStatsFlowModelFile)) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit RaReadFlowModelFile returned NULL %s\n", strerror(errno));

      if ((RaAvailableQueue = RaNewQueue()) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit RaNewQueue returned NULL %s\n", strerror(errno));

      if ((RaFailedQueue = RaNewQueue()) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit RaNewQueue returned NULL %s\n", strerror(errno));

      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaHashTable.size = RA_HASHTABLESIZE;
      }

      bzero ((char *)RaSortAlgorithms, sizeof(RaSortAlgorithms));
      RaSortAlgorithms[0] = RaSortPktsCount;
      RaSortAlgorithms[1] = RaSortByteCount;


      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
      (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
      (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
      (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);

      if (Sflag || Cflag) {
         RaClientTimeout.tv_sec  = 0;
         RaClientTimeout.tv_usec = 10000;
      } else {
         RaClientTimeout.tv_sec  = 0;
         RaClientTimeout.tv_usec = 10000;
      }

      RaInitialized++;

      if (nflag > 0)
         hfield = 15;

      if (nflag > 1)
         pfield = 5;

      RaCumulativeMerge = 0;
      RaInitCurses ();

      RaTopLoop();
   }
}

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

int RaTopCompleted = 0;
int RaTasksToDo = 1;

struct ARGUS_INPUT *RaCurrentInput = NULL;
void RaRefreshDisplay(void);
void RaOutputHelpScreen(void);
void RaOutputModifyScreen(void);


void
RaTopLoop()
{
   int retn = 0, fd = -1;

   while (!(RaTopCompleted)) {
      if (RaTasksToDo) {
         if (Sflag || Cflag) {
            mvwprintw (RaHeaderWindow, 4, 1, "RaTopLoop() Processing Remote Server");
            if ((RaCurrentInput = ArgusRemoteHostList) != NULL) {
               if ((RaCurrentInput->fd = ArgusGetServerSocket (RaCurrentInput)) >= 0) {
                  if ((ArgusReadConnection (RaCurrentInput, NULL)) >= 0) {
                     ArgusRemoteFDs[ArgusActiveServers++] = RaCurrentInput;
                     ArgusReadStream();
                  } else
                     mvwprintw (RaHeaderWindow, 4, 1, "%s", strerror(errno));
               }

               RaTasksToDo = 0;
               ArgusDeleteHostList();
            }

         } else {
            if ((RaCurrentInput = ArgusInputFileList) != NULL) {
               mvwprintw (RaHeaderWindow, 4, 1, "RaTopLoop() Processing Input File List");
               while (RaCurrentInput) {
                  if (strcmp (RaCurrentInput->filename, "-")) {
                     if ((RaCurrentInput->fd = open(RaCurrentInput->filename, O_RDONLY)) >= 0) {
                        if (((ArgusReadConnection (RaCurrentInput, RaCurrentInput->filename)) >= 0)) {
                           ArgusRemoteFDs[0] = RaCurrentInput;
                           ArgusReadStream();
                           if (RaCurrentInput)
                              close(RaCurrentInput->fd);
                        }

                     } else {
                        fprintf (stderr, "%s: open '%s': %s\n", ArgusProgramName,
                                                   RaCurrentInput->filename, strerror(errno));
                        _exit(1);
                     }

                  } else {
                     RaCurrentInput->fd = 0;
                     if (((ArgusReadConnection (RaCurrentInput, NULL)) >= 0)) {
                        ArgusRemoteFDs[0] = RaCurrentInput;
                        ArgusReadStream();
                        if (RaCurrentInput)
                           close(RaCurrentInput->fd);
                     }
                  }
                  if (RaCurrentInput)
                     RaCurrentInput = RaCurrentInput->nxt;
               }

            } else {
               mvwprintw (RaHeaderWindow, 4, 1, "RaTopLoop() Idle");
               RaRefreshDisplay();
            }

            ArgusClientTimeout ();
            RaTasksToDo = 0;
            RaWindowModified = RA_MODIFIED;
            RaWindowImmediate = TRUE;
            RaRefreshDisplay();
         }

      } else {
         usleep(50000);
         ArgusClientTimeout ();
      }
   }

   if (fd >= 0) {
      ArgusShutDown (0);
   } else
      retn = 1;

#ifdef ARGUSDEBUG
   ArgusDebug (1, "main () exiting with %d\n", retn);
#endif

   exit (retn);
}


int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   int retn = 0;

#ifdef ARGUS_CURSES
   if (RaCursesInit) {
      mvwprintw (RaWindow, LINES - 1, 0, "\n");
      endwin();
      refresh();
      if ((retn = tcsetattr(0, TCSAFLUSH, &RaOrigTty)) == ERR)
         ArgusLog (LOG_ERR, "tcsetattr error %s\n", strerror(errno));
      printf("\n");
   }

#endif

   switch (sig) {
      case SIGHUP:
      case SIGTERM:
      case SIGINT:
         RaTopCompleted = 1;
         _exit(0);
         break;
   }
}


extern char version[];
extern struct ARGUS_INPUT *ArgusInput;
struct timeval RaProbeUptime = {0, 0};

int RaActiveCons = 0;
int RaClosingCons = 0;
int RaNoResponse = 0;

void RaUpdateWindow (WINDOW *, struct RaQueueStruct *);

char *RaModifyCommandArray[] = {
   "RA_CISCONETFLOW_SOURCE=",
   "RA_NO_OUTPUT=",
   "RA_USER_AUTH=",
   "RA_AUTH_PASS=",
   "RA_OUTPUT_FILE=",
   "RA_EXCEPTION_OUTPUT_FILE=",
   "RA_TIMERANGE=",
   "RA_RUN_TIME=",
   "RA_FIELD_DELIMITER=",
   "RA_TIME_FORMAT=",
   "RA_USEC_PRECISION=",
   "RAGATOR_TIME_SERIES=",
   "RAGATOR_VALIDATE=",
   "RAMON_NUMBER=",
   "RA_DEBUG_LEVEL=",
   "RA_USERDATA_ARG=",
   "RA_USERDATA_ENCODE=",
   "RA_HOST_FIELD_LENGTH=",
   "RA_PORT_FIELD_LENGTH=",
};

char *RaGetCiscoServers(void); 
char *RaGetNoOutputStatus(void); 
char *RaGetUserAuth(void); 
char *RaGetUserPass(void); 
char *RaGetOutputFile(void);
char *RaGetExceptionOutputFile(void); 
char *RaGetTimeRange(void); 
char *RaGetRunTime(void); 
char *RaGetFieldDelimiter(void); 
char *RaGetTimeFormat(void); 
char *RaGetPrecision(void); 
char *RaGetTimeSeries(void); 
char *RaGetValidateStatus(void); 
char *RaGetNumber(void); 
char *RaGetDebugLevel(void); 
char *RaGetUserDataArg(void); 
char *RaGetUserDataEncode(void); 
char *RaGetHostFieldLength(void); 
char *RaGetPortFieldLength(void); 

#define ARGUS_MAX_MODIFY_ELEMENTS 19
strproc RaModifyCommandValueArray[ARGUS_MAX_MODIFY_ELEMENTS] = {
   RaGetCiscoServers,
   RaGetNoOutputStatus,
   RaGetUserAuth,
   RaGetUserPass,
   RaGetOutputFile,
   RaGetExceptionOutputFile,
   RaGetTimeRange,
   RaGetRunTime,
   RaGetFieldDelimiter,
   RaGetTimeFormat,
   RaGetPrecision,
   RaGetTimeSeries,
   RaGetValidateStatus,
   RaGetNumber,
   RaGetDebugLevel,
   RaGetUserDataArg,
   RaGetUserDataEncode,
   RaGetHostFieldLength,
   RaGetPortFieldLength,
};

#define RANEWCOMMAND    1
#define RAGETTINGc      2
#define RAGETTINGf      3
#define RAGETTINGF      4
#define RAGETTINGh      5
#define RAGETTINGm      6
#define RAGETTINGN      7
#define RAGETTINGo      8
#define RAGETTINGs      9
#define RAGETTINGt      10
#define RAREADINGSTR    11
#define RAGETTINGu      12

#define RAGETTINGcSTR           "Connect to: "
#define RAGETTINGfSTR           "Specify filter: "
#define RAGETTINGtSTR           "Specify time range: "
#define RAGETTINGFSTR           "Specify fields: "
#define RAGETTINGhSTR           "Help menu (press any key to continue): "
#define RAGETTINGmSTR           "Use arrow keys to select variable (use d when done)."
#define RAGETTINGNSTR           "Connections to display: "
#define RAGETTINGoSTR           "Read file: "
#define RAGETTINGSSTR           "Switching to Server Mode: "
#define RAGETTINGSORTSTR        "Specify Sort fields: "
#define RAGETTINGUPDATESTR      "Specify Update value: "

unsigned int RaUpdateCount = 0;

int RaInputStatus = RANEWCOMMAND;
char RaCommandInputStr[MAXSTRLEN];
int RaCommandIndex = 0;
int RaCommandLines = 0;
int RaMinCommandLines = 0;
int RaMaxCommandLines = 0;
WINDOW *RaCommandWindow = NULL;
char **RaCommandArray = NULL;
strproc *RaCommandValueArray = NULL;

void
RaOutputModifyScreen ()
{
   int i = 0;
   werase(RaAvailableWindow);

   for (i = RaMinCommandLines; i < (RaMaxCommandLines + 1); i++) {
      mvwprintw (RaAvailableWindow, i, 1, RaCommandArray[i - RaMinCommandLines]);
      if (i == RaMinCommandLines)
         wstandout(RaAvailableWindow);
      wprintw (RaAvailableWindow, "%s", RaCommandValueArray[i - RaMinCommandLines]());
      if (i == RaMinCommandLines)
         wstandend(RaAvailableWindow);
   }
}

void
RaOutputHelpScreen ()
{
   werase(RaAvailableWindow);

   mvwprintw (RaAvailableWindow, 1, 1, "RaTop Version %s\n", version);
   mvwprintw (RaAvailableWindow, 2, 1, "Key Commands: h,q,n,s");
   mvwprintw (RaAvailableWindow, 3, 1, "   h - Print help screen.");
   mvwprintw (RaAvailableWindow,27, 1, "   c - connect to remote Argus");
   mvwprintw (RaAvailableWindow,26, 1, "   d - disconnect from remote Argus");
   mvwprintw (RaAvailableWindow,29, 1, "   f - specify filter expression");
   mvwprintw (RaAvailableWindow,12, 1, "   F - Specify fields to print.");
   mvwprintw (RaAvailableWindow,13, 1, "         +[#]field - add field to optional column # or end of line");
   mvwprintw (RaAvailableWindow,14, 1, "         -field    - remove field from display");
   mvwprintw (RaAvailableWindow,15, 1, "          field    - reset fields and add to display");
   mvwprintw (RaAvailableWindow,16, 1, "             available fields are:");
   mvwprintw (RaAvailableWindow,18, 1, "               startime, lasttime, trans, dur, avgdur");
   mvwprintw (RaAvailableWindow,19, 1, "               saddr, daddr, proto, sport, dport");
   mvwprintw (RaAvailableWindow,20, 1, "               stos, dtos, sttl, dttl, bytes, sbytes");
   mvwprintw (RaAvailableWindow,21, 1, "               dbytes, pkts, spkts, dpkts");
   mvwprintw (RaAvailableWindow,22, 1, "               load, loss, rate, srcid, ind, mac, dir");
   mvwprintw (RaAvailableWindow,23, 1, "               jitter, status, user, tcpext, win");
   mvwprintw (RaAvailableWindow, 5, 1, "   n - Specify the number of entries to print.");
   mvwprintw (RaAvailableWindow,28, 1, "   o - open Argus data file");
   mvwprintw (RaAvailableWindow, 4, 1, "   q - Quit the program.");
   mvwprintw (RaAvailableWindow, 6, 1, "   s - Specify sort fields.");
   mvwprintw (RaAvailableWindow, 7, 1, "         available fields are:");
   mvwprintw (RaAvailableWindow, 8, 1, "            time, startime, lasttime, trans, dur, avgdur");
   mvwprintw (RaAvailableWindow, 9, 1, "            saddr, daddr, proto, sport, dport");
   mvwprintw (RaAvailableWindow,10, 1, "            stos, dtos, sttl, dttl, bytes, sbytes, dbytes");
   mvwprintw (RaAvailableWindow,11, 1, "            pkts, spkts, dpkts, load, loss, rate, transref, seq");
   mvwprintw (RaAvailableWindow,24, 1, "   S - Toggle server display mode. When on displays entries that have multiple sources");
   mvwprintw (RaAvailableWindow,25, 1, "   t - Specify time range. same as -t command line option. ");
   mvwprintw (RaAvailableWindow,25, 1, "   u - Specify the window update timer, in seconds (default 0.5s)");
   wnoutrefresh(RaWindow);
   doupdate();
}

void
RaRefreshDisplay ()
{
   struct timeval tvp, *last, *start;
   int dsec, dusec, srcaddr;
   char timebuf[32], tbuf[MAXSTRLEN];
   struct tm *tm;

   if (RaTopStartTime.tv_sec == 0)
      RaTopStartTime = ArgusGlobalTime;

   tvp = ArgusRealTime;

   if (RaUpdateTime.tv_sec == 0)
      RaUpdateTime = tvp;
   
   if ((RaWindowImmediate) ||
      ((RaUpdateTime.tv_sec < tvp.tv_sec) ||
      ((RaUpdateTime.tv_sec == tvp.tv_sec) &&
       (RaUpdateTime.tv_usec <= tvp.tv_usec)))) {

      RaUpdateWindow(RaAvailableWindow, RaAvailableQueue);

      RaUpdateTime = tvp;

      RaUpdateTime.tv_sec  += RaUpdateTimeInterval.tv_sec;
      RaUpdateTime.tv_usec += RaUpdateTimeInterval.tv_usec;

      if (RaUpdateTime.tv_usec >= 1000000) {
         RaUpdateTime.tv_sec  += 1;
         RaUpdateTime.tv_usec -= 1000000;
      }

      RaWindowImmediate = 0;
   }

   if (ArgusInput && (RaProbeUptime.tv_sec == 0)) {
      start = &ArgusInput->ArgusInitCon.argus_mar.startime;
      last  = &ArgusInput->ArgusInitCon.argus_mar.now;

      tvp.tv_sec  = ntohl(last->tv_sec)  - ntohl(start->tv_sec);
      tvp.tv_usec = ntohl(last->tv_usec) - ntohl(start->tv_usec);

      if (tvp.tv_usec < 0) {
         tvp.tv_sec--;
         tvp.tv_usec += 1000000;
      }

      if (tvp.tv_usec >= 1000000) {
         tvp.tv_sec++;
         tvp.tv_usec -= 1000000;
      }

      RaProbeUptime = tvp;
   }

   dsec  = RaProbeUptime.tv_sec  + (tvp.tv_sec  - RaTopStartTime.tv_sec);
   dusec = RaProbeUptime.tv_usec + (tvp.tv_usec - RaTopStartTime.tv_usec);

   if (dusec < 0) {
      dsec--;
      dusec += 1000000;
   }

   if (dusec >= 1000000) {
      dsec++;
      dusec -= 1000000;
   }

   tm = localtime((time_t *) &tvp.tv_sec);
   strftime ((char *) timebuf, 32, "%d/%m/%Y %I:%M%P", tm);

#ifdef ARGUS_CURSES
   mvwprintw (RaWindow, LINES - 1, 0, "\n");

   if (strlen(ArgusProgramArgs) > 60)
      ArgusProgramArgs[60] = '\0';

   mvwaddnstr (RaHeaderWindow, 0, 1, ArgusProgramArgs, COLS - 1);

   if (ArgusInput != NULL) {
      srcaddr = ntohl(ArgusInput->ArgusInitCon.argus_mar.argusid);
      if (RaTasksToDo) {
         bzero (tbuf, MAXSTRLEN);
         sprintf (tbuf, "Source %s  Version %d.%d  %s up %d days, %02d:%02d:%02d  Far %d secs Mar %d secs",
                          ipaddr_string(&srcaddr),
                          ArgusInput->ArgusInitCon.argus_mar.major_version,
                          ArgusInput->ArgusInitCon.argus_mar.minor_version,
                          timebuf, dsec / 86400, (dsec % 86400)/3600, ((dsec % 86400) % 3600)/60,
                          (((dsec % 86400) % 3600) % 60),
                          ntohs(ArgusInput->ArgusInitCon.argus_mar.reportInterval),
                          ntohs(ArgusInput->ArgusInitCon.argus_mar.argusMrInterval));

         mvwaddnstr (RaHeaderWindow, 1, 1, tbuf, COLS - 1);
         wclrtoeol(RaHeaderWindow);

         bzero (tbuf, MAXSTRLEN);
         sprintf (tbuf, "%6d Flows   %6d Active   %6d Closing  %12d Total Records", 
               RaAvailableQueue->count + RaFailedQueue->count, RaActiveCons, RaClosingCons, totalrecords);

         mvwaddnstr (RaHeaderWindow, 2, 1, tbuf, COLS - 1);
         wclrtoeol(RaHeaderWindow);
      }

   } else {
         mvwaddnstr (RaHeaderWindow, 1, 1, "Source Not Connected", COLS - 3);
   }

   wnoutrefresh(RaHeaderWindow);
#endif

#ifdef ARGUS_CURSES
   if (RaWindowStatus && (RaLabel != NULL)) {
      if (RaServerMode)
         mvwaddnstr (RaAvailableWindow, 0, 1, "Services", COLS - 2);
      else 
         mvwaddnstr (RaAvailableWindow, 0, 1, "Flows", COLS - 2);

      if (strlen(RaLabel) > (COLS - 6))
         RaLabel[COLS - 6] = '\0';

      sprintf (tbuf, "Rank %s", RaLabel);
      mvwaddnstr (RaAvailableWindow, 1, 1, tbuf, COLS - 2);
      wnoutrefresh(RaAvailableWindow);
   }

   wnoutrefresh(RaWindow);
   doupdate();
#endif

}


extern void ArgusCloseInput(struct ARGUS_INPUT *);

#include <pcap.h>
#include <glob.h>

extern unsigned char *ArgusRemoteFilter;

int RaScreenLines = 0;
int RaScreenColumns = 0;


void
ArgusClientTimeout ()
{
#ifdef ARGUS_CURSES
   struct ArgusRecordStore *store = NULL;
   struct ARGUS_INPUT *addr, *tmp;
   struct timeval tv;
   fd_set in;
   int c;
   unsigned char *ptr = NULL;
   int i;

   tv.tv_sec  = 0;
   tv.tv_usec = 0;
   FD_ZERO(&in);
   FD_SET(0, &in);

   if (RaScreenLines == 0)
      RaScreenLines = LINES;
   if (RaScreenColumns == 0)
      RaScreenColumns = COLS;

   if ((RaScreenLines != LINES) || (RaScreenColumns != COLS)) {
      RaScreenLines = LINES;
      RaScreenColumns = COLS;
      RaWindowLines = LINES - (HEADERWINSIZE + 1);
      Nflag = RaWindowLines - 3;

#if !defined(HAVE_SOLARIS)
      wresize(RaWindow, LINES, COLS);
      wresize(RaHeaderWindow, HEADERWINSIZE, COLS);
      wresize(RaAvailableWindow, RaWindowLines, COLS);
#endif

      RaLabel = NULL;

      RaWindowModified = RA_MODIFIED;
      RaWindowImmediate = TRUE;
      RaRefreshDisplay();
   }

   switch (RaInputStatus) {
      case RANEWCOMMAND: {
         if ((c = wgetch(RaWindow)) != ERR) {
            mvwaddnstr (RaHeaderWindow, 4, 1, "  ", COLS - 2);
            wclrtoeol(RaHeaderWindow);
            
            if (RaCommandIndex != 0) {
               RaCommandIndex = 0;
               bzero (RaCommandInputStr, MAXSTRLEN);
            }
            switch (c) {
               case 0x1B:
                  RaWindowModified = RA_MODIFIED;
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;

               case 'k':
               case 0x19:
               case KEY_UP:
                  if (RaWindowStartLine > 0) {
                     RaWindowStartLine--;
                     RaWindowModified = RA_MODIFIED;
                     RaWindowImmediate = TRUE;
                     wscrl(RaAvailableWindow, -1);
                     RaRefreshDisplay();
                  }
                  break;

               case 'j':
               case 0x05:
               case KEY_DOWN:
                  if ((RaAvailableQueue->count - RaWindowStartLine) > Nflag) {
                     RaWindowStartLine++;
                     RaWindowModified = RA_MODIFIED;
                     wscrl(RaAvailableWindow, 1);
                     RaWindowImmediate = TRUE;
                     RaRefreshDisplay();
                  }
                  break;

               case 'g':
               case KEY_HOME:
                  if (RaWindowStartLine != 0) {
                     werase(RaWindow);
                     RaWindowModified = RA_MODIFIED;
                     RaWindowStartLine = 0;
                     RaWindowImmediate = TRUE;
                     RaRefreshDisplay();
                  }
                  break;

               case 'G':
               case KEY_END:
                  if (RaWindowStartLine != (RaAvailableQueue->count - Nflag)) {
                     RaWindowStartLine = RaAvailableQueue->count - Nflag;
                     if (RaWindowStartLine < 0)
                        RaWindowStartLine = 0;

                     werase(RaWindow);
                     RaWindowModified = RA_MODIFIED;
                     RaWindowImmediate = TRUE;
                     RaRefreshDisplay();
                  }
                  break;

               case 0x06:
               case ' ':
               case KEY_NPAGE:
                  if ((RaAvailableQueue->count - RaWindowStartLine) > Nflag) {
                     RaWindowStartLine += Nflag;
                     wscrl(RaWindow, Nflag);
                     RaWindowModified = RA_MODIFIED;
                     RaWindowImmediate = TRUE;
                     RaRefreshDisplay();
                  }
                  break;

               case 0x02:
               case KEY_PPAGE:
                  if (RaWindowStartLine > 0) {
                     wscrl(RaWindow, (Nflag > RaWindowStartLine) ? -RaWindowStartLine : -Nflag);
                     RaWindowStartLine -= Nflag;
                     if (RaWindowStartLine < 0)
                        RaWindowStartLine = 0;
                     RaWindowModified = RA_MODIFIED;
                     RaWindowImmediate = TRUE;
                     RaRefreshDisplay();
                  }
                  break;

               case 'n':
                  RaInputStatus = RAGETTINGN;
                  mvwaddnstr (RaHeaderWindow, 4, 1, RAGETTINGNSTR, COLS - 2);
                  break;

               case 'c':
                  if ((ArgusRemoteHostList) == NULL) {
                     RaInputStatus = RAGETTINGc;
                     mvwaddnstr (RaHeaderWindow, 4, 1, RAGETTINGcSTR, COLS - 2);
                  } else
                     mvwaddnstr (RaHeaderWindow, 4, 1, "Connected", COLS - 2);

                  RaWindowModified = RA_MODIFIED;
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;

               case 'd':
                  if (RaCurrentInput != NULL) {
                     for (i = 0; i < ARGUS_MAX_REMOTE_CONN; i++) {
                        if (ArgusRemoteFDs[i] != NULL) {
                           ArgusCloseInput(ArgusRemoteFDs[i]);
                           ArgusRemoteFDs[i] = NULL;
                        }
                     }
                    
                     RaCurrentInput = NULL;
                     Sflag = 0;
                     Cflag = 0;
                  
                     if ((ArgusRemoteHostList) != NULL) {
                        ArgusDeleteHostList();
                        ArgusActiveServers = 0;
                        ArgusRemoteHostList = NULL;
                     }

                     if ((addr = ArgusInputFileList) != NULL) {
                        while (addr) {
                           tmp = addr->nxt;
                           ArgusFree(addr);
                           addr = tmp;
                        }
                        ArgusInputFileList = NULL;
                     }

                     RaWindowModified = RA_MODIFIED;
                     werase(RaHeaderWindow);
                     while ((store = (struct ArgusRecordStore *) RaPopQueue(RaAvailableQueue)))
                        RaTimeoutArgusStore(store);

                     while ((store = (struct ArgusRecordStore *) RaPopQueue(RaFailedQueue)))
                        RaTimeoutArgusStore(store);

                     mvwaddnstr (RaHeaderWindow, 4, 1, "Disonnecting", COLS - 2);
                     RaWindowModified = RA_MODIFIED;
                     RaWindowImmediate = TRUE;
                     RaRefreshDisplay();

                  } else
                     mvwaddnstr (RaHeaderWindow, 4, 1, "Not Connected", COLS - 2);

                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;

               case 'f': {
                  char tbuf[MAXSTRLEN];

                  bzero (tbuf, MAXSTRLEN);
                  RaInputStatus = RAGETTINGf;
                  if (*RaInputFilter != NULL) {
                     sprintf (tbuf, "%s %s", RAGETTINGfSTR, *RaInputFilter);
                     mvwaddnstr (RaHeaderWindow, 4, 1, tbuf, COLS - 2);
                  } else {
                     sprintf (tbuf, "%s %s", RAGETTINGfSTR, "(null)");
                     mvwaddnstr (RaHeaderWindow, 4, 1, tbuf, COLS - 2);
                  }

                  break;
               }

               case 't':
                  RaInputStatus = RAGETTINGt;
                  if (timearg != NULL) {
                     char tbuf[MAXSTRLEN];
                     bzero (tbuf, MAXSTRLEN);
                     sprintf (tbuf, "%s %s", RAGETTINGtSTR, timearg);
                     mvwaddnstr (RaHeaderWindow, 4, 1, tbuf, COLS - 2);
                  } else
                     mvwaddnstr (RaHeaderWindow, 4, 1, RAGETTINGtSTR, COLS - 2);

                  break;

               case 'o':
                  if (RaCurrentInput == NULL) {
                     RaInputStatus = RAGETTINGo;
                     mvwaddnstr (RaHeaderWindow, 4, 1, RAGETTINGoSTR, COLS - 2);
                  } else
                     mvwaddnstr (RaHeaderWindow, 4, 1, "Connected", COLS - 2);
                  break;

               case '+': {
                  if (++RaThisFlowModelIndex > (MAXFLOWMODELS - 1))
                     RaThisFlowModelIndex = 0;

                  while ((store = (struct ArgusRecordStore *) RaPopQueue(RaAvailableQueue)))
                     RaTimeoutArgusStore(store);

                  while ((store = (struct ArgusRecordStore *) RaPopQueue(RaFailedQueue)))
                     RaTimeoutArgusStore(store);

                  werase(RaAvailableWindow);
                  mvwaddnstr (RaHeaderWindow, 4, 1, RaSrvStatsFlowModelDesc[RaThisFlowModelIndex], COLS - 2);
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;
               }

               case '-': {
                  struct ArgusRecordStore *store = NULL;

                  if (--RaThisFlowModelIndex < 0)
                     RaThisFlowModelIndex = MAXFLOWMODELS - 1;

                  while ((store = (struct ArgusRecordStore *) RaPopQueue(RaAvailableQueue)))
                     RaTimeoutArgusStore(store);

                  while ((store = (struct ArgusRecordStore *) RaPopQueue(RaFailedQueue)))
                     RaTimeoutArgusStore(store);

                  werase(RaAvailableWindow);
                  mvwaddnstr (RaHeaderWindow, 4, 1, RaSrvStatsFlowModelDesc[RaThisFlowModelIndex], COLS - 2);
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;
               }

               case 'S': {
                  char tbuf[MAXSTRLEN];
                  bzero (tbuf, MAXSTRLEN);
                  if ((RaServerMode = (RaServerMode > 0) ? 0 : 1) > 0)
                     ptr = (unsigned char *) "on";
                  else
                     ptr = (unsigned char *) "off";

                  sprintf (tbuf, "%s %s", RAGETTINGSSTR, ptr);
                  mvwaddnstr (RaHeaderWindow, 4, 1, tbuf, COLS - 2);
                  break;
               }

               case 's':
                  RaInputStatus = RAGETTINGs;
                  mvwaddnstr (RaHeaderWindow, 4, 1, RAGETTINGSORTSTR, COLS - 2);
                  break;

               case 'u':
                  RaInputStatus = RAGETTINGu;
                  mvwaddnstr (RaHeaderWindow, 4, 1, RAGETTINGUPDATESTR, COLS - 2);
                  break;

               case 'F':
                  RaInputStatus = RAGETTINGF;
                  mvwaddnstr (RaHeaderWindow, 4, 1, RAGETTINGFSTR, COLS - 2);
                  break;

               case 'q':
                  RaParseComplete(SIGINT);
                  break;

               case 'h':
                  RaInputStatus = RAGETTINGh;
                  mvwaddnstr (RaHeaderWindow, 4, 1, RAGETTINGhSTR, COLS - 2);
                  RaWindowStatus = 0;
                  werase(RaAvailableWindow);
                  RaOutputHelpScreen();
                  RaWindowModified = RA_MODIFIED;
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;

               case 'm':
                  RaInputStatus = RAGETTINGm;
                  RaMinCommandLines = 2;
                  RaMaxCommandLines = 20;
                  RaCommandArray = RaModifyCommandArray;
                  RaCommandValueArray = RaModifyCommandValueArray;
                  RaCommandLines = RaMinCommandLines;
                  mvwaddnstr (RaHeaderWindow, 4, 1, RAGETTINGmSTR, COLS - 2);
                  RaWindowStatus = 0;
                  werase(RaAvailableWindow);
                  RaOutputModifyScreen();
                  RaWindowModified = RA_MODIFIED;
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;

               default:
                  break;
            }
         }
         break;
      }

      case RAGETTINGc: {
         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {
               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  RaCommandInputStr[RaCommandIndex] = '\0';
                  if (!(ArgusAddHostList (RaCommandInputStr, ARGUS_DATA_SOURCE))) {
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s host not found", RAGETTINGcSTR, RaCommandInputStr);
                  } else {
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGcSTR, RaCommandInputStr);
                     RaTasksToDo = 1;
                     totalrecords = 0;
                     Sflag = 1;
                     RaClientTimeout.tv_sec  = 0;
                     RaClientTimeout.tv_usec = 10000;

                     while ((store = (struct ArgusRecordStore *) RaPopQueue(RaAvailableQueue)))
                        RaTimeoutArgusStore(store);

                     while ((store = (struct ArgusRecordStore *) RaPopQueue(RaFailedQueue)))
                        RaTimeoutArgusStore(store);
                  }

                  RaInputStatus = RANEWCOMMAND;

                  werase(RaAvailableWindow);
                  RaWindowModified = RA_MODIFIED;
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;

               } else {
                  if (iscntrl((int)RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGcSTR, RaCommandInputStr);

                     } else {
                        RaInputStatus = RANEWCOMMAND;
                        mvwaddnstr (RaHeaderWindow, 4, 1, " ", COLS - 2);
                        wclrtoeol(RaHeaderWindow);
                        RaWindowImmediate = TRUE;
                        RaRefreshDisplay();
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGcSTR, RaCommandInputStr);
                  }
               }
            } else
               break;
         }
         break;
      }

      case RAGETTINGf: {
         struct bpf_program localfilter;
         char *str = NULL;

         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {
               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  RaInputStatus = RANEWCOMMAND;
                  werase(RaAvailableWindow);
                  RaCommandInputStr[RaCommandIndex] = '\0';

                  if (*RaInputFilter !=  NULL)
                     free(*RaInputFilter);

                  str = strdup(RaCommandInputStr);

                  bzero ((char *) &localfilter, sizeof (localfilter));

                  if (ArgusFilterCompile (&localfilter, str, 1, ArgusNetMask) < 0) {
                     mvwprintw (RaHeaderWindow, 4, 1, "%-*s syntax error", strlen(RAGETTINGfSTR) + strlen(RaCommandInputStr), str);
                  } else {
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s %s", RAGETTINGfSTR, RaCommandInputStr, "Filter accepted");

                     *RaInputFilter = strdup(str);

                     setArgusRemoteFilter((unsigned char *) str);
                     if (ArgusFilterCode.bf_insns != NULL)
                        free (ArgusFilterCode.bf_insns);
                 
                     bcopy((char *)&localfilter, (char *)&ArgusFilterCode, sizeof(localfilter));
                     while ((store = (struct ArgusRecordStore *) RaPopQueue(RaAvailableQueue)))
                        RaTimeoutArgusStore(store);

                     while ((store = (struct ArgusRecordStore *) RaPopQueue(RaFailedQueue)))
                        RaTimeoutArgusStore(store);
                  }
                  RaWindowModified = RA_MODIFIED;
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;

               } else {
                  if (iscntrl((int)RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGfSTR, RaCommandInputStr);

                     } else {
                        RaInputStatus = RANEWCOMMAND;
                        mvwaddnstr (RaHeaderWindow, 4, 1, " ", COLS - 2);
                        wclrtoeol(RaHeaderWindow);
                        RaWindowImmediate = TRUE;
                        RaRefreshDisplay();
                        break;
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGfSTR, RaCommandInputStr);
                  }
               }
            } else
               break;
         }
      }

      case RAGETTINGt: {
         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {
               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  RaInputStatus = RANEWCOMMAND;
                  werase(RaAvailableWindow);
                  RaTmStruct = localtime((time_t *) &ArgusGlobalTime.tv_sec);
                  RaCommandInputStr[strlen(RaCommandInputStr) - 1] = '\0';
                  if (check_time_format (RaTmStruct, RaCommandInputStr)) {
                     mvwprintw (RaHeaderWindow, 4, 1, "%-*s", COLS, "Time format error");
                  } else {
                     mvwprintw (RaHeaderWindow, 4, 1, "%-*s", COLS, "Time range accepted");
                     tflag++;
                     if (timearg != NULL)
                        free(timearg);
                     timearg = strdup(RaCommandInputStr);
                  }

                  RaWindowModified = RA_MODIFIED;
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;

               } else {
                  if (iscntrl((int)RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGtSTR, RaCommandInputStr);

                     } else {
                        RaInputStatus = RANEWCOMMAND;
                        mvwaddnstr (RaHeaderWindow, 4, 1, " ", COLS - 2);
                        wclrtoeol(RaHeaderWindow);
                        RaWindowImmediate = TRUE;
                        RaRefreshDisplay();
                        break;
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGtSTR, RaCommandInputStr);
                  }
               }
            } else
               break;
         }
      }

      case RAGETTINGo: {
         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {

               glob_t globbuf;

               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  RaCommandInputStr[RaCommandIndex] = '\0';

                  glob(RaCommandInputStr, 0, NULL, &globbuf);
                  ArgusDeleteFileList();
                  if (globbuf.gl_pathc > 0) {
                     for (i = 0; i < globbuf.gl_pathc; i++) {
                        if (!(ArgusAddFileList (globbuf.gl_pathv[i], (Cflag ? ARGUS_CISCO_DATA_SOURCE : ARGUS_DATA_SOURCE), -1, -1))) {
                           mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGoSTR, RaCommandInputStr, strerror(errno));
                        } else {
                           werase(RaHeaderWindow);
                           mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGoSTR, RaCommandInputStr);
                           RaTasksToDo = 1;
                           totalrecords = 0;
                           RaClientTimeout.tv_sec  = 0;
                           RaClientTimeout.tv_usec = 10000;
                           if (ArgusProgramArgs != NULL)
                              ArgusFree(ArgusProgramArgs);
                           if ((ArgusProgramArgs = ArgusCalloc (1, MAXSTRLEN)) != NULL) {
                              sprintf (ArgusProgramArgs, "%s -r %s", ArgusProgramName, RaCommandInputStr);
                              if (tflag) {
                                 sprintf  (&ArgusProgramArgs[strlen(ArgusProgramArgs)], " -t %s", timearg);
                              }

                              if (*RaInputFilter != NULL) {
                                 sprintf  (&ArgusProgramArgs[strlen(ArgusProgramArgs)], " - %s", *RaInputFilter);
                              }
                           }

                           while ((store = (struct ArgusRecordStore *) RaPopQueue(RaAvailableQueue)))
                              RaTimeoutArgusStore(store);

                           while ((store = (struct ArgusRecordStore *) RaPopQueue(RaFailedQueue)))
                              RaTimeoutArgusStore(store);
                        }
                     }
                  } else
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGoSTR, RaCommandInputStr, " no match");

                  RaInputStatus = RANEWCOMMAND;

                  werase(RaAvailableWindow);

                  RaWindowModified = RA_MODIFIED;
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;

               } else {
                  if (iscntrl((int)RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGoSTR, RaCommandInputStr);

                     } else {
                        RaInputStatus = RANEWCOMMAND;
                        mvwaddnstr (RaHeaderWindow, 4, 1, " ", COLS - 2);
                        wclrtoeol(RaHeaderWindow);
                        RaWindowImmediate = TRUE;
                        RaRefreshDisplay();
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGoSTR, RaCommandInputStr);
                  }
               }
            } else
               break;
         }
         break;
      }


      case RAGETTINGF: {
         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {
               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  char strbuf[1024], *ptr = strbuf, *tok;
                  int ind = 0;

                  RaInputStatus = RANEWCOMMAND;
                  werase(RaAvailableWindow);
                  RaCommandInputStr[RaCommandIndex] = '\0';
                  strcpy (ptr, RaCommandInputStr);
                  bzero ((char *)RaSOptionStrings, sizeof(RaSOptionStrings));
                 
                  while ((tok = strtok(ptr, " ")) != NULL) {
                     RaSOptionStrings[ind++] = tok;
                     ptr = NULL;
                  }

                  RaLabel = NULL;
                  RaProcessSOptions();
                  werase(RaAvailableWindow);
                  RaWindowModified = RA_MODIFIED;
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;

               } else {
                  if (iscntrl((int)RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGFSTR, RaCommandInputStr);

                     } else {
                        RaInputStatus = RANEWCOMMAND;
                        mvwaddnstr (RaHeaderWindow, 4, 1, " ", COLS - 2);
                        wclrtoeol(RaHeaderWindow);
                        RaWindowImmediate = TRUE;
                        RaRefreshDisplay();
                        break;
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGFSTR, RaCommandInputStr);
                  }
               }
            } else
               break;
         }
      }

      case RAGETTINGh: {
         if (select(1, &in, 0, 0, &tv) > 0 && read(0, &c, 1) == 1) {
            RaWindowStatus = 1;
            RaInputStatus = RANEWCOMMAND;
            mvwaddnstr (RaHeaderWindow, 4, 1, " ", COLS - 2);
            wclrtoeol(RaHeaderWindow);
            werase(RaAvailableWindow);
            RaWindowModified = RA_MODIFIED;
            RaWindowImmediate = TRUE;
            RaRefreshDisplay();
         }

         break;
      }

      case RAGETTINGm: {
         if ((c = wgetch(RaWindow)) != ERR) {
            switch (c) {
               case KEY_UP:
               case 'k':
                   mvwprintw (RaAvailableWindow, RaCommandLines, 1, RaCommandArray[RaCommandLines - RaMinCommandLines]);
                   wprintw (RaAvailableWindow, "%s", RaCommandValueArray[RaCommandLines - RaMinCommandLines]());
                   RaCommandLines = (RaCommandLines > RaMinCommandLines) ? RaCommandLines - 1: RaMinCommandLines;
                   mvwprintw (RaAvailableWindow, RaCommandLines, 1, RaCommandArray[RaCommandLines - RaMinCommandLines]);
                   wstandout(RaAvailableWindow);
                   wprintw (RaAvailableWindow, "%s", RaCommandValueArray[RaCommandLines - RaMinCommandLines]());
                   wstandend(RaAvailableWindow);
                   break;

               case KEY_DOWN:
               case 'j':
                   mvwprintw (RaAvailableWindow, RaCommandLines, 1, RaCommandArray[RaCommandLines - RaMinCommandLines]);
                   wprintw (RaAvailableWindow, "%s", RaCommandValueArray[RaCommandLines - RaMinCommandLines]());
                   RaCommandLines = (RaCommandLines < RaMaxCommandLines) ? RaCommandLines + 1: RaMaxCommandLines;
                   mvwprintw (RaAvailableWindow, RaCommandLines, 1, RaCommandArray[RaCommandLines - RaMinCommandLines]);
                   wstandout(RaAvailableWindow);
                   wprintw (RaAvailableWindow, "%s", RaCommandValueArray[RaCommandLines - RaMinCommandLines]());
                   wstandend(RaAvailableWindow);
                   break;

               case 'D':
               case 'd':
                  RaWindowStatus = 1;
                  RaInputStatus = RANEWCOMMAND;
                  mvwaddnstr (RaHeaderWindow, 4, 1, " ", COLS - 2);
                  wclrtoeol(RaHeaderWindow);
                  werase(RaAvailableWindow);
                  break;

               default:
                  break;
            }
         }

         RaWindowModified = RA_MODIFIED;
         RaWindowImmediate = TRUE;
         RaRefreshDisplay();
      }
      break;

      case RAGETTINGu: {
         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {
               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  char strbuf[1024], *endptr = NULL;
                  double value = 0.0, ivalue, fvalue;

                  RaInputStatus = RANEWCOMMAND;
                  RaCommandInputStr[RaCommandIndex] = '\0';

                  value = strtod(RaCommandInputStr, &endptr);

                  if (RaCommandInputStr != endptr) {
                     fvalue = modf(value, &ivalue);

                     RaUpdateTimeInterval.tv_sec  = (int) ivalue;
                     RaUpdateTimeInterval.tv_usec = (int) (fvalue * 1000000.0);

                     bzero (strbuf, 1024);
                     sprintf (strbuf, "%s %6.6f %s", RAGETTINGUPDATESTR, value, " accepted");
                     mvwaddnstr (RaHeaderWindow, 4, 1, strbuf, COLS - 2);

                     RaUpdateTime = ArgusRealTime;
                     RaWindowModified = RA_MODIFIED;
                     RaWindowImmediate = TRUE;
                     RaRefreshDisplay();

                  } else {
                     sprintf (strbuf, "%s %s %s", RAGETTINGUPDATESTR, RaCommandInputStr, " syntax error");
                     mvwaddnstr (RaHeaderWindow, 4, 1, strbuf, COLS - 2);
                     RaRefreshDisplay();
                  }

                  break;

               } else {
                  if (iscntrl((int)RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGUPDATESTR, RaCommandInputStr);

                     } else {
                        RaInputStatus = RANEWCOMMAND;
                        mvwaddnstr (RaHeaderWindow, 4, 1, " ", COLS - 2);
                        wclrtoeol(RaHeaderWindow);
                        RaWindowImmediate = TRUE;
                        RaRefreshDisplay();
                        break;
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGUPDATESTR, RaCommandInputStr);
                  }
               }

            } else
               break;
         }
      }
      break;

      case RAGETTINGs: {
         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {
               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  char strbuf[1024], *ptr = strbuf, *tok;
                  int x, ind = 0;

                  RaInputStatus = RANEWCOMMAND;
                  RaCommandInputStr[RaCommandIndex] = '\0';
                  strcpy (ptr, RaCommandInputStr);
                  bzero ((char *)RaSortAlgorithms, sizeof(RaSortAlgorithms));

                  while ((tok = strtok(ptr, " ")) != NULL) {
                     for (x = 0; x < MAX_SORT_ALG_TYPES; x++) {
                        if (!strncmp (RaSortKeyWords[x], tok, strlen(RaSortKeyWords[x]))) {
                           RaSortAlgorithms[ind++] = RaSortAlgorithmTable[x];
                           break;
                        }
                     }
                     ptr = NULL;
                  }

                  RaWindowModified = RA_MODIFIED;
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();

                  break;

               } else {
                  if (iscntrl((int)RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGSORTSTR, RaCommandInputStr);

                     } else {
                        RaInputStatus = RANEWCOMMAND;
                        mvwaddnstr (RaHeaderWindow, 4, 1, " ", COLS - 2);
                        wclrtoeol(RaHeaderWindow);
                        RaWindowImmediate = TRUE;
                        RaRefreshDisplay();
                        break;
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGSORTSTR, RaCommandInputStr);
                  }
               }
            } else
               break;
         }
      }
      break;

      case RAGETTINGN: {
         while (select(1, &in, 0, 0, &tv) > 0) {
            if (read (0, &RaCommandInputStr[RaCommandIndex], 1)) {
               if ((RaCommandInputStr[RaCommandIndex] == '\n') || (RaCommandInputStr[RaCommandIndex] == '\r')) {
                  Nflag = ((atoi(RaCommandInputStr) < (LINES - (HEADERWINSIZE + 1)) - 3) ?
                            atoi(RaCommandInputStr) : (LINES - (HEADERWINSIZE + 1)) - 3);
                  RaInputStatus = RANEWCOMMAND;
                  werase(RaAvailableWindow);
                  RaWindowModified = RA_MODIFIED;
                  RaWindowImmediate = TRUE;
                  RaRefreshDisplay();
                  break;

               } else {
                  if (iscntrl((int)RaCommandInputStr[RaCommandIndex])) {
                     if ((RaCommandInputStr[RaCommandIndex] == '\b') ||
                         (RaCommandInputStr[RaCommandIndex] == 0x7F)) {
                        RaCommandInputStr[RaCommandIndex--] = '\0';
                        RaCommandInputStr[RaCommandIndex] = ' ';
                        if (RaCommandIndex < 0) RaCommandIndex = 0;
                        mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGNSTR, RaCommandInputStr);

                     } else {
                        RaInputStatus = RANEWCOMMAND;
                        mvwaddnstr (RaHeaderWindow, 4, 1, " ", COLS - 2);
                        wclrtoeol(RaHeaderWindow);
                        RaWindowImmediate = TRUE;
                        RaRefreshDisplay();
                        break;
                     }
                  } else {
                     RaCommandIndex++;
                     mvwprintw (RaHeaderWindow, 4, 1, "%s %s", RAGETTINGNSTR, RaCommandInputStr);
                  }
               }
            } else
               break;
         }
      }
      break;
   }

   wclrtoeol(RaHeaderWindow);
#endif

   RaUpdateCount++;
   RaProcessQueue (RaAvailableQueue, ARGUS_STATUS);
   RaProcessQueue (RaFailedQueue, ARGUS_STATUS);
   
#ifdef ARGUS_CURSES
   if (RaCursesInit)
#endif
      RaRefreshDisplay();

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));

   if (ArgusHourlyUpdate++ == 3600) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));
      ArgusHourlyUpdate = 1;
   }
#endif
}



void
parse_arg (argc, argv)
int argc;
char**argv;
{}

void
usage ()
{

   fprintf (stderr, "RaTop Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusProgramName);

   fprintf (stderr, "options: -a             print record summaries on termination.\n");
   fprintf (stderr, "         -A             print application bytes.\n");
   fprintf (stderr, "         -b             dump packet-matching code.\n");
   fprintf (stderr, "         -c             print packet and byte counts.\n");
   fprintf (stderr, "         -C             treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "         -d <bytes>     print number of <bytes> from user data capture buffer.\n");
   fprintf (stderr, "               format:  num | s<num> | d<num> | s<num>:d<num>\n");
#if defined (ARGUSDEBUG)
   fprintf (stderr, "         -D <level>     specify debug level\n");
#endif
   fprintf (stderr, "         -e <encode>    convert user data using <encode> method.\n");
   fprintf (stderr, "                        Supported types are <Ascii> and <Encode64>.\n");
   fprintf (stderr, "         -E <file>      write records that are rejected by the filter into <file>\n");
   fprintf (stderr, "         -F <conffile>  read configuration from <conffile>.\n");
   fprintf (stderr, "         -g             print record time duration.\n");
   fprintf (stderr, "         -G             print both start and last time values.\n");
   fprintf (stderr, "         -h             print help.\n");
   fprintf (stderr, "         -i             print source probe id.\n");
   fprintf (stderr, "         -I             print transaction state and option indicators.\n");
   fprintf (stderr, "         -l             print last time values [default is start time].\n");
   fprintf (stderr, "         -m             print MAC addresses.\n");
   fprintf (stderr, "         -n             don't convert numbers to names.\n");
   fprintf (stderr, "         -p <digits>    print fractional time with <digits> precision.\n");
   fprintf (stderr, "         -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "         -q             quiet mode. don't print record outputs.\n");
   fprintf (stderr, "         -r <file>      read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -R             print out response data when availabile.\n");
   fprintf (stderr, "         -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "         -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "               format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                        timeSpecification: [[[yyyy/]mm/]dd.]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                             [yyyy/]mm/dd\n");
   fprintf (stderr, "         -T <secs>      attach to remote server for T seconds.\n");
   fprintf (stderr, "         -u             print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth> specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "         -w <file>      write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "         -z             print Argus TCP state changes.\n");
   fprintf (stderr, "         -Z <s|d|b>     print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}

#ifdef ARGUS_CURSES
int RaPrintCursesStr (char *);

int
RaPrintCursesStr (char *str)
{
   if (!(RaCursesInit))
      RaInitCurses ();
 
   return (1);
}
#endif


void
RaProcessRecord (struct ArgusRecord *argus)
{
   if (argus->ahdr.type & ARGUS_MAR)
      RaProcessManRecord (argus);

   else {
      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            switch (argus->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_TCP:
                  RaProcessTCPRecord (argus);
                  break;

               case IPPROTO_UDP:
                  RaProcessUDPRecord (argus);
                  break;

               case IPPROTO_ICMP:
                  RaProcessICMPRecord (argus);
                  break;

               default:
                  RaProcessIPRecord (argus);
                  break;
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            RaProcessARPRecord (argus);
            break;

         default:
            RaProcessNonIPRecord (argus);
            break;
      }
   }
}

void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus;
   struct ArgusRecordStore *store;
   struct ArgusRecordData *data = NULL;
   struct RaPolicyStruct *rap;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   int RaTimeout = -1, RaIdleTimeout = 30;

   RaThisActiveIndex = 0;

   bcopy((char *) flow, (char *) RaArgusFlow, sizeof(struct ArgusFlow));

   rap =  RaFlowModel->policy[RaThisFlowModelIndex];
   if (RaPolicyMatch (argus, rap)) {
      RaModifyFlow(rap, argus, RaFlowModel);
      if (Sflag || Cflag) {
         RaTimeout = rap->ArgusTimeout;
         RaIdleTimeout = rap->ArgusIdleTimeout;
      } else {
         RaTimeout = 1000000;
         RaIdleTimeout = 1000000;
      }

   } else {
      if (Sflag || Cflag) {
         RaTimeout =  60;
         RaIdleTimeout = 120;
      } else {
         RaTimeout = 60;
         RaIdleTimeout = 120;
      }

      RaModifyDefaultFlow(argus);
   }

   if ((store = RaFindArgusRecord(&RaHashTable, argus)) == NULL)
      if ((store = RaFindRevArgusRecord(&RaHashTable, argus)) != NULL) {
      }

   if (store) {
      RaThisArgusStore = store;

      if (RaCheckTimeout (store, argus))
         RaSendArgusRecord(store);

      if ((data = store->data[0]) == NULL) {
         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;
            data->status |= srv->status & RA_SVCTEST;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *) &data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.meanval * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.meanval * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[0] = data;
         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

      } else
         RaMergeArgusRecord(argus, store, RaThisActiveIndex);

      RaUpdateArgusStore(argus, store);

      if (RaServerMode) {
         if ((data = store->data[0]) != NULL) {
            if ((data->status & ARGUS_FAR_SRCADDR_MODIFIED) && (store->status & RA_SVCPASSED)) {
               if (store->qhdr.queue == RaFailedQueue) {
                  RaRemoveFromQueue(RaFailedQueue, &store->qhdr);
                  RaAddToQueue(RaAvailableQueue, &store->qhdr);
               }
            } else {
               if (store->qhdr.queue == RaAvailableQueue) {
                  RaRemoveFromQueue(RaAvailableQueue, &store->qhdr);
                  RaAddToQueue(RaFailedQueue, &store->qhdr);
               }
            }
         }
      }

   } else {
      if ((store = RaNewArgusStore(argus)) != NULL) {
         struct ArgusRecordData *data = NULL;

         store->status |= RA_MODIFIED;
         store->status |= srv->status & RA_SVCTEST;

         store->ArgusTimeout = RaTimeout;
         store->ArgusIdleTimeout = RaIdleTimeout;
         RaThisArgusStore = store;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.meanval * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.meanval * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[0] = data;

            if ((store->rahtblhdr = RaAddHashEntry (&RaHashTable, store)) != NULL) {
               if (!(RaServerMode) && (store->status & RA_SVCPASSED))
                  RaAddToQueue(RaAvailableQueue, &store->qhdr);
               else
                  RaAddToQueue(RaFailedQueue, &store->qhdr);
            }

         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

         RaUpdateArgusStore(argus, store);

      } else
         ArgusLog (LOG_ERR, "RaNewArgusStore failed %s\n", strerror(errno));
   }

   RaWindowModified = RA_MODIFIED;

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessSrvRecord: done\n");
#endif
}


void
RaProcessManRecord (struct ArgusRecord *argus)
{
   char *str = NULL;

   if (!(qflag || wfile)) {
      str = get_man_string (argus);
#ifdef ARGUS_CURSES
      RaPrintCursesStr (str);
#endif
   }
}


struct ArgusServiceRecord ArgusThisSrv;

void
RaProcessTCPRecord (struct ArgusRecord *argus)
{
   char *str = NULL;

   if (!(qflag || wfile)) {
      str = get_tcp_string (argus);
#ifdef ARGUS_CURSES
      RaPrintCursesStr (str);
#endif
   }

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessICMPRecord (struct ArgusRecord *argus)
{
   char *str = NULL;

   if (!(qflag || wfile)) {
      str = get_icmp_string (argus);
#ifdef ARGUS_CURSES
      RaPrintCursesStr (str);
#endif
   }

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessUDPRecord (struct ArgusRecord *argus)
{
   char *str = NULL;

   if (!(qflag || wfile)) {
      str = get_udp_string (argus);
#ifdef ARGUS_CURSES
      RaPrintCursesStr (str);
#endif
   }

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
         argus->argus_far.flow.ip_flow.tp_p = 0;

   if (Rflag) 
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessIPRecord (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessSrvRecord (&ArgusThisSrv);
}


void
RaProcessARPRecord (struct ArgusRecord *argus)
{
}


void
RaProcessNonIPRecord (struct ArgusRecord *argus)
{
}


int
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   unsigned char buf[4098];
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   struct ArgusAGRStruct *agr = NULL;
   int retn = 0, i;

   for (i = 0; i < RaHistoTimeSeries; i++) {
      if ((data = store->data[i]) != NULL) {
         argus = data->argus;

         if (argus && (data->status & RA_MODIFIED)) {
            if (data->act.n > 0) {
               data->agr.act.n = data->act.n;
               data->agr.act.meanval = data->act.sumtime/data->act.n;
               data->agr.act.stdev = (unsigned int) sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
            }
            if (data->idle.n > 0) {
               data->agr.idle.n = data->idle.n;
               data->agr.idle.meanval = data->idle.sumtime/data->idle.n;
               data->agr.idle.stdev = (unsigned int) sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
            }

            ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

            if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
               bcopy ((char *)&data->agr, (char *)agr, data->agr.length);
            } else {
               if (ArgusFlowModelFile && (data->agr.count > 1)) {
                  bzero (buf, sizeof(buf));
                  bcopy ((char *)argus, buf, argus->ahdr.length);
                  argus = (struct ArgusRecord *) buf;
                  bcopy ((char *)&data->agr, &buf[argus->ahdr.length], data->agr.length);
                  argus->ahdr.length += data->agr.length;
                  argus->ahdr.status |= ARGUS_MERGED;
                  ArgusFree (data->argus);
                  data->argus = RaCopyArgusRecord(argus);
                  ArgusThisFarStatus = ArgusIndexRecord(data->argus, ArgusThisFarHdrs);
               }
            }
         }
      }
   }

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif

   return (retn);
}


#include <stdio.h>
#include <errno.h>


#define RA_MAXQSCAN  100000
 
void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordData *data = NULL;
   struct ArgusRecordStore *obj = NULL;
   int cnt = 0;
 
   switch (status) {
      case ARGUS_STOP:
         while ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL)
            RaTimeoutArgusStore(obj);
         break;

      default:
         if ((cnt = ((queue->count > RA_MAXQSCAN) ? RA_MAXQSCAN : queue->count)) != 0) {
            while (cnt--) {
               if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
                  if (RaCheckTimeout(obj, NULL)) {
                     RaTimeoutArgusStore(obj);
                  } else {
                     if ((RaServerMode) && (data = obj->data[0]) != NULL) {
                        if ((data->status & ARGUS_FAR_SRCADDR_MODIFIED) && (obj->status & RA_SVCPASSED)) {
                           RaAddToQueue(RaAvailableQueue, &obj->qhdr);
                        } else {
                           RaAddToQueue(RaFailedQueue, &obj->qhdr);
                        }
                     } else
                        RaAddToQueue(queue, &obj->qhdr);
                  }

               } else
                  cnt++;
            }
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}


void
RaUpdateWindow (WINDOW *window, struct RaQueueStruct *queue)
{
   struct ArgusAGRStruct *agr = NULL;
   struct ArgusRecord *argus = NULL;
   struct ArgusQueueHeader *qhdr;
   struct ArgusFlow *flow = NULL;
   int i, x, cnt, attr = A_NORMAL;
   short color = COLOR_WHITE;
   char *str = NULL;
   char buf[2048], tbuf[MAXSTRLEN];

   if (RaWindowModified) {
      if (RaWindowStatus && ((qhdr = queue->start) != NULL)) {
         RaSortQueue (queue);
         RaActiveCons = 0; RaClosingCons = 0; RaNoResponse = 0;
         cnt = ((Nflag > 0) ? Nflag : RaWindowLines - 1);
         cnt = (cnt > (queue->count - RaWindowStartLine)) ? (queue->count - RaWindowStartLine) : cnt;

         if (cnt > 0) {
            mvwaddnstr (RaHeaderWindow, 4, 1, RaSrvStatsFlowModelDesc[RaThisFlowModelIndex], COLS - 2);
            wclrtoeol(RaHeaderWindow);
         }

         for (x = 0, i = RaWindowStartLine; x < cnt; x++, i++) {
            attr = A_NORMAL;
            color = COLOR_WHITE;
            if (queue->array[i] != NULL) {
               struct ArgusRecordStore *store = (struct ArgusRecordStore *) queue->array[i];
               struct ArgusRecordData *data = store->data[0];

               if (data && ((argus = data->argus) != NULL)) {
                  if (RaLabel == NULL)
                     RaLabel = RaGenerateLabel(argus);

                  if (data->act.n > 0) {
                     data->agr.act.n = data->act.n;
                     data->agr.act.meanval = data->act.sumtime/data->act.n;
                     data->agr.act.stdev = (unsigned int) sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
                  }
                  if (data->idle.n > 0) {
                     data->agr.idle.n = data->idle.n;
                     data->agr.idle.meanval = data->idle.sumtime/data->idle.n;
                     data->agr.idle.stdev = (unsigned int) sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
                  }

                  ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

                  if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL)
                     bcopy ((char *)&data->agr, (char *)agr, agr->length);
                  else {
                     bcopy ((char *) argus, buf, argus->ahdr.length);
                     argus = (struct ArgusRecord *) buf;

                     if (ArgusFlowModelFile && (data->agr.count > 1)) {
                        bcopy ((char *)&data->agr, &buf[argus->ahdr.length], data->agr.length);
                        argus->ahdr.length += data->agr.length;
                        argus->ahdr.status |= ARGUS_MERGED;
                        ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

                        ArgusFree (data->argus);
                        data->argus = RaCopyArgusRecord(argus);
                        ArgusThisFarStatus = ArgusIndexRecord(data->argus, ArgusThisFarHdrs);
                        argus = data->argus;
                     }
                  }

                  flow = &argus->argus_far.flow;

                  if (!(data->status & ARGUS_FAR_SRCADDR_MODIFIED))
                     flow->ip_flow.ip_src = data->flow.ip_flow.ip_src;

                  if (!(data->status & ARGUS_FAR_DSTADDR_MODIFIED))
                     flow->ip_flow.ip_dst = data->flow.ip_flow.ip_dst;

                  if (!(data->status & ARGUS_FAR_PROTO_MODIFIED))
                     flow->ip_flow.ip_p = data->flow.ip_flow.ip_p;

                  if (!(data->status & ARGUS_FAR_SRCPORT_MODIFIED))
                     flow->ip_flow.sport = data->flow.ip_flow.sport;

                  if (!(data->status & ARGUS_FAR_DSTPORT_MODIFIED))
                     flow->ip_flow.dport = data->flow.ip_flow.dport;

                  if (!(data->status & ARGUS_FAR_TPVAL_MODIFIED))
                     flow->ip_flow.tp_p = data->flow.ip_flow.tp_p;

                  switch (argus->ahdr.status & 0xFFFF) {
                     case ETHERTYPE_IP:
                        switch (argus->argus_far.flow.ip_flow.ip_p) {
                           case IPPROTO_TCP: {
                              int status;
                              struct ArgusTCPObject *tcp = NULL;

                              if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS)
                                 tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

                              if ((tcp != NULL) && ((status = tcp->state) != 0)) {
                                 if ((status & ARGUS_NORMAL_CLOSE) || (status & ARGUS_RESET) ||
                                     (status & ARGUS_FIN) || (status & ARGUS_FIN_ACK)) {
                                    RaClosingCons++;
                                 } else 
                                    if (status & ARGUS_CON_ESTABLISHED) {
                                       RaActiveCons++;
                                    }

                              } else {
                                 if (argus->argus_far.src.count && argus->argus_far.dst.count) {
                                    RaActiveCons++;
                                    } else {
                                    RaNoResponse++;
                                 }
                              }
                              break;
                           }
            
                           case IPPROTO_UDP:
                           case IPPROTO_ICMP:
                              if (argus->argus_far.src.count && argus->argus_far.dst.count) {
                                 RaActiveCons++;
                              } else {
                                 RaNoResponse++;
                              }
                              break;

                           default:
                              if (argus->argus_far.src.count && argus->argus_far.dst.count) {
                                 RaActiveCons++;
                              } else {
                                 RaNoResponse++;
                              }
                              break;
                        }
                        break;

                     case ETHERTYPE_ARP:
                     case ETHERTYPE_REVARP:
                     default:
                        break;
                  }

                  str = get_argus_string (argus);
                  attr = A_NORMAL;

#ifdef ARGUS_CURSES
                  bzero (tbuf, MAXSTRLEN);
                  sprintf (tbuf, "%3d %s", i + 1, str);
                  mvwaddnstr (window, x + 2, 2, tbuf, COLS - 3);
                  wclrtoeol(window);
#else
                  fprintf (stdout, "%3d %s", i, str);
#endif
               }
            } else
               break;
         }
#ifdef ARGUS_CURSES
         if (i < queue->count)
            mvwaddnstr (window, x + 2, 0, "-more-\n", COLS - 3);

         wclrtobot(window);
#endif
      }

#ifdef ARGUS_CURSES
      wnoutrefresh(window);
#endif
   }

   RaWindowModified = 0;
}


char RaGetStrBuf[MAXSTRLEN];

char *
RaGetCiscoServers(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%s", (Cflag ? "yes" : "no"));
   return(retn);
}

char *
RaGetNoOutputStatus(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%s", (qflag ? "yes" : "no"));
   return(retn);
}

char *
RaGetUserAuth(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%s", ustr);
   return(retn);
}

char *
RaGetUserPass(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%s", pstr);
   return(retn);
}

char *
RaGetOutputFile(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%s", wfile);
   return(retn);
}

char *
RaGetExceptionOutputFile(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%s", exceptfile);
   return(retn);
}

char *
RaGetTimeRange(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%s", timearg);
   return(retn);
}

char *
RaGetRunTime(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%d", Tflag);
   return(retn);
}

char *
RaGetFieldDelimiter(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   if (RaFieldDelimiter == '\0')
      sprintf(RaGetStrBuf, "'\\0'");
   else
      sprintf(RaGetStrBuf, "'%c'", RaFieldDelimiter);
   return(retn);
}

char *
RaGetTimeFormat(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%s", RaTimeFormat);
   return(retn);
}

char *
RaGetPrecision(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%d", pflag);
   return(retn);
}

char *
RaGetTimeSeries(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%s", (Hflag ? "yes" : "no"));
   return(retn);
}

char *
RaGetValidateStatus(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%s", (Vflag ? "yes" : "no"));
   return(retn);
}

char *
RaGetNumber(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%d", Nflag);
   return(retn);
}

char *
RaGetDebugLevel(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%d", Argusdflag);
   return(retn);
}

char *
RaGetUserDataArg(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   if (ArgusSrcUserDataLen || ArgusDstUserDataLen) {
      if (ArgusSrcUserDataLen == ArgusDstUserDataLen)
         sprintf (RaGetStrBuf, "d%d", ArgusSrcUserDataLen);
      else {
         sprintf (RaGetStrBuf, "d");
         if (ArgusSrcUserDataLen)
            sprintf (&RaGetStrBuf[strlen(RaGetStrBuf)], "s:%d", ArgusSrcUserDataLen);
         if (ArgusDstUserDataLen)
            sprintf (&RaGetStrBuf[strlen(RaGetStrBuf)], "d:%d", ArgusDstUserDataLen);
      }
   } else
      retn = NULL;

   return (retn);
}

char *
RaGetUserDataEncode(void)
{
   char *retn = RaGetStrBuf, *str = NULL;

   bzero(RaGetStrBuf, MAXSTRLEN);
   switch (eflag) {
      case ARGUS_ENCODE_ASCII:
         str = "ascii"; break;
      case ARGUS_ENCODE_32:
         str = "encode32"; break;
      case ARGUS_ENCODE_64:
         str = "encode64"; break;
   }

   sprintf(RaGetStrBuf, "%s", str);
   return(retn);
}

char *
RaGetHostFieldLength(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%d", hfield);
   return(retn);
}

char *
RaGetPortFieldLength(void)
{
   char *retn = RaGetStrBuf;

   bzero(RaGetStrBuf, MAXSTRLEN);
   sprintf(RaGetStrBuf, "%d", pfield);
   return(retn);
}

