// ---------------------------------------------------------------------------
// - Geometry.cpp                                                            -
// - afnix:geo service - base geometry class implementation                  -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2019 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Geometry.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default geometry

  Geometry::Geometry (void) {
  }

  // copy construct this geometry

  Geometry::Geometry (const Geometry& that) {
    that.rdlock ();
    try {
      d_uuid = that.d_uuid;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }
  
  // copy move this geometry

  Geometry::Geometry (Geometry&& that) noexcept :
    Nameable (static_cast<Nameable&&>(that)) {
  }
  
  // assign a geometry into this one

  Geometry& Geometry::operator = (const Geometry& that) {
    // check for self-assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      d_uuid = that.d_uuid;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }

  // move this geometry into this one

  Geometry& Geometry::operator = (Geometry&& that) noexcept {
    Nameable::operator = (static_cast<Nameable&&>(that));
    return *this;
  }

  // get the geometry uuid

  Uuid Geometry::getuuid (void) const {
    rdlock ();
    try {
      Uuid result = d_uuid;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 4;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_RESET   = zone.intern ("reset");
  static const long QUARK_GETUUID = zone.intern ("get-uuid");
  static const long QUARK_SETPLST = zone.intern ("set-plist");
  static const long QUARK_GETPLST = zone.intern ("get-plist");

  // return true if the given quark is defined
  
  bool Geometry::isquark (const long quark, const bool hflg) const {
    rdlock ();
    try {
      if (zone.exists (quark) == true) {
	unlock ();
	return true;
      }
      bool result = hflg ? Nameable::isquark (quark, hflg) : false;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // apply this object with a set of arguments and a quark

  Object* Geometry::apply (Runnable* robj, Nameset* nset, const long quark,
			Vector* argv) {
    // get the number of arguments
    long argc = (argv == nullptr) ? 0 : argv->length ();
    
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETUUID)   return new Uuid  (getuuid ());
      if (quark == QUARK_GETPLST)   return new Plist (getplst ());
      if (quark == QUARK_RESET) {
	reset ();
	return nullptr;
      }
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETPLST) {
	Object* obj = argv->get (0);
        Plist* alst = dynamic_cast <Plist*> (obj);
        if ((obj != nullptr) && (alst == nullptr)) {
          throw Exception ("type-error", "invalid object with set-plist",
                           Object::repr (obj));
        }
        setplst (*alst);
        return nullptr;
      }
    }
    // call the nameable method
    return Nameable::apply (robj, nset, quark, argv);
  }
}
