/* newtestwin.c - window for creating a new test
 *
 * Copyright 2010 Petteri Hintsanen <petterih@iki.fi>
 *
 * This file is part of abx.
 *
 * abx is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * abx is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with abx.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "test.h"
#include "gtkui.h"

static GtkWidget *window;
static GtkWidget *main_box;
static GtkWidget *first_file_selector;
static GtkWidget *second_file_selector;
static GtkWidget *first_file_box;
static GtkWidget *second_file_box;
static GtkWidget *file_box;
static GtkObject *adjustment;
static GtkWidget *spinner;
static GtkWidget *spinner_box;
static GtkWidget *ok;
static GtkWidget *cancel;
static GtkWidget *button_box;

/*
 * Try to initialize a new test.
 *
 * Destroy window if test setup was successful.  Otherwise display
 * error dialog.
 */
static void
init_new_test(void)
{
    char *a;
    char *b;
    char *s;
    int testsucc;

    a = gtk_file_chooser_get_filename
        (GTK_FILE_CHOOSER(first_file_selector));
    b = gtk_file_chooser_get_filename
        (GTK_FILE_CHOOSER(second_file_selector));

    if (a == NULL) {
        error_dialog(window, "You have not chosen sample file A.");
        return;
    }

    if (b == NULL) {
        error_dialog(window, "You have not chosen sample file B.");
        return;
    }

    testsucc = init_test(a, b, gtk_spin_button_get_value_as_int
                         (GTK_SPIN_BUTTON(spinner)), -1);

    switch (testsucc) {
    case 0:
        gtk_widget_destroy(window);
        window = NULL;
        reset_test();
        break;
    case 1:
        b = a;                  /* allow uniform error handling */
    case 2:
        s = g_strdup_printf("An error occurred while setting up "
                            "playback for sample '%s'.", b);
        error_dialog(window, s);
        g_free(s);
        break;
    case 3:
        error_dialog(window, 
                     "The chosen samples have different duration.  "
                     "Both samples must have exactly the same duration.");
        break;
    case 4:
        error_dialog(window, "Invalid number of trials.");
        break;
    }
}

static gboolean 
delete_event_handler(GtkWidget *widget, GdkEvent *event, gpointer data)
{
    window = NULL;
    return FALSE;
}

static void
button_clicked(GtkWidget *button, gpointer data)
{
    if (button == cancel) { 
        gtk_widget_destroy(window);
        window = NULL;
    } else if (button == ok) {
        init_new_test();
    }
}

static void 
create_new_test_window(void)
{
    GtkWidget *sample_a_label = gtk_label_new_with_mnemonic("Sample _A:");
    GtkWidget *sample_b_label = gtk_label_new_with_mnemonic("Sample _B:");
    GtkWidget *trials_label = (gtk_label_new_with_mnemonic
                               ("Number of _trials:"));

    /* create filename boxes and selector */
    first_file_selector = (gtk_file_chooser_button_new
                           ("Choose audio sample A", 
                            GTK_FILE_CHOOSER_ACTION_OPEN));
    second_file_selector = (gtk_file_chooser_button_new
                            ("Choose audio sample B", 
                             GTK_FILE_CHOOSER_ACTION_OPEN));
    gtk_label_set_mnemonic_widget(GTK_LABEL(sample_a_label), 
                                  first_file_selector);
    gtk_label_set_mnemonic_widget(GTK_LABEL(sample_b_label), 
                                  second_file_selector);

    first_file_box = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(first_file_box), sample_a_label,
                       FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(first_file_box), 
                       GTK_WIDGET(first_file_selector), TRUE, TRUE, 0);
    second_file_box = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(second_file_box), sample_b_label,
                       FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(second_file_box), 
                       second_file_selector, TRUE, TRUE, 0);
    file_box = gtk_vbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(file_box),
                       first_file_box, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(file_box),
                       second_file_box, TRUE, TRUE, 0);

    /* trials */
    adjustment = gtk_adjustment_new(1, 1, 50, 1.0, 10.0, 0);
    spinner = gtk_spin_button_new(GTK_ADJUSTMENT(adjustment), 1, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(trials_label),
                                  spinner);
    spinner_box = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(spinner_box), trials_label,
                       TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(spinner_box), spinner, TRUE, TRUE, 0);

    /* buttons */
    ok = gtk_button_new_from_stock(GTK_STOCK_OK);
    cancel = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
    button_box = gtk_hbox_new(TRUE, 5);
    gtk_box_pack_start(GTK_BOX(button_box), ok, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(button_box), cancel, TRUE, TRUE, 0);
    g_signal_connect(ok, "clicked", G_CALLBACK(button_clicked), NULL);
    g_signal_connect(cancel, "clicked", G_CALLBACK(button_clicked),
                     NULL);

    /* main window */
    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), "New test");
    main_box = gtk_vbox_new(FALSE, 5);
    gtk_container_set_border_width(GTK_CONTAINER(main_box), 10);
    gtk_box_pack_start(GTK_BOX(main_box), file_box, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(main_box), spinner_box, TRUE, TRUE, 5);
    gtk_box_pack_start(GTK_BOX(main_box), button_box, TRUE, TRUE, 0);
    g_signal_connect(G_OBJECT(window), "delete_event",
                     G_CALLBACK(delete_event_handler), NULL);
    gtk_container_add(GTK_CONTAINER(window), main_box);
}

void
show_new_test_window(void)
{
    if (!window) {
        create_new_test_window();
        gtk_widget_show_all(window);
    } else {
        gtk_window_present(GTK_WINDOW(window));
    }
}
