#ifndef INCLUDE_MODULE_ACPI_H
#define INCLUDE_MODULE_ACPI_H
/**
 * @brief Definitions and prototypes for module_acpi.c
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/module_acpi.h
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */

#include <glib.h>
#include "debug.h"

#if defined(DEBUG) && SIMU_ACPI
#  define PATH_ACPI		"/home/grimm/acpi"
#else
#  define PATH_ACPI		"/proc/acpi"
#endif

#define ACPID_SOCKET	"/var/run/acpid.socket"

/**
 * @brief  Battery data
 *
 * This structure contains data of a battery. This data is used
 * from the power management for battery warnings.
 *
 * This structure is currently filled only by the ACPI module
 * but could also be used by the PMU module.
 */
struct battery {
	/**
	 * @brief TRUE, if the battery is present
	 */
	gboolean present;
	/**
	 * @brief TRUE, if the battery is charging 
	 */
	gboolean charging;
	/**
	 * @brief voltage of the battery in mV
	 */
	int voltage;       /* in mV */
	/**
	 * @brief Current sourced or sinked by the battery in mA
	 *
	 * Some ACPI systems report the current rate as power
	 * consumption in mW. In that case this field contains
	 * the power consumption rather than a current.
	 *
	 * @todo  calculate the current out of power consumption
	 *        and battery voltage.
	 */
	int current;       /* in mA */
	/**
	 * @brief Remaining battery capacity in mAh
	 *
	 * Those ACPI systems that report their current rate in 
	 * mW report the capacity in mWh.
	 *
	 * @todo  recalculate this value to mAh if given in mWh
	 */
	int capacity;      /* in mAh */
	/**
	 * @brief Last full battery capacity in mAh
	 *
	 * The battery capacity will decrease over time. This value
	 * shows the capacity the battery have had after the last
	 * charge process.
	 *
	 * Those ACPI systems that report their current rate in 
	 * mW report the capacity in mWh.
	 *
	 * @todo  recalculate this value to mAh if given in mWh
	 */
	int full_capacity; /* in mAh */
	/**
	 * @brief Source of battery data
	 *
	 * This field contains the source path to the battery data.
	 * For ACPI systems this path link to /proc/acpi/battery/..
	 * and for PMU systems it could show /proc/pmu/battery/..
	 */
	char *path;
};

/**
 * @brief ACPI event
 *
 * This structure holds a decoded ACPI event. Each ACPI event
 * has four segments:
 *   @li device which triggers the event. This is a string which
 *       holds the class and the event source divided by a slash.
 *   @li bus identifier
 *   @li type This value is set by the specific ACPI driver
 *   @li data Free to choose data set by the specific ACPI driver
 *
 * The complete event is sent as a string of following format by
 * the kernel via /proc/acpi/event:
 *              button/lid  PFCD  00000040 00000001
 */
struct acpi_event {
	char *device;
	char *bus;
	int  type;
	int  data;
};

/**
 * @brief Structure to store module specific flags
 */
struct modflags_acpi {
	unsigned int coveropen :1;
	unsigned int ac_power:1;
	unsigned int sleepsupported:1;
	unsigned int :0;
};

/* public prototypes */
int acpi_init ();
void acpi_exit ();

/* private prototypes */
void acpi_query (struct tagitem *taglist);
void acpi_configure (struct tagitem *taglist);
void acpi_handle_tags (int cfgure, struct tagitem *taglist);
gboolean acpi_event_handler (int fd, gpointer user_data);
void acpi_timer1000 (struct tagitem *taglist);
int acpi_calc_timeleft (GList *batlist);
GList *acpi_update_batteries (GList *batlist);
void acpi_free_battery (struct battery *bat);
gboolean acpi_read_ac_adapter ();
gboolean acpi_read_cover_status ();
GList *acpi_collect_batteries (GList *batlist);
int acpi_open_socket (char *name);
GList *acpi_get_file_list (char *startdir);
void acpi_free_file_list (GList *list);

#if defined(DEBUG) && ACPI_BATTERY
void acpi_print_battery (GList *batlist);
#endif

#endif    /* INCLUDE_MODULE_ACPI_H */
