/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#include <libwpd/WPXDocumentInterface.h>

#include <libwpd-stream/libwpd-stream.h>

#include "libebook_utils.h"
#include "EBOOKMemoryStream.h"
#include "ZVRParser.h"

using boost::shared_ptr;

using std::string;

namespace libebook
{

ZVRParser::ZVRParser(WPXInputStream *input, WPXDocumentInterface *document)
  : m_input(input)
  , m_document(document)
{
  unsigned char c = 0;
  do
  {
    c = readU8(m_input);
  }
  while ('\n' != c);
}

bool ZVRParser::parse()
{
  readReplacementTable();

  const boost::shared_ptr<WPXInputStream> input = uncompress();

  m_document->startDocument();
  m_document->setDocumentMetaData(WPXPropertyList());
  m_document->openPageSpan(WPXPropertyList());

  writeText(input);

  m_document->closePageSpan();
  m_document->endDocument();

  return false;
}

void ZVRParser::readReplacementTable()
{
  for (size_t i = 1; i != EBOOK_NUM_ELEMENTS(m_replacementTable); ++i)
  {
    string replacement;
    unsigned char c = '\n';

    while (!m_input->atEOS())
    {
      c = readU8(m_input);
      if ('\n' == c)
      {
        if (replacement.empty())
          m_replacementTable[i] = static_cast<char>(i);
        else
        {
          m_replacementTable[i] = replacement;
          replacement.clear();
        }
        break;
      }
      else
        replacement.push_back(c);
    }
  }
}

shared_ptr<WPXInputStream> ZVRParser::uncompress()
{
  string text;

  while (!m_input->atEOS())
  {
    const unsigned char c = readU8(m_input);
    text.append(m_replacementTable[c]);
  }

  shared_ptr<WPXInputStream> strm(new EBOOKMemoryStream(reinterpret_cast<const unsigned char *>(text.data()), text.size()));
  return strm;
}

void ZVRParser::writeText(const shared_ptr<WPXInputStream> input)
{
  string text;

  while (!input->atEOS())
  {
    const unsigned char c = readU8(input.get());
    if ('\n' == c)
    {
      m_document->openParagraph(WPXPropertyList(), WPXPropertyListVector());
      if (!text.empty())
      {
        m_document->openSpan(WPXPropertyList());
        m_document->insertText(WPXString(text.c_str()));
        m_document->closeSpan();

        text.clear();
      }
      m_document->closeParagraph();
    }
    else
      text.push_back(c);
  }
}

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
