// SPDX-FileCopyrightText: 2006 Istituto Nazionale di Fisica Nucleare
//
// SPDX-License-Identifier: Apache-2.0

package org.italiangrid.voms.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.net.URISyntaxException;

import org.italiangrid.voms.VOMSError;
import org.italiangrid.voms.request.VOMSServerInfo;
import org.italiangrid.voms.request.impl.VOMSESLineParser;
import org.junit.Test;

public class TestVOMSESLineParser {

  @Test
  public void nullLineFailure() throws URISyntaxException {

    String line = null;
    try {

      VOMSESLineParser p = new VOMSESLineParser();
      p.parse(line);
      fail("No error raised.");

    } catch (VOMSError e) {
      assertNotNull("Got a null error message", e.getMessage());
      assertEquals("Cannot parse a null VOMSES line", e.getMessage());
    }
  }

  @Test
  public void emptyLineReturnsNull() throws URISyntaxException {

    String line = "";

    VOMSESLineParser p = new VOMSESLineParser();
    VOMSServerInfo i = p.parse(line);
    assertNull(i);

  }

  @Test
  public void emptyAlias() {

    String line = "\"\"";
    try {
      VOMSESLineParser p = new VOMSESLineParser();
      p.parse(line);
      fail("No error raised.");
    } catch (VOMSError e) {
      assertNotNull("Got a null error message", e.getMessage());
      assertTrue(e.getMessage().contains("Invalid VOMSES line: empty 'vo alias' field."));
    }
  }

  @Test
  public void incompleteAlias() {

    String line = "\"incomplete-alias";
    try {
      VOMSESLineParser p = new VOMSESLineParser();
      p.parse(line);
      fail("No error raised.");
    } catch (VOMSError e) {
      assertNotNull("Got a null error message", e.getMessage());
      assertTrue(e.getMessage().contains("Invalid VOMSES line: incomplete 'vo alias' field."));
    }
  }

  @Test
  public void incompleteHost() {

    String line = "\"alias\" \"voms.cnaf.infn.it";
    try {
      VOMSESLineParser p = new VOMSESLineParser();
      p.parse(line);
      fail("No error raised.");
    } catch (VOMSError e) {
      assertNotNull("Got a null error message", e.getMessage());
      assertTrue(e.getMessage().contains("Invalid VOMSES line: incomplete 'voms host' field."));
    }
  }

  @Test
  public void onlyAlias() {

    String line = "\"ciccio\" ";
    try {
      VOMSESLineParser p = new VOMSESLineParser();
      p.parse(line);
      fail("No error raised.");
    } catch (VOMSError e) {
      assertNotNull("Got a null error message", e.getMessage());
      assertTrue(e.getMessage().contains("Invalid VOMSES line: incomplete information"));
    }
  }

  @Test
  public void minimumInfoFailure() {

    String line = "\t\"a\"  \"voms.cern.ch\" \t \"15000\" \"DN=Illo\"";

    try {
      VOMSESLineParser p = new VOMSESLineParser();
      p.parse(line);
      fail("No error raised.");
    } catch (VOMSError e) {

      assertNotNull("Got a null error message", e.getMessage());
      assertTrue(e.getMessage().contains("Invalid VOMSES line: incomplete information"));
    }
  }

  @Test
  public void minimumInfo() {

    String line = "\t\"a\"  \"voms.cern.ch\" \t \"15000\" \"DN=Illo\" \"alice\" ";

    VOMSESLineParser p = new VOMSESLineParser();
    VOMSServerInfo i = p.parse(line);

    assertEquals("a", i.getAlias());
    assertEquals("voms://voms.cern.ch:15000", i.getURL().toString());
    assertEquals("alice", i.getVoName());
    assertEquals("DN=Illo", i.getVOMSServerDN());
  }

  @Test
  public void whitespaceHandling() {

    String line = "\t\"a\"  \"voms.cern.ch\" \t \"15000\" \"DN=Illo\" \"alice\" \"24\"";

    VOMSESLineParser p = new VOMSESLineParser();
    VOMSServerInfo i = p.parse(line);

    assertEquals("a", i.getAlias());
    assertEquals("voms://voms.cern.ch:15000", i.getURL().toString());
    assertEquals("alice", i.getVoName());
    assertEquals("DN=Illo", i.getVOMSServerDN());
  }

  @Test
  public void tooManyFields() {

    String line = "\t\"a\"  \"voms.cern.ch\" \t \"15000\" \"DN=Illo\" \"alice\" \"24\" \"Too much\"";

    try {
      VOMSESLineParser p = new VOMSESLineParser();
      p.parse(line);
      fail("No error raised.");
    } catch (VOMSError e) {
      assertNotNull("Got a null error message", e.getMessage());
      assertTrue(e.getMessage().contains("Invalid VOMSES line: too many fields!"));
    }

  }

  @Test
  public void invalidPort() {

    String line = "\t\"a\"  \"voms.cern.ch\" \t \"ciccio\" \"DN=Illo\" \"alice\"";

    try {
      VOMSESLineParser p = new VOMSESLineParser();
      p.parse(line);
      fail("No error raised.");
    } catch (VOMSError e) {
      assertNotNull("Got a null error message", e.getMessage());
      assertTrue(e.getMessage().contains("Invalid VOMSES line: invalid port number."));
    }

  }

  @Test
  public void portOutOfRange1() {

    String line = "\t\"a\"  \"voms.cern.ch\" \t \"-1\" \"DN=Illo\" \"alice\"";

    try {
      VOMSESLineParser p = new VOMSESLineParser();
      p.parse(line);
      fail("No error raised.");
    } catch (VOMSError e) {
      assertNotNull("Got a null error message", e.getMessage());
      assertTrue(e.getMessage().contains("Invalid VOMSES line: invalid port number: -1"));
    }
  }

  @Test
  public void portOutOfRange2() {

    String line = "\t\"a\"  \"voms.cern.ch\" \t \"65536\" \"DN=Illo\" \"alice\"";

    try {
      VOMSESLineParser p = new VOMSESLineParser();
      p.parse(line);
      fail("No error raised.");
    } catch (VOMSError e) {
      assertNotNull("Got a null error message", e.getMessage());
      assertTrue(e.getMessage().contains("Invalid VOMSES line: invalid port number: 65536"));
    }
  }

  @Test
  public void tooMultiCall() {

    String line0 = "\"a\" \"voms.cern.ch\" \"15000\" \"DN=Illo\" \"alice\"";
    String line1 = "\"b\" \"voms.cern.ch\" \"15001\" \"DN=IllY\" \"bolice\"";

    VOMSESLineParser p = new VOMSESLineParser();
    VOMSServerInfo i0 = p.parse(line0);
    VOMSServerInfo i1 = p.parse(line1);

    assertEquals("a", i0.getAlias());
    assertEquals("voms://voms.cern.ch:15000", i0.getURL().toString());
    assertEquals("alice", i0.getVoName());

    assertEquals("b", i1.getAlias());
    assertEquals("voms://voms.cern.ch:15001", i1.getURL().toString());
    assertEquals("bolice", i1.getVoName());
  }
}
