/* Tic-Tac-Toe plasmoid
 * Copyright (C) 2009 Mauro E. Bender <mbender@dc.uba.ar>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "plasma-tictactoe.h"

#include <QPainter>
#include <QPointF>
#include <QFontMetrics>
#include <QSizeF>
#include <QGraphicsSceneMouseEvent>
#include <QString>
#include <QList>
#include <QAction>
#include <KIcon>
#include <KStandardDirs>
#include <KConfigDialog>

PlasmaTictactoe::PlasmaTictactoe( QObject *parent, const QVariantList &args ) : Plasma::Applet(parent, args){
	_tictactoe = new Tictactoe;
	
	QAction *resetAction = new QAction(this);
	resetAction->setText( i18n( "&Reset" ) );
	resetAction->setIcon ( KIcon( "view-refresh" ) );
	connect ( resetAction, SIGNAL(triggered()), this, SLOT(resetGame()) );
	_contextualActions.append ( resetAction );
	/*edit-redo
	  edit-undo
	  system-restart
	  view-refresh*/
	
	QString xofile = KStandardDirs::locate ( "data", "plasma-tictactoe/xo.svg" );
	
	_xo.setImagePath( xofile );
	
	if( !_xo.isValid() )
		setFailedToLaunch( true, i18n("Couldn't find xo.svg!") );
	
	_tictactoe->setPlayerName(1, i18n( "Player 1" ).toUtf8().constData() );
	_tictactoe->setPlayerName(2, i18n( "Player 2" ).toUtf8().constData() );
	
	_currentTheme = GRAY;
	setMinimumSize(200,200);
	resize(300, 300);
}

PlasmaTictactoe::~PlasmaTictactoe(){
	if( hasFailedToLaunch() ){}
	else{}
}

void PlasmaTictactoe::init(){
	if( !_tictactoe )
		setFailedToLaunch( true, i18n("Unknown error!") );
}

void PlasmaTictactoe::mousePressEvent(QGraphicsSceneMouseEvent *e){
	if( !_tictactoe || e->button() != Qt::LeftButton )
		return;
	
	if( _tictactoe->gameStatus() != INGAME )
		_tictactoe->reset();
	else{
		//Get the coordinates where the user made click in the plasmoid
		QPointF posicion = e->pos() - contentsRect().topLeft();
		int xcoord = (int) posicion.x();
		int ycoord = (int) posicion.y();
	
		//Get the dimensions of my plasmoid
		int height = contentsRect().height();
		int width = contentsRect().width();
	
		int hSpace = width / 3;
		int vSpace = height / 3;
	
		if( ycoord < vSpace + 1 ){						//Is in the first row
			if( xcoord < width - 5)
				_tictactoe->touch(0, xcoord / hSpace);
		}
		else if ( ycoord < 2 * vSpace + 1 ){		//Is in the second row
			if( xcoord < width - 5)
				_tictactoe->touch(1, xcoord / hSpace);
		}
		else if ( ycoord < 3 * vSpace + 1){			//Is in the third row
			if( xcoord < width - 5)
				_tictactoe->touch(2, xcoord / hSpace);
		}
	}
	
	update();
}

void PlasmaTictactoe::paintInterface( QPainter *painter, const QStyleOptionGraphicsItem *options, const QRect &contentsRect ){
  Q_UNUSED ( options );
  
	painter->setRenderHints( QPainter::Antialiasing );
	painter->setPen( Qt::white );

	//Get the dimensions of my plasmoid
	int height = contentsRect.size().height();
	int width = contentsRect.size().width();
	
	//Calculating the space between lines
	int hSpace = width / 3;
	int vSpace = height / 3;
	
	//Paint the tictactoe lines on the plasmoid
	painter->drawLine( contentsRect.topLeft() + QPoint(hSpace, 0), contentsRect.bottomLeft() + QPoint(hSpace, 0) );
	painter->drawLine( contentsRect.topLeft() + QPoint(hSpace * 2, 0), contentsRect.bottomLeft() + QPoint(hSpace * 2, 0) );
	painter->drawLine( contentsRect.topLeft() + QPoint(0, vSpace), contentsRect.topRight() + QPoint(0, vSpace) );
	painter->drawLine( contentsRect.topLeft() + QPoint(0, vSpace * 2), contentsRect.topRight() + QPoint(0, vSpace * 2) );
	
	
	QRect r;
	int topLeftX = contentsRect.topLeft().x();
	int topLeftY = contentsRect.topLeft().y();
	for(int i = 0; i < 3; i++){
		for(int j = 0; j < 3; j++){
			r.setCoords(  topLeftX + 2 + j * hSpace, topLeftY +2 + i * vSpace, topLeftX + (j + 1) * hSpace - 2, topLeftY + ( i + 1) * vSpace - 2 );
			if( _tictactoe->getValue(i, j) == 1 ){
				QPixmap pixmap = _xo.pixmap( QString("x_%1").arg( _currentTheme ) );
				painter->drawPixmap( r, pixmap, pixmap.rect() );
			}
			else if ( _tictactoe->getValue(i, j) == 2 ){
				r.adjust(5,0,-5,0);
				QPixmap pixmap = _xo.pixmap( QString("o_%1").arg( _currentTheme ) );
				painter->drawPixmap( r, pixmap, pixmap.rect() );
			}
		}
	}
	
	
	GameStatus status = _tictactoe->gameStatus();
	if( status != INGAME){
		r.setTopLeft( contentsRect.topLeft() + QPoint(width / 8, height / 4) );
		r.setBottomRight( contentsRect.bottomRight() - QPoint(width / 8, height / 4) );
		painter->setBrush( QColor(200,200,200,235) );
		painter->drawRect( r );
		if( status == WINNER ){
			r.adjust( 2, 2, -2, -2);
			painter->setPen( Qt::white );
			painter->drawText ( r, Qt::AlignCenter | Qt::TextWordWrap,
				i18n( "%1 has won the game!!!\n\n"
				"Click on the plasmoid to start a new game.", QString::fromUtf8( _tictactoe->winner().c_str() ) ) );
		}
		else{
			painter->drawText ( r, Qt::AlignCenter | Qt::TextWordWrap,
				i18n ( "Tie!\n\n"
					"Click on the plasmoid to start a new game." ) );
		}
	}
}

void PlasmaTictactoe::resetGame(){
	_tictactoe->reset();
	
	update();
}

void PlasmaTictactoe::createConfigurationInterface ( KConfigDialog *parent ){
	_playersSettings = new PlayersSettings ( &_xo, _currentTheme );
	_playersSettings->setName(1, QString::fromUtf8 ( _tictactoe->getPlayerName(1).c_str () ) );
	_playersSettings->setName(2, QString::fromUtf8 ( _tictactoe->getPlayerName(2).c_str () ) );
	parent->addPage( _playersSettings, i18n("Players"), icon() );
	
	_appearanceSettings = new AppearanceSettings( &_xo, _currentTheme );
	parent->addPage( _appearanceSettings, i18n("Appearence"), "applications-graphics" );
	
	connect ( parent, SIGNAL(okClicked()), this, SLOT(applyConfiguration()) );
	connect ( parent, SIGNAL(applyClicked()), this, SLOT(applyConfiguration()) );
}

void PlasmaTictactoe::applyConfiguration(){
	_tictactoe->setPlayerName(1, _playersSettings->name(1).toUtf8().constData() );
	_tictactoe->setPlayerName(2, _playersSettings->name(2).toUtf8().constData() );
	_currentTheme = _appearanceSettings->getCurrentTheme();
}

#include "plasma-tictactoe.moc"
