///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __TARGET_CAMERA_CREATION_MODE_H
#define __TARGET_CAMERA_CREATION_MODE_H

#include <core/Core.h>
#include <core/scene/objects/CreationMode.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/snapping/SnappingManager.h>
#include <core/scene/objects/SceneObject.h>
#include <core/scene/SceneRoot.h>
#include <core/data/DataSetManager.h>
#include <core/gui/properties/PropertiesPanel.h>
#include "../StdObjects.h"
#include "CameraObject.h"
#include "../helpers/TargetObject.h"

namespace StdObjects {

/******************************************************************************
* This input mode lets the user create a target camera object.
******************************************************************************/
class TargetCameraCreationMode : public CreationMode
{
public:
	/// Default constructor.
	TargetCameraCreationMode() : CreationMode(), clickCount(0), createOperation(NULL) {}

	/// Returns the activation behaviour of this input handler.
	virtual InputHandlerType handlerActivationType() { return NORMAL; }
	
protected:

	/// Will be called when the user presses the left mouse button.
	virtual void onMousePressed(QMouseEvent* event) {
		if(clickCount != 0) return;

		try {
			OVITO_ASSERT(createOperation == NULL);
			OVITO_ASSERT(adjustOperations.empty());
			
			// Begin recording operation.
			createOperation = UNDO_MANAGER.beginCompoundOperation(tr("Create Camera"));
			
			// Do not create any animation keys.
			AnimationSuspender animSuspender;

			SceneRoot* scene = DATASET_MANAGER.currentSet()->sceneRoot();
			CHECK_OBJECT_POINTER(scene);
		
			{
				UndoSuspender unsoSuspender;	// Do not create undo records for this. 
				
				// Create target helper.
				target = new TargetObject();
				targetNode = new ObjectNode();				
				targetNode->setSceneObject(target);
				targetNode->setDisplayColor(Color(0.2f,0.9f,1.0f));
				
				// Create camera object.
				camera = new CameraObject();
				cameraNode = new ObjectNode();				
				cameraNode->setSceneObject(camera);
				cameraNode->setDisplayColor(targetNode->displayColor());
				
				// Make the camera look at target.
				cameraNode->bindToTarget(targetNode.get());
				
				// Give the new nodes a name.
				cameraNode->setName(scene->makeNameUnique(tr("Camera")));
				targetNode->setName(cameraNode->name() + tr(".Target"));
			}
			
			// Insert nodes into scene.
			scene->addChild(cameraNode);
			scene->addChild(targetNode);
			
			// Select new node.
			DATASET_MANAGER.currentSelection()->clear();
			DATASET_MANAGER.currentSelection()->add(cameraNode);
			
			// Take click point as initial camera position
			cameraPos = ORIGIN;
			if(!viewport()->snapPoint(event->pos(), cameraPos))
				onAbort();
				
			// Position camera and target node.
			AffineTransformation tm = viewport()->grid().gridMatrix() * AffineTransformation::translation(cameraPos - ORIGIN);			
			cameraNode->transformationController()->setValue(0, tm);
			targetNode->transformationController()->setValue(0, tm);
			
			// Show properties of new object.
			if(propertiesPanel() != NULL)
				propertiesPanel()->setEditObject(camera);
			
			// Begin recording adjusting target position
			beginAdjustOperation();

			// This is the first click.
			clickCount = 1;
		}
		catch(const Exception& ex) {
			onAbort();
			ex.showError();
		}
	}

	/// Will be called when the user releases the left mouse button.
	virtual void onMouseReleased(QMouseEvent* event) {
		OVITO_ASSERT(createOperation != NULL);
		clickCount++;
		if(clickCount >= 2) {
			if(targetPos != cameraPos)
				onFinish();
			else
				onAbort();
		}
	}

	/// Will be called when the user moves the mouse while 
	/// the operation is active.
	virtual void onMouseDrag(QMouseEvent* event) {
		if(clickCount == 1) {
			// Take mouse position as new target position
			targetPos = ORIGIN;
			if(!viewport()->snapPoint(event->pos(), targetPos))
				return;
			
			abortAdjustOperation();
			beginAdjustOperation();

			// Position targetPos node.
			AffineTransformation tm = viewport()->grid().gridMatrix() * AffineTransformation::translation(targetPos - ORIGIN);			
			targetNode->transformationController()->setValue(0, tm);

			// Update viewports immediately.
			VIEWPORT_MANAGER.processViewportUpdates();
		}
	}

	/// Will be called when the user moves the mouse while
	/// the operation is not active.
	virtual void onMouseFreeMove(Viewport& vp, QMouseEvent* event)
	{
		snapPreview(vp, event);		
	}
	
	/// Starts a new adjust operation.
	void beginAdjustOperation()
	{
		CompoundOperation* adjustOperation = UNDO_MANAGER.beginCompoundOperation("Create Object Adjust Operation");
		adjustOperations.push(adjustOperation);
	}

	/// Undoes the current adjust operation and removes it from the undo stack.
	void abortAdjustOperation()
	{
		OVITO_ASSERT(!adjustOperations.empty());
		CompoundOperation* adjustOperation = adjustOperations.top();
		adjustOperations.pop_back();
		adjustOperation->clear();
		UNDO_MANAGER.endCompoundOperation();
	}	

	/// Finishes an adjust operation and puts it on the undo stack.
	void CommitAdjustOperation()
	{
		OVITO_ASSERT(!adjustOperations.empty());
		adjustOperations.pop_back();
		UNDO_MANAGER.endCompoundOperation();
	}	

	/// Is called when the user aborts the operation by clicking the right
	/// mouse button or activating another input mode. When overriden,
	/// the base implementation should be called.
	virtual void onAbort()
	{
		SimpleInputHandler::onAbort();
	
		if(createOperation == NULL) return;
		OVITO_ASSERT(!adjustOperations.empty());
		OVITO_ASSERT(camera);
	
		// Close properties editor
		if(propertiesPanel() != NULL)
			propertiesPanel()->setEditObject(NULL);
	
		while(!adjustOperations.empty())
			abortAdjustOperation();
		createOperation->clear();
		UNDO_MANAGER.endCompoundOperation();	
		createOperation = NULL;
		cameraNode = NULL;
		camera = NULL;
		targetNode = NULL;
		target = NULL;
		clickCount = 0;
	}

	/// Is called when the user finishes the operation. When overriden,
	/// the base implementation should be called.
	virtual void onFinish()
	{
		SimpleInputHandler::onFinish();
	
		if(createOperation == NULL) return;
		OVITO_ASSERT(!adjustOperations.empty());
		OVITO_ASSERT(camera);
	
		while(!adjustOperations.empty())
			CommitAdjustOperation();
		UNDO_MANAGER.endCompoundOperation();
		createOperation = NULL;
		cameraNode = NULL;
		camera = NULL;
		targetNode = NULL;
		target = NULL;
		clickCount = 0;
	}	

private:

	/// For undoing the complete create operation.
	CompoundOperation* createOperation;

	/// For undoing the position/resize operations.
	QStack<CompoundOperation*> adjustOperations;

	/// The object node.
	ObjectNode::SmartPtr cameraNode;

	/// The camera object.
	CameraObject::SmartPtr camera;

	/// The target node.
	ObjectNode::SmartPtr targetNode;

	/// The target object.
	TargetObject::SmartPtr target;

	/// The number of clicks during the operation.
	int clickCount;

	Point3 cameraPos;
	Point3 targetPos;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(TargetCameraCreationMode)
};

};

#endif 	// __TARGET_CAMERA_CREATION_MODE_H

