# +=========================================================================+
# || Module/Build/CipUX.pm                                                 ||
# ||                                                                       ||
# || Extension for building CipUX plugins and core packages.               ||
# || Inspired by Module::Build::IkiWiki by Victor Moral,                   ||
# || Copyright (C) 2008, licensed under GNU GPL version 2 or any later     ||
# || version.                                                              ||
# ||                                                                       ||
# || Copyright (C) 2008 - 2009 by Christian Kuelker. All rights reserved!  ||
# ||                                                                       ||
# || License: GNU GPL version 2 or any later version.                      ||
# ||                                                                       ||
# +=========================================================================+
# ID:       $Id$
# Revision: $Revision$
# Head URL: $HeadURL$
# Date:     $Date$

package Module::Build::CipUX;
use base qw(Module::Build);
use warnings;
use strict;
use Carp;
use Data::Dumper;
use File::Basename;
use File::Spec;
use Readonly;
use version; our $VERSION = qv('0.4.0');

# +=========================================================================+
# || CONSTANTS                                                             ||
# +=========================================================================+
Readonly::Scalar my $EMPTY_STRING     => q{};
Readonly::Scalar my $MODE_RWX_RWX_R_X => oct 755;
Readonly::Scalar my $MODE_RWX_R_X____ => oct 750;
Readonly::Scalar my $MODE_R__________ => oct 600;

# flag for fake installations
my $_fake_install = 0;

my @cipux_elements
    = qw( setup schema conf644 conf600 webcgi webtpl webimg weblog
    webconf locale rpcconf sbin750 sbin755 bin750 bin755 initscript
    apacheconf bootcfg bootcfgstorage bootcfgobject bootcfgtask
    bootcfgaccess);

my @cipux_types = ();    # cipux_*
foreach my $e (@cipux_elements) {
    push @cipux_types, 'cipux_' . $e;
}
my @cipux_build_targets = ();    # *_files
foreach my $e (@cipux_elements) {
    push @cipux_build_targets, $e . '_files';
}

# This overwrites the KEYs in INSTALL target of Build.PL
my %types = (    # translate types/dirs to those expected by CipUX
    lib     => '/usr/share/perl5',
    arch    => '/arch',
    bin     => '/bin',
    script  => '/usr/bin',
    bindoc  => '/usr/share/man/man1',
    libdoc  => '/usr/share/man/man3',
    binhtml => undef,
    libhtml => undef,
    conf644 => '/etc/cipux/',
    conf600 => '/etc/cipux/',
    schema  => '/etc/ldap/schema',
    setup   => '/etc/cipux/setup',
    webcgi  => '/usr/share/cipux-cat-web',
    webtpl  => '/usr/share/cipux-cat-web/tpl',
    webimg  => '/usr/share/cipux-cat-web/img',
    weblog  => '/var/log/cipux-cat-web',
    webconf => '/etc/cipux-cat-web',
    locale  => '/usr/share/locale',            # de_DE/LC_MESSAGES/<PACKET>.mo
    rpcconf => '/etc/cipux-rpc',
    sbin750 => '/usr/sbin',
    sbin755 => '/usr/sbin',
    bin750  => '/usr/bin',
    bin755  => '/usr/bin',
    apacheconf     => '/etc/apache2/conf.d',
    initscript     => '/etc/init.d',
    bootcfg        => '/usr/share/cipux/etc',
    bootcfgstorage => '/usr/share/cipux/etc/cipux-storage.d',
    bootcfgaccess => '/usr/share/cipux/etc/cipux-access.d',
    bootcfgobject  => '/usr/share/cipux/etc/cipux-object.d',
    bootcfgtask    => '/usr/share/cipux/etc/cipux-task.d',
);

my %cipux_include = (
    webcgi_files => {
        source_dir => 'cgi',         # directory prefix under src
        source_ext => qr{[.]pl$}smx, # search regex
        build_dir  => 'webcgi',      # directory prefix in blib
        build_ext  => '.cgi',        # replace string (define only if replace)
        build_exec => 1,
    },
    webtpl_files => {
        source_dir => 'tpl',
        source_ext => qr{[.]html$|[.]jpg$|[.]png$|[.]css$|[.]js$}smx,
        build_dir  => 'webtpl',
    },
    webimg_files => {
        source_dir => 'img',
        source_ext => qr{[.]jpg$|[.]png$}smx,
        build_dir  => 'webimg',
    },
    webconf_files => {
        source_dir => 'etc/cipux-cat-web',
        source_ext => qr{[.]conf$}smx,
        build_dir  => 'webconf',
    },

   # 'etc/apache2/conf.d/cipux_apache.conf' => 'apacheconf/cipux_apache.conf',
    apacheconf_files => {
        source_dir => 'etc/apache2/conf.d',
        source_ext => qr{[.]conf$}smx,
        build_dir  => 'apacheconf',
    },
    sbin750_files => {
        source_dir => 'sbin750',
        source_ext => qr{cipux_}smx,
        build_dir  => 'sbin750',
        build_mode => $MODE_RWX_R_X____,    # octal number NOT string
        build_exec => 1,
    },
    sbin755_files => {
        source_dir => 'sbin',
        source_ext => qr{cipux_}smx,
        build_dir  => 'sbin755',
        build_exec => 1,
    },
    bin750_files => {
        source_dir => 'bin750',
        source_ext => qr{cipux_}smx,
        build_dir  => 'bin750',
        build_mode => $MODE_RWX_R_X____,    # octal number NOT string
        build_exec => 1,
    },
    bin755_files => {
        source_dir => 'bin',
        source_ext => qr{cipux_}smx,
        build_dir  => 'bin755',
        build_exec => 1,
    },
    schema_files => {
        source_dir => 'etc/ldap/schema',
        source_ext => qr{[.]schema$}smx,
        build_dir  => 'schema',
    },
    setup_files => {
        source_dir => 'etc/cipux/setup',
        source_ext =>
            qr{[.]schema$|[.]ldif$|[.]conf$|[.]acl$|[.]patched$|[.]diff$}smx,
        build_dir => 'setup',

    },
    conf644_files => {
        source_dir => 'etc/cipux',
        source_ext => qr{[.]conf$}smx,
        build_dir  => 'conf644',
    },
    conf600_files => {
        source_dir => 'etc600/cipux',
        source_ext => qr{[.]conf$}smx,
        build_dir  => 'conf600',
        build_mode => $MODE_R__________,    # octal number NOT string
    },
    initscript_files => {
        source_dir => 'etc/init.d',
        source_ext => qr{cipux-}smx,
        build_dir  => 'initscript',
        build_exec => 1,
    },
    locale_files => {
        source_dir => 'po',
        source_ext => qr{[.]po$}smx,
        build_dir  => 'locale',
    },
    bootcfg_files => {
        source_dir => 'usr/share/cipux/etc',
        source_ext =>
            qr{etc/[a-zA-Z_-]+\.(cfgperl|perl|pl|ini|cnf|yml|yaml|json|jsn|xml)$}smx,
        build_dir => 'bootcfg',
    },
    bootcfgstorage_files => {
        source_dir => 'usr/share/cipux/etc/cipux-storage.d',
        source_ext =>
            qr{\.(cfgperl|perl|pl|ini|cnf|yml|yaml|json|jsn|xml)$}smx,
        build_dir => 'bootcfgstorage',
    },
    bootcfgaccess_files => {
        source_dir => 'usr/share/cipux/etc/cipux-access.d',
        source_ext =>
            qr{\.(cfgperl|perl|pl|ini|cnf|yml|yaml|json|jsn|xml)$}smx,
        build_dir => 'bootcfgaccess',
    },
    bootcfgobject_files => {
        source_dir => 'usr/share/cipux/etc/cipux-object.d',
        source_ext =>
            qr{\.(cfgperl|perl|pl|ini|cnf|yml|yaml|json|jsn|xml)$}smx,
        build_dir => 'bootcfgobject',
    },
    bootcfgtask_files => {
        source_dir => 'usr/share/cipux/etc/cipux-task.d',
        source_ext =>
            qr{\.(cfgperl|perl|pl|ini|conf|cnf|yml|yaml|json|jsn|xml)$}smx,
        build_dir => 'bootcfgtask',
    },

    #rpcconf_files => {
    #    source_dir => 'etc/cipux',
    #    source_ext => qr{[.]conf$}smx,
    #    build_dir  => 'rpccpn',
    #},

    #script_files =>{ }, # disable scripts, because they will be installed
    # sometimes to different location /user/local/bin

    #weblog     => 'var/log/cipux-cat-web',
    #rpcconf    => 'etc/cipux-rpc',

);

# Module implementation here
sub new {
    my ( $class, @params ) = @_;
    my $self = $class->SUPER::new(@params);
    my $prop = $self->_get_prop();

    my %cipux_paths = %types;

    # checking default values
    if ( not defined $prop->{cipux_paths} ) {
        $prop->{cipux_paths} = \%cipux_paths;
    }

    # checking file types
    # setup, schema, conf644,  ...
    foreach my $other (@cipux_types) {
        if ( not exists $prop->{$other} ) {
            $prop->{$other} = [];
        }
    }

    # (1) add all build elements (so that they can build)
    # (2) add all CipUX file types to install_path, if
    # there are no definition already
    # setup, schema, conf644,  ...
    foreach my $ftype (@cipux_elements) {
        push @{ $prop->{build_elements} }, $ftype;
        if ( not exists $prop->{install_path}->{$ftype} ) {
            $prop->{install_path}->{$ftype} = $types{$ftype};
        }
    }

    # include all standard CipUX files in the build
    foreach my $build_target (@cipux_build_targets) {
        if ( not exists $prop->{$build_target} ) {
            $prop->{$build_target} = {};

            # if we have defined a standard location
            if ( exists $cipux_include{$build_target} ) {
                my $sp = $cipux_include{$build_target}->{source_dir};
                my $se = $cipux_include{$build_target}->{source_ext};
                my $bp = $cipux_include{$build_target}->{build_dir};

                next if not -d $sp;

                # get all files of a special kind
                my $files_ar = $self->rscan_dir( $sp, $se );
                foreach my $f ( @{$files_ar} ) {
                    next if $f =~ m/[.]svn/smx;
                    my $dest = $f;
                    $dest =~ s{^$sp}{$bp}xms;

                    # if build ext => replace source ext with build ext
                    if ( exists $cipux_include{$build_target}->{build_ext} ) {
                        my $be = $cipux_include{$build_target}->{build_ext};
                        $dest =~ s{$se}{$be}xms;
                    }
                    $prop->{$build_target}->{$f} = $dest;
                }
            }
        }
    }

    # PODs use default location
    push @{ $prop->{bindoc_dirs} }, $self->blib . '/bin755';
    push @{ $prop->{bindoc_dirs} }, $self->blib . '/bin750';
    push @{ $prop->{bindoc_dirs} }, $self->blib . '/sbin755';
    push @{ $prop->{bindoc_dirs} }, $self->blib . '/sbin750';
    push @{ $prop->{bindoc_dirs} }, $self->blib . '/conf644';
    push @{ $prop->{bindoc_dirs} }, $self->blib . '/conf600';

    return $self;
}

sub ACTION_cipuxpo2mo {

    # API
    my ( $self, @params ) = @_;

    $self->depends_on('build');

    # get all *.po
    my $files_ar = $self->rscan_dir( 'po', qr{[.]po$}smx );

    # de_DE/LC_MESSAGES/<PACKET>.mo
    my $podir = File::Spec->catfile( $self->blib, 'locale' );

    while ( not -d $podir ) {
        File::Path::mkpath( $podir, 0, $MODE_RWX_RWX_R_X )
            or croak "Couldn't mkdir $podir: $!";
    }

    foreach my $f ( @{$files_ar} ) {
        my $dest = $f;
        $dest =~ s/[.]po$/.mo/gsmx;    # *.po -> *.mo
        $dest =~ s{^po/}{}gsmx;        # rm ^po/
        $dest = "$podir/$dest";        # blib/locale/*.mo

        $self->do_system("msgfmt -o $dest $f");
    }

    return;
}

sub ACTION_cipuxrelease {
    my ( $self, @params ) = @_;
    my $prop = $self->_get_prop();

    $self->depends_on('build');
    $self->depends_on('cipuxpo2mo');

    # e.g. CipUX-3.16.tar.gz
    my $crfn = $self->dist_dir;      # CipUX release file name
    my $mn   = $self->module_name;
    my $pn   = lc $mn;               # Package name CipUX -> cipux::cipux
    $pn =~ s/::/-/gmx;               # cipux::cipux -> cipux

    $self->delete_filetree($crfn);
    $self->add_to_cleanup( $crfn, "$crfn.tar.gz" );

    foreach my $type ( $self->install_types ) {
        next if exists( $types{$type} ) && !defined $types{$type};

        #$self->log_info("Using type $type\n");

        my $dir = File::Spec->catdir( $self->blib, $type );
        next if not -e $dir;

        #$self->log_info("Using dir $dir\n");

        my $files = $self->rscan_dir($dir);

        foreach my $file ( @{$files} ) {
            next if not -f $file;
            my $rel_file = File::Spec->abs2rel( File::Spec->rel2abs($file),
                File::Spec->rel2abs($dir) );
            my $to_file = File::Spec->catdir( $crfn, $EMPTY_STRING,
                exists( $types{$type} ) ? $types{$type} : $type, $rel_file );
            $self->copy_if_modified( from => $file, to => $to_file );
        }
    }

    # create a tarball;
    # the directory tar'ed is not blib so we need not do a chdir first
    my $start_wd = $self->cwd;

    $self->make_tarball( $crfn, File::Spec->catfile( $start_wd, $crfn ) );

    return;
}

# inspired by Module::Build::IkiWiki
sub _get_prop {

    my $self = shift;

    return $self->{properties};    ##Violates 'ProhibitAccessOfPrivateData'
}

# ARRAY ref to cipux file types
sub get_all_elements {

    my ( $self, $arg_r ) = @_;

    return \@cipux_elements;
}

# add all cipux file types to the build
sub add_all_elements {

    my ( $self, $arg_r ) = @_;

    my $elem_ar = $self->get_all_elements;

    foreach my $e ( @{$elem_ar} ) {
        $self->add_build_element($e);
    }

    return;
}

sub search_files {

    my ($arg_r) = @_;
    my $sp      = $arg_r->{source_prefix};            # cgi
    my $se      = $arg_r->{source_extension};         # pl
    my $dp      = $arg_r->{destination_prefix};       # webcgi
    my $de      = $arg_r->{destination_extension};    # cgi
    my $verbose = exists $arg_r->{verbose} ? $arg_r->{verbose} : 0;    # 1|0

    #if ($verbose) {
    #    print "se [$se]\n";
    #    if ( defined $de ) {
    #        print "de [$de]\n";
    #    }
    #    print "sp [$sp]\n";
    #    print "dp [$dp]\n";
    #}

    my $glob   = $sp . '/*.' . $se;    # ('cgi/*.pl')
    my @files  = glob $glob;
    my %return = ();

    #if ($verbose) {
    #    print "glob [$glob]\n";
    #}

    foreach my $file (@files) {

        #if ($verbose) {
        #    print "file [$file]\n";
        #}

        my $destination = $file;
        if ( defined $de ) {
            $destination =~ s{[.]$se$}{.$de}smx;
        }
        $destination =~ s{^$sp}{$dp}smx;
        $return{$file} = $destination;
    }
    return \%return;
}

sub var_dump {

    # API
    my ( $self, $arg_r ) = @_;
    my $var_r
        = exists $arg_r->{var_r} ? $arg_r->{var_r} : $self->perr('var_r');
    my $name = exists $arg_r->{name} ? $arg_r->{name} : $self->perr('name');
    my $file = exists $arg_r->{file} ? $arg_r->{file} : 0;
    my $fh   = exists $arg_r->{fh}   ? $arg_r->{fh}   : *STDOUT;

    # dump
    my $c = Data::Dumper->new( [$var_r], ["*$name"] );
    $c->Varname($name);
    $c->Purity(1);
    $c->Deepcopy(1);

    my $dump = $c->Dump;

    if ( defined $file and $file and -w $file ) {

        my $msg = "Can not write to dump file [$file]!\n";
        open $fh, '>>', $file or croak $msg;
        print {$fh} $c->Dump or croak 'Can not dump to file handle!';
        close $fh or croak 'Close of dump was not possible!';

    }
    else {

        print $dump or croak 'Can not dump to STDOUT!';

    }

    return 1;
}

# special file treatment
# inspired by Module::Build::Base::process_script_files
# bin750
sub find_bin750_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'bin750' } );
}

sub process_bin750_files {

    my $self = shift;

    $self->process_files_element( { element => 'bin750' } );

    return;
}

# bin755
sub find_bin755_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'bin755' } );
}

sub process_bin755_files {

    my $self = shift;

    $self->process_files_element( { element => 'bin755' } );

    return;
}

# sbin750
sub find_sbin750_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'sbin750' } );
}

sub process_sbin750_files {

    my $self = shift;

    $self->process_files_element( { element => 'sbin750' } );

    return;
}

# sbin755
sub find_sbin755_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'sbin755' } );
}

sub process_sbin755_files {

    my $self = shift;

    $self->process_files_element( { element => 'sbin755' } );

    return;
}

# webcgi
sub find_webcgi_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'webcgi' } );
}

sub process_webcgi_files {

    my $self = shift;

    $self->process_files_element( { element => 'webcgi' } );

    return;
}

# initscript
sub find_initscript_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'initscript' } );
}

sub process_initscript_files {

    my $self = shift;

    $self->process_files_element( { element => 'initscript' } );

    return;
}

# conf644
sub find_conf644_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'conf644' } );
}

sub process_conf644_files {

    my $self = shift;

    $self->process_files_element( { element => 'conf644' } );

    return;
}

# conf600
sub find_conf600_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'conf600' } );
}

sub process_conf600_files {

    my $self = shift;

    $self->process_files_element( { element => 'conf600' } );

    return;
}

# bootcfg (Perl and other bootstrap configuration)
sub find_bootcfg_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'bootcfg' } );
}

sub process_bootcfg_files {

    my $self = shift;

    $self->process_files_element( { element => 'bootcfg' } );

    return;
}

# bootcfgstorage (Perl and other bootstrap configuration for storage layer)
sub find_bootcfgstorage_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'bootcfgstorage' } );
}

sub process_bootcfgstorage_files {

    my $self = shift;

    $self->process_files_element( { element => 'bootcfgstorage' } );

    return;
}

# bootcfgaccess (Perl and other bootstrap configuration for access layer)
sub find_bootcfgaccess_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'bootcfgaccess' } );
}

sub process_bootcfgaccess_files {

    my $self = shift;

    $self->process_files_element( { element => 'bootcfgaccess' } );

    return;
}

# bootcfgobject (Perl and other bootstrap configuration for object layer)
sub find_bootcfgobject_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'bootcfgobject' } );
}

sub process_bootcfgobject_files {

    my $self = shift;

    $self->process_files_element( { element => 'bootcfgobject' } );

    return;
}

# bootcfgtask (Perl and other bootstrap configuration for task layer)
sub find_bootcfgtask_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'bootcfgtask' } );
}

sub process_bootcfgtask_files {

    my $self = shift;

    $self->process_files_element( { element => 'bootcfgtask' } );

    return;
}

# locale
sub find_locale_files {

    my $self = shift;

    return $self->find_files_by_element( { element => 'locale' } );
}

sub process_locale_files {

    my $self = shift;

    $self->process_files_element( { element => 'locale' } );

    return;
}

###
sub find_files_by_element {

    my ( $self, $arg_r ) = @_;
    my $element = $arg_r->{element};

    my $dir = $cipux_include{ $element . '_files' }->{source_dir};    # 'cgi'
    my $ext = $cipux_include{ $element . '_files' }
        ->{source_ext};    # qr{[.]pl$}smx,
    return if not -d $dir;
    my $files_ar = $self->rscan_dir( $dir, $ext );

    #return { map { $_, $_ } @{$files_ar} };
    return { map { $_ => $_ } @{$files_ar} };
}
###
sub process_files_element {

    #my ( $self, $arg_r ) = (@_);
    my ( $self, $arg_r ) = @_;
    my $element = $arg_r->{element};    # 'sbin755'
    my $prop    = $self->_get_prop();

    my $method = "find_${element}_files";
    my $files = $self->$method( { element => $element } );

    # inspired by Module::Build::Base::process_script_files
    return if not keys %{$files};

    my $script_dir = File::Spec->catdir( $self->blib, $element );
    File::Path::mkpath($script_dir);

    foreach my $file ( keys %{$files} ) {
        next if $file =~ m/[.]svn/smx;

        # treat *.po files different
        if ( $file =~ m{^po/(\w+)[.]po$}smx ) {
            my $locale = $1;

            # scan for the translation realm
            my $pot_ar = $self->rscan_dir( 'po', qr{[.]pot$}smx );
            die "Count of *.pot files to high in po/!\n"
                if scalar( @{$pot_ar} ) > 1;
            my $pot = $pot_ar->[0];
            $pot =~ s/[.]pot$//msx;
            $pot =~ s{^po/}{}msx;     # remove ^po/

            # file: po/de_DE.po
            my $dest = "$pot.mo";     # cipux-cat-web.mo

            #/usr/share/locale/de_DE/LC_MESSAGES/cipux-cat-web.mo
            # locale -> /usr/share/locale
            #$dest =~ s{^po/}{$pot}msx; # ^po/ -> cipux-cat-web
            my $path = File::Spec->catdir( $self->blib, 'locale', $locale,
                'LC_MESSAGES' );
            File::Path::mkpath($path);
            my $to = File::Spec->catfile( $path, $dest );
            next if $self->up_to_date( $file, $to );    # Already fresh

            $self->do_system("msgfmt -o $to $file");

            next;    # *po, *.mo files need not be copied, or chmod
        }
        else {

            my $result = undef;

            # change extension in some cases
            if ( exists $cipux_include{ $element . '_files' }->{build_ext} ) {

                # we will not copy if not defined under *_files
                next if not exists $prop->{ $element . '_files' }->{$file};

                # we can also not copy if we have no target
                next if not defined $prop->{ $element . '_files' }->{$file};

                my $dest = $prop->{ $element . '_files' }->{$file};

                $result = $self->copy_if_modified(
                    from => $file,
                    to   => File::Spec->catfile( $self->blib, $dest )
                ) or next;

            }
            else {

                $result
                    = $self->copy_if_modified( $file, $script_dir, 'flatten' )
                    or next;

            }

            # $self->fix_shebang_line($result) unless $self->is_vmsish;
            # make some executable
            if ( exists $cipux_include{ $element . '_files' }->{build_exec} )
            {
                $self->make_executable($result);
            }

            # change some mode
            if ( exists $cipux_include{ $element . '_files' }->{build_mode} )
            {
                my $mode
                    = $cipux_include{ $element . '_files' }->{build_mode};
                chmod $mode, $result;
            }
        }
    }

    return;

}

1;    # Magic true value required at end of module

__END__

=pod

=head1 NAME

Module::Build::CipUX - Extension for building CipUX


=head1 VERSION

This document describes Module::Build::CipUX version 0.4.0


=head1 SYNOPSIS

    use Module::Build::CipUX;

    my $build = Module::Build::CipUX->new(
        module_name     =>  'xxxx',
        license         =>  'gpl',
        ...
        cipux_paths       =>  {
            'schema' =>  q(/etc/ldap/schema),
            'setup'  =>  q(/etc/cipux/setup),
        },
        # cipux_schema   =>  [ glob('extras/*.schema') ],
        # cipux_setup =>  [ glob('extras/*.setup') ],

        # build target
        schema_files => {
             'etc/ldap/schema/cipux.schema' => 'schema/cipux.schema',
        },
        setup_files => {
            'etc/cipux/setup/courier.schema'        => 'setup/courier.schema',
            'etc/cipux/setup/debian-cipux.conf'     => 'setup/debian-cipux.conf',
            'etc/cipux/setup/debian-edu-cipux.conf' => 'setup/debian-edu-cipux.conf',
            'etc/cipux/setup/debian-edu-ldap.acl'   => 'setup/debian-edu-ldap.acl',
            'etc/cipux/setup/debian-edu-ldap.ldif'  => 'setup/debian-edu-ldap.ldif',
            'etc/cipux/setup/debian-ldap.acl'       => 'setup/debian-ldap.acl',
            'etc/cipux/setup/debian-ldap.ldif'      => 'setup/debian-ldap.ldif',
            'etc/cipux/setup/lis.schema'            => 'setup/lis.schema',
        },

                );

    my $elem_ar = $builder->get_all_elements;
    foreach my $e (@{$elem_ar}){
        $builder->add_build_element($e);
    }

    $build->create_build_script();


=head1 DESCRIPTION

The goal of this module is build and install CipUX plugins in Perl and also
handle CipUX core modules, subclassing the Module::Build and adding some extra
funtionalities to it.

For a description of the interface see Module::Build::API.

This is a list of new parameters in the Module::Build::new method:

cipux_paths

    Define the install paths of the components using a hash with the following keys:

    setup

        The default value is /etc/cipux/setup.

    schema

        The default value is /etc/ldap/schema.

cipux_setup

    List of setup files to install.

cipux_schema

    List of LDAP schema files to install.


=head1 INTERFACE


=over

=item cipuxrelease

  [version 0.1.0]

  Make a cipux release tarball

=back


=head1 CONSTRUCTOR


=head2 new

New inscance of Module::Build::CipUX.

  my $build = Module::Build::CipUX->new();

=head1 SUBROUTINES/METHODS


=head2 ACTION_cipuxpo2mo( @params )

Build action which converts *.po files to *.mo files using gettext.


=head2 ACTION_cipuxrelease( @params )

Build a simple CipUX release tarball.


=head2 _get_prop( $self )

Returns private properties from the hash in Build.PL. This was inspired
by Module::Build::IkiWiki. This violates ProhibitAccessOfPrivateData.
If someone knows how to do better, drop a mail to cipux-devel mailing list.


=head2 $cipux_elements_ar = get_all_elements()

Retrieve all elements and return a array reference to it.

 $cipux_elements_ar = get_all_elements();


=head2 add_all_elements()

Call get_all_elements and add all elements to the build list by add_build_element.


=head2 $results_hr = search_files({ PARAM })

PARAM:
    source_prefix              # cgi
    source_extension           # pl
    destination_prefix         # webcgi
    destination_extension      # cgi
    verbose                    # 1|0

Search a given directory for files of given extension, change the extension
and construct a new path with the file name. It returns a hash reference.


=head2 var_dump({ PARAM })

PARAM:
    var_r    (mandatory)                          reference to variable
    name     (mandatory)                          name of variable
    file     (optional)                           filename if wish is to dump to file
    fh       (optional) if not defined *STDOUT;   other file handle or handle

Dump a variable to STDOUT. Or if parameter is given, dump it to a file or to
another location given by a handle.


=head2 $files_hr = find_bin750_files()

Find all files for mode 750 and return hash reference.  Uses
find_files_by_element.


=head2 process_bin750_files()

Process files.


=head2 $files_hr = find_bin755_files()

Find all files for mode 755 and return hash reference.  Uses
find_files_by_element.

=head2 process_bin755_files

Process files.


=head2 $files_hr = find_sbin750_files()

Find all files for mode 750 and return hash reference.  Uses
find_files_by_element.


=head2 process_sbin750_files

Process files.


=head2 $files_hr = find_sbin755_files()

Find all files for mode 755 and return hash reference.  Uses
find_files_by_element.


=head2 process_sbin755_files

Process files.


=head2 $files_hr = find_webcgi_files()

Find all files for element webcgi and return hash reference.  Uses
find_files_by_element.


=head2 process_webcgi_files

Process files.


=head2 $files_hr = find_initscript_files()

Find all files for element initscript and return hash reference.  Uses
find_files_by_element.


=head2 process_initscript_files

Process files.


=head2 $files_hr = find_conf644_files()

Find all files for element conf644 and return hash reference.  Uses
find_files_by_element.


=head2 process_conf644_files

Process files.


=head2 $files_hr = find_conf600_files()

Find all files for element conf600 and return hash reference.  Uses
find_files_by_element.


=head2 process_conf600_files

Process files.

=head2 find_bootcfg_files

Find files.

=head2 process_bootcfg_files

Process files.

=head2 find_bootcfgobject_files

Find files.

=head2 find_bootcfgstorage_files

Find files.

=head2 find_bootcfgaccess_files

Find files.

=head2 find_bootcfgtask_files

Find files.

=head2 process_bootcfgobject_files

Process files.

=head2 process_bootcfgstorage_files

Process files.

=head2 process_bootcfgaccess_files

Process files.

=head2 process_bootcfgtask_files

Process files.

=head2 $files_hr = find_locale_files()

Find all files for element locale and return hash reference.  Uses
find_files_by_element.


=head2 process_locale_files

Process files.


=head2 $files_hr|undef = find_files_by_element( { element=>$element } )

Add $element . '_files' to include list.  Returns reference to hash of
all files in that directory if the directory exists; otherwise returns
undef


=head2 @files = process_files_element( { element=>$element } )

Execute find_${elelemt}_files method.  Returns list of files
for trivial files. For non-trivial files it does...:

General:

 make a path
 skip .svn dirs
 change extension for some files, copy otherwise
 make some files executable
 change some modes

PO:

 check number of *.pot files
 identify the *.pot file
 build file name from *.pot file for *.mo file
 build *.mo file from *.po file with msgfmt


=head1 DIAGNOSTICS

Error messages are from the base class. This package generates no exceptions.

=over

=item C<< Couldn't mkdir $podir: $!";  >>

Occurs if it is not possible to make a directory in blib/po

=back


=head1 CONFIGURATION AND ENVIRONMENT

Module::Build::CipUX requires no configuration files or environment variables.


=head1 DEPENDENCIES

Module::Build


=head1 INCOMPATIBILITIES

None reported.


=head1 BUGS AND LIMITATIONS

No bugs have been reported.

Please report any bugs or feature requests to
C<bug-module-build-cipux@rt.cpan.org>, or through the web interface at
L<http://rt.cpan.org>.


=head1 AUTHOR

Christian Kuelker  C<< <christian.kuelker@cipworx.org> >>


=head1 LICENSE AND COPYRIGHT

Copyright (C) 2008 - 2009, Christian Kuelker
C<< <christian.kuelker@cipworx.org> >>.  All rights reserved.

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License or any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
Street, Fifth Floor, Boston, MA 02110-1301 USA.


=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY FOR THE
SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN OTHERWISE
STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES PROVIDE THE
SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS FOR A PARTICULAR PURPOSE. THE ENTIRE RISK AS TO THE QUALITY AND
PERFORMANCE OF THE SOFTWARE IS WITH YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE,
YOU ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING WILL ANY
COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR REDISTRIBUTE THE
SOFTWARE AS PERMITTED BY THE ABOVE LICENCE, BE LIABLE TO YOU FOR DAMAGES,
INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
OUT OF THE USE OR INABILITY TO USE THE SOFTWARE (INCLUDING BUT NOT LIMITED TO
LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR
THIRD PARTIES OR A FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE),
EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
DAMAGES.

=cut
