/* Ubuntu TV Media Scanner
 * Copyright (C) 2012 Canonical Ltd. - All Rights Reserved
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 */

// GLib related libraries
#include <grilo.h>

// Boost C++
#include <boost/algorithm/string.hpp>
#include <boost/bind.hpp>
#include <boost/foreach.hpp>

// C++ Standard Library
#include <algorithm>
#include <map>
#include <set>
#include <string>

// Media Scanner Library
#include "mediascanner/locale.h"
#include "mediascanner/mediaindex.h"
#include "mediascanner/propertyschema.h"
#include "mediascanner/settings.h"

namespace mediascanner {
namespace internal {

class GenerateSchemaDox {
public:
    GenerateSchemaDox();

    int Run();

    struct less_by_name {
        bool operator()(const Property &a, const Property &b) const {
            return a.metadata_key().name() < b.metadata_key().name();
        }
    };

private:
    static std::string gtype_name(GType gtype);

    static std::wstring category_name(Property::Category category);
    static std::wstring category_title(Property::Category category);

    bool collect_property(const Property &prop);
    void print_dox(const mediascanner::Property &prop);

    std::wostream& out_;

    typedef std::set<Property, less_by_name> PropertySet;
    typedef std::map<Property::Category, PropertySet> CategoryMap;
    CategoryMap categories_;
};

GenerateSchemaDox::GenerateSchemaDox()
    : out_(std::wcout) {
}

std::string GenerateSchemaDox::gtype_name(GType type) {
    const char* name = g_type_name(type);

    if (!name)
        return "Unknown type";

    if (type == G_TYPE_STRING)
        return "String (" + std::string(name) + ")";

    return name;
}

std::wstring GenerateSchemaDox::category_name(Property::Category category) {
    std::wstring name = category_title(category);
    boost::algorithm::to_lower(name);
    return name;
}

std::wstring GenerateSchemaDox::category_title(Property::Category category) {
    if (category == Property::Category::Generic)
        return L"Generic";
    if (category == Property::Category::File)
        return L"File";
    if (category == Property::Category::Media)
        return L"Media";
    if (category == Property::Category::Music)
        return L"Music";
    if (category == Property::Category::Image)
        return L"Image";
    if (category == Property::Category::Photo)
        return L"Photo";
    if (category == Property::Category::Movie)
        return L"Movie";

    return L"Other";
}

bool GenerateSchemaDox::collect_property(const Property &prop) {
    categories_[prop.category()].insert(prop);
    return false;
}

static std::wstring to_string(Property::MergeStrategy strategy) {
    switch (strategy) {
    case Property::MergeAppend:
        return L"appending the new value";
    case Property::MergeReplace:
        return L"replacing the old value with the new value";
    case Property::MergePreserve:
        return L"ignoring the new value and preserving the old one";
    }

    return L"unknown strategy";
}

void GenerateSchemaDox::print_dox(const Property &prop) {
    const Property::MetadataKey& key = prop.metadata_key();

    const std::string name = key.name();
    out_ << "- @anchor " << name.c_str()
         << " Key: <b>@c " << name.c_str() << "</b>" << std::endl
         // (The convention is for GObject property descriptions to
         // have no . at the end, so we add one.
         << "  - Description: " << key.description().c_str() << "." << std::endl
         << "  - Type: @c " << gtype_name(key.gtype()).c_str() << std::endl
         << "  - Stored in Lucene as @c \"" << prop.field_name() << "\"."
         << std::endl
         << "  - Merge conflicts are resolved by "
         << to_string(prop.merge_strategy()) << "." << std::endl;

    out_ << "  - Supports "
         << "@ref grl-mediascanner-range-filters \"range filters\"."
         << std::endl;

    if (prop.supports_full_text_search()) {
        out_ << "  - Covered by "
             << "@ref grl-mediascanner-full-text-search \"full text search\"."
             << std::endl;
    }

    const std::set<std::string> origins = prop.origins();

    if (not origins.empty()) {
        std::set<std::string>::const_iterator it = origins.begin();

        out_ << "  - Retrieved from " << it->c_str();

        while (++it != origins.end())
            out_ << ", " << it->c_str();

        out_ << "." << std::endl;
    } else {
        std::wcerr << "Warning: No origin for " << prop.field_name()
                   << std::endl;
    }

    const GList *const relation = prop.metadata_key().relation();

    if (relation && relation->next) {
        bool separator_needed = false;
        out_ << "  - Related to ";

        for (const GList *l = relation; l; l = l->next) {
            const GrlKeyID related_key = GRLPOINTER_TO_KEYID(l->data);

            if (related_key == prop.metadata_key().id())
                continue;

            if (separator_needed) {
                out_ << ", ";
            } else {
                separator_needed = true;
            }

            out_ << "@ref " << grl_metadata_key_get_name(related_key);
        }

        out_ << "." << std::endl;
    }
}

int GenerateSchemaDox::Run() {
    ::setenv("XDG_DATA_DIRS", USER_DATA_DIR ":/usr/share", true);

    Settings settings;
    settings.LoadMetadataSources();

    Property::VisitAll(boost::bind(&GenerateSchemaDox::collect_property,
                                   this, _1));

    out_ << "/**" << std::endl
         << "\n" << std::endl
         << "@page properties-schema Properties Schema" << std::endl
         << std::endl
         << "These are the fields that may be used with the @ref grilo-plugin, "
         << "for instance with "
         << "<a href=\"http://developer.gnome.org/grilo/unstable/GrlSource.html"
         << "#grl-source-query\">"
         << "grl_source_query()</a>."
         << std::endl
         << "@tableofcontents"
         << std::endl << std::endl;

    for (const auto &p: categories_) {
        const Property::Category category = p.first;
        const PropertySet &properties = p.second;

        out_ << "@section properties-" << category_name(category)
             << " " << category_title(category) << " Properties"
             << std::endl << std::endl;

        std::for_each(properties.begin(), properties.end(),
                      boost::bind(&GenerateSchemaDox::print_dox, this, _1));

        out_ << std::endl;
    }

    out_ << "*/" << std::endl;

    return EXIT_SUCCESS;
}

} // namespace internal
} // namespace mediascanner

int main(int argc, char *argv[]) {
    mediascanner::SetupLocale();
    grl_init(&argc, &argv);

    return mediascanner::internal::GenerateSchemaDox().Run();
}
