/**************************************************************************
*   Copyright (C) 2007-2010 by Thomas Thelliez aka jblud                  *
*   Contact : <admin.kontrol@gmail.com>                                   *
*                                                                         *
* This program is free software; you can redistribute it and/or           *
* modify it either under the terms of the GNU Lesser General Public       *
* License version 3 as published by the Free Software Foundation          *
* (the "LGPL") or, at your option, any later version.                     *
* If you do not alter this notice, a recipient may use your version       *
* of this file under the LGPL.                                            *
*                                                                         *
* You should have received a copy of the LGPL along with this library     *
* in the file COPYING; if not, write to the Free Software                 *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA *
*                                                                         *
* This software is distributed on an "AS IS" basis, WITHOUT WARRANTY      *
* OF ANY KIND, either express or implied. See the LGPL for                *
* the specific language governing rights and limitations.                 *
**************************************************************************/

#include "auth_server.h"

auth_server::auth_server(
        QString server_password,
        QObject *parent) :
        password(server_password),
        QTcpServer(parent)
{
    client_list = new QMap<int, auth_session*>();
    QObject::connect(this,
                     SIGNAL(newConnection()),
                     this, SLOT(add_new_client()));
    if (server_password.length() > 30)
        emit wrong_password("Password too long. Password should be between 8 and 30 digits.");
    nbr_client = 0;
}

void auth_server::add_new_client()
{
    nbr_client++;
    QTcpSocket *socket = this->nextPendingConnection();
    auth_session *remote_client =
            new auth_session(socket,
                             nbr_client, password, this);
    QObject::connect(socket,
                     SIGNAL(readyRead()), remote_client,
                     SLOT(read_results()));
    QObject::connect(remote_client,
                     SIGNAL(emit_tcp_state(int,int)), this,
                     SLOT(tcp_state(int,int)));
    QObject::connect(remote_client,
                     SIGNAL(emit_error(int)), this,
                     SLOT(auth_error_handler(int)));
    QObject::connect(remote_client,
                     SIGNAL(auth_suceeded(int)), this,
                     SLOT(auth_suceeded(int)));
    QObject::connect(remote_client,
                     SIGNAL(received_stream(QString, int)), this,
                     SLOT(received_stream_slot(QString, int)));
    QObject::connect(remote_client,
                     SIGNAL(remove_client(int)), this,
                     SLOT(remove_client(int)));
    client_list->insert(nbr_client, remote_client);

    // Start the authentification process
    // sending TOKEN to remote client.
    remote_client->handle_crypted_auth();
}

int auth_server::write_encrypted_data_to_socket(char* buffer, int nbr)
{
    if (client_list->contains (nbr)) {
        auth_session *client =
                client_list->value(nbr);
        if (client->auth_status == AUTH_SUCCESS) {
            client->write_data_to_socket(client->auth_aes_encrypt(buffer).toUtf8().data());
            client->get_socket()->flush();
            return 1;
        }
    }
    return 0;
}

int auth_server::write_data_to_socket(char* buffer, int nbr)
{
    if (client_list->contains (nbr)) {
        auth_session *client =
                client_list->value(nbr);
        client->get_socket()->write(QString(buffer).toUtf8());
        client->get_socket()->flush();
        return 1;
    }
    return 0;
}

QString auth_server::encrypt_message(char* buffer, int nbr)
{
    if (client_list->contains (nbr)) {
        auth_session *client =
                client_list->value(nbr);
        return client->auth_aes_encrypt(buffer);
    }
}

void auth_server::disconnect_client(int nbr)
{
    if (client_list->contains(nbr)) {
        auth_session *client =
                client_list->value(nbr);
        client->get_socket()->close();
        client->get_socket()->disconnect();
        this->remove_client(nbr);
    }
}

void auth_server::remove_client(int nbr)
{
    if (client_list->contains(nbr)) {
        client_list->remove(nbr);
        nbr_client--;
    }
}

int auth_server::stop_server()
{
    nbr_client = 0;
    if (!client_list->empty()) {
        for (int j = 0; j <= 100; ++j) {
            if (client_list->contains(j)) {
                auth_session *client =
                        client_list->value(j);
                client->get_socket()->close();
                client->get_socket()->disconnect();
                remove_client(j);
            }
        }
    }
    close();
    this->close();
    this->disconnect();
    return 0;
}

/*
    #define TOKEN_WELCOME 0
    #define RSA_PUBLIC_KEY_EXCHANGE 1
    #define PASSWORD_VALIDATION 2
    #define AES_KEY_RECEPTION 3
*/
void auth_server::tcp_state(int state, int client_nbr)
{
    switch (state)
    {
    case TOKEN_WELCOME : {
            qDebug("Server notification : Status TOKEN_WELCOME");
            emit signals_state(state, client_nbr);
            break;
        }
    case RSA_PUBLIC_KEY_RECEPTION : {
            qDebug("Server notification : Status RSA_PUBLIC_KEY_EXCHANGE");
            emit signals_state(state, client_nbr);
            break;
        }
    case PASSWORD_VALIDATION : {
            qDebug("Server notification : Status PASSWORD_VALIDATION");
            emit signals_state(state, client_nbr);
            break;
        }
    case AES_KEY_RECEPTION : {
            qDebug("Server notification : Status AES_KEY_RECEPTION");
            emit signals_state(state, client_nbr);
            break;
        }
    case 4 : {
            qDebug("Server notification : Status 4");
            emit signals_state(state, client_nbr);
            break;
        }
    case 5 : {
            qDebug("Server notification : Status 5");
            emit signals_state(state, client_nbr);
            break;
        }
    default:
        ;
    }
}

/*
Connect the app to the received_stream_signal
to be aware of reception and handle streams.
*/
void auth_server::received_stream_slot(QString message, int client_nbr)
{
    emit received_stream_signal(message, client_nbr);
}

void auth_server::auth_error_handler(int client_nbr)
{
    qDebug("Server notification : Auth error");
    this->remove_client(client_nbr);
    emit emit_error(client_nbr);
}

/*
Connect to the auth_succeeded
to be aware of new client authentification state.
*/
void auth_server::auth_suceeded(int client_nbr)
{
    qDebug("Server notification : Auth suceeded");
    emit auth_succeeded(client_nbr);
}

int auth_server::get_client_number()
{
    return client_list->size();
}

QMap<int, auth_session*>* auth_server::get_client_list()
{
    return client_list;
}

