/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#include "common.h"
#include "types.h"
#include "modules.h"
#include "bitops.h"
#include "convert.h"
#include "shared.h"
#include "memory.h"

static const u32   ATTACK_EXEC    = ATTACK_EXEC_OUTSIDE_KERNEL;
static const u32   DGST_POS0      = 0;
static const u32   DGST_POS1      = 1;
static const u32   DGST_POS2      = 2;
static const u32   DGST_POS3      = 3;
static const u32   DGST_SIZE      = DGST_SIZE_4_16;
static const u32   HASH_CATEGORY  = HASH_CATEGORY_FDE;
static const char *HASH_NAME      = "LUKS v1 SHA-256 + AES";
static const u64   KERN_TYPE      = 14621; // old kernel used here
static const u32   OPTI_TYPE      = OPTI_TYPE_ZERO_BYTE
                                  | OPTI_TYPE_SLOW_HASH_SIMD_LOOP;
static const u64   OPTS_TYPE      = OPTS_TYPE_STOCK_MODULE
                                  | OPTS_TYPE_PT_GENERATE_LE;
static const u32   SALT_TYPE      = SALT_TYPE_EMBEDDED;
static const char *ST_PASS        = "hashcat";
static const char *ST_HASH        = "$luks$1$sha256$aes$cbc-plain64$128$92180$10f2691b0afce5da52a9fa18b03d288cc4e3e17cca1efcf85d578fd8b4ec65fa$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$d979dbd38edbf4c80312b4ab25460b69930e7fd98f8830af673adc8e318dd832bfc9845b46eb2f0551ed497cfeb92fe1e59660122f14205e96759cee59eaf714e1476c00b85b53bca9e7e9aa9ddf4ce0c1e9662313ce7a1787c43d188735bf94d32306a66bc65a2db6d770f6511992536c70bafd7df8eaa03c5baacdaa72843001fe4a56a75a696badb52fbedc5d09d951a2f71d5be7b8c368388a5907731b23b7edfabe247169941ca6d7d6686192d07c0f5d24c8da38e3716afd79c5251ea7adfc8cdf99a0bf2bee6564e32f4ca72862797d07001785586d38ea7ed5474a5e3cc9320aeea1c089422e98d443ac9f8c8ca63cc6b064c3dca00ba6f3892a8bf3f466cc8e3a98fb0c889c6f8337e7b84148f58ed4337e694f00e383286b6bc0d84d02d652134387b47c9b36ec0fc9f524b870a64162303c20a46c1e1ab9cb510ebb385292d32ae6774e9759d5641121ea9c4420bf74219a822e282a1313e0d7b24fd7761a8d6b53b565b7f9ef45766b8d1dbe24454167c42d4eb3a14c0b22270d8ea63fdd34b950d3ce696a3ea197f40542c792dd446fc52fc72aca851f50e03b5292274e095217df717bd411bde787f5bd4974e94ff6928ea4940caee91ec91da2d14b060cb577fe320c899b90d0f35ae02ff861881a4308ee0c5bcb9ee2e159e8c4f8c807916d9e40151e82915716237025deb2f13b0f69383fa97dd9ac3f5c";

u32         module_attack_exec    (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ATTACK_EXEC;     }
u32         module_dgst_pos0      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS0;       }
u32         module_dgst_pos1      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS1;       }
u32         module_dgst_pos2      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS2;       }
u32         module_dgst_pos3      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_POS3;       }
u32         module_dgst_size      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return DGST_SIZE;       }
u32         module_hash_category  (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_CATEGORY;   }
const char *module_hash_name      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return HASH_NAME;       }
u64         module_kern_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return KERN_TYPE;       }
u32         module_opti_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTI_TYPE;       }
u64         module_opts_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return OPTS_TYPE;       }
u32         module_salt_type      (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return SALT_TYPE;       }
const char *module_st_hash        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_HASH;         }
const char *module_st_pass        (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra) { return ST_PASS;         }

#define LUKS_STRIPES        (                                   4000)
#define LUKS_SALT_LEN       (                                     32)
#define LUKS_SALT_HEX_LEN   (                      LUKS_SALT_LEN * 2)
#define LUKS_CT_LEN         (                                    512)
#define LUKS_CT_HEX_LEN     (                        LUKS_CT_LEN * 2)
#define LUKS_AF_MIN_LEN     (HC_LUKS_KEY_SIZE_128 / 8 * LUKS_STRIPES)
#define LUKS_AF_MIN_HEX_LEN (                    LUKS_AF_MIN_LEN * 2)
#define LUKS_AF_MAX_LEN     (HC_LUKS_KEY_SIZE_512 / 8 * LUKS_STRIPES)
#define LUKS_AF_MAX_HEX_LEN (                    LUKS_AF_MAX_LEN * 2)

typedef enum hc_luks_hash_type
{
  HC_LUKS_HASH_TYPE_SHA1      = 1,
  HC_LUKS_HASH_TYPE_SHA256    = 2,
  HC_LUKS_HASH_TYPE_SHA512    = 3,
  HC_LUKS_HASH_TYPE_RIPEMD160 = 4,
  HC_LUKS_HASH_TYPE_WHIRLPOOL = 5,

} hc_luks_hash_type_t;

typedef enum hc_luks_key_size
{
  HC_LUKS_KEY_SIZE_128 = 128,
  HC_LUKS_KEY_SIZE_256 = 256,
  HC_LUKS_KEY_SIZE_512 = 512,

} hc_luks_key_size_t;

typedef enum hc_luks_cipher_type
{
  HC_LUKS_CIPHER_TYPE_AES     = 1,
  HC_LUKS_CIPHER_TYPE_SERPENT = 2,
  HC_LUKS_CIPHER_TYPE_TWOFISH = 3,

} hc_luks_cipher_type_t;

typedef enum hc_luks_cipher_mode
{
  HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256 = 1,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN        = 2,
  HC_LUKS_CIPHER_MODE_CBC_PLAIN64      = 3,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN        = 4,
  HC_LUKS_CIPHER_MODE_XTS_PLAIN64      = 5,

} hc_luks_cipher_mode_t;

typedef struct luks
{
  int hash_type;   // hc_luks_hash_type_t
  int key_size;    // hc_luks_key_size_t
  int cipher_type; // hc_luks_cipher_type_t
  int cipher_mode; // hc_luks_cipher_mode_t

  u32 ct_buf[LUKS_CT_LEN / 4];

  u32 af_buf[LUKS_AF_MAX_LEN / 4];
  u32 af_len;

} luks_t;

typedef struct luks_tmp
{
  u32 ipad32[8];
  u64 ipad64[8];

  u32 opad32[8];
  u64 opad64[8];

  u32 dgst32[32];
  u64 dgst64[16];

  u32 out32[32];
  u64 out64[16];

} luks_tmp_t;

static const char *SIGNATURE_LUKS = "$luks$1$sha256$aes$";

bool module_unstable_warning (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra, MAYBE_UNUSED const hc_device_param_t *device_param)
{
  if ((device_param->opencl_platform_vendor_id == VENDOR_ID_APPLE) && (device_param->opencl_device_type & CL_DEVICE_TYPE_GPU))
  {
    if (device_param->is_metal == true)
    {
      if (strncmp (device_param->device_name, "Intel", 5) == 0)
      {
        // Intel Iris Graphics, Metal Version 244.303: failed to create 'm14621_init' pipeline, timeout reached
        return true;
      }
    }
  }

  return false;
}

u64 module_esalt_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 esalt_size = (const u64) sizeof (luks_t);

  return esalt_size;
}

u64 module_tmp_size (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const user_options_t *user_options, MAYBE_UNUSED const user_options_extra_t *user_options_extra)
{
  const u64 tmp_size = (const u64) sizeof (luks_tmp_t);

  return tmp_size;
}

int module_hash_decode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED void *digest_buf, MAYBE_UNUSED salt_t *salt, MAYBE_UNUSED void *esalt_buf, MAYBE_UNUSED void *hook_salt_buf, MAYBE_UNUSED hashinfo_t *hash_info, const char *line_buf, MAYBE_UNUSED const int line_len)
{
  luks_t *luks = (luks_t *) esalt_buf;

  hc_token_t token;

  memset (&token, 0, sizeof (hc_token_t));

  token.token_cnt  = 7;

  token.signatures_cnt    = 1;
  token.signatures_buf[0] = SIGNATURE_LUKS;

  // signature with hash and cipher type
  token.len[0]     = 19;
  token.attr[0]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_SIGNATURE;

  // cipher mode
  token.sep[1]     = '$';
  token.len_min[1] = 9;
  token.len_max[1] = 16;
  token.attr[1]    = TOKEN_ATTR_VERIFY_LENGTH;

  // key size
  token.sep[2]     = '$';
  token.len[2]     = 3;
  token.attr[2]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // iter
  token.sep[3]     = '$';
  token.len_min[3] = 1;
  token.len_max[3] = 10;
  token.attr[3]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_DIGIT;

  // salt
  token.sep[4]     = '$';
  token.len[4]     = LUKS_SALT_HEX_LEN;
  token.attr[4]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // af
  token.sep[5]     = '$';
  token.len_min[5] = LUKS_AF_MIN_HEX_LEN;
  token.len_max[5] = LUKS_AF_MAX_HEX_LEN;
  token.attr[5]    = TOKEN_ATTR_VERIFY_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  // ct
  token.sep[6]     = '$';
  token.len[6]     = LUKS_CT_HEX_LEN;
  token.attr[6]    = TOKEN_ATTR_FIXED_LENGTH
                   | TOKEN_ATTR_VERIFY_HEX;

  const int rc_tokenizer = input_tokenizer ((const u8 *) line_buf, line_len, &token);

  if (rc_tokenizer != PARSER_OK) return (rc_tokenizer);

  // hash type

  luks->hash_type = HC_LUKS_HASH_TYPE_SHA256;

  // cipher type

  luks->cipher_type = HC_LUKS_CIPHER_TYPE_AES;

  // cipher mode

  const u8 *cipher_mode_pos = token.buf[1];
  const u32 cipher_mode_len = token.len[1];

  if ((strncmp ((const char *) cipher_mode_pos, "cbc-essiv:sha256", 16) == 0) && (cipher_mode_len == 16))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "cbc-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_CBC_PLAIN64;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain",    9) == 0) && (cipher_mode_len ==  9))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN;
  }
  else if ((strncmp ((const char *) cipher_mode_pos, "xts-plain64", 11) == 0) && (cipher_mode_len == 11))
  {
    luks->cipher_mode = HC_LUKS_CIPHER_MODE_XTS_PLAIN64;
  }
  else
  {
    return (PARSER_LUKS_CIPHER_MODE);
  }

  // key size

  const u8 *key_size_pos = token.buf[2];

  const u32 key_size = hc_strtoul ((const char *) key_size_pos, NULL, 10);

  if (key_size == 128)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_128;
  }
  else if (key_size == 256)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_256;
  }
  else if (key_size == 512)
  {
    luks->key_size = HC_LUKS_KEY_SIZE_512;
  }
  else
  {
    return (PARSER_LUKS_KEY_SIZE);
  }

  // salt

  const u8 *salt_pos = token.buf[4];

  salt->salt_len = hex_decode (salt_pos, LUKS_SALT_HEX_LEN, (u8 *) salt->salt_buf);

  // iter

  const u8 *iter_pos = token.buf[3];

  const u32 iter = hc_strtoul ((const char *) iter_pos, NULL, 10);

  salt->salt_iter = iter - 1;

  // af

  u32 af_len = token.len[5] / 2;

  if (af_len != (key_size / 8) * LUKS_STRIPES) return (PARSER_LUKS_FILE_SIZE);

  const u8 *af_pos = token.buf[5];

  luks->af_len = hex_decode (af_pos, token.len[5], (u8 *) luks->af_buf);

  // ct

  u32 ct_len = token.len[6] / 2;

  if (ct_len != LUKS_CT_LEN) return (PARSER_LUKS_FILE_SIZE);

  const u8 *ct_pos = token.buf[6];

  hex_decode (ct_pos, LUKS_CT_HEX_LEN, (u8 *) luks->ct_buf);

  return (PARSER_OK);
}

int module_hash_encode (MAYBE_UNUSED const hashconfig_t *hashconfig, MAYBE_UNUSED const void *digest_buf, MAYBE_UNUSED const salt_t *salt, MAYBE_UNUSED const void *esalt_buf, MAYBE_UNUSED const void *hook_salt_buf, MAYBE_UNUSED const hashinfo_t *hash_info, char *line_buf, MAYBE_UNUSED const int line_size)
{
  const luks_t *luks = (const luks_t *) esalt_buf;

  // cipher mode

  const char *cipher_mode = "";

  switch (luks->cipher_mode)
  {
    case HC_LUKS_CIPHER_MODE_CBC_ESSIV_SHA256: cipher_mode = "cbc-essiv:sha256"; break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN:        cipher_mode = "cbc-plain";        break;
    case HC_LUKS_CIPHER_MODE_CBC_PLAIN64:      cipher_mode = "cbc-plain64";      break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN:        cipher_mode = "xts-plain";        break;
    case HC_LUKS_CIPHER_MODE_XTS_PLAIN64:      cipher_mode = "xts-plain64";      break;
    default:                                                                     return 0;
  }

  // key size

  u32 key_size = 0;

  switch (luks->key_size)
  {
    case HC_LUKS_KEY_SIZE_128: key_size = 128; break;
    case HC_LUKS_KEY_SIZE_256: key_size = 256; break;
    case HC_LUKS_KEY_SIZE_512: key_size = 512; break;
    default:                                   return 0;
  }

  // iterations

  u32 iterations = salt->salt_iter + 1;

  // salt

  char salt_buf[LUKS_SALT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) salt->salt_buf, LUKS_SALT_LEN, (u8 *) salt_buf);

  // af

  char af_buf[LUKS_AF_MAX_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->af_buf, luks->af_len, (u8 *) af_buf);

  // ct

  char ct_buf[LUKS_CT_HEX_LEN + 1] = { 0 };

  hex_encode ((const u8 *) luks->ct_buf, LUKS_CT_LEN, (u8 *) ct_buf);

  // output

  int line_len = snprintf (line_buf, line_size, "%s%s$%u$%u$%s$%s$%s",
    SIGNATURE_LUKS,
    cipher_mode,
    key_size,
    iterations,
    salt_buf,
    af_buf,
    ct_buf);

  return line_len;
}

void module_init (module_ctx_t *module_ctx)
{
  module_ctx->module_context_size             = MODULE_CONTEXT_SIZE_CURRENT;
  module_ctx->module_interface_version        = MODULE_INTERFACE_VERSION_CURRENT;

  module_ctx->module_attack_exec              = module_attack_exec;
  module_ctx->module_benchmark_esalt          = MODULE_DEFAULT;
  module_ctx->module_benchmark_hook_salt      = MODULE_DEFAULT;
  module_ctx->module_benchmark_mask           = MODULE_DEFAULT;
  module_ctx->module_benchmark_charset        = MODULE_DEFAULT;
  module_ctx->module_benchmark_salt           = MODULE_DEFAULT;
  module_ctx->module_bridge_name              = MODULE_DEFAULT;
  module_ctx->module_bridge_type              = MODULE_DEFAULT;
  module_ctx->module_build_plain_postprocess  = MODULE_DEFAULT;
  module_ctx->module_deep_comp_kernel         = MODULE_DEFAULT;
  module_ctx->module_deprecated_notice        = MODULE_DEFAULT;
  module_ctx->module_dgst_pos0                = module_dgst_pos0;
  module_ctx->module_dgst_pos1                = module_dgst_pos1;
  module_ctx->module_dgst_pos2                = module_dgst_pos2;
  module_ctx->module_dgst_pos3                = module_dgst_pos3;
  module_ctx->module_dgst_size                = module_dgst_size;
  module_ctx->module_dictstat_disable         = MODULE_DEFAULT;
  module_ctx->module_esalt_size               = module_esalt_size;
  module_ctx->module_extra_buffer_size        = MODULE_DEFAULT;
  module_ctx->module_extra_tmp_size           = MODULE_DEFAULT;
  module_ctx->module_extra_tuningdb_block     = MODULE_DEFAULT;
  module_ctx->module_forced_outfile_format    = MODULE_DEFAULT;
  module_ctx->module_hash_binary_count        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_parse        = MODULE_DEFAULT;
  module_ctx->module_hash_binary_save         = MODULE_DEFAULT;
  module_ctx->module_hash_decode_postprocess  = MODULE_DEFAULT;
  module_ctx->module_hash_decode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_decode_zero_hash    = MODULE_DEFAULT;
  module_ctx->module_hash_decode              = module_hash_decode;
  module_ctx->module_hash_encode_status       = MODULE_DEFAULT;
  module_ctx->module_hash_encode_potfile      = MODULE_DEFAULT;
  module_ctx->module_hash_encode              = module_hash_encode;
  module_ctx->module_hash_init_selftest       = MODULE_DEFAULT;
  module_ctx->module_hash_mode                = MODULE_DEFAULT;
  module_ctx->module_hash_category            = module_hash_category;
  module_ctx->module_hash_name                = module_hash_name;
  module_ctx->module_hashes_count_min         = MODULE_DEFAULT;
  module_ctx->module_hashes_count_max         = MODULE_DEFAULT;
  module_ctx->module_hlfmt_disable            = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_size    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_init    = MODULE_DEFAULT;
  module_ctx->module_hook_extra_param_term    = MODULE_DEFAULT;
  module_ctx->module_hook12                   = MODULE_DEFAULT;
  module_ctx->module_hook23                   = MODULE_DEFAULT;
  module_ctx->module_hook_salt_size           = MODULE_DEFAULT;
  module_ctx->module_hook_size                = MODULE_DEFAULT;
  module_ctx->module_jit_build_options        = MODULE_DEFAULT;
  module_ctx->module_jit_cache_disable        = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_accel_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_max         = MODULE_DEFAULT;
  module_ctx->module_kernel_loops_min         = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_max       = MODULE_DEFAULT;
  module_ctx->module_kernel_threads_min       = MODULE_DEFAULT;
  module_ctx->module_kern_type                = module_kern_type;
  module_ctx->module_kern_type_dynamic        = MODULE_DEFAULT;
  module_ctx->module_opti_type                = module_opti_type;
  module_ctx->module_opts_type                = module_opts_type;
  module_ctx->module_outfile_check_disable    = MODULE_DEFAULT;
  module_ctx->module_outfile_check_nocomp     = MODULE_DEFAULT;
  module_ctx->module_potfile_custom_check     = MODULE_DEFAULT;
  module_ctx->module_potfile_disable          = MODULE_DEFAULT;
  module_ctx->module_potfile_keep_all_hashes  = MODULE_DEFAULT;
  module_ctx->module_pwdump_column            = MODULE_DEFAULT;
  module_ctx->module_pw_max                   = MODULE_DEFAULT;
  module_ctx->module_pw_min                   = MODULE_DEFAULT;
  module_ctx->module_salt_max                 = MODULE_DEFAULT;
  module_ctx->module_salt_min                 = MODULE_DEFAULT;
  module_ctx->module_salt_type                = module_salt_type;
  module_ctx->module_separator                = MODULE_DEFAULT;
  module_ctx->module_st_hash                  = module_st_hash;
  module_ctx->module_st_pass                  = module_st_pass;
  module_ctx->module_tmp_size                 = module_tmp_size;
  module_ctx->module_unstable_warning         = module_unstable_warning;
  module_ctx->module_warmup_disable           = MODULE_DEFAULT;
}
