/* 
 * gpsk31  - PSK31 for Linux with a GTK+ Interface
 * 
 * Copyright (C) 2000 Luc Langehegermann, LX2GT
 * Copyright (C) 2008 Thomas Ries <tries@gmx.net>
 * Copyright (C) 2005,2006,2007,2008 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * The main author can be reached at pg4i@amsat.org or by smail-mail:
 * Joop Stakenborg, Bramengaarde 24, 3992KG Houten, The Netherlands.
 * 
 */

/*
 * Functions for the spectrum analyzer
 */

#include <unistd.h>
#include <sys/stat.h>
#include <ctype.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <sys/soundcard.h>

#include <gtk/gtk.h>
#include <stdio.h>
#include <math.h>
#include "main_screen.h"
#include "spectrum.h"
#include "globals.h"

#include "server.h"

float luc;			/* Position of the red line in the spectrum */
int ffts;			/* Number of filters */
float centrefreq;		/* Centre freq of the spectrum */
float fc;			/* Currently RX freq */
float deltaf;			/* Hz per fft sample */
int delta;			/* Offset into ffts */
double max;			/* Max value in values? */
int maxi;			/* Index to max */
int row;			/* First row in circular buffer */
int samples;			/* Number of samles */
int cnt;
float pwr[SPECTRUM_WIDTH+1];	/* Array where we put fft data */
GdkGC *gc_tune_line;		/* GC's of the spectrum */
GdkGC *gc_spectrum;
GdkGC *gc_center_line;		/* GC's of the center spectrum */

/* Waterfall display control... 0: display spectrum, 1: display waterfall */
int display_waterfall;
/* Waterfall Display mode: 0 b/w, 1: colored */
int waterfall_colored;

/* colors for RX and center line in display */
static guchar rx_color[3];
static guchar center_color[3];

/* Waterfall window pixbuf */
static guchar *gbl_wfall_pixels;
static gint
  gbl_wfall_rowstride,
  gbl_wfall_n_channels,
  gbl_wfall_width,
  gbl_wfall_height;

static void color_pixel(float val, guchar rgb[]);


/*
 * setup for spectrum GC's
 */
void
spectrum_setup ()
{
  GdkColor
    color;
  GdkColormap *
    cmap;

  cmap = gdk_colormap_get_system ();
  gdk_color_parse (ini_settings.spectrum_color, &color);

  if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
  {
    g_error ("Couldn't allocate color");
  }

  gc_spectrum = gdk_gc_new (main_screen.spectrum);
  gdk_gc_set_foreground (gc_spectrum, &color);

  gdk_color_parse (ini_settings.spectrum_tune_line_color, &color);
  if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
  {
    g_error ("Couldn't allocate color");
  }

  gc_tune_line = gdk_gc_new (main_screen.spectrum);
  gdk_gc_set_foreground (gc_tune_line, &color);

  /* get RGB values and remember for waterfall display */
  rx_color[0]=(color.red/256);
  rx_color[1]=(color.green/256);
  rx_color[2]=(color.blue/256);

  /* Center frequency color line */
  gdk_color_parse (ini_settings.spectrum_center_line_color, &color);
  if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
  {
    g_error ("Couldn't allocate color");
  }
  gc_center_line = gdk_gc_new (main_screen.spectrum);
  gdk_gc_set_foreground (gc_center_line, &color);

  /* get RGB values and remember for waterfall display */
  center_color[0]=(color.red/256);
  center_color[1]=(color.green/256);
  center_color[2]=(color.blue/256);

  //fft_setup (SAMPS , SPEED);		/* HARD CODED FOR NOW? */

  /* start with spectrum display first in colored mode */
  display_waterfall=1;
  waterfall_colored=1;

  /* Create waterfall pixbuf */
  main_screen.gbl_wfall_pixbuf = gdk_pixbuf_new(
				GDK_COLORSPACE_RGB, FALSE, 8,
				SPECTRUM_WIDTH, SPECTRUM_HEIGHT );
  if (main_screen.gbl_wfall_pixbuf == NULL)
  {
    g_error ("Couldn't allocate pixbuf");
  }

  gbl_wfall_pixels = gdk_pixbuf_get_pixels( main_screen.gbl_wfall_pixbuf );
  gbl_wfall_width  = gdk_pixbuf_get_width ( main_screen.gbl_wfall_pixbuf );
  gbl_wfall_height = gdk_pixbuf_get_height( main_screen.gbl_wfall_pixbuf );
  gbl_wfall_rowstride  = gdk_pixbuf_get_rowstride( main_screen.gbl_wfall_pixbuf);
  gbl_wfall_n_channels = gdk_pixbuf_get_n_channels( main_screen.gbl_wfall_pixbuf );

  /* clear pixbuff */
  gdk_pixbuf_fill( main_screen.gbl_wfall_pixbuf, 0x00000000 );

}


/*
 * fft_setup - set fft values for sample sizes
 */

void
fft_setup (int samps, int speed)
{
	/*set center line style*/
	gdk_gc_set_line_attributes (gc_center_line, 1, 
		GDK_LINE_SOLID, GDK_CAP_ROUND, GDK_JOIN_ROUND);

	if (samps > 3000) 
	gdk_gc_set_line_attributes (gc_tune_line, 2, 
		GDK_LINE_SOLID, GDK_CAP_ROUND, GDK_JOIN_ROUND);
	else
	gdk_gc_set_line_attributes (gc_tune_line, 1, 
                GDK_LINE_SOLID, GDK_CAP_ROUND, GDK_JOIN_ROUND);
	static int fftspeed=0;
	if(samps>0) {
		samples = samps;
	}
	float expo = 8+(100-speed)/99.0*3;  // 8..11
	    fftspeed = (int)pow(2.0, expo);
	
	ffts = samples/2;
	deltaf = (float)8000 / samples;

	delta = (int)(fc / deltaf - SPECTRUM_WIDTH/2);
	cnt = 0;
	commControl(COMM_FFTCH, COMM_FFTN, samples);
	commControl(COMM_FFTCH, COMM_FFTOVERLAP, fftspeed);	
}


/*
 * new_fc - set new center frequency
 */

float
spectrum_new_fc (int x)
{
  fc = centrefreq + deltaf * (x - SPECTRUM_WIDTH/2);
  return (fc);
}

void spectrum_updateFFT(float *val, int len)
{
	if(len!=samples) {
                g_warning ("FFT length mismatch");
	}
	spectrum_calc_power(val);
	spectrum_draw_display();
}

/*
 * calc power - pwr array
 * just calculate power for the 200 we need. 
 */

void
spectrum_calc_power (float *buffer)
{
  int i, j;
  for(i=delta,j=0; i<delta+SPECTRUM_WIDTH; i++,j++) {
    if(i<0||i>samples/2) {
      pwr[j]=0;
    } else {
      pwr[j] = buffer[i];
    }
  }
}

/*
 * draw_display method - draws display as spectrum or waterfall
 */

void
spectrum_draw_display ()
{
  PSK31info rxinfo;
  int x;
  float my_value;

  gdk_draw_rectangle (main_screen.spectrum,
		      main_screen.spectrum_drawing->style->black_gc,
		      TRUE,
		      0, 0,
		      main_screen.spectrum_drawing->allocation.width,
		      main_screen.spectrum_drawing->allocation.height);

  if (display_waterfall == 0) {
    /* draw the spectrum */
    for (x = 0; x < SPECTRUM_WIDTH ; x = x + 1) {
      my_value = (42.0 * (log10 (pwr[x] + 1)));
      gdk_draw_line (main_screen.spectrum, gc_spectrum, x, SPECTRUM_HEIGHT,
                     x, SPECTRUM_HEIGHT - (int) my_value);
    }

    commGetInfo (COMM_RXCH, &rxinfo, sizeof(rxinfo));

    /* draw center line - before the RX line*/
    gdk_draw_line (main_screen.spectrum, gc_center_line,
		   SPECTRUM_WIDTH/2, SPECTRUM_HEIGHT,
		   SPECTRUM_WIDTH/2, 0);

    /*draw RX frequency line */
    luc = 100 + ((rxinfo.freq/100) - centrefreq) / deltaf + 0.5;
    gdk_draw_line (main_screen.spectrum, gc_tune_line,
		  (int) luc, SPECTRUM_HEIGHT,
                  (int) luc, 0);

    gdk_draw_drawable (main_screen.spectrum_drawing->window,
		       main_screen.spectrum_drawing->
		       style->fg_gc[GTK_WIDGET_STATE
				(main_screen.spectrum_drawing)],
		       main_screen.spectrum, 0, 0, 0, 0,
		       SPECTRUM_WIDTH, SPECTRUM_HEIGHT);
  } else {
    /* draw waterfall */
    int idh, idv;	/* loop variable for moving PixMap content */

    /* Positions to draw red line in waterfall */
    int red_line;
    int center_line;

    /* Pointer to current pixel */
    static guchar *pix;

    /* calculate current tuning position */
    commGetInfo (COMM_RXCH, &rxinfo, sizeof(rxinfo));

    red_line = (int) (SPECTRUM_WIDTH/2 + ((rxinfo.freq/(SPECTRUM_WIDTH/2)) -
                                          centrefreq) / deltaf + 0.5);
    center_line = SPECTRUM_WIDTH/2;

    /* Copy each line of waterfall to next one */
    for( idv = (gbl_wfall_height-1); idv > 0; idv-- ) {
      pix = gbl_wfall_pixels + gbl_wfall_rowstride * idv;
      for( idh = 0; idh < SPECTRUM_WIDTH; idh++ ) {
        *pix = *( pix - gbl_wfall_rowstride ); pix++;
        *pix = *( pix - gbl_wfall_rowstride ); pix++;
        *pix = *( pix - gbl_wfall_rowstride );
        pix += gbl_wfall_n_channels - 2;
      }
    }

    /* Go to top left of pixbuf */
    pix = gbl_wfall_pixels;

    for (x = 0; x < SPECTRUM_WIDTH ; x = x + 1) {
      my_value = 50.0 * (log10 (pwr[x] + 1)); /* ~ 0..100 */

      /*&&& I may want to do some range scaling.
            Remember min/max value of each incoming row
            and then calculate the min/max value of the
	    whole waterfall picture and scale the new
	    to-be-drawn pixel value relative in this range. */

      /* "false" coloring of pixels or b/w? */
      if (waterfall_colored) {
        color_pixel(my_value, pix);
      } else {
        pix[0]=pix[1]=pix[2]=(int)(my_value*2.55);
      }

      pix += gbl_wfall_n_channels;
    }

    /* Draw a vertical orange line in waterfall at middle freq. */
    pix = gbl_wfall_pixels;
    pix += gbl_wfall_n_channels * center_line;
    pix[0] = center_color[0];
    pix[1] = center_color[1];
    pix[2] = center_color[2];

    /* Draw a vertical red line in waterfall at detector's freq. */
    pix = gbl_wfall_pixels;
    pix += gbl_wfall_n_channels * red_line;
    pix[0] = rx_color[0];
    pix[1] = rx_color[1];
    pix[2] = rx_color[2];

    /* Draw waterfall */
    gdk_draw_pixbuf(main_screen.spectrum_drawing->window,
		NULL, main_screen.gbl_wfall_pixbuf,
		0,0,0,0,SPECTRUM_WIDTH, SPECTRUM_HEIGHT,
		GDK_RGB_DITHER_NONE, 0, 0 );
  }



}

/*
 * offset method - sets delta to starting index value of values array
 */
void
spectrum_offset (float freq)
{
  fc = freq;
  centrefreq = freq;
  delta = (int) (fc / deltaf - 100);
}

/*
 * Returns the number of samles
 */

int getsamplecnt()
{
  return samples;
}

/* 
 * creates "false coloring" from b/w value 0..100
 * return value is int[3] of R/G/B values
 */
static void color_pixel(float val, guchar rgb[]) {
  guchar r,g,b;

  float L1=20; /* black to blue  */
  float L2=40; /* blue to green  */
  float L3=60; /* green to cyan	 */
  float L4=80; /* cyan to yellow */
  float L5=95; /* yellow to white*/

  /* HB9XAR: This is just a quick hack to get something colored.
             A better/nicer solution is welcome ;-) */

  /* for testing: this will goive a plain greyscale image */
  if (0) {
    int t;
    t=(int)(val*(255/70));
    r=g=b=(t>255)?255:t;
    r=0;
    rgb[0]=r;
    rgb[1]=g;
    rgb[2]=b;
    return;
  }
  val = val - 2.0;
  if (val <0) val=0;

  if (val < L1) {		/* black to blue */
    r=0;
    g=0;
    b=(int)((val/(L1))*200);
  } else if (val  < L2) {	/* blue to green */
    r=0;
    g=50+(int)(((val-L1)/(L2-L1))*205);
    b=200-(int)((val/(L2))*200);
  } else if (val  < L3) {	/* green to cyan */
    r=0;
    g=255;
    b=50+(int)(((val-L2)/(L3-L2))*205);
  } else if (val  < L4) {	/* cyan to yellow */
    r=50+(int)(((val-L3)/(L4-L3))*205);
    g=255;
    b=255-(int)(((val-L3)/(L4-L3))*255);
  } else if (val  < L5) {	/* yellow to white */
    r=255;
    g=255;
    b=50+(int)(((val-L4)/(L5-L4))*205);
  } else {
    r=255;
    g=255;
    b=255;
  }

  rgb[0]=r;
  rgb[1]=g;
  rgb[2]=b;
  return;
}
