/* $Id: player.c,v 1.33 2003/06/11 11:14:33 paul Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include "global.h"
#include "model.h"
#include "object.h"
#include "food.h"
#include "player.h"
#include "collision.h"

#define MDEBUG(...)      DEBUG(DMODEL, "Model", __VA_ARGS__)


/*****************************************************************************/
/* Player object (un)conscious functions                                     */
/*****************************************************************************/

static void
player_make_unconscious(Model *m, Model_object *p) {
  int r = FALSE;
  
  assert(!IS_UNCONSCIOUS(p) && !IS_DEAD(p));

  p->state.ps = PS_UNCONSCIOUS;
  if (IS_ACTIVE(p)) {
    r = list_del(m->act_players, p);
    list_append(m->act_uplayers, p);
  }
  else {
    r = list_del(m->pas_players, p);
    list_append(m->pas_uplayers, p);
  }

  assert(r && "Player couldn't be removed from players sublist.");
  MDEBUG("Made player %p unconscious.", p);
}

static void
player_make_conscious(Model *m, Model_object *p) {
  int r = FALSE;

  assert(IS_UNCONSCIOUS(p));

  p->state.ps = PS_NORMAL;
  if (IS_ACTIVE(p)) {
    r = list_del(m->act_uplayers, p);
    list_append(m->act_players, p);
  }
  else {
    r = list_del(m->pas_uplayers, p);
    list_append(m->pas_players, p);
  }

  assert(r && "Player couldn't be removed from uplayers sublist.");
  MDEBUG("Made player %p conscious.", p);
}


/*****************************************************************************/
/* Player object friction functions                                          */
/*****************************************************************************/

/* Friction on player when walking. If players X speed is zero, set
 * friction deceleration to zero. */
static void
player_friction_walkstand(Model_object *o) {
  o->accel->x = -SGN(o->speed->x) * 
                (o->friction_ground + list_length(o->foodstack));
}

/* Friction on player when flying. If players X speed is zero,
 * set friction deceleration to zero. */
static void
player_friction_fly(Model_object *o) {
  o->accel->x = -SGN(o->speed->x) * o->friction_air;
}

/* Select right friction routine, based on whether player is standing. */
static void
player_friction(Model_object *o) {
  if (IS_STANDING(o))
    player_friction_walkstand(o);
  else
    player_friction_fly(o);
}


/*****************************************************************************/
/* Player functions for collision detection/response                         */
/*****************************************************************************/

int
player_pickup_food(Model *m, Model_object *p, Model_object *f) {
  /* Put food object type on player's foodstack if it isn't full. */
  if (list_length(p->foodstack) >= m->settings->foodstack_size &&
      f->type != OT_POWERUP)
    return FALSE;
  else {
    /* Food object will be pickup up, delete in next tick and from screen now. */
    f->gfx_state = GFX_DELETED;
    
    if (f->type == OT_POWERUP) {
      /* Picked up powerup, set frame for losing ability and request new food
       * spawned. */
      if (m->settings->spawn_pickup_powerup)
	m->spawn_food++;
      p->events |= EVF_POWERUP;
      p->state.ps = PS_POWERUP;
      p->frame_powerup_exp = m->framecounter + m->settings->powerup_length;

      MDEBUG("Player is powered up until frame %d!",
             p->frame_powerup_exp);
    }
    else {
      /* Picked up normal food. */
      Object_type *t = malloc(sizeof(Object_type));
      assert(t != NULL);

      *t = f->type;
      list_prepend(p->foodstack, t);

      MDEBUG("Player picks up food:");
      object_dump(p, NULL);
    }
    /* Set event and increase player's score. */
    p->events |= EVF_PICKUP;
    p->score += SC_PICKUP;
    
    return TRUE;
  }
}

int
player_gets_hit(Model *m, Model_object *p, Model_object *o) {
  /* Player hit is not unconscious */
  if (IS_UNCONSCIOUS(p) || IS_DEAD(p)) {
    WARN("Player %p is unconscious or dead and can't get hit:", p);
    list_foreach(m->act_players,  object_dump, "Active  player:  ");
    list_foreach(m->pas_players,  object_dump, "Passive player:  ");
    list_foreach(m->act_uplayers, object_dump, "Active  Uplayer: ");
    list_foreach(m->pas_uplayers, object_dump, "Passive Uplayer: ");
    
    assert(!IS_UNCONSCIOUS(p) && !IS_DEAD(p));
  }
    
  /* No hit when owner is not set or player itself. */
  if (o->owner == NULL || o->owner == p || o->disown)
    return FALSE;
    
  /* Update score of owner of food, set hit flag, event and frame to revive. */
  o->owner->score += SC_HIT;
  o->gfx_state = GFX_DELETED;
  p->hit = TRUE;
  p->events |= EVF_HIT;
  p->frame_revive = m->framecounter + m->settings->unconc_length;
  player_make_unconscious(m, p);

  /* Spawn food if required. */
  if (m->settings->spawn_hit)
    m->spawn_food++;

  /* Postpone losing food till next tick (player_hit_tick). */
  
  MDEBUG("Player %p got hit by a %s and will be unconscious until frame %d...", 
         p, object_type_string[o->type], p->frame_revive);
  
  return TRUE;
}

/*****************************************************************************/
/* Player action function                                                    */
/*****************************************************************************/

/* Create food object on right location, give it initial speed
 * and power it up when necessary. */
static Model_object *
player_throw_food(Model *m, Model_object *p, Object_type t,
                  int32_t x, int32_t y, int32_t w, int32_t h) {
  Model_object *food;

  /* Create food and accelerate it. */
  food = new_object(m, t, p, x, y, w, h);

  food->speed->y = -p->throw_speed->y;
  food->speed->x = (LOOKS_LEFT(p) ? -p->throw_speed->x
                                  : +p->throw_speed->x) + 
		    (2 * p->speed->x) / 3;

  /* Powerup the food. */
  if (IS_POWEREDUP(p))
    food_powerup(m, food, p);
		  
  /* Set event. */
  p->events |= EVF_THROW;

  return food;
}


/*****************************************************************************/
/* Player object tick functions                                              */
/*****************************************************************************/

/* If player got hit in previous frame, set event and frame to revivie.
 * Let player `lose' food (if any). */
static void
player_tick_hit(Model *m, Model_object *o) {
  Object_type *t;
  Vector *pos, *size;

  /* If player isn't hit, nothing has to be done. */
  if (!o->hit)
    return;
    
  /* Player drops two food objects from foodstack (if any)! */
  /* Get first food object out of stack (if possible). */
  if (list_length(o->foodstack) >= 1) {
    t = (Object_type *)list_pop(o->foodstack);
    size = m->reg_types[*t];
    pos = new_vector(o->pos->x - size->x, o->pos->y);
    
    food_generate(m, *t, NULL, pos, size);
    
    del_vector(pos);
    free(t);
  }

  /* Get second food object out of stack (if possible). */
  if (list_length(o->foodstack) >= 1) {
    t = (Object_type *)list_pop(o->foodstack);
    size = m->reg_types[*t];
    pos = new_vector(o->pos->x + o->size->x, o->pos->y);
    
    food_generate(m, *t, NULL, pos, size);
    
    del_vector(pos);
    free(t);
  }

  o->hit = FALSE;
}


/* Handles the part of player tick if player is unconscious or dead. */
static void
player_tick_unconscious(Model *m, Model_object *o) {
  assert(IS_UNCONSCIOUS(o) || IS_DEAD(o));

  /* If player has been PL_MAX_UNCONC frames unconscious,
     unfreeze player. */
  if (IS_DEAD(o)) {
    player_friction(o);
    o->action = FALSE; /* Can't perform action while unconscious */
  }
  else if (m->framecounter >= o->frame_revive &&
           !collides_with_objects(m->act_players, o->pos, o->size) &&
           !collides_with_objects(m->pas_players, o->pos, o->size) &&
           !collides_with_objects(m->boxes,       o->pos, o->size)) {
	     /* Time to revive player and nothing is in the way */
	     player_make_conscious(m, o);
	     MDEBUG("Player %p regained consciousness.", o);
  }
  else { /* m->framecounter < o->frame_revive */
    MDEBUG("Player %p is still unconscious or can't regain consciousness.",
           o);
    player_friction(o);
    o->action = FALSE; /* Can't perform action while unconscious */
  }
}

static void
player_tick_walkstand(Model_object *o) {
  assert(o->left || o->right || (!o->left && !o->right));

  if (o->left) {
    /* Move left faster (if possible). */
    o->accel->x = -o->accel_ground + list_length(o->foodstack);
    o->lookdir = LD_LEFT;
  }
  else if (o->right) {
    /* Move right faster (if possible). */
    o->accel->x = +o->accel_ground - list_length(o->foodstack);
    o->lookdir = LD_RIGHT;
  }
  else
    /* Player doesn't want to move, apply air/floor friction. */
    player_friction_walkstand(o);
}

static void
player_tick_fly(Model_object *o) {
  assert(o->left || o->right || (!o->left && !o->right));

  if (o->left) {
    /* Move left faster (if possible). */
    o->accel->x = -o->accel_air -(SGN(o->speed->x) * o->friction_air);
    o->lookdir = LD_LEFT;
  }
  else if (o->right) {
    /* Move right faster (if possible). */
    o->accel->x = +o->accel_air -(SGN(o->speed->x) * o->friction_air);
    o->lookdir = LD_RIGHT;
  }
  else {
    /* Player doesn't want to move, apply air friction. */
    player_friction_fly(o);
  }
}

/* Handle jumping of player. */
static void
player_tick_jump(Model_object *o) {
  assert(!IS_UNCONSCIOUS(o) && !IS_DEAD(o));

  if (o->jump && IS_STANDING(o)) {
    /* Set event. */
    o->events |= EVF_JUMP;
    
    /* Player is not standing any more, give initial X, Y speed for jump. */
    o->speed->y = -o->jump_speed->y;
    if (o->speed->x < 0) {
      o->speed->x = MAX(o->speed->x / 3 - o->jump_speed->x, -o->max_speed->x);
    }
    else if (o->speed->x > 0) {
      o->speed->x = MIN(o->speed->x / 3 + o->jump_speed->x, +o->max_speed->x);
    }
  }
}

/* Handle action (if any) of player. */
static void
player_tick_action(Model *m, Model_object *o) {
  Object_type *type;
  Vector *pos, *size;

  assert(!IS_UNCONSCIOUS(o) && !IS_DEAD(o));

  if (o->action) {
    if (list_length(o->foodstack) > 0) {
      /* Pop object type from stack and  get size from registry. */
      type = (Object_type *)list_pop(o->foodstack);
      size = m->reg_types[*type];
      pos = new_vector(o->pos->x + (LOOKS_LEFT(o) ? -size->x : o->size->x), 
		       o->pos->y);

      if (!collides_with_objects(m->act_food, pos, size) &&
	  !collides_with_objects(m->pas_food, pos, size) &&
	  !collides_with_objects(m->boxes,    pos, size)) {
	/* If food available in stack and no collisions if food were
	 * thrown, throw it. */
	player_throw_food(m, o, *type, pos->x, pos->y, size->x, size->y);
	free(type);
      }
      else
	/* Can't throw it, put type back in stack. */
	list_prepend(o->foodstack, type);

      del_vector(pos);
    }

    o->action = FALSE;
  }
}

void
player_tick(Model *m, Model_object *o) {
  /* Player may be hit and should lose food and become unconscious. */
  player_tick_hit(m, o);

  /* Player is unconscious or can move around */
  if (IS_UNCONSCIOUS(o) || IS_DEAD(o)) {
    /* Player is unconscious, revive if this is the time, otherwise
       apply friction. */
    player_tick_unconscious(m, o);
  }
  else {
    /* Player is alive, accelerate if left/right/jump flag set
       and perform action */
    if (IS_STANDING(o))
      player_tick_walkstand(o);
    else
      player_tick_fly(o);

    player_tick_jump(o);
    player_tick_action(m, o);

    /* Players power up expires. */
    if (IS_POWEREDUP(o) && m->framecounter == o->frame_powerup_exp)
      o->state.ps = PS_NORMAL;
  }

  /* If player is not dead, make unconscious and let player never revive. */
  if (!IS_DEAD(o) && o->life > 0) {
     /* Set in model at least one (this) player is alive. */
    m->alive++;

    /* Decrease life with PL_DEC_LIFE units and increase score
     * every MODEL_UPD_LIFE ticks. */
    if (m->framecounter % MODEL_UPD_LIFE == 0) {
      o->life = MAX(o->life - (MODEL_UPD_LIFE * PL_DEC_LIFE), 0);
      o->score += MODEL_UPD_LIFE * SC_TICK;
    }

    MDEBUG("Player %p's life is now %d units.", o, o->life);
  }
  else if (!IS_DEAD(o) && o->life == 0) {
    if (!IS_UNCONSCIOUS(o))
      player_make_unconscious(m, o);

    /* Set player status to dead, decrease the min food. */
    o->state.ps = PS_DEAD;
    m->min_food -= m->settings->min_food;

    /* Set event. */
    o->events |= EVF_DEAD;
    
    MDEBUG("Player %p is dead.", o);
  } 
}
