/*
Copyright (C) 2000-2001 Jos Roberto B. de A. Monteiro <jrm@autsens.com>
                        and Pedro Zorzenon Neto <pzn@autsens.com>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "avrserial.h"
#include "intelhex.h"
#include "devices.h"
#include "microdelay.h"
#include "avrserial.h"
#include "avrconf.h"
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/io.h>
#include <sys/types.h>
#include <string.h>

int main(int argc, char **argv) {

    /* parallel programmer variables */
    ASPP *aspp;
    int data;
    unsigned int parallel_port=0;

    /* intelhex variables */
    inteldata ihex;
    inteldata *ihexptr = &ihex;
    int readhex_ok=1;

    /* variables for options */
    int opt_c, opt_reset_pin=-1;
    int opt_read=0, opt_write=0, opt_erase=0, opt_verify=0, opt_reset=0;
    int opt_lock=-1;
    int opt_autodevice=1;
    int opt_verbose=0;

    /* other variables */
    int n;
    int eeprom_mode=0;

    /* variables for chip identification */
    unsigned char signature_bytes[4]={-1,-1,-1,-1};
    char chip_id[256];
    int program_memsize=-1;
    int eeprom_memsize=-1;

    /* Parsing options... */
    opterr=0;
    while ((opt_c = getopt(argc, argv, "hrwceEd:p:RL:b:Vv")) != -1) {
	switch (opt_c) {
	case 'h':
	    printf("usage: %s [-r|-w|-c|-e|-b x|-L n] [-E] [-d device] [-R] [-p port] [-R]\n",PACKAGE);
            printf("               [-L mode] [-v] [-V]\n\n");
	    printf("-r        reads data from chip and send to stdout.\n");
	    printf("-w        reads data from stdin and write to chip.\n");
	    printf("-c        reads from stdin and from chip and compare if data is the same.\n");
	    printf("          returns an error if data is not the same. bytes that are not in the\n");
	    printf("          stdin are not compared.\n");
	    printf("-e        erases chip (Program Memory and Eeprom Memory) and exit.\n");
	    printf("-E        reads/writes Eeprom Memory (default is to read/write Program Memory)\n");
	    printf("-d device specify which device you are using. use '-d ?' to see which devices\n");
	    printf("          are supported. if you do not use -d, then the program tries to\n");
	    printf("          autodetect the device.\n");
	    printf("-p port   programmer parallel port address in hexadecimal format. see also\n");
            printf("          configuration file 'avrprog.conf'.\n");
	    printf("-R        set avr_reset_pin to 1 before leaving the program, enabling avr\n");
	    printf("          to start software execution. The default is to leave reset=0.\n");
            printf("-b n      just change state of reset pin to n (0 or 1) and exit.\n");
	    printf("-L n      set lock bits mode. if n=0 then LB1=LB2=0. if n=1 then LB1=1, LB2=0.\n");
	    printf("          if n=2 then LB1=0, LB2=1. if n=3 then LB1=LB2=1. See chip docs.\n");
	    printf("          WARNING: in some devices, even erasing chip (-e) didn't clear the\n");
	    printf("          lock bits... so we lost the chips... use it at your own risk.\n");
	    printf("-v        verbose mode.\n");
            printf("-V        show program version and copyright.\n");
	    printf("-h        prints this usage message\n\n");
	    printf("examples: %s -r > file.hex        [reads chip and saves to file.hex]\n",PACKAGE);
	    printf("          %s -w < file.hex        [reads file.hex and saves to chip]\n\n",PACKAGE);
	    printf("stdin/stdout data are Intel hexadecimal file format.\n");
	    printf("in input data (stdin), LF and CR+LF are accepted as a newline mark.\n");
	    printf("output data (stdout) is generated with LF as a newline mark.\n");
	    printf("if you run %s without arguments, just a device autodetection is made (and\n",PACKAGE);
            printf("by default, software execution is stoped - see option -R)\n");
	    printf("if you specify at the same time the arguments -b -e -r -w -c -L, it is executed\n");
	    printf("just the first argument found in the following order: -b -c -r -L -e -w.\n");
	    return 1;
	case 'V':
	    printf("%s %s (c)2000-2001 Jose Roberto B. A. Monteiro and Pedro Zorzenon Neto\n",PACKAGE,VERSION);
	    printf("this software and hardware are copyrighted under the terms of GNU GPL license.\n");
	    printf("Send coments or bug reports to 'jrm' or 'pzn' @autsens.com\n");
	    printf("\n%s compiled %s %s\ngcc version %s\n",PACKAGE,__DATE__,__TIME__,__VERSION__);
            return 1;
  	case 'r':
	    opt_read=1;
	    break;
	case 'w':
	    opt_write=1;
	    break;
	case 'c':
            opt_verify=1;
	    break;
	case 'e':
	    opt_erase=1;
	    break;
	case 'E':
            eeprom_mode=1;
            break;
	case 'd':
	    strcpy(chip_id,optarg);
            opt_autodevice=0;
	    break;
	case 'v':
	    opt_verbose=1;
            break;
	case 'p':
	    parallel_port=(unsigned int)strtoul(optarg,(char **)NULL,16);
	    if (parallel_port==0) {
		fprintf(stderr,"ERROR: Sorry... didn't understand the port address. Try something like '-p 378' or see the man pages.\n");
		abort();
	    }
            break;
	case 'R':
            opt_reset=1;
            break;
	case 'L':
            opt_lock=atoi(optarg);
	    break;
	case 'b':
	    if (strcmp(optarg,"1")==0) opt_reset_pin=1;
	    else opt_reset_pin=0;
            break;
	case '?':
	    if (isprint (optopt))
		fprintf (stderr, "Unknown option '-%c'. Try -h for help.\n", optopt);
	    else
		fprintf (stderr, "Unknown option character '\\x%x'. Try -h for help.\n", optopt);
	    return 1;
	default:
	    abort ();
	}
    }
    /* end of parsing options */

    /* ****** */
    /* palellel port checking */
    /* ****** */
    if (parallel_port == 0) {
	/* loads config file default value for parallel port */
        /* if option '-p' wasn't specified */
        parallel_port=avrprog_conf_port_default();
    }
    /* verify if port number is allowed in config file */
    if (avrprog_conf_port_allowed(parallel_port)==0) {
	fprintf(stderr,"ERROR: specified port address (0x%x) not allowed. check 'avrprog.conf' file.\n",parallel_port);
	abort();
    }

    /* check root privileges */
    if (geteuid()!=0) {
	fprintf(stderr,"ERROR: this programs needs root permisions to access the parallel port directly. Please use root user to run it, or configure the 'sudo' program. You can also make it setuid root (not recommended - all users will have direct access to ioport/parport).\n");
        abort();
    }
    ioperm(parallel_port,3,1);

    aspp = aspp_init(aspp_new(),parallel_port);

    /* change reset_pin state and exits (option '-b x') */
    if (opt_reset_pin>-1) {
	if (opt_reset_pin==1) {
	    change_reset_pin(aspp,1);
	    fprintf(stderr,"RESET pin changed to 1 - software running.\n"); }
	else {
	    change_reset_pin(aspp,0);
	    fprintf(stderr,"RESET pin changed to 0 - software halted.\n"); }
        exit(0);
    }
    /* end of change reset_pin_state option */

    microdelay_init(); /* initialize microdelay function */
    power_up(aspp);
    data = programming_enable(aspp);
    if (data != 1) { fprintf(stderr,"chip not found...\n"); exit(0); }

    /* ***************************************** */
    /* recognition/autodetection/signature bytes */
    /* ***************************************** */
    /*  reading signature bytes */
    fprintf(stderr,"Chip signature bytes: ");
    for (n=0; n<4; n++) {
	signature_bytes[n]=(unsigned char) read_signature_byte(aspp, n);
	fprintf(stderr,"0x%02x ", signature_bytes[n]); }
    fprintf(stderr,"\n");
    if (!opt_autodevice) {
	search_chip_by_name(signature_bytes, chip_id, &program_memsize, &eeprom_memsize);
    } else {
	search_chip_by_signature(signature_bytes, chip_id, &program_memsize, &eeprom_memsize);
    }
    if (program_memsize<0 || eeprom_memsize<0) abort();
    fprintf(stderr,"Chip identification: %s - %d bytes program; %d bytes eeprom\n", chip_id, program_memsize, eeprom_memsize);

    /* ******************************** */
    /*  verifies if the data read in the chip and file are the same (program_memory) */
    /* ******************************** */
    if (opt_verify && !eeprom_mode) {
	intelhex_fileopen(&ihex,"/dev/stdin",1); /* open file for reading */
	readhex_ok=intelhex_readline(&ihex); /* reads first line of hex file */
        if (readhex_ok>1) abort();
	fprintf(stderr,"Compare program memory\n");
	if (opt_verbose) fprintf(stderr," offset reclen success\n");
	{   int full_vrfy_errors=0;
	    while (readhex_ok==1) {
		int highbyte, addr, vrfy_errors=0;
		if (ihexptr->offset+ihexptr->reclen>program_memsize) {
		    fprintf(stderr,"ERROR: address greater than memory size\n"); abort(); }
		for (n=0; n<ihexptr->reclen; n++) {
		    /* program memory is 16bit, so need to find if is the high or low 8-bits */
		    addr = (n+ihexptr->offset)/2;
		    highbyte = ((addr*2) != (n+ihexptr->offset));
		    if (ihexptr->recdata[n]!=read_program_memory(aspp, highbyte, addr))
			vrfy_errors++;
		}
		if (opt_verbose) {
		    fprintf(stderr," 0x%04x  0x%02x  ",ihexptr->offset/2,ihexptr->reclen/2);
		    if (vrfy_errors==0)
			fprintf(stderr,"  :-)\n");
		    else
			fprintf(stderr," -%d bytes do not match-\n", vrfy_errors);
		}
		full_vrfy_errors+=vrfy_errors;
		readhex_ok=intelhex_readline(&ihex); /* reads next line of hex file */
		if (readhex_ok>1) abort();
	    }
	if (full_vrfy_errors==0)
	    fprintf(stderr,"Compare successfull.\n");
	else
	    fprintf(stderr,"ERROR: Compare failure. %d bytes do not match\n",full_vrfy_errors);
	intelhex_fileclose(&ihex);
	power_off(aspp,opt_reset);
	if (full_vrfy_errors==0) exit (0); /* exit with success */
	exit(1); /* exit with error */
	}
    }

    /* ******************************** */
    /*  verifies if the data read in the chip and file are the same (eeprom_memory) */
    /* ******************************** */
    if (opt_verify && eeprom_mode) {
	intelhex_fileopen(&ihex,"/dev/stdin",1); /* open file for reading */
	readhex_ok=intelhex_readline(&ihex); /* reads first line of hex file */
	if (readhex_ok>1) abort();
	fprintf(stderr,"Compare eeprom memory\n");
	if (opt_verbose) fprintf(stderr," offset reclen success\n");
	{   int full_vrfy_errors=0;
	    while (readhex_ok==1) {
		int vrfy_errors=0;
		if (ihexptr->offset+ihexptr->reclen>eeprom_memsize) {
		    fprintf(stderr,"ERROR: address greater than memory size\n"); abort(); }
		for (n=0; n<ihexptr->reclen; n++) {
		    if (ihexptr->recdata[n]!=read_eeprom_memory(aspp, ihexptr->offset+n))
			vrfy_errors++;
		}
		if (opt_verbose) {
		    fprintf(stderr," 0x%04x  0x%02x  ",ihexptr->offset/2,ihexptr->reclen/2);
		    if (vrfy_errors==0)
			fprintf(stderr,"  :-)\n");
		    else
			fprintf(stderr," -%d bytes do not match-\n", vrfy_errors);
		}
		full_vrfy_errors+=vrfy_errors;
		readhex_ok=intelhex_readline(&ihex); /* reads next line of hex file */
		if (readhex_ok>1) abort();
	    }
	if (full_vrfy_errors==0)
	    fprintf(stderr,"Compare successfull.\n");
	else
	    fprintf(stderr,"ERROR: Compare failure. %d bytes do not match\n",full_vrfy_errors);
	intelhex_fileclose(&ihex);
	power_off(aspp,opt_reset);
	if (full_vrfy_errors==0) exit (0); /* exit with success */
	exit(1); /* exit with error */
	}
    }

    /* ******************************** */
    /*  read the chip and save to file (program_memory) */
    /* ******************************** */
    if (opt_read && !eeprom_mode) {
	intelhex_fileopen(&ihex,"/dev/stdout", 0); /* open file for writing */
	ihexptr->reclen=16;
	ihexptr->offset=0x0000;
	fprintf(stderr,"Reading chip program memory from 0x%04x ",ihexptr->offset);
	while (ihexptr->offset<(program_memsize-1)) {
	    int highbyte, addr;
	    for (n=0; n<ihexptr->reclen; n++) {
		/* program memory is 16bit, so need to find if is the high or low 8-bits */
		addr = (n+ihexptr->offset)/2;
		highbyte = ((addr*2) != (n+ihexptr->offset));
		ihexptr->recdata[n]=read_program_memory(aspp, highbyte, addr); }
	    intelhex_writeline(&ihex);
	    ihexptr->offset+=ihexptr->reclen;
	}
	intelhex_writeeof(&ihex);
	intelhex_fileclose(&ihex);
	fprintf(stderr,"to 0x%04x. done.\n",ihexptr->offset-1);
	power_off(aspp,opt_reset);
        exit(0);
    }

    /* ******************************** */
    /*  read the chip and save to file (eeprom_memory) */
    /* ******************************** */
    if (opt_read && eeprom_mode) {
	intelhex_fileopen(&ihex,"/dev/stdout", 0); /* open file for writing */
	ihexptr->reclen=16;
	ihexptr->offset=0x0000;
	fprintf(stderr,"Reading chip eeprom memory from 0x%04x ",ihexptr->offset);
	while (ihexptr->offset<(eeprom_memsize-1)) {
	    for (n=0; n<ihexptr->reclen; n++)
		ihexptr->recdata[n]=read_eeprom_memory(aspp, ihexptr->offset+n);
	    intelhex_writeline(&ihex);
	    ihexptr->offset+=ihexptr->reclen;
	}
	intelhex_writeeof(&ihex);
	intelhex_fileclose(&ihex);
	fprintf(stderr,"to 0x%04x. done.\n",ihexptr->offset-1);
	power_off(aspp,opt_reset);
        exit(0);
    }

    /* *********************** */
    /*  program the lock bits */
    /* *********************** */
    if (opt_lock!=-1) {
	fprintf(stderr,"Writing lock bits...");
	write_lock_bits(aspp,opt_lock);
	power_off(aspp,opt_reset);
	fprintf(stderr," done.\n");
        exit(0);
    }

    /* **************** */
    /*  erase the chip */
    /* **************** */
    if (opt_erase) {
	fprintf(stderr,"Erasing chip...");
	chip_erase(aspp);
	power_off(aspp,opt_reset);
	fprintf(stderr," done.\n");
        exit(0);
    }

    /* ********************************* */
    /*  read the file and write to chip (program_memory) */
    /* ********************************* */
    if (opt_write && !eeprom_mode) {
	intelhex_fileopen(&ihex,"/dev/stdin",1); /* open file for reading */
	readhex_ok=intelhex_readline(&ihex); /* reads first line of hex file */
	if (readhex_ok>1) abort();
	fprintf(stderr,"Writing program memory\n");
	if (opt_verbose) fprintf(stderr," offset reclen success\n");
	{   int full_write_ok=1;
	    while (readhex_ok==1) {
		int highbyte, addr, write_ok=1;
		if (ihexptr->offset+ihexptr->reclen>program_memsize) {
		    fprintf(stderr,"ERROR: address greater than memory size\n"); abort(); }
		for (n=0; n<ihexptr->reclen; n++) {
		    /* program memory is 16bit, so need to find if is the high or low 8-bits */
		    addr = (n+ihexptr->offset)/2;
		    highbyte = ((addr*2) != (n+ihexptr->offset));
		    write_ok&=write_program_memory(aspp, highbyte, addr , ihexptr->recdata[n]); }
		if (opt_verbose) {
		fprintf(stderr," 0x%04x  0x%02x  ",ihexptr->offset/2,ihexptr->reclen/2);
		if (write_ok==1)
		    fprintf(stderr,"  :-)\n");
		else
		    fprintf(stderr," -fail-\n");
		}
		full_write_ok&=write_ok;
		readhex_ok=intelhex_readline(&ihex); /* reads next line of hex file */
		if (readhex_ok>1) abort();
	    }
	if (full_write_ok==1)
	    fprintf(stderr,"Writing successfull.\n");
	else
	    fprintf(stderr,"ERROR: Writing failure.\n");
	intelhex_fileclose(&ihex);
	power_off(aspp,opt_reset);
        if (full_write_ok==1) exit(0); /* exit with success */
	exit(1); /* exit with error */
	}
    }

    /* ********************************* */
    /*  read the file and write to chip (eeprom_memory) */
    /* ********************************* */
    if (opt_write && eeprom_mode) {
	intelhex_fileopen(&ihex,"/dev/stdin",1); /* open file for reading */
	readhex_ok=intelhex_readline(&ihex); /* reads first line of hex file */
	if (readhex_ok>1) abort();
	fprintf(stderr,"Writing eeprom memory\n");
	if (opt_verbose) fprintf(stderr," offset reclen success\n");
	{   int full_write_ok=1;
	    while (readhex_ok==1) {
		int write_ok=1;
		if (ihexptr->offset+ihexptr->reclen>eeprom_memsize) {
		    fprintf(stderr,"ERROR: address greater than memory size\n"); abort(); }
		for (n=0; n<ihexptr->reclen; n++) {
		    write_ok&=write_eeprom_memory(aspp, ihexptr->offset+n, ihexptr->recdata[n]); }
		if (opt_verbose) {
		fprintf(stderr," 0x%04x  0x%02x  ",ihexptr->offset,ihexptr->reclen);
		if (write_ok==1)
		    fprintf(stderr,"  :-)\n");
		else
		    fprintf(stderr," -fail-\n");
		}
		full_write_ok&=write_ok;
		readhex_ok=intelhex_readline(&ihex); /* reads next line of hex file */
		if (readhex_ok>1) abort();
	    }
	if (full_write_ok==1)
	    fprintf(stderr,"Writing eeprom successfull.\n");
	else
	    fprintf(stderr,"ERROR: Writing eeprom failure.\n");
	intelhex_fileclose(&ihex);
	power_off(aspp,opt_reset);
        if (full_write_ok==1) exit(0); /* exit with success */
	exit(1); /* exit with error */
	}
    }

    power_off(aspp,opt_reset);
    exit(0);
}
