## Stateless Mode

Configure your webserver to respond statelessly to challenges for a given account key. This requires nothing more than a one-time web server configuration change and no "moving parts".

1. First get your account key thumbprint:
    ```
    root@ed:~# acme.sh --register-account
    [Mon Feb  6 21:40:18 CST 2017] Registering account
    [Mon Feb  6 21:40:19 CST 2017] Already registered
    [Mon Feb  6 21:40:21 CST 2017] Update success.
    [Mon Feb  6 21:40:21 CST 2017] ACCOUNT_THUMBPRINT='6fXAG9VyG0IahirPEU2ZerUtItW2DHzDzD9wZaEKpqd'
    ```
    Remember the thumbprint in the last line:
    `
    6fXAG9VyG0IahirPEU2ZerUtItW2DHzDzD9wZaEKpqd
    `
1. Configure the web server to return the account key thumbprint:
    ### NGINX
    Add something similar to your `nginx.conf`:
    ```nginx
    http {
    ...
      server {
      ...
        location ~ ^/\.well-known/acme-challenge/([-_a-zA-Z0-9]+)$ {
          default_type text/plain;
          return 200 "$1.6fXAG9VyG0IahirPEU2ZerUtItW2DHzDzD9wZaEKpqd";
        }
      ...
      }
    }
    ```
    ### CADDY
    Add something similar to your `Caddyfile`:
    ```Caddyfile
    example.com {
      @achallenge {
        path_regexp ch ^/\.well-known/acme-challenge/([-_a-zA-Z0-9]+)$
     }
     respond @achallenge "{re.ch.1}.6fXAG9VyG0IahirPEU2ZerUtItW2DHzDzD9wZaEKpqd"
    ```
    ### APACHE

    _Tested on Apache 2.4.62_  

    Add something similar to your `httpd.conf`:
    ```apache
    ...
    <VirtualHost *:80>
        ...
        <LocationMatch "/.well-known/acme-challenge/(?<challenge>[-_a-zA-Z0-9]+)">
            RewriteEngine On
            RewriteRule "^([-_a-zA-Z0-9]+)$" "$1" [E=challenge:$1]
            ErrorDocument 200 "%{ENV:MATCH_CHALLENGE}.6fXAG9VyG0IahirPEU2ZerUtItW2DHzDzD9wZaEKpqd"
            RewriteRule ^ - [L,R=200]
        </LocationMatch>
        ...
    </VirtualHost>
    ...
    ```

    ### HAPROXY
    Add the http-request return rule to your configuration:
    ```haproxy
    global
        setenv ACCOUNT_THUMBPRINT '6fXAG9VyG0IahirPEU2ZerUtItW2DHzDzD9wZaEKpqd'
        log stderr local0
        stats socket /var/run/haproxy.sock level admin mode 0666

    frontend web
        log global
        option httplog
        mode  http
        bind :80
        bind :443 ssl crt /etc/haproxy/certs/
        http-request return status 200 content-type text/plain lf-string "%[path,field(-1,/)].${ACCOUNT_THUMBPRINT}\n" if { path_reg '^/.well-known/acme-challenge/[-_a-zA-Z0-9]+$' }
    ```
3. Ok, you can issue cert now.
  ```
acme.sh --issue -d example.com  --stateless
```


