/* MediaManager.m - Window and menu support
   class to handle disk, cartridge, cassette,
   and executable file management and support 
   functions for the Macintosh OS X SDL port 
   of Atari800
   Mark Grebe <atarimac@cox.net>
   
   Based on the Preferences pane of the
   TextEdit application.

*/
#import <Cocoa/Cocoa.h>
#import "MediaManager.h"
#import "atari.h"
#import "binload.h"
#import "cartridge.h"
#import "cassette.h"
#import "sio.h"

/* Definition of Mac native keycodes for characters used as menu shortcuts the bring up a windo. */
#define QZ_c			0x08
#define QZ_d			0x02
#define QZ_o			0x1F
#define QZ_r			0x0F
#define QZ_1			0x12

/* ATR Disk header */
typedef struct {
    UBYTE id[4];
    UBYTE type[4];
    UBYTE checksum[4];
    UBYTE gash[4];
} Header;

extern void PauseAudio(int pause);
extern UnitStatus drive_status[MAX_DRIVES];
extern char sio_filename[MAX_DRIVES][FILENAME_MAX];
extern char atari_disk_dirs[][FILENAME_MAX];
extern char atari_diskset_dir[FILENAME_MAX];
extern char atari_rom_dir[FILENAME_MAX];
extern char atari_exe_dir[FILENAME_MAX];
extern int cart_type;
extern int requestCaptionChange;

/* Arrays which define the cartridge types for each size */
static int CART8KTYPES[] = {CART_STD_8, CART_5200_8, CART_RIGHT_8};
static int CART16KTYPES[] = {CART_STD_16, CART_OSS_16, CART_5200_EE_16, 
                             CART_OSS2_16, CART_5200_NS_16, CART_MEGA_16};
static int CART32KTYPES[] = {CART_5200_32, CART_DB_32, CART_XEGS_32, 
                             CART_WILL_32, CART_MEGA_32, CART_SWXEGS_32};
static int CART40KTYPES[] = {CART_5200_40, CART_BBSB_40};
static int CART64KTYPES[] = {CART_WILL_64, CART_EXP_64, CART_DIAMOND_64, 
                             CART_SDX_64, CART_XEGS_64, CART_MEGA_64, CART_SWXEGS_64};
static int CART128KTYPES[] = {CART_XEGS_128, CART_ATRAX_128, CART_MEGA_128, CART_SWXEGS_128};
static int CART256KTYPES[] = {CART_XEGS_256, CART_MEGA_256, CART_SWXEGS_256};
static int CART512KTYPES[] = {CART_XEGS_512, CART_MEGA_512, CART_SWXEGS_512};
static int CART1024KTYPES[] = {CART_XEGS_1024, CART_MEGA_1024, CART_SWXEGS_1024};

/* Functions which provide an interface for C code to call this object's shared Instance functions */
void UpdateMediaManagerInfo() {
    [[MediaManager sharedInstance] updateInfo];
}

void MediaManagerRunDiskManagement() {
    [[MediaManager sharedInstance] showManagementPanel:nil];
}

void MediaManagerInsertCartridge() {
    [[MediaManager sharedInstance] cartInsert:nil];
}

void MediaManagerRemoveCartridge() {
    [[MediaManager sharedInstance] cartRemove:nil];
}

void MediaManagerInsertDisk(int diskNum) {
    [[MediaManager sharedInstance] diskInsertKey:diskNum];
}

void MediaManagerRemoveDisk(int diskNum) {
    if (diskNum == 0)
        [[MediaManager sharedInstance] diskRemoveAll:nil];
    else
        [[MediaManager sharedInstance] diskRemoveKey:diskNum];
}

void MediaManagerLoadExe() {
    [[MediaManager sharedInstance] loadExeFile:nil];
}


@implementation MediaManager

static MediaManager *sharedInstance = nil;

+ (MediaManager *)sharedInstance {
    return sharedInstance ? sharedInstance : [[self alloc] init];
}

- (id)init {
    if (sharedInstance) {
	[self dealloc];
    } else {
        [super init];
        sharedInstance = self;
        /* load the nib and all the windows */
        if (!d1DiskField) {
            if (![NSBundle loadNibNamed:@"MediaManager" owner:self])  {
                NSLog(@"Failed to load MediaManager.nib");
                NSBeep();
                return nil;
                }
            }
	[[diskFmtMatrix window] setExcludedFromWindowsMenu:YES];
	[[diskFmtMatrix window] setMenu:nil];
	[[d1DiskField window] setExcludedFromWindowsMenu:YES];
	[[d1DiskField window] setMenu:nil];
	[[errorButton window] setExcludedFromWindowsMenu:YES];
	[[errorButton window] setMenu:nil];
	[[cart8KMatrix window] setExcludedFromWindowsMenu:YES];
	[[cart8KMatrix window] setMenu:nil];
	[[cart16KMatrix window] setExcludedFromWindowsMenu:YES];
	[[cart16KMatrix window] setMenu:nil];
	[[cart32KMatrix window] setExcludedFromWindowsMenu:YES];
	[[cart32KMatrix window] setMenu:nil];
	[[cart40KMatrix window] setExcludedFromWindowsMenu:YES];
	[[cart40KMatrix window] setMenu:nil];
	[[cart64KMatrix window] setExcludedFromWindowsMenu:YES];
	[[cart64KMatrix window] setMenu:nil];
	[[cart128KMatrix window] setExcludedFromWindowsMenu:YES];
	[[cart128KMatrix window] setMenu:nil];
	[[cart256KMatrix window] setExcludedFromWindowsMenu:YES];
	[[cart256KMatrix window] setMenu:nil];
	[[cart512KMatrix window] setExcludedFromWindowsMenu:YES];
	[[cart512KMatrix window] setMenu:nil];
	[[cart1024KMatrix window] setExcludedFromWindowsMenu:YES];
	[[cart1024KMatrix window] setMenu:nil];
        }
    return sharedInstance;
}

- (void)dealloc {
}


/*------------------------------------------------------------------------------
*  displayError - This method displays an error dialog box with the passed in
*     error message.
*-----------------------------------------------------------------------------*/
- (void)displayError:(NSString *)errorMsg {
    [errorField setStringValue:errorMsg];
    [NSApp runModalForWindow:[errorButton window]];
}

/*------------------------------------------------------------------------------
*  updateInfo - This method is used to update the disk management window GUI.
*-----------------------------------------------------------------------------*/
- (void)updateInfo {
    int i;
    int noDisks = TRUE;
    
    for (i=0;i<8;i++) {
        if (drive_status[i] == Off)
            strcpy(sio_filename[i],"Off");
        switch(i) {
            case 0:
                [d1DiskField setStringValue:[NSString stringWithCString:sio_filename[0]]];
                [d1DriveStatusPulldown selectItemAtIndex:drive_status[0]];
                if (drive_status[0] == Off || drive_status[0] == NoDisk)
                    [removeD1Item setTarget:nil];
                else {
                    [removeD1Item setTarget:self];
                    noDisks = FALSE;
                    }
                break;
            case 1:
                [d2DiskField setStringValue:[NSString stringWithCString:sio_filename[1]]];
                [d2DriveStatusPulldown selectItemAtIndex:drive_status[1]];
                if (drive_status[1] == Off || drive_status[1] == NoDisk)
                    [removeD2Item setTarget:nil];
                else {
                    [removeD2Item setTarget:self];
                    noDisks = FALSE;
                    }
            case 2:
                [d3DiskField setStringValue:[NSString stringWithCString:sio_filename[2]]];
                [d3DriveStatusPulldown selectItemAtIndex:drive_status[2]];
                if (drive_status[2] == Off || drive_status[2] == NoDisk)
                    [removeD3Item setTarget:nil];
                else {
                    [removeD3Item setTarget:self];
                    noDisks = FALSE;
                    }
                break;
            case 3:
                [d4DiskField setStringValue:[NSString stringWithCString:sio_filename[3]]];
                [d4DriveStatusPulldown selectItemAtIndex:drive_status[3]];
                if (drive_status[3] == Off || drive_status[3] == NoDisk)
                    [removeD4Item setTarget:nil];
                else {
                    [removeD4Item setTarget:self];
                    noDisks = FALSE;
                    }
                break;
            case 4:
                [d5DiskField setStringValue:[NSString stringWithCString:sio_filename[4]]];
                [d5DriveStatusPulldown selectItemAtIndex:drive_status[4]];
                if (drive_status[4] == Off || drive_status[4] == NoDisk)
                    [removeD5Item setTarget:nil];
                else {
                    [removeD5Item setTarget:self];
                    noDisks = FALSE;
                    }
                break;
            case 5:
                [d6DiskField setStringValue:[NSString stringWithCString:sio_filename[5]]];
                [d6DriveStatusPulldown selectItemAtIndex:drive_status[5]];
                if (drive_status[5] == Off || drive_status[5] == NoDisk)
                    [removeD6Item setTarget:nil];
                else {
                    [removeD6Item setTarget:self];
                    noDisks = FALSE;
                    }
                break;
            case 6:
                [d7DiskField setStringValue:[NSString stringWithCString:sio_filename[6]]];
                [d7DriveStatusPulldown selectItemAtIndex:drive_status[6]];
                if (drive_status[6] == Off || drive_status[6] == NoDisk)
                    [removeD7Item setTarget:nil];
                else {
                    [removeD7Item setTarget:self];
                    noDisks = FALSE;
                    }
                break;
            case 7:
                [d8DiskField setStringValue:[NSString stringWithCString:sio_filename[7]]];
                [d8DriveStatusPulldown selectItemAtIndex:drive_status[7]];
                if (drive_status[7] == Off || drive_status[7] == NoDisk)
                    [removeD8Item setTarget:nil];
                else {
                    [removeD8Item setTarget:self];
                    noDisks = FALSE;
                    }
                break;
            }
        }
        if (noDisks) 
            [removeMenu setTarget:nil];
        else 
            [removeMenu setTarget:self];
        if (cart_type == CART_NONE)
            [removeCartItem setTarget:nil];
        else
            [removeCartItem setTarget:self];
        if ((strcmp(cassette_filename, "None") == 0) || (strlen(cassette_filename) == 0)) {
            [removeCassItem setTarget:nil];
            [rewindCassItem setTarget:nil];
            }
        else {
            [removeCassItem setTarget:self];
            [rewindCassItem setTarget:self];
            }

}

/*------------------------------------------------------------------------------
*  browseFileInDirectory - This allows the user to chose a file to read in from
*     the specified directory.
*-----------------------------------------------------------------------------*/
- (NSString *) browseFileInDirectory:(NSString *)directory {
    NSOpenPanel *openPanel = nil;
    
    openPanel = [NSOpenPanel openPanel];
    [openPanel setCanChooseDirectories:NO];
    [openPanel setCanChooseFiles:YES];
    
    if ([openPanel runModalForDirectory:directory file:nil types:nil] == NSOKButton)
        return([[openPanel filenames] objectAtIndex:0]);
    else
        return nil;
    }

/*------------------------------------------------------------------------------
*  browseFileTypeInDirectory - This allows the user to chose a file of a 
*     specified typeto read in from the specified directory.
*-----------------------------------------------------------------------------*/
- (NSString *) browseFileTypeInDirectory:(NSString *)directory:(NSString *)type {
    NSOpenPanel *openPanel = nil;
    
    openPanel = [NSOpenPanel openPanel];
    [openPanel setCanChooseDirectories:NO];
    [openPanel setCanChooseFiles:YES];
    
    if ([openPanel runModalForDirectory:directory file:nil 
            types:[NSArray arrayWithObjects:type, nil]] == NSOKButton)
        return([[openPanel filenames] objectAtIndex:0]);
    else
        return nil;
    }

/*------------------------------------------------------------------------------
*  saveFileInDirectory - This allows the user to chose a filename to save in from
*     the specified directory.
*-----------------------------------------------------------------------------*/
- (NSString *) saveFileInDirectory:(NSString *)directory:(NSString *)type {
    NSSavePanel *savePanel = nil;
    
    savePanel = [NSSavePanel savePanel];
    
    [savePanel setRequiredFileType:type];
    
    if ([savePanel runModalForDirectory:directory file:nil] == NSOKButton)
        return([savePanel filename]);
    else
        return nil;
    }

/*------------------------------------------------------------------------------
*  cancelDisk - This method handles the cancel button from the disk image
*     creation window.
*-----------------------------------------------------------------------------*/
- (IBAction)cancelDisk:(id)sender
{
    [NSApp stopModal];
    [[diskFmtMatrix window] close];
}

/*------------------------------------------------------------------------------
*  cartInsert - This method inserts a cartridge image into the emulator
*-----------------------------------------------------------------------------*/
- (IBAction)cartInsert:(id)sender
{
    NSString *filename;
    char cfilename[FILENAME_MAX];
    int cartSize;

    PauseAudio(1);
    filename = [self browseFileInDirectory:[NSString stringWithCString:atari_rom_dir]];
    if (filename != nil) {
        [filename getCString:cfilename];
        cartSize = CART_Insert(cfilename);
        if (cartSize > 0) 
            cart_type = [self cartSelect:cartSize];
        if (cart_type != CART_NONE) {
            int for5200 = CART_IsFor5200(cart_type);
            if (for5200 && machine_type != MACHINE_5200) {
                machine_type = MACHINE_5200;
                ram_size = 16;
                Atari800_InitialiseMachine();
                requestCaptionChange = 1;
                }
            else if (!for5200 && machine_type == MACHINE_5200) {
                machine_type = MACHINE_XLXE;
                ram_size = 64;
                Atari800_InitialiseMachine();
                requestCaptionChange = 1;
                }
            }
        Coldstart();
        }
    [self updateInfo];
    [self releaseCmdKeys:@"o":QZ_o];
    PauseAudio(0);
}

/*------------------------------------------------------------------------------
*  cartInsertFile - This method inserts a cartridge image into the emulator,
*     given it's filename.
*-----------------------------------------------------------------------------*/
- (void)cartInsertFile:(NSString *)filename
{
    char cfilename[FILENAME_MAX];
    int cartSize;

    if (filename != nil) {
        [filename getCString:cfilename];
        cartSize = CART_Insert(cfilename);
        if (cartSize > 0) {
            cart_type = [self cartSelect:cartSize];
            }
        if (cart_type != CART_NONE) {
            int for5200 = CART_IsFor5200(cart_type);
            if (for5200 && machine_type != MACHINE_5200) {
                machine_type = MACHINE_5200;
                ram_size = 16;
                Atari800_InitialiseMachine();
                }
            else if (!for5200 && machine_type == MACHINE_5200) {
                machine_type = MACHINE_XLXE;
                ram_size = 64;
                Atari800_InitialiseMachine();
                }
            }
        Coldstart();
        }
    [self updateInfo];
}

/*------------------------------------------------------------------------------
*  cassRemove - This method removes a cartridge image from the emulator
*-----------------------------------------------------------------------------*/
- (IBAction)cartRemove:(id)sender
{
    CART_Remove();
    [self updateInfo];
    Coldstart();
}

/*------------------------------------------------------------------------------
*  cassSelect - This method displays a dialog box so that the user can select
*     a cartridge type based on the passed in size in Kbytes.
*-----------------------------------------------------------------------------*/
- (int)cartSelect:(int)cartSize 
{
    int cartType;
    NSWindow *theWindow;

    switch (cartSize) {
        case 8:
            theWindow = [cart8KMatrix window];
            break;
        case 16:
            theWindow = [cart16KMatrix window];
            break;
        case 32:
            theWindow = [cart32KMatrix window];
            break;
        case 40:
            theWindow = [cart40KMatrix window];
            break;
        case 64:
            theWindow = [cart64KMatrix window];
            break;
        case 128:
            theWindow = [cart128KMatrix window];
            break;
        case 256:
            theWindow = [cart256KMatrix window];
            break;
        case 512:
            theWindow = [cart512KMatrix window];
            break;
        case 1024:
            theWindow = [cart1024KMatrix window];
            break;
        default:
            return(CART_NONE);
        }
    cartType = [NSApp runModalForWindow:theWindow];
    return(cartType);             
}

/*------------------------------------------------------------------------------
*  cartSelectOK - This method handles the OK button from the cartridge selection
*     window, and sets the cartridge type appropriately.
*-----------------------------------------------------------------------------*/
- (IBAction)cartSelectOK:(id)sender 
{
    int cartSize = [sender tag];
    int cartType;
    
    switch(cartSize) {
        case 8:
            cartType = CART8KTYPES[[[cart8KMatrix selectedCell] tag]];
            break;
        case 16:
            cartType = CART16KTYPES[[[cart16KMatrix selectedCell] tag]];
            break;
        case 32:
            cartType = CART32KTYPES[[[cart32KMatrix selectedCell] tag]];
            break;
        case 40:
            cartType = CART40KTYPES[[[cart40KMatrix selectedCell] tag]];
            break;
        case 64:
            cartType = CART64KTYPES[[[cart64KMatrix selectedCell] tag]];
            break;
        case 128:
            cartType = CART128KTYPES[[[cart128KMatrix selectedCell] tag]];
            break;
        case 256:
            cartType = CART256KTYPES[[[cart256KMatrix selectedCell] tag]];
            break;
        case 512:
            cartType = CART512KTYPES[[[cart512KMatrix selectedCell] tag]];
            break;
        case 1024:
            cartType = CART1024KTYPES[[[cart1024KMatrix selectedCell] tag]];
            break;
        default:
            cartType = 0;
        }
    
    [NSApp stopModalWithCode:cartType];
    [[sender window] close];
}

/*------------------------------------------------------------------------------
*  cartSelectOK - This method handles the cancel button from the cartridge 
*     selection window, and sets the cartridge type appropriately.
*-----------------------------------------------------------------------------*/
- (IBAction)cartSelectCancel:(id)sender
{
    [NSApp stopModalWithCode:CART_NONE];
    [[sender window] close];
}

/*------------------------------------------------------------------------------
*  cassInsert - This method inserts a cassette image into the emulator
*-----------------------------------------------------------------------------*/
- (IBAction)cassInsert:(id)sender
{
    NSString *filename;
    char tapename[FILENAME_MAX+1];
    int ret = FALSE;
    
    PauseAudio(1);
    filename = [self browseFileInDirectory:[NSString stringWithCString:atari_exe_dir]];
    if (filename != nil) {
        [filename getCString:tapename];
        ret = CASSETTE_Insert(tapename);
        if (! ret) 
            [self displayError:@"Unable to Insert Cassette!"];
        }
    PauseAudio(0);
    [self updateInfo];
}

/*------------------------------------------------------------------------------
*  cassInsertFile - This method inserts a cassette image into the emulator, 
*     given it's filename
*-----------------------------------------------------------------------------*/
- (void)cassInsertFile:(NSString *)filename
{
    char tapename[FILENAME_MAX+1];
    int ret = FALSE;
    
    if (filename != nil) {
        [filename getCString:tapename];
        ret = CASSETTE_Insert(tapename);
        if (! ret) 
            [self displayError:@"Unable to Insert Cassette!"];
        }
    [self updateInfo];
}

/*------------------------------------------------------------------------------
*  cassRewind - This method removes the inserted cassette.
*-----------------------------------------------------------------------------*/
- (IBAction)cassRemove:(id)sender
{
    CASSETTE_Remove();
    [self updateInfo];
}

/*------------------------------------------------------------------------------
*  cassRewind - This method rewinds the inserted cassette.
*-----------------------------------------------------------------------------*/
- (IBAction)cassRewind:(id)sender
{
    cassette_current_block = 1;
}

/*------------------------------------------------------------------------------
*  convertCartRom - This method converts a .cart image into a ROM dump file.
*-----------------------------------------------------------------------------*/
- (IBAction)convertCartRom:(id)sender
{
    UBYTE* image;
    NSString *filename;
    Header header;
    char cfilename[FILENAME_MAX+1];
    int nbytes;
    FILE *f;
    
    filename = [self browseFileInDirectory:[NSString stringWithCString:atari_rom_dir]];
    if (filename != nil) {
        [filename getCString:cfilename];
        f = fopen(cfilename, "rb");
	if (!f) {
            [self displayError:@"Unable to Open Cartridge File!"];
            return;
            }
        image = malloc(CART_MAX_SIZE+1);
        if (image == NULL) {
            fclose(f);
            [self displayError:@"Unable to Create ROM Image!"];
            }
        nbytes = fread((char *) &header, 1, sizeof(Header), f);
        nbytes = fread(image, 1, CART_MAX_SIZE + 1, f);

        fclose(f);
        filename = [self saveFileInDirectory:[NSString stringWithCString:atari_rom_dir]:@"rom"];
                
        if (filename == nil) {
            free(image);
            return;
            }
                    
        [filename getCString:cfilename];
	f = fopen(cfilename, "wb");
	if (f) {
            fwrite(image, 1, nbytes, f);
            fclose(f);
            }
	free(image);
	}
    
}

/*------------------------------------------------------------------------------
*  convertRomCart - This method converts a ROM dump file into a .cart image.
*-----------------------------------------------------------------------------*/
- (IBAction)convertRomCart:(id)sender
{
    UBYTE *image;
    int nbytes;
    Header header;
    int checksum;
    int type;
    FILE *f;
    NSString *filename;
    char cfilename[FILENAME_MAX+1];
    
    filename = [self browseFileInDirectory:[NSString stringWithCString:atari_rom_dir]];
    if (filename != nil) {
        [filename getCString:cfilename];
	f = fopen(cfilename, "rb");
	if (!f) {
            [self displayError:@"Unable to Open ROM File!"];
            return;
            }
	image = malloc(CART_MAX_SIZE+1);
	if (image == NULL) {
            fclose(f);
            [self displayError:@"Unable to Create Cart File"];
            return;
            }
	nbytes = fread(image, 1, CART_MAX_SIZE + 1, f);
	fclose(f);
	if ((nbytes & 0x3ff) == 0) {
            type = [self cartSelect:(nbytes/1024)];
            if (type != CART_NONE) {
                checksum = CART_Checksum(image, nbytes);

                header.id[0] = 'C';
                header.id[1] = 'A';
                header.id[2] = 'R';
                header.id[3] = 'T';
                header.type[0] = (type >> 24) & 0xff;
                header.type[1] = (type >> 16) & 0xff;
                header.type[2] = (type >> 8) & 0xff;
                header.type[3] = type & 0xff;
                header.checksum[0] = (checksum >> 24) & 0xff;
                header.checksum[1] = (checksum >> 16) & 0xff;
                header.checksum[2] = (checksum >> 8) & 0xff;
                header.checksum[3] = checksum & 0xff;
                header.gash[0] = '\0';
                header.gash[1] = '\0';
                header.gash[2] = '\0';
                header.gash[3] = '\0';
                filename = [self saveFileInDirectory:[NSString stringWithCString:atari_rom_dir]:@"car"];
                
                if (filename == nil) {
                    free(image);
                    return;
                    }
                    
                [filename getCString:cfilename];
                f = fopen(cfilename, "wb");
                if (f) {
                    fwrite(&header, 1, sizeof(header), f);
                    fwrite(image, 1, nbytes, f);
                    fclose(f);
		}
            }
	}
	free(image);
    }
}

/*------------------------------------------------------------------------------
*  createDisk - This method responds to the create disk button push in the disk
*     creation window, and actually creates the disk image.
*-----------------------------------------------------------------------------*/
- (IBAction)createDisk:(id)sender
{
    ULONG bytesInBootSector;
    ULONG bytesPerSector;
    ULONG sectors;
    ULONG imageLength;
    FILE *image = NULL;
    NSString *filename;
    char cfilename[FILENAME_MAX];
    struct ATR_Header atrHeader;
    int diskMounted;
    int i;
    
    bytesInBootSector = ([diskFmtDDBytesPulldown indexOfSelectedItem] + 1) * 128;
    bytesPerSector = ([diskFmtCusBytesPulldown indexOfSelectedItem] + 1) * 128;
    sectors = [diskFmtCusSecField intValue];
    
    if (sectors <= 3)
        imageLength = sectors * bytesInBootSector / 16;
    else
        imageLength = ((sectors - 3) * bytesPerSector + 3 * bytesInBootSector) / 16;
    
    filename = [self saveFileInDirectory:[NSString stringWithCString:atari_disk_dirs[0]]:@"atr"];
    if (filename != nil) {
        [filename getCString:cfilename];
        image = fopen(cfilename, "wb");
        if (image == NULL) {
            [self displayError:@"Unable to Create Disk Image!"];
            }
        else {
            atrHeader.magic1 = MAGIC1;
            atrHeader.magic2 = MAGIC2;
            atrHeader.secsizelo = bytesPerSector & 0x00ff;
            atrHeader.secsizehi = (bytesPerSector & 0xff00) >> 8;
            atrHeader.seccountlo = imageLength & 0x00ff;
            atrHeader.seccounthi = (imageLength & 0xff00) >> 8;
            atrHeader.hiseccountlo = (imageLength & 0x00ff0000) >> 16;
            atrHeader.hiseccounthi = (imageLength & 0xff000000) >> 24;
            for (i=0;i<8;i++)
                atrHeader.gash[i] = 0;
            atrHeader.writeprotect = 0;
            
            fwrite(&atrHeader, sizeof(struct ATR_Header), 1, image);
            
            for (i = 0; i < imageLength; i++)
                fwrite("\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000",16,1,image);
                
            fflush(image);
            fclose(image);
            }
        }
        
    if ([diskFmtInsertNewButton state] == NSOnState) {
        diskMounted = SIO_Mount([diskFmtInsertDrivePulldown indexOfSelectedItem] + 1, cfilename, 0);
        if (!diskMounted)
            [self displayError:@"Unable to Mount Disk Image!"];
        [self updateInfo];
        }
    
    [NSApp stopModal];
    [[diskFmtMatrix window] close];
}

/*------------------------------------------------------------------------------
*  diskInsert - This method inserts a floppy disk in the specified drive in
*     response to a menu.
*-----------------------------------------------------------------------------*/
- (IBAction)diskInsert:(id)sender
{
    int diskNum = [sender tag] - 1;
    int readOnly;
    NSString *filename;
    char cfilename[FILENAME_MAX];
    int diskMounted;
    
    PauseAudio(1);
    readOnly = (drive_status[diskNum] == ReadOnly ? TRUE : FALSE);
    filename = [self browseFileInDirectory:[NSString stringWithCString:atari_disk_dirs[0]]];
    if (filename != nil) {
        SIO_Dismount(diskNum + 1);
        [filename getCString:cfilename];
        diskMounted = SIO_Mount(diskNum + 1, cfilename, readOnly);
        if (!diskMounted)
            [self displayError:@"Unable to Mount Disk Image!"];
        [self updateInfo];
        }
    PauseAudio(0);
}

/*------------------------------------------------------------------------------
*  diskRotate - This method rotates the floppy disks between drivers in
*     response to a menu.
*-----------------------------------------------------------------------------*/
- (IBAction)diskRotate:(id)sender
{
    Rotate_Disks();
    [self updateInfo];
}

/*------------------------------------------------------------------------------
*  diskInsertFile - This method inserts a floppy disk into drive 1, given its
*     filename.
*-----------------------------------------------------------------------------*/
- (void)diskInsertFile:(NSString *)filename
{
    int readOnly;
    char cfilename[FILENAME_MAX];
    int diskMounted;
    
    readOnly = (drive_status[0] == ReadOnly ? TRUE : FALSE);
    if (filename != nil) {
        SIO_Dismount(1);
        [filename getCString:cfilename];
        diskMounted = SIO_Mount(1, cfilename, readOnly);
        if (!diskMounted)
            [self displayError:@"Unable to Mount Disk Image!"];
        [self updateInfo];
        Coldstart();
        }
}

/*------------------------------------------------------------------------------
*  diskInsertKey - This method inserts a floppy disk in the specified drive in
*     response to a keyboard shortcut.
*-----------------------------------------------------------------------------*/
- (IBAction)diskInsertKey:(int)diskNum
{
    int readOnly;
    NSString *filename;
    char cfilename[FILENAME_MAX];
    int diskMounted;
    
    PauseAudio(1);
    readOnly = (drive_status[diskNum] == ReadOnly ? TRUE : FALSE);
    filename = [self browseFileInDirectory:[NSString stringWithCString:atari_disk_dirs[0]]];
    if (filename != nil) {
        SIO_Dismount(diskNum);
        [filename getCString:cfilename];
        diskMounted = SIO_Mount(diskNum, cfilename, readOnly);
        if (!diskMounted)
            [self displayError:@"Unable to Mount Disk Image!"];
        [self updateInfo];
        }
    PauseAudio(0);
    [self releaseCmdKeys:@"1":(QZ_1 + diskNum - 1)];
}

/*------------------------------------------------------------------------------
*  diskRemove - This method removes a floppy disk in the specified drive in
*     response to a menu.
*-----------------------------------------------------------------------------*/
- (IBAction)diskRemove:(id)sender
{
    int diskNum = [sender tag] - 1;

    SIO_Dismount(diskNum + 1);
    [self updateInfo];
}

/*------------------------------------------------------------------------------
*  diskRemoveKey - This method removes a floppy disk in the specified drive in
*     response to a keyboard shortcut.
*-----------------------------------------------------------------------------*/
- (IBAction)diskRemoveKey:(int)diskNum
{
    SIO_Dismount(diskNum );
    [self updateInfo];
}

/*------------------------------------------------------------------------------
*  diskRemoveAll - This method removes disks from all of the floppy drives.
*-----------------------------------------------------------------------------*/
- (IBAction)diskRemoveAll:(id)sender
{
    int i;

    for (i=0;i<8;i++)
        SIO_Dismount(i + 1);
    [self updateInfo];
}

/*------------------------------------------------------------------------------
*  diskSetSave - This method saves the names of the mounted disks to a file
*      chosen by the user.
*-----------------------------------------------------------------------------*/
- (IBAction)diskSetSave:(id)sender
{
    NSString *filename;
    char cfilename[FILENAME_MAX+1];
    FILE *f;
    int i;

    filename = [self saveFileInDirectory:[NSString stringWithCString:atari_diskset_dir]:@"set"];
    
    if (filename == nil)
        return;
                    
    [filename getCString:cfilename];
    f = fopen(cfilename, "w");
    if (f) {
        for (i=0;i<8;i++) {
            fputs(sio_filename[i],f);
            fprintf(f,"\n");
            }
        fclose(f);
        }
}

/*------------------------------------------------------------------------------
*  diskSetLoad - This method mounts the set of disk images from a file
*      chosen by the user.
*-----------------------------------------------------------------------------*/
- (IBAction)diskSetLoad:(id)sender
{
    NSString *filename;
    char cfilename[FILENAME_MAX+1];
    char diskname[FILENAME_MAX+1];
    FILE *f;
    int i, mounted, readOnly;
    int numMountErrors = 0;
    int mountErrors[8];

    filename = [self browseFileTypeInDirectory:
                  [NSString stringWithCString:atari_diskset_dir]:@"set"];
    
    if (filename == nil)
        return;
    
    [filename getCString:cfilename];
    f = fopen(cfilename, "r");
    if (f) {
        for (i=0;i<8;i++) {
            fgets(diskname,FILENAME_MAX,f);
            if (strlen(diskname) != 0)
                diskname[strlen(diskname)-1] = 0;
            if ((strcmp(diskname,"Off") != 0) && (strcmp(diskname,"Empty") != 0)) {
                readOnly = (drive_status[i] == ReadOnly ? TRUE : FALSE);
                SIO_Dismount(i+1);
                mounted = SIO_Mount(i+1, diskname, readOnly);
                if (!mounted) {
                    numMountErrors++;
                    mountErrors[i] = 1;
                    }
                else
                    mountErrors[i] = 0;
                }
            else
                mountErrors[i] = 0;
            }
        fclose(f);
        if (numMountErrors != 0) 
            [self displayError:@"Unable to Mount Disk Image!"];
        [self updateInfo];
        }
}

/*------------------------------------------------------------------------------
*  diskSetLoad - This method mounts the set of disk images from a file
*      specified by the filename parameter.
*-----------------------------------------------------------------------------*/
- (IBAction)diskSetLoadFile:(NSString *)filename
{
    char cfilename[FILENAME_MAX+1];
    char diskname[FILENAME_MAX+1];
    FILE *f;
    int i, readOnly;

    [filename getCString:cfilename];
    f = fopen(cfilename, "r");
    if (f) {
        for (i=0;i<8;i++) {
            fgets(diskname,FILENAME_MAX,f);
            if (strlen(diskname) != 0)
                diskname[strlen(diskname)-1] = 0;
            if ((strcmp(diskname,"Off") != 0) && (strcmp(diskname,"Empty") != 0)) {
                readOnly = (drive_status[i] == ReadOnly ? TRUE : FALSE);
                SIO_Dismount(i+1);
                SIO_Mount(i+1, diskname, readOnly);
                }
            }
        fclose(f);
        [self updateInfo];
        }
}

/*------------------------------------------------------------------------------
*  driveStatusChange - This method handles changes in the drive status controls 
*     in the disk management window.
*-----------------------------------------------------------------------------*/
- (IBAction)driveStatusChange:(id)sender
{
    int diskNum = [sender tag] - 1;
    char tempFilename[FILENAME_MAX];
    int readOnly;
    
    readOnly = (drive_status[diskNum] == ReadOnly ? TRUE : FALSE);
    
    switch([sender indexOfSelectedItem]) {
        case 0:
            if (drive_status[diskNum] == ReadOnly || drive_status[diskNum] == ReadWrite) 
                SIO_Dismount(diskNum+1);
            SIO_DisableDrive(diskNum+1);
            break;
        case 1:
            if (drive_status[diskNum] == ReadOnly || drive_status[diskNum] == ReadWrite) 
                SIO_Dismount(diskNum+1);
            else {
                drive_status[diskNum] = NoDisk;
                strcpy(sio_filename[diskNum],"Empty");
                }
            break;
        case 2:
            if (drive_status[diskNum] == ReadWrite) {
                strcpy(tempFilename, sio_filename[diskNum]);
                SIO_Dismount(diskNum+1);
                SIO_Mount(diskNum+1, tempFilename, TRUE);
                }
            else
                [sender selectItemAtIndex:drive_status[diskNum]];
            break;
        case 3:
            if (drive_status[diskNum] == ReadOnly) {
                strcpy(tempFilename, sio_filename[diskNum]);
                SIO_Dismount(diskNum+1);
                SIO_Mount(diskNum+1, tempFilename, FALSE);
                }
            else
                [sender selectItemAtIndex:drive_status[diskNum]];
            break;
        }
    [self updateInfo];
}

/*------------------------------------------------------------------------------
*  loadExeFile - This method loads an Atari executable file into the emulator.
*-----------------------------------------------------------------------------*/
- (IBAction)loadExeFile:(id)sender
{
    NSString *filename;
    char exename[FILENAME_MAX+1];
    int ret = FALSE;
    
    PauseAudio(1);
    filename = [self browseFileInDirectory:[NSString stringWithCString:atari_exe_dir]];
    if (filename != nil) {
        [filename getCString:exename];
        ret = BIN_loader(exename);
        if (! ret) 
            [self displayError:@"Unable to Load Binary File!"];
    }
    PauseAudio(0);
    [self releaseCmdKeys:@"r":QZ_r];
}

/*------------------------------------------------------------------------------
*  loadExeFileFile - This method loads an Atari executable file into the 
*       emulator given its filename.
*-----------------------------------------------------------------------------*/
- (void)loadExeFileFile:(NSString *)filename
{
    char exename[FILENAME_MAX+1];
    int ret = FALSE;
    
    if (filename != nil) {
        [filename getCString:exename];
        ret = BIN_loader(exename);
        if (! ret) 
            [self displayError:@"Unable to Load Binary File!"];
    }
}

/*------------------------------------------------------------------------------
*  miscUpdate - This method handles control updates in the disk image creation
*     window.
*-----------------------------------------------------------------------------*/
- (IBAction)miscUpdate:(id)sender
{
    if (sender == diskFmtMatrix) {
        switch([[diskFmtMatrix selectedCell] tag]) {
            case 0:
                [diskFmtCusBytesPulldown selectItemAtIndex:0];
                [diskFmtCusSecField setIntValue:720];
                [diskFmtDDBytesPulldown selectItemAtIndex:0];
                [diskFmtCusBytesPulldown setEnabled:NO];
                [diskFmtCusSecField setEnabled:NO];
                [diskFmtDDBytesPulldown setEnabled:NO];
                break;
            case 1:
                [diskFmtCusBytesPulldown selectItemAtIndex:0];
                [diskFmtCusSecField setIntValue:1040];
                [diskFmtDDBytesPulldown selectItemAtIndex:0];
                [diskFmtCusBytesPulldown setEnabled:NO];
                [diskFmtCusSecField setEnabled:NO];
                [diskFmtDDBytesPulldown setEnabled:NO];
                break;
            case 2:
                [diskFmtCusBytesPulldown selectItemAtIndex:1];
                [diskFmtCusSecField setIntValue:720];
                [diskFmtDDBytesPulldown selectItemAtIndex:0];
                [diskFmtCusBytesPulldown setEnabled:NO];
                [diskFmtCusSecField setEnabled:NO];
                [diskFmtDDBytesPulldown setEnabled:YES];
                break;
            case 3:
                [diskFmtCusBytesPulldown setEnabled:YES];
                [diskFmtCusSecField setEnabled:YES];
                [diskFmtDDBytesPulldown setEnabled:YES];
                break;
            }
        }
    else if (sender == diskFmtInsertNewButton) {
        if ([diskFmtInsertNewButton state] == NSOnState)
            [diskFmtInsertDrivePulldown setEnabled:YES];
        else
            [diskFmtInsertDrivePulldown setEnabled:NO];        
        }
}

/*------------------------------------------------------------------------------
*  ok - This method handles the OK button press from the disk managment window.
*-----------------------------------------------------------------------------*/
- (IBAction)ok:(id)sender
{
    [NSApp stopModal];
    [[d1DiskField window] close];
    PauseAudio(0);
}

/*------------------------------------------------------------------------------
*  errorOK - This method handles the OK button press from the error window.
*-----------------------------------------------------------------------------*/
- (IBAction)errorOK:(id)sender;
{
    [NSApp stopModal];
    [[errorButton window] close];
}

/*------------------------------------------------------------------------------
*  showCreatePanel - This method displays a window which allows the creation of
*     blank floppy images.
*-----------------------------------------------------------------------------*/
- (IBAction)showCreatePanel:(id)sender
{
    [diskFmtMatrix selectCellWithTag:0];
    [diskFmtCusBytesPulldown setEnabled:NO];
    [diskFmtCusSecField setEnabled:NO];
    [diskFmtDDBytesPulldown setEnabled:NO];
    [diskFmtInsertDrivePulldown setEnabled:NO];
    [diskFmtCusBytesPulldown selectItemAtIndex:0];
    [diskFmtCusSecField setIntValue:720];
    [diskFmtDDBytesPulldown selectItemAtIndex:0];
    [diskFmtInsertNewButton setState:NSOffState];
    [NSApp runModalForWindow:[diskFmtMatrix window]];
}

/*------------------------------------------------------------------------------
*  showManagementPanel - This method displays the disk management window for
*     managing the Atari floppy drives.
*-----------------------------------------------------------------------------*/
- (IBAction)showManagementPanel:(id)sender
{
    [self updateInfo];
    PauseAudio(1);
    [NSApp runModalForWindow:[d1DiskField window]];
    [self releaseCmdKeys:@"d":QZ_d];
}

/*------------------------------------------------------------------------------
*  releaseCmdKeys - This method fixes an issue when modal windows are used with
*     the Mac OSX version of the SDL library.
*     As the SDL normally captures all keystrokes, but we need to type in some 
*     Mac windows, all of the control menu windows run in modal mode.  However, 
*     when this happens, the release of the command key and the shortcut key 
*     are not sent to SDL.  We have to manually cause these events to happen 
*     to keep the SDL library in a sane state, otherwise only everyother shortcut
*     keypress will work.
*-----------------------------------------------------------------------------*/
- (void) releaseCmdKeys:(NSString *)character:(int)keyCode
{
    NSEvent *event1, *event2;
    NSPoint point;
    
    event1 = [NSEvent keyEventWithType:NSKeyUp location:point modifierFlags:0
                    timestamp:nil windowNumber:0 context:nil characters:character
                    charactersIgnoringModifiers:character isARepeat:NO keyCode:keyCode];
    [NSApp postEvent:event1 atStart:NO];
    
    event2 = [NSEvent keyEventWithType:NSFlagsChanged location:point modifierFlags:0
                    timestamp:nil windowNumber:0 context:nil characters:nil
                    charactersIgnoringModifiers:nil isARepeat:NO keyCode:0];
    [NSApp postEvent:event2 atStart:NO];
}

@end
