/*
 * gen_crc32_tables.c - a program for CRC-32 table generation
 *
 * Copyright 2016 Eric Biggers
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdio.h>

#include "../common_defs.h"

#define CRCPOLY	0xEDB88320 /* G(x) without x^32 term */

static u32
crc32_update_bit(u32 remainder, u8 next_bit)
{
	return (remainder >> 1) ^ (((remainder ^ next_bit) & 1) ? CRCPOLY : 0);
}

static u32
crc32_update_byte(u32 remainder, u8 next_byte)
{
	for (int j = 0; j < 8; j++, next_byte >>= 1)
		remainder = crc32_update_bit(remainder, next_byte & 1);
	return remainder;
}

static void
print_256_entries(const u32 *entries)
{
	for (size_t i = 0; i < 256 / 4; i++) {
		printf("\t");
		for (size_t j = 0; j < 4; j++) {
			printf("0x%08x,", entries[i * 4 + j]);
			if (j != 3)
				printf(" ");
		}
		printf("\n");
	}
}

int
main(void)
{
	u32 crc32_table[0x800];

	/* crc32_table[i] for 0 <= i < 0x100 is the CRC-32 of byte i. */
	for (int i = 0; i < 0x100; i++)
		crc32_table[i] = crc32_update_byte(0, i);

	/*
	 * crc32_table[i] for 0x100 <= i < 0x800 is the CRC-32 of byte i % 0x100
	 * followed by i / 0x100 zero bytes.
	 */
	for (int i = 0x100; i < 0x800; i++)
		crc32_table[i] = crc32_update_byte(crc32_table[i - 0x100], 0);

	printf("/*\n");
	printf(" * crc32_tables.h - data tables for CRC-32 computation\n");
	printf(" *\n");
	printf(" * THIS FILE WAS GENERATED BY gen_crc32_tables.c.  DO NOT EDIT.\n");
	printf(" */\n");
	printf("\n");
	/*
	 * Although crc32_slice1_table is the same as the first 256 entries of
	 * crc32_slice8_table, we output these tables separately because any
	 * combo of (slice1, slice8, slice1 && slice8, nothing) might be needed,
	 * and it's simplest to let the compiler optimize out any unused tables.
	 */
	printf("static const u32 crc32_slice1_table[] MAYBE_UNUSED = {\n");
	print_256_entries(&crc32_table[0x000]);
	printf("};\n");
	printf("\n");
	printf("static const u32 crc32_slice8_table[] MAYBE_UNUSED = {\n");
	print_256_entries(&crc32_table[0x000]);
	print_256_entries(&crc32_table[0x100]);
	print_256_entries(&crc32_table[0x200]);
	print_256_entries(&crc32_table[0x300]);
	print_256_entries(&crc32_table[0x400]);
	print_256_entries(&crc32_table[0x500]);
	print_256_entries(&crc32_table[0x600]);
	print_256_entries(&crc32_table[0x700]);
	printf("};\n");
	return 0;
}
