/* pgtable.h: page table stuff for the MN10300 arch
 *
 * Copyright (C) 2004 Red Hat, Inc. All Rights Reserved.
 * Written by David Howells (dhowells@redhat.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#ifndef _ASM_MN10300_PGTABLE_H
#define _ASM_MN10300_PGTABLE_H

#include <asm/cpu-regs.h>

/*
 * The Linux memory management assumes a three-level page table setup. On
 * the i386, we use that, but "fold" the mid level into the top-level page
 * table, so that we physically have the same two-level page table as the
 * i386 mmu expects.
 *
 * This file contains the functions and defines necessary to modify and use
 * the i386 page table tree.
 */
#ifndef __ASSEMBLY__
#include <asm/processor.h>
#include <asm/cache-regs.h>
#include <linux/threads.h>

#include <asm/bitops.h>

#define _LVM_H_INCLUDE			/* get the definition of list_head */
#include <linux/list.h>

/*
 * ZERO_PAGE is a global shared page that is always zero: used
 * for zero-mapped memory areas etc..
 */
#define ZERO_PAGE(vaddr) (virt_to_page(empty_zero_page))
extern unsigned long empty_zero_page[1024];
extern pgd_t swapper_pg_dir[1024];
extern struct page *pgd_list;

void pgtable_cache_init(void);
void paging_init(void);

#endif /* !__ASSEMBLY__ */

/*
 * The Linux mn10300 paging architecture only implements both the traditional
 * 2-level page tables
 */
#define PGDIR_SHIFT	22
#define PTRS_PER_PGD	1024
#define PTRS_PER_PUD	1		/* we don't really have any PUD physically */
#define PTRS_PER_PMD	1		/* we don't really have any PMD physically */
#define PTRS_PER_PTE	1024

#define PGD_SIZE	PAGE_SIZE
#define PMD_SIZE	(1UL << PMD_SHIFT)
#define PGDIR_SIZE	(1UL << PGDIR_SHIFT)
#define PGDIR_MASK	(~(PGDIR_SIZE - 1))

#define USER_PTRS_PER_PGD	(TASK_SIZE / PGDIR_SIZE)
#define FIRST_USER_PGD_NR	0

#define USER_PGD_PTRS		(PAGE_OFFSET >> PGDIR_SHIFT)
#define KERNEL_PGD_PTRS		(PTRS_PER_PGD - USER_PGD_PTRS)

#define TWOLEVEL_PGDIR_SHIFT	22
#define BOOT_USER_PGD_PTRS	(__PAGE_OFFSET >> TWOLEVEL_PGDIR_SHIFT)
#define BOOT_KERNEL_PGD_PTRS	(1024 - BOOT_USER_PGD_PTRS)

/*
 * Unfortunately, due to the way the MMU works on the MN10300, the vmalloc VM
 * area has to be in the lower half of the virtual address range (the upper
 * half is not translated through the TLB).
 *
 * So in this case, the vmalloc area goes at the bottom of the address map
 * (leaving a hole at the very bottom to catch addressing errors), and
 * userspace starts immediately above.
 *
 * The vmalloc() routines also leaves a hole of 4kB between each vmalloced
 * area to catch addressing errors.
 */
#define VMALLOC_OFFSET	(8 * 1024 * 1024)
#define VMALLOC_START	(0x70000000)
#define VMALLOC_END	(0x7C000000)

#ifndef __ASSEMBLY__
extern pte_t kernel_vmalloc_ptes[(VMALLOC_END - VMALLOC_START) / PAGE_SIZE];
#endif

#ifndef __ASSEMBLY__

#define VMALLOC_VMADDR(x) ((unsigned long)(x))

/*
 * Define this to warn about kernel memory accesses that are
 * done without a 'verify_area(VERIFY_WRITE,..)'
 */
#undef TEST_VERIFY_AREA

#define pages_to_mb(x) ((x) >> (20 - PAGE_SHIFT))

#define kern_addr_valid(addr)	(1)

#define io_remap_page_range(vma, vaddr, paddr, size, prot) \
	remap_pfn_range(vma, vaddr, (paddr) >> PAGE_SHIFT, size, prot)

#define __HAVE_ARCH_PTEP_TEST_AND_CLEAR_YOUNG
#define __HAVE_ARCH_PTEP_TEST_AND_CLEAR_DIRTY
#define __HAVE_ARCH_PTEP_GET_AND_CLEAR
#define __HAVE_ARCH_PTEP_SET_WRPROTECT
#define __HAVE_ARCH_PTEP_MKDIRTY
#define __HAVE_ARCH_PTE_SAME
#include <asm/generic/pgtable.h>

#endif /* !__ASSEMBLY__ */

#endif /* _ASM_MN10300_PGTABLE_H */
