# -*- coding: utf-8 -*-
# vim:fenc=utf-8

# Copyright (C) 2012-2020 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# X2Go Session Broker is free software; you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# X2Go Session Broker is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

# modules
import pwd
import grp

# Python X2GoBroker modules
from . import base_nameservice as base

_pwd = pwd.getpwall()
_grp = grp.getgrall()

class X2GoBrokerNameService(base.X2GoBrokerNameService):

    def get_users(self):
        """\
        Retrieve list of users from the POSIX nameservices system.

        :returns: list of known user names
        :rtype: ``list``

        """
        return [ p.pw_name for p in _pwd ]

    def get_primary_group(self, username):
        """\
        Get the primary group of a given POSIX user.

        :param username: name of the user to get the primary group for
        :type username: ``str``

        :returns: name of the primary group of the given user
        :rtype: ``str``

        """
        prim_gid_number = [ p.pw_gid for p in _pwd if p.pw_name == username ][0]
        return [ g.gr_name for g in _grp if g.gr_gid == prim_gid_number ][0]

    def get_groups(self):
        """\
        Retrieve list of groups from the POSIX nameservices system.

        :returns: list of known group names
        :rtype: ``list``

        """
        return [ g.gr_name for g in _grp ]

    def get_group_members(self, group, primary_groups=False):
        """\
        Retrieve a list of POSIX users being members of a given POSIX
        group.

        Optionally, primary group memberships can be considered (or not).

        :param group: name of the group to retrieve members of
        :type group: ``str``
        :param primary_groups: take primary group membership into consideration
            or not
        :type primary_groups: ``bool``

        :returns: list of users that are members of the given group
        :rtype: ``list``

        """
        _members_from_primgroups = []
        if primary_groups:
            for username in self.get_users():
                if group == self.get_primary_group(username):
                    _members_from_primgroups.append(username)
        return [ u for u in grp.getgrnam(group).gr_mem ] + _members_from_primgroups

