/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2017 - 2020 by Kay F. Jahnke                    */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/// \file grok.cc
///
/// \brief demonstrates use of vspline::grok_type

#include <iostream>
#include <vspline/vspline.h>

/// grokkee_type is a vspline::unary_functor returning twice it's input

template < size_t vsize >
struct grokkee_type
: public vspline::unary_functor < double , double , vsize >
{
  template < typename IN , typename OUT >
  void eval ( const IN & in , OUT & out ) const
  {
    std::cout << "call to grokkee_type::eval, in = " << in ;
    out = in + in ;
    std::cout << ", out = " << out << std::endl ;
  }
} ;

/// 'regular' function template doing the same

template < typename T >
void twice ( const T & in , T & out )
{
  std::cout << "call to twice, in = " << in ;
  out = in + in ;
  std::cout << ", out = " << out << std::endl ;
}

/// test 'groks' a grokkee_type object 'grokkee' as a vspline::grok_type
/// and calls the resulting object, which behaves just as a grokkee_type.

template < size_t vsize >
void test()
{
  std::cout << "testing grok_type with vsize " << vsize << std::endl ;

  typedef vspline::grok_type < double , double , vsize > grok_t ;
  
  grokkee_type<vsize> grokkee ; 
  
  grok_t gk ( grokkee ) ;  

  double x = 1.0 ;
  
  std::cout << x << std::endl << " -> " << gk ( x ) << std::endl ;

  typedef typename grok_t::in_v argtype ;
  
  argtype xx = 1.0 ;
  
  std::cout << xx << std::endl << " -> " << gk ( xx ) << std::endl ;
}

int main ( int argc , char * argv[] )
{
  test<1>() ;
  test<4>() ;
  test<8>() ;
  
  // with vsize == 1, we can construct a grok_type with a single
  // std::function:

  typedef vspline::grok_type < double , double , 1 > gk1_t ;
  
  typedef gk1_t::eval_type ev_t ;
  ev_t ev = & twice<double> ;
  
  vspline::grok_type < double , double , 1 > gk1 ( ev ) ;
  std::cout << "calling grok_type with vsize 1" << std::endl ;
  gk1 ( 1.23 ) ;
  
  // with vsize greater than 1, we may pass another std::function
  // for vectorized evaluation. If we omit it, it will be generated
  // automatically, but then it won't use vector code.
  
  typedef vspline::grok_type < double , double , 8 > gk8_t ;
  
  typedef gk8_t::v_eval_type v_ev_t ;
  typedef gk8_t::in_v v8_t ;
  
  v_ev_t vev = & twice<v8_t> ;
  
  vspline::grok_type < double , double , 8 > gk8a ( ev , vev ) ;
  vspline::grok_type < double , double , 8 > gk8b ( ev ) ;

  typedef typename vspline::grok_type < double , double , 8 > :: in_v v_t ;
  std::cout << "calling grok_type using vector code" << std::endl ;
  gk8a ( v_t ( 1.23 ) ) ;
  std::cout << "calling grok_type using broadcasting" << std::endl ;
  gk8b ( v_t ( 1.23 ) ) ;
}
