/*
	SRG - Squid Report Generator
	Copyright 2005 University of Waikato

	This file is part of SRG.

	SRG is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	SRG is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with SRG; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

*/
#include "srg.h"
#include "libconfig.h"
#include "prototypes.h"
#include <getopt.h>

/* Tell people how to use srg */
void usage(void)
{
  printf("usage: %s [options] <logfile> [logfile] ...\n"
  "Options:\n"
  "-C <config file>              Location of the srg config file.\n"
  "-d                            Enable debugging output\n"
  "-f <date>[time]               Start time to begin analysation at\n"
  "--help -h                     Display this help\n"
  "-H                            Generate HTML rather than PHP reports\n"
  "-m <days>                     Specify maximum number of days to keep a\n"
  "                              report\n"
  "-M                            Output a summary report suitable for \n"
  "                              emailing.\n"
  "-o <directory>                Output Directory\n"
  "-O <URL>                      URL where reports will be located\n"
  "-t <date>[time]               End time to stop analysation at\n"
  "-v                            Verbose mode. Print relevant messages.\n"
  "-V                            Print Version and exit.\n"
  "\n"
  "Date and time should be specified in the following format for -f and -t\n"
  " Date: YYYY-MM-DD\n"
  " Time: :HH:MM:SS - note leading :\n"
  " eg. 2005-01-08:15:46:30 (Jan 8 2005 15:46:30)\n"
  "Logfile(s) to process should be specified at the end of the options\n"
  , progname);

}

/* Configure srg */
bool do_configuration(int argc, char **argv) {

	char ch;
	
	/* Initialise configuration options */
	set_defaults();
	
	/* The real config options */
	char *conffile = NULL;

	/* Temp config options for config *f*ile */
	char *accesslogf = NULL;
	char *outputdirf = NULL;
	char *outputurlf = NULL;
	char *titlef = NULL;
	char *filterbyf = NULL;
	char *filtercritf = NULL;
	char *groupbyf = NULL;
	char *groupbynetf = NULL;
	char *ip2userf = NULL;
	char *sitefilterf = NULL;
	char *resourcedirf = NULL;
	int debugf = 0;
    int sitereportf = 0;
	int maxagef = 0;
	int authf = 0;
	int verbosef = 0;
	int emailreportf = 0;
	int sortcolumnsf = 0;
	int disable_phpf = 0;
        
	/* temp config options for *c*ommand line */
	char *accesslogc = NULL;
	char *outputdirc = NULL;
	char *outputurlc = NULL;
	int maxagec = 0;
	int debugc = 0;
	int verbosec = 0;
	int emailreportc = 0;
    char *time_fromc = NULL;
	char *time_toc = NULL;
	int disable_phpc = 0;
	
	/* Config options from the config file */
	config_t main_config[] = {
		{"log_file", TYPE_STR|TYPE_NULL, &accesslogf},
		{"output_dir", TYPE_STR|TYPE_NULL, &outputdirf},
		{"output_url", TYPE_STR|TYPE_NULL, &outputurlf},
		{"title", TYPE_STR|TYPE_NULL, &titlef},
		{"emailreport", TYPE_BOOL|TYPE_NULL, &srg.emailreport}, 
		{"filterby", TYPE_STR|TYPE_NULL, &filterbyf},
		{"filtercrit", TYPE_STR|TYPE_NULL, &filtercritf},
		{"groupby", TYPE_STR|TYPE_NULL, &groupbyf},
		{"groupbynetmask", TYPE_STR|TYPE_NULL, &groupbynetf},
		{"disable_php", TYPE_BOOL|TYPE_NULL, &disable_phpf},
		{"debug", TYPE_BOOL|TYPE_NULL, &debugf},
		{"location", TYPE_BOOL|TYPE_NULL, &srg.locationStats},
        {"disable_sitereport", TYPE_BOOL|TYPE_NULL, &sitereportf},
		{"show_times", TYPE_BOOL|TYPE_NULL, &srg.showtimes},
		{"show_rates", TYPE_BOOL|TYPE_NULL, &srg.showrates},
		{"nonameexists_showip", TYPE_BOOL|TYPE_NULL, &srg.nonameshowip},
		{"iptouser_file", TYPE_STR|TYPE_NULL, &ip2userf},
		{"destinationsitefilter_file", TYPE_STR|TYPE_NULL, &sitefilterf},
		{"dontshow_onlydeniedusers", TYPE_BOOL|TYPE_NULL, &srg.hideDeniedOnly},
		{"php_authentication", TYPE_BOOL|TYPE_NULL, &authf},
		{"max_age", TYPE_INT|TYPE_NULL, &maxagef},
		{"lookup_hosts", TYPE_BOOL|TYPE_NULL, &srg.lookupHosts},
		{"verbose", TYPE_BOOL|TYPE_NULL, &verbosef},
		{"sort_columns", TYPE_BOOL|TYPE_NULL, &sortcolumnsf},
		{"resource_dir", TYPE_STR|TYPE_NULL, &resourcedirf},
		{"min_connects", TYPE_INT|TYPE_NULL, &srg.minimumConnects},
		{NULL, 0, NULL}
	};
	
	static struct option long_options[] = {
		{"help", 0, 0, 'h'},
		{0, 0, 0, 0}
	};
	int optindex=0;

	/* Parse command line arguments */
	while((ch = getopt_long(argc, argv, "C:df:hHm:Mo:O:t:vV", long_options, 
					&optindex)) != -1) {
		switch(ch)
		{
		case 'C':
			conffile = strdup(optarg);
			break;
		case 'd':
			debugc++;
			break;
		case 'f':
			time_fromc = strdup(optarg);
			break;
		case 'h':
			usage();
			exit(0);
			break;
		case 'H':
			disable_phpc++;
			break;
		case 'm':
			maxagec = atoi(optarg);
			break;
		case 'M':
			emailreportc++;
			break;
		case 'o':
			outputdirc = strdup(optarg);
			break;
		case 'O':
			outputurlc = strdup(optarg);
			break;
		case 't':
			time_toc = strdup(optarg);
			break;
		case 'v':
			verbosec = 1;
			break;
		case 'V':
			printf("srg (Squid Report Generator) %s\n\t%s\n", 
					version, cvsid);
			exit(1);
		case '?':
			usage();
			exit(1);
		break;
		}
	}
	
	/* Check for extra non options on the command line */
	while (optind < argc) {
		/* Assume it is a filename to process and add it to the list */
		if (accesslogc) {
			char *temp = strdup(accesslogc);
			free(accesslogc);
			asprintf(&accesslogc, "%s %s", temp, argv[optind]);
			free(temp);
		} else {
			accesslogc = strdup(argv[optind]);
		}
		optind++;
	}
	
	/* Try and read a configuration file */
	if (conffile == NULL) {
		conffile = strdup(CONFFILE);
	}

	/* Check if configuration file exists */
	if (access(conffile, R_OK)!=0) {
		fprintf(stderr, "WARNING: Configuration file not found!\n");
	} else {
		if (parse_config(main_config,conffile)) {
			fprintf(stderr, "Error in configuration file (%s)!\n", conffile);
			exit(1);
		}
	}
	free(conffile);
	
	/* Check if debugging information should be printed */
	if (debugc > 0 || debugf > 0) {
		fprintf(stderr, "Setting 'debug' and 'verbose' rules to true\n");
		srg.debug++;
		srg.verbose = 1;
	}

	if (verbosec > 0 || verbosef > 0) {
		if (srg.debug)
			fprintf(stderr, "Setting verbose to true\n");
		srg.verbose = 1;
	}

	if (resourcedirf != NULL) {
		if (srg.debug)
			fprintf(stderr, "Setting resource dir to %s from config file\n",
					resourcedirf);
		srg.resourceDir = strdup(resourcedirf);
	} else {
		srg.resourceDir = strdup(RESOURCEDIR);
	}
	free(resourcedirf);
		
	if (titlef != NULL) {
		if (srg.debug)
			fprintf(stderr, "Setting title to %s from config file\n",titlef);
		srg.title = strdup(titlef);
	}
	free(titlef);
	
	if (accesslogc != NULL) {
		if (srg.debug)
			fprintf(stderr, "Setting input file(s) to %s from "
					"command line\n",accesslogc);
		srg.accessLog = strdup(accesslogc);
	} else if (accesslogf != NULL) {
		if (srg.debug)
			fprintf(stderr, "Setting input file(s) to %s from "
					"config file\n",accesslogf);
		srg.accessLog = strdup(accesslogf);
	}
	free(accesslogc);
	free(accesslogf);
	
	if (outputdirc != NULL) {
		if (srg.debug)
			fprintf(stderr, "Setting output directory to %s "
					"from command line\n",outputdirc);
		srg.outputDir = strdup(outputdirc);
	} else if (outputdirf != NULL) {
		if (srg.debug)
			fprintf(stderr, "Setting output directory to %s "
					"from config file\n",outputdirf);
		srg.outputDir = strdup(outputdirf);
	}
	free(outputdirc);
	free(outputdirf);
	
	if (outputurlc != NULL) {
		if (srg.debug)
			fprintf(stderr, "Setting output URL to %s "
					"from command line\n",outputurlc);
		srg.outputURL = strdup(outputurlc);
	} else if (outputurlf != NULL) {
		if (srg.debug)
			fprintf(stderr, "Setting output URL to %s "
					"from config file\n",outputurlf);
		srg.outputURL = strdup(outputurlf);
	}
	free(outputurlc);
	free(outputurlf);

	if (disable_phpc==1 || disable_phpf==1) {
		srg.outputMode = OUTPUT_HTML;
		srg.indexfname = strdup("index.html");
	}
	
	if (ip2userf != NULL) {
		if (srg.debug)
			fprintf(stderr, "Setting IP to user file to %s "
					"from config file\n",ip2userf);
		srg.ip2user = strdup(ip2userf);
	}
	free(ip2userf);

	if (sitefilterf != NULL) {
		if (srg.debug)
			fprintf(stderr, "Setting destination site filter file to %s "
					"from config file\n", sitefilterf);
		srg.sitefilter = strdup(sitefilterf);
	}
	free(sitefilterf);
	
    if (sitereportf > 0) {
        if (srg.debug)
            fprintf(stderr, "Setting disable site report to true from " 
                    "configu file\n");
        srg.siteStats = 0;
    }
    
	if (maxagec > 0) {
		if (srg.debug)
			fprintf(stderr, "Setting maximum report age to "
					"%d from command line\n",maxagec);
		srg.maxreportage = maxagec;
	} else if (maxagef > 0) {
		if (srg.debug)
			fprintf(stderr, "Setting maximum report age to "
					"%d from config file\n",maxagef);
		srg.maxreportage = maxagef;
	}

	if (filterbyf != NULL && filtercritf != NULL) {
		if (strcmp(filterbyf,"BY_IP")==0)
			srg.filter.by = BY_IP;
		else if (strcmp(filterbyf, "BY_SUBNET")==0)
			srg.filter.by = BY_SUBNET;
		else if (strcmp(filterbyf, "BY_USER")==0)
			srg.filter.by = BY_USER;
		else {
			fprintf(stderr, "Error in config file: '%s' is "
					"an invalid option to filterby!\n", filterbyf);
			usage();
			exit(1);
		}
		if (srg.debug)
			fprintf(stderr, "Setting 'filter' to %s from "
					"config file\n",filterbyf);
	}
	free(filterbyf);	
	
	if (filtercritf != NULL && filterbyf != NULL) {
		if (srg.debug)
			fprintf(stderr, "Setting filter criteria to %s "
					"from config file\n",filtercritf);
		switch (srg.filter.by) {
		case BY_USER:
			srg.filter.user = strdup(filtercritf);
			break;
		case BY_IP:
			if (inet_aton(filtercritf, &srg.filter.address)==0) {
				fprintf(stderr, "Invalid filter criteria for "
						"address filter: %s\n", filtercritf);
				exit(1);
			}
			break;
		case BY_SUBNET:
			char *networkAddress;
			char *netmask;
			char *tmp = filtercritf;
			netmask = break_string(tmp, '/');
			networkAddress = tmp;
			if (inet_aton(netmask, &srg.filter.netmask)==0) {
				fprintf(stderr, "Invalid filter criteria for "
						"subnet filter (netmask): %s\n", tmp);
				exit(1);
			}
			if (inet_aton(networkAddress, 
					&srg.filter.network)==0) {
				fprintf(stderr, "Invalid filter criteria for "
						"subnet filter (network): %s\n", tmp);
				exit(1);
			}
			break;
		}	
	}
	free(filtercritf);
	filtercritf = NULL;
	
	if (groupbyf != NULL) {
		if (strcmp(groupbyf,"A")==0)
			srg.groupBy += BY_IP;
		else if (strcmp(groupbyf, "S")==0) {
			srg.groupBy += BY_SUBNET;
			if (inet_aton(groupbynetf, &srg.groupByNetmask)==0) {
				fprintf(stderr, "Error in config file: %s is "
						"not a valid grouping netmask!\n", 
						groupbynetf);
				exit(1);
			}
		}
		else if (strcmp(groupbyf, "U")==0)
			srg.groupBy += BY_USER;
		else {
			fprintf(stderr, "Error in config file: '%s' is "
					"an invalid option to groupby!\n", groupbyf);
			exit(1);
		}
		if (srg.debug)
			fprintf(stderr, "Setting 'groupby' rule to %s "
					"from config file\n",groupbyf);
	}
	free(groupbyf);
	free(groupbynetf);
	
	if (authf > 0) {
		if (srg.outputMode != OUTPUT_PHP) {
		fprintf(stderr, "Authentication cannot be used with HTML output!");
			usage();
			exit(1);
		}
		if (srg.debug)
			fprintf(stderr, "Setting 'PHP authentication' option to true\n");
		srg.authenticate++;
	}
	if (emailreportc || emailreportf) 
		srg.emailreport++;
	if (sortcolumnsf) {
		if (srg.debug)
			fprintf(stderr, "Setting 'column sorting' rule to "
					"true\n");
		srg.sortcolumns++;
		srg.usejs++;
	}

	/* if the period is specified at all, set the srg variables so. */
	if (time_fromc) {
		srg.startTime = parse_timestring(time_fromc);
		if (srg.startTime == (time_t)-1) {
			fprintf(stderr, "%s: could not parse %s to a valid start time!\n",
					argv[0], time_fromc);
			exit(1);
		}
		if (srg.debug) {
			fprintf(stderr, "Starting at %d (%s)\n", srg.startTime,
					time_fromc);
		}
	}
	if (time_toc) {
		srg.endTime = parse_timestring(time_toc);
		if (srg.endTime == (time_t)-1) {
			fprintf(stderr, "%s: could not parse %s to a valid end time!\n",
					argv[0], time_toc);
			exit(1);
		}
		if (srg.debug) {
			fprintf(stderr, "Finishing at %d (%s)\n", srg.endTime,
					time_toc);
		}
	}
	
	if (srg.startTime != (time_t)-1 && srg.endTime != (time_t)-1) {
		/* Check that the end time is after the start time */
		if (srg.startTime == srg.endTime) {
			struct tm * starttm = localtime(&srg.startTime);
			if (starttm->tm_hour == 0 && starttm->tm_min == 0 && 
					starttm->tm_sec == 0) {
				if (srg.debug)
					fprintf(stderr,"No times specified but same date. "
							"Assuming start at 00:00 and end at 23:59\n");
				srg.endTime += 86399;
			}
		}
		if ((srg.startTime >= srg.endTime) && (srg.endTime != (time_t)-1)) {
			fprintf(stderr, "%s: End Time must be after Start Time!\n", 
					argv[0]);
			exit(1);
		}
	}
	free(time_fromc);
	free(time_toc);
    
	/* Only allowed to filter on one Criteria */
	if (srg.groupBy > BY_MAX) {
		fprintf(stderr, "%s: -a -s -u options are mutually " 
				"exclusive!\n", argv[0]);
		usage();
		exit(1);
	}

	/* Check required parameters are set */
	if (srg.groupBy == BY_SUBNET) {
		if (srg.groupByNetmask.s_addr==0) {
			fprintf(stderr, "%s: please specify a netmask "
					"with -n!\n", argv[0]);
			usage();
			exit(1);
		}
	}
	
	if (strlen(srg.accessLog)<=0) {
		fprintf(stderr, "%s: Please specify a logfile to analyse!\n", argv[0]);
		usage();
		exit(1);
	}

	return true;

}

/* Set program defaults */
void set_defaults(void) {

	/* General Options */
	srg.debug=0;
	srg.verbose=0;
	srg.groupBy=0;
	srg.showtimes=0;
	srg.showrates=0;
	srg.nonameshowip=0;
	srg.startTime=(time_t)-1;
	srg.endTime=(time_t)-1;
	srg.minTime=(time_t)-1;
	srg.maxTime=(time_t)-1;
	srg.title="SRG - Squid Log Analysis";
	srg.ip2user=NULL;
	srg.locationStats=1;
    srg.siteStats=1;
	srg.hideDeniedOnly=0;
	srg.authenticate=0;
	srg.lookupHosts=0;
	srg.emailreport=0;
	srg.maxreportage=0;
	srg.minimumConnects=0;
	
	srg.accessLog="";

	/* Grouping & Filtering Options */
	srg.groupByNetmask.s_addr=0;
	srg.sortcolumns=0;
	srg.filter.by= BY_NONE;
	srg.filter.user = NULL;
	srg.filter.address.s_addr=0;
	srg.filter.network.s_addr=0;
	srg.filter.netmask.s_addr=0;
	
	/* Output Options */
	srg.outputDir="srg_reports";
	srg.outputURL="/";
	srg.outputMode = OUTPUT_PHP;
	srg.resourceDir="";
	srg.usejs=0;
	srg.indexfname="index.php";
	srg.phpheader="header.php";
	srg.phpfooter="footer.php";
	srg.cssfile=SRG_CSS_FILE;
	srg.jsfile=SRG_JS_FILE;

}
