use nettle_sys::{
    nettle_hash, nettle_sha512_256, nettle_sha512_256_digest,
    nettle_sha512_256_init, nettle_sha512_update, sha512_ctx,
};
use std::default::Default;
use std::mem::zeroed;

use crate::hash::NettleHash;
use crate::hash::Hash;

/// 512 bit variant of the Secure Hash Algorithm 2 (SHA-2) defined in FIPS 180-4, truncated to 256
/// bit.
#[allow(non_camel_case_types)]
pub struct Sha512_256 {
    context: sha512_ctx,
}

impl_write_for_hash!(Sha512_256);

impl Clone for Sha512_256 {
    fn clone(&self) -> Self {
        use std::intrinsics::copy_nonoverlapping;

        unsafe {
            let mut ctx: sha512_ctx = zeroed();
            copy_nonoverlapping(&self.context, &mut ctx, 1);

            Sha512_256 { context: ctx }
        }
    }
}

impl Default for Sha512_256 {
    fn default() -> Self {
        let mut ctx = unsafe { zeroed() };

        unsafe {
            nettle_sha512_256_init(&mut ctx as *mut _);
        }

        Sha512_256 { context: ctx }
    }
}

impl Hash for Sha512_256 {
    fn digest_size(&self) -> usize {
        ::nettle_sys::SHA512_256_DIGEST_SIZE as usize
    }

    fn update(&mut self, data: &[u8]) {
        unsafe {
            nettle_sha512_update(
                &mut self.context as *mut _,
                data.len(),
                data.as_ptr(),
            );
        }
    }

    fn digest(&mut self, digest: &mut [u8]) {
        unsafe {
            nettle_sha512_256_digest(
                &mut self.context as *mut _,
                digest.len(),
                digest.as_mut_ptr(),
            );
        }
    }

    fn box_clone(&self) -> Box<dyn Hash> {
        Box::new(self.clone())
    }
}

impl NettleHash for Sha512_256 {
    type Context = sha512_ctx;

    unsafe fn nettle_hash() -> &'static nettle_hash {
        &nettle_sha512_256
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn do_nothing() {
        let _ = Sha512_256::default();
    }

    //  CAVS 21.1
    //  "SHA-512/256 ShortMsg" information for "sha_values"
    //  SHA-512/256 tests are configured for BYTE oriented implementations
    //  Generated on Mon Jul 10 14:34:21 2017
    #[test]
    fn nist_cavs_short_msg() {
        let mut ctx = Sha512_256::default();
        let mut digest = vec![0u8; ctx.digest_size()];

        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc6\x72\xb8\xd1\xef\x56\xed\x28\xab\x87\xc3\x62\x2c\x51\x14\x06\x9b\xdd\x3a\xd7\xb8\xf9\x73\x74\x98\xd0\xc0\x1e\xce\xf0\x96\x7a"[..]);

        ctx.update(b"\xfa");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc4\xef\x36\x92\x3c\x64\xe5\x1e\x87\x57\x20\xe5\x50\x29\x8a\x5a\xb8\xa3\xf2\xf8\x75\xb1\xe1\xa4\xc9\xb9\x5b\xab\xf7\x34\x4f\xef"[..]);

        ctx.update(b"\x74\xe4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0c\x99\x42\x28\xb8\xd3\xbd\x5e\xa5\xb5\x25\x91\x57\xa9\xbb\xa7\xa1\x93\x11\x8a\xd2\x28\x17\xe6\xfb\xed\x2d\xf1\xa3\x2a\x41\x48"[..]);

        ctx.update(b"\x6f\x63\xb4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa9\xe2\x42\x7c\xec\x31\x4b\x28\x14\xaa\xba\x87\x03\x94\x85\xfc\x8d\x3a\xde\x99\x2f\xa1\xd9\xac\xbb\x7f\x67\x69\x46\x0a\x73\x17"[..]);

        ctx.update(b"\x2b\xf9\x8e\x7f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x97\xd4\x55\xe6\x88\x1a\xdb\xb8\xbf\x26\xdd\xc3\xef\x12\x0d\xf4\xab\xb3\x30\xad\xf8\xed\x2a\xc7\xa4\xb6\xcc\xe4\xa1\x72\x70\x68"[..]);

        ctx.update(b"\x77\xd2\xee\x3f\xd1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x32\x9d\x0a\xe2\x6d\xf3\x64\x6f\xb7\xdb\x0b\x32\x2c\xf1\xf3\x48\x8d\x53\x81\xf6\x6e\xb3\x76\xfe\xe4\x0f\x05\x96\xd6\xd9\xdc\x4b"[..]);

        ctx.update(b"\x7d\xae\x8f\xc0\x20\xd9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x44\x70\x44\xf0\x3b\xc3\x0e\x2c\xaa\x24\x5d\x26\xce\x4c\x72\xc1\x45\x4f\x70\x8c\xfc\xd9\xa2\x15\x84\x1a\x88\xcf\x5e\xcd\x20\x95"[..]);

        ctx.update(b"\x5b\x48\xb8\xb0\x1a\x59\xd5");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x1a\x86\xb4\xd3\x4c\xd1\x04\xc1\x3b\x5b\x8d\x41\x97\x84\xce\x4c\x6d\x35\x59\x4f\x2a\x93\x00\xe8\x14\x68\xf4\xdc\x06\x34\x83\x2a"[..]);

        ctx.update(b"\xb4\xe2\xe8\x50\x1f\x54\xbe\x91");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd2\x52\x65\xbf\x9c\xbc\x0d\xd2\xf1\x08\xa2\xf5\xe8\xf6\x9d\xb7\xd1\x5e\x5b\x8f\xe9\x10\x0f\xe8\x87\xda\xe2\x0b\x6e\x05\x4f\xe8"[..]);

        ctx.update(b"\x90\x1d\xb0\xf7\x0f\x2d\x10\x62\x6a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf8\x4c\xe7\xa7\x60\xc8\x5e\x67\x87\x2f\xfc\xa1\x72\x2b\x1b\x77\x41\x03\x2d\x91\x39\x21\x10\x77\xf9\x63\xa3\x2d\x9f\x84\x77\x1b"[..]);

        ctx.update(b"\xf2\x4d\x98\x59\x1e\xe3\xc7\x32\xa1\x72");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xda\x23\x4c\x1d\x6d\x18\xad\x8f\x53\x07\x0e\x4e\x5b\x9c\xff\x75\x38\x5c\x13\xf3\xe4\x4b\x01\x4f\xc4\x5c\x35\x09\xd0\xd0\x6a\xdb"[..]);

        ctx.update(b"\xbf\xb2\x72\x53\x7f\x02\x1a\xde\x9b\xb3\x30");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7c\x49\xa5\x9a\x3c\x3f\xd8\xe2\x4d\xe8\x58\x54\xf6\x8a\x3b\x7e\xc6\x0c\x20\x31\x74\x2a\x9a\x2d\xab\xf4\x91\x99\xd8\x8c\x2d\xb5"[..]);

        ctx.update(b"\x0a\x5f\x74\x24\xba\xbb\x5c\xb8\x57\x15\xfb\x17");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5c\xd4\x0c\x51\x50\x26\x00\xf3\xa0\xee\xc7\xb3\xa3\xad\xdb\xf6\x67\x2e\xee\xcc\x50\xb3\xde\x2f\xd8\x6d\xd1\xc0\xa4\x7e\x8a\x3f"[..]);

        ctx.update(b"\x57\xa1\x03\x08\xaa\x20\x60\xae\xca\xd0\x5a\xdb\x2d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa5\x20\x8e\xce\x1d\xe1\x34\x2d\x02\xea\x9a\xb7\xcf\xe5\x37\xa2\xc8\x1f\xa0\xea\xed\xb4\x27\xa7\xb3\xcb\x6c\x29\x19\x59\xa6\xa8"[..]);

        ctx.update(b"\x5e\x46\x74\xe9\x7a\x25\x7c\x23\x1d\x8e\x91\x39\x1d\x0a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa8\x8d\x7c\xbb\x6e\x31\xe8\x7a\x6c\x99\x73\x4f\x59\x9d\x15\x88\xcb\x94\x39\x97\x9b\x82\x10\x80\x0e\xca\x0a\xa2\x06\x7c\xd5\x56"[..]);

        ctx.update(
            b"\xba\x10\xb0\xda\x70\xe2\xa0\x17\xcd\xc6\x3f\x99\x6d\xbc\x1b",
        );
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf8\xa5\xaa\x22\x93\x74\x53\x71\x9e\xb4\x0b\x1d\xcf\x68\xa3\x46\xe2\x02\xba\x22\x5f\x4d\x7a\xcc\x87\xe5\xa7\x36\xab\xed\x34\x23"[..]);

        ctx.update(
            b"\xa3\x48\x19\x96\xc1\x09\xc9\xef\xb6\x69\x2b\x82\x17\x39\x21\xb6",
        );
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xfb\xa0\xf0\x81\x3c\xa5\xc5\x13\xd0\xb5\x47\x41\x9a\x15\xe2\x7a\x70\x97\xd3\xeb\x28\x0a\x29\x8c\x1c\x69\x3b\x55\xe0\x3d\x2d\x2c"[..]);

        ctx.update(b"\x08\x77\xc2\x0c\xdd\xcc\x86\xb8\x33\xf7\xbc\x2d\xcf\x7a\x70\x9a\x2e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf4\xc3\xa2\x60\x4e\x27\x25\xe4\xd1\x79\xed\xd1\xb4\x45\xba\x48\x99\x79\xf4\xea\x59\x37\xe0\xc6\xb2\x0b\x42\x2c\x72\x0d\x8b\xfa"[..]);

        ctx.update(b"\x69\x44\x5b\xd1\x2f\x02\x2f\x6a\x66\x9b\xeb\xe0\x46\x77\x29\xf4\xdc\x7c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x26\xbf\x70\x9b\x34\x74\x55\x4a\xb8\xdd\x12\xd9\x52\x4f\xac\xa4\x30\x34\x11\x8b\xd9\x55\xfc\x52\xcb\x1f\x35\x5c\x6d\xf2\x23\x91"[..]);

        ctx.update(b"\xf7\x51\xf4\x28\xa0\xaf\x0b\x90\xa9\x12\xa3\xbe\x78\xda\xf1\x2a\x76\x5b\x43");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5e\x0f\x8a\x87\x39\xf5\x56\xda\xba\x4d\xa4\xe8\x61\xf3\x9d\x1e\xda\xae\xce\x02\x04\x96\x39\x42\xa4\xe6\x6a\x11\xec\x1f\xcf\x13"[..]);

        ctx.update(b"\x19\x82\x9a\x8c\x72\x3c\x58\xca\x95\x80\xcd\xf0\xda\xd2\xb6\xd3\x8c\xdf\x60\xa5");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3a\xa7\x91\xb3\x96\xab\x81\xdf\x91\x07\x27\x8b\x8d\xd1\xc4\x6e\xe5\x75\x79\xdb\x4c\xf5\xf9\xb8\x3c\x54\xb3\x7b\x0a\x60\x1f\xc9"[..]);

        ctx.update(b"\xae\x43\x13\x2a\xc3\x0d\xe1\x80\xc2\x4d\xad\xb8\x86\x02\xca\xea\x91\x36\x6b\x8e\x51");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb9\xf0\x14\x1b\xa0\x06\xa3\x98\x39\x72\xdb\xaa\x5e\x2b\xfe\x71\x52\xe3\x81\x10\xe0\x24\x75\xf0\x3d\x84\xb0\xa3\xc7\x43\xcd\xa9"[..]);

        ctx.update(b"\x8d\x27\x82\xa7\x84\x3a\xa4\x77\xb8\xf3\xbc\xa9\xf9\xf2\xba\xfb\x58\x13\xdb\x4c\x8c\x43");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x54\x24\xb3\x19\x89\x03\x18\x09\xb5\xfc\x89\x69\xb7\xc4\x8b\x5d\xc0\xc2\x33\xfc\x34\xff\xb5\xb2\x23\xcd\x5f\x3a\x97\x12\xa8\xd6"[..]);

        ctx.update(b"\x72\x32\x1f\xff\xd7\x91\x8d\x4f\x5c\xea\x10\x5c\xb0\xb8\x84\xa2\x35\x6a\xf1\x1e\xbe\x12\x67");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x17\xe3\xfb\x33\xe7\xf8\x5b\x1a\xd5\x63\x4f\xbb\xce\x9b\x5f\x99\x09\xf3\x32\x9f\x4d\x74\xac\xad\x3d\xd0\x94\x1c\x15\x31\xcf\x54"[..]);

        ctx.update(b"\xa7\xef\xce\x3e\x31\x43\x3f\xe6\x8d\x75\x14\x5d\x84\xbb\x02\x96\x0f\xb9\x68\x87\x1a\x3c\x2f\x7f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8f\xf0\xbe\x06\x01\xe3\xae\x92\x90\x71\x5a\xed\x30\x63\x4b\x13\xb8\xeb\x6f\x3c\x8b\x32\x94\xee\x9b\x1f\xc3\x42\xc2\x60\x31\x3e"[..]);

        ctx.update(b"\xd4\xd0\x22\xcf\x90\xe6\x43\x7a\xa0\xbf\xdb\x89\x28\x42\x44\xc1\x6d\xee\x0b\x48\x8b\xc3\x45\xab\x66");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xcb\x3a\x62\xfe\xb0\x48\x4b\xe0\xff\x9b\x56\xea\x92\x41\x4c\x70\x83\xf2\xb5\x77\x6e\x88\x35\x08\xb3\xa6\x1e\x57\x30\x5e\xe8\x3c"[..]);

        ctx.update(b"\x95\xde\x55\x28\x7a\xd3\xcf\xf6\x9e\xfe\xc6\xe9\x7c\x81\x24\x56\xe4\x7b\xe2\x5e\x43\x34\x70\xc3\x25\x9b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x41\xc7\x02\x7d\x16\xe3\x72\x59\x64\x5d\x01\x73\xc8\x61\x41\xf3\x8d\x80\x8e\x9e\x27\xdc\x2d\xfe\xea\xf3\x35\xed\x7c\x99\x49\x0c"[..]);

        ctx.update(b"\x63\xd8\xcf\xd7\x27\x68\xc4\x49\x20\xd7\xb0\x15\x46\x04\x89\xad\x57\x8c\x06\x3b\xe1\x90\x53\x88\x9c\xb8\x09");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x87\x6e\x59\xc8\xa6\x4f\xaf\x9d\x66\x5f\x7c\xde\x5d\x42\xfb\xb3\x31\xba\x81\x8d\xdc\xd2\x84\x49\x1a\xc5\x1e\xd5\x0e\x16\x13\xbe"[..]);

        ctx.update(b"\xbd\x2c\x5a\x91\x16\xbf\x0d\x75\xb3\x49\x8c\xe2\xee\x4b\xa2\x86\x8e\x06\xe0\xc0\x47\xec\x40\xb1\x81\x54\x08\x96");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xeb\x53\x7d\xf0\x4d\x51\x2c\x2f\xc8\x44\x02\x0f\x8f\xf3\x2e\x04\x3c\x04\x61\x4b\x88\xce\xa0\xa4\xe0\x5e\x0f\xf3\x2c\xcf\xfa\x06"[..]);

        ctx.update(b"\x37\xa4\x56\x43\xa2\x80\xa9\xd1\xa1\xc8\x3e\x09\x4e\x01\x17\x3a\xcd\x8a\xa1\x68\x25\x25\x2d\xee\x4e\xb3\xa3\x67\x8c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x80\xbc\xa5\xba\x3e\x63\x2e\x8b\xad\x15\x89\xcd\x02\xd7\x5d\x7d\x3e\x6a\x1a\xa9\xff\x64\xc2\xb9\x82\xd5\x72\xf3\xd0\x0d\x94\x7d"[..]);

        ctx.update(b"\xa7\xbe\x84\x01\x4e\x8d\x09\xd8\xd5\x16\x57\x78\x9c\x07\xf2\x9f\x7d\xa4\x9c\x5b\x5c\x0f\x83\x6f\xff\xc6\xba\xf8\x20\x6d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc5\x75\xb2\xba\x07\x5a\x63\xc4\x54\x00\x84\x49\x36\x36\xde\xab\xe9\xb8\xdf\x61\x95\x2f\x07\x15\x59\x39\x4d\x4d\xab\x81\x0e\x08"[..]);

        ctx.update(b"\x93\xc4\x5f\xb9\x1d\xcb\xa4\x20\x82\x0b\xd3\xfe\xe3\x34\xb4\x8f\xdf\x7b\x73\xd7\xb4\x31\xeb\xdb\xa7\x9b\x23\x10\xa7\xfe\x19");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x98\x15\x18\x39\xb8\xf3\x57\x4e\x0a\x0b\x9f\x19\xe8\x4a\xfe\xb1\xc6\x20\xfb\x2d\xce\xf2\x48\x14\x9c\x00\x61\x32\x33\x33\x79\x97"[..]);

        ctx.update(b"\x1b\x77\xc8\xdc\xfd\x2f\xc4\xb5\x46\x17\x05\x4f\xa6\xb1\x4d\x6e\x9d\x09\xce\x91\x85\xa3\x4a\x7f\xd2\xb2\x79\x23\x99\x8a\xab\x99");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x68\x3b\x48\x68\x61\xe5\x98\xda\xbb\xa7\x40\xac\x91\x95\x22\xcf\x3b\x60\x9c\x18\x20\x5b\x6b\xec\xa4\xcc\xbe\x6b\x0f\x6d\xc6\xdb"[..]);

        ctx.update(b"\x8d\x5c\xb9\x33\x40\x1b\x35\xdc\x34\xb1\x5d\xe0\xf5\x47\x4d\x3c\x9b\x6a\xeb\x90\x57\x53\x1b\x7f\x4b\x6b\x32\x59\x9f\x0e\xae\xf7\xd1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x16\xc5\x23\xd2\xb8\x83\x0e\xa9\x0d\xd6\xb6\xb1\xca\x49\xe4\xb5\x16\x73\x00\x6c\xa3\x0f\xf4\x7a\x4d\xcd\x4e\x7b\x35\xad\x4f\x62"[..]);

        ctx.update(b"\xd6\xa6\x6d\x66\xbf\x54\x6b\x97\xeb\xd0\x33\xcb\xed\x35\x04\x93\x1e\x45\x29\x63\x34\xae\x28\xf4\x17\x8d\x3d\xe6\xf0\x31\x06\xaa\x82\x21");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3f\x60\x7a\x5c\x7a\xef\x13\xfe\x98\xdc\x9f\x6b\x6a\x4b\x52\x97\x9e\x7c\x68\x78\x1e\xf5\xe1\x2f\x69\xde\xd2\x0c\x49\x7e\xda\x47"[..]);

        ctx.update(b"\xce\x88\x5f\xfb\x34\x0f\xd7\xa4\x3e\x21\x50\x69\x9b\x3f\x98\xef\x33\x42\x70\x03\x67\x94\xec\x75\x9a\xa0\x1d\x29\xf0\x87\xed\x6b\x55\x15\x49");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x10\x71\xf8\xb5\x73\x28\xb7\x67\x20\x9c\x15\xa4\x3e\x1d\x84\x52\x76\x38\x65\xac\x93\xd5\x8f\x93\xc8\x40\x87\x6c\x1a\x5b\x21\xe0"[..]);

        ctx.update(b"\xf2\xc2\x8d\x68\x54\x86\xbe\x99\xaf\xe8\x7b\xf9\x0c\x9f\x30\x93\x57\xb4\x4f\x60\xb6\x88\x44\x6a\x19\x18\xaa\x32\x11\x80\xbf\x4f\xe5\xbe\x15\x59");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x35\xbd\x19\x2c\xc1\x0f\xa2\x4b\xd7\x95\x4e\x0e\x0e\xe2\x0c\x09\x44\x0e\xfc\x64\x16\xe9\x4c\xa2\x5a\x47\xe7\x65\xbf\x32\xef\x1d"[..]);

        ctx.update(b"\x96\xe8\xe3\x0c\x75\xfe\xc8\x2b\xef\x4d\xe7\xec\x8e\x05\x3e\xa6\x61\xcb\x72\xbc\x4e\xb9\xc1\xc1\xaa\xa6\x38\xdf\x07\xa8\x26\x79\x4d\xd8\xd0\x0f\x0d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb5\x56\x79\x8f\x76\xef\x74\x2c\xc1\xeb\x5d\x10\x73\x2a\x92\x5f\xd8\x20\x8b\x4f\x07\x9a\x6c\x0e\x1f\xe0\xd9\x2f\xd8\x75\x64\xa5"[..]);

        ctx.update(b"\x29\x15\xc9\xb5\x88\x35\xf0\xec\x1e\x01\x1f\xe5\xe6\xdb\x40\xae\x3e\xf3\xd8\x90\xe2\x68\x93\x4d\x1b\xb0\x24\xa6\x34\x39\x4b\x20\x2e\x50\x37\x08\xb5\x43");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x38\x68\x88\x5b\x4e\x3a\xe2\x17\x23\x05\xac\xaa\xd2\x5f\xfe\x90\x94\xc7\x23\x48\xee\x9d\x7b\x0a\xcc\x20\x02\x9e\x26\xa7\xd0\x13"[..]);

        ctx.update(b"\x67\x3c\xcc\xe2\x80\x7c\xb4\x76\x22\xb7\xe0\x9e\xe8\xef\xd8\x12\xb4\x4f\xae\xef\x43\x6c\x30\x15\x05\xcc\x63\xcd\x2c\xa5\x2d\xb0\x03\xe6\x52\xc4\x82\x7b\xa2");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc7\x75\x20\x98\xc3\xb5\xfc\xea\x98\x42\x5a\x18\x0a\x5c\x96\x14\x6b\x44\x3a\xa8\xd7\x6a\x0c\x79\xfd\xcf\x37\x16\xbd\xd2\xb9\x06"[..]);

        ctx.update(b"\xc1\xa6\xac\x69\xf4\x80\x08\xdb\x50\x95\x86\x65\xdf\x9d\x91\x1b\x78\xdf\xe5\x19\xd9\xf3\x43\x66\xf6\xcd\x69\x1f\x90\xf7\x41\xd1\x08\x1e\x59\x26\x76\xd0\x93\xc4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xbd\x18\xaf\x5b\x2b\x24\xd7\x69\xea\xde\x52\x5c\x8f\xcd\x53\x52\x12\xab\xf8\xe5\xdf\x18\x0d\x7f\x3e\x3d\x02\x30\x33\xfc\xe1\x92"[..]);

        ctx.update(b"\x3c\xd1\x23\xae\x7b\xfd\x2b\x31\x61\x3e\xcc\x5e\xe8\xb8\x6e\x8e\x59\xa5\x2f\xf4\x6a\xeb\x2b\xcf\xa9\xf4\xfb\xe4\x1e\xba\x7b\x2c\x2c\x21\xd0\xca\xb9\x8a\xaa\x3f\x8f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7c\xe1\x97\x14\xe1\xd3\xbe\x68\x0d\xde\x0e\x91\x9d\x90\x15\x7a\xab\xca\x76\x43\x9d\xf8\xf0\x97\x9d\x19\xb3\xa6\x86\xdf\x6b\x28"[..]);

        ctx.update(b"\xa5\x2d\xde\x5a\xea\x1f\x04\x39\x9c\x5d\x91\xfb\x5c\x4e\x62\xda\x06\xb7\x3a\x5d\x9b\xdc\x5a\x92\x7f\xe8\x52\x98\xe5\x8e\x16\x61\x87\xf1\x54\xb6\x9a\xc1\x05\x7c\x01\xa7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x03\x5d\x36\x2a\x42\x65\x03\x14\x34\xc3\xaa\x15\x3a\xb4\xae\xf6\xb0\x0d\x51\x76\x22\x71\x97\x43\x0b\xb6\x28\x84\xa0\x85\xc5\xfe"[..]);

        ctx.update(b"\x38\x19\x06\x80\x83\x94\xc5\xf9\x38\x5a\x06\xbc\x9b\x6e\x52\x80\x95\xe6\xf6\xac\x07\xf0\xfc\x75\x5e\x13\x59\x47\x68\xb7\x68\x67\x66\xe8\x97\xd6\x62\x57\x61\x78\xc2\xa1\x7c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x22\xea\x52\x98\xaf\x2a\xa3\x11\xa5\xb5\x3f\x8b\x28\x38\xdf\xc0\x08\xae\x47\x0b\xf9\x6d\x2b\xb6\x79\x47\x35\xea\x72\x10\x88\x79"[..]);

        ctx.update(b"\xa0\x4a\x5d\x5d\x61\x3d\x23\x70\x3a\xb8\xdd\x00\x93\xd3\x54\x82\x9c\x54\x49\x82\x3b\xcf\x09\xeb\x49\x12\x41\x83\xfd\xc0\x14\xb5\xa8\x3b\x6b\x93\xa3\xc7\x8c\x1b\x86\xed\xef\xb9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8a\xe6\x88\xd5\xe9\xd0\xde\x9d\xe8\x34\xbe\xfe\x5c\x08\x29\x64\x4d\xef\x9c\x5d\xa7\x1a\xe0\xda\x4f\x79\xd1\x89\x04\x87\xe2\x5b"[..]);

        ctx.update(b"\x17\x5e\x10\x59\x93\x1a\x94\xe6\xe3\x7e\xa5\x3c\x87\xe3\xc7\x1d\x46\xbe\x2f\x41\x0b\x21\x31\xf8\xf2\x85\xf8\x10\xa3\xc6\x79\xf9\xdf\x6f\xa9\x27\x58\x38\xee\xb8\xc6\xcd\xd4\x57\x37");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x56\x41\x03\x6e\xf6\x22\x1e\x08\xe3\xc7\x95\xb2\xad\x56\x6a\xac\x9a\x69\xf8\x5d\xf1\xa8\xd3\x5b\x3e\xa2\xb7\x80\xce\xd3\x34\x21"[..]);

        ctx.update(b"\x3c\xe3\x8a\x2e\x28\x7f\xda\x13\x38\x4c\xb3\x06\x23\x7e\x8c\x74\x35\x78\x48\x81\x8d\x34\x0a\x94\x88\xe6\x4a\x15\x7d\xdc\x29\x75\xab\xa9\x01\x6f\x6f\x82\x41\x8e\xbb\xe8\x78\xf0\xc3\x88");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xbc\x20\x41\x02\x9b\x16\x89\x5a\xfb\x1f\x59\xf7\x28\x0f\x8a\x8f\x09\xbe\x5d\xd2\x41\xc6\x75\x34\x3f\x7a\xd8\xd7\x5b\x83\x4b\x75"[..]);

        ctx.update(b"\x96\x28\xa1\x95\xca\x6e\x70\x8c\xfc\xcf\x9f\x43\x4d\xbe\x6c\x51\x05\x32\x7f\x0f\x28\xee\x1c\xfd\x4a\x06\x9d\x62\xa3\x9f\x20\x33\x66\xf6\x7d\x36\x4f\xa4\x7b\x67\x7b\x20\x43\x9b\x7d\x85\x14");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdf\x6d\x89\x1d\x2c\x20\x43\x54\xb8\x27\xf5\x30\xc5\x46\x52\x94\x0c\xb4\x83\xf5\x8d\x6b\x43\x39\x73\x75\x2c\x1f\xd1\xd2\xf8\x4d"[..]);

        ctx.update(b"\xd8\x5b\xdb\x2d\xf5\xd7\x2f\x21\x90\xe4\xb3\xd4\xfa\x5b\x6d\x84\x8a\x08\x8b\xfc\xf0\xaf\x52\xb8\xc6\x4c\x58\x4a\x68\x9f\xf3\xfc\x06\xe8\xf1\xb7\xf7\xf0\x89\x23\x50\x48\x92\x10\xb6\xd5\xf3\x15");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xeb\x26\x81\xed\xf9\x2a\x1f\x55\x59\xd1\x01\x77\x11\x0a\xd2\x1f\xfd\x3f\xc1\x29\x6e\x3c\xc8\x3e\xce\x32\xe4\x2f\x56\xef\x7d\xed"[..]);

        ctx.update(b"\xd0\x78\xa5\xdc\x6e\xc5\x89\x11\x71\x19\xba\x3c\xfb\x9b\x11\xde\x57\xfc\xcb\x56\x3b\xad\xe8\xb3\x4a\x4f\xd2\xc7\xdd\x6b\x3f\x7c\x03\x63\x58\x6c\xc7\xa8\x1b\xab\xb1\x2e\xfe\xe7\x8f\x1d\xd5\xb4\x74");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0f\xc7\x04\x6f\x02\xd0\x70\x64\x2d\x82\x1c\x74\x67\x85\xf2\x09\xbb\x33\x26\x90\xbb\x34\x33\x5e\x6c\x64\x28\x05\x17\x93\xe1\xa9"[..]);

        ctx.update(b"\xb7\xae\x60\x37\xd9\xb4\x52\x99\x4f\xb7\x55\xaa\x0c\xf5\x7f\xbc\x68\x72\x98\x06\x0a\x47\x1f\x9b\x0c\x3d\x8e\x9d\x5a\x9a\xe0\xad\x5b\x3b\x4f\xa7\xd4\xe3\x62\x8c\x09\xd3\x27\xc0\x25\x9b\xbd\x76\xdb\x2f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xcd\x10\x40\xa0\xd9\x43\x82\xa6\x44\x6a\x69\x1c\x0a\x07\xc4\xf6\x43\xed\xc1\x1e\x31\x03\xe3\x43\x60\x86\x03\xb5\xbd\x10\x1b\xe3"[..]);

        ctx.update(b"\x62\xf0\x16\xdf\xc3\x8a\x4c\x70\xb7\xf9\x5d\x66\x9e\x2d\x47\x57\x04\x90\x9e\x13\xc9\x97\xd3\xda\x7e\x96\xb6\xcd\x11\xd8\x3a\xa3\xad\xef\x78\x9d\xdc\x49\xf7\xbc\xef\x04\x2b\x31\x9f\x03\x3b\x15\xf4\x42\x08");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8a\xee\x50\x57\x2b\x6b\x15\x5c\xf4\x0d\x6e\xd6\xc7\x38\xbd\xbc\xaf\xf2\xad\xe0\x9b\xd4\xd8\xbc\x9c\x2f\x9a\x67\x33\x73\xc4\x4f"[..]);

        ctx.update(b"\xbe\x89\x08\x98\xb0\x3f\x7c\xb5\x2a\x9d\x43\x00\xac\x2e\x8a\x80\x71\x51\x02\xb1\x53\x8f\xe3\x7c\xe9\x2c\x31\x65\xf2\xfc\x3e\x07\xf6\x78\x70\xbd\x2a\x11\x44\xa3\x95\xc5\x93\xbd\x71\xef\x14\x75\xd4\x63\x23\x4b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xad\x15\x6f\x2f\x41\x6c\xce\xeb\x9a\x6a\x25\x20\xe7\x9b\xba\x52\x95\xb5\x84\xc5\x96\x06\xfb\xdf\xda\x6e\xd1\x3e\xd9\xd9\x9a\xf0"[..]);

        ctx.update(b"\xc1\x21\x49\xee\xd4\x11\x3e\xfa\x31\xbb\x3c\x4b\x9a\xa9\x33\x63\x9a\x0e\x1c\xa7\x16\x06\x2e\xbe\x25\x1e\x55\x3a\x22\x17\xe0\x7a\xff\xe7\x79\xce\xdb\x01\x5b\xdd\xd7\xd6\xbf\x44\x26\x9c\x97\x80\x64\x4e\x7f\x0e\x29");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb4\x94\x15\x56\xb3\x7c\x22\xc9\xd8\x41\xbd\x0b\x1c\x4e\x5d\x87\x78\xb1\xc5\x98\xd8\xb1\xec\x55\xc4\xcf\xf1\x1c\xc7\x40\x2d\x05"[..]);

        ctx.update(b"\xc5\x79\xbf\xfa\x93\x96\x17\x2a\xa1\xd6\x54\xb4\x79\x30\xe6\x5f\x95\xc0\xc7\x4b\x6a\xbd\xa3\x82\xa5\xab\x3b\xd0\xb4\xd2\x0b\xd5\x88\xee\x2a\x58\x99\x28\x73\xde\x2b\x45\xe2\x7b\xd0\x1e\x40\xe9\x2f\x50\x03\xd7\x03\x34");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x63\xf8\x98\xb4\xfe\x7a\x7d\x8c\x7a\xc7\x76\xed\x78\x11\xa9\x22\x05\x27\x91\x40\x64\x29\x31\x8f\xd5\x38\x6a\x8f\x78\x64\xcf\xc6"[..]);

        ctx.update(b"\x71\xb5\x8f\x40\xe2\x8b\xd9\x42\x7f\x0f\x78\x21\xec\xe9\xc6\xe0\xf3\x4b\x7f\x5c\xc9\x24\x22\x26\x92\x97\x9a\x8e\x28\xab\x1e\x00\x18\x44\x56\x94\xee\x52\xbf\xe0\xb3\x48\x50\x11\xb3\xd0\x5e\x3a\x84\xbc\xbd\xeb\x43\x40\x28");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x76\x14\x58\x2f\x42\xd5\xa9\x07\xd4\xce\xba\x00\x25\x79\x8e\xe6\x1d\x38\x07\x24\x85\xf5\x4b\xa3\xd7\x42\xf2\x95\x03\xf1\x91\x1f"[..]);

        ctx.update(b"\x59\x23\x35\x63\x9c\xd4\xf2\xf9\xae\x66\x24\xe8\x95\xd6\xa7\x29\xb5\x07\x9e\xe1\xf4\x5a\xba\x91\x1b\x7f\xb9\x8c\x7f\x8b\x2c\xaf\xe2\x4a\x7e\x4c\x94\xd6\x32\x67\xdd\x68\x6c\x25\xa0\x6f\xb3\x81\x38\xdf\xf5\x41\xa6\xcc\xcb\x28");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x68\xd6\x4c\xb4\x33\x1b\x71\xe7\x2b\x42\xc8\x8b\x8f\x65\x13\xf4\xbc\x17\xd9\x95\x25\x4f\x04\xdc\x2b\x7c\xac\x02\x2a\xd5\xed\xfa"[..]);

        ctx.update(b"\xe2\x6e\x1d\xc8\x93\xfd\xce\xd8\x33\xea\xe1\x5a\xc0\x4e\xd4\x62\x4a\x07\xe3\xc8\x9d\xc3\xd1\xd8\x20\x5b\xe4\x4b\xfd\x63\x32\x77\x82\xff\x1b\xc2\xc8\xf7\x17\x5c\x92\x0e\xb2\x2e\x77\x90\xd4\x0d\x44\x2b\x46\x34\x9c\xff\x72\xf1\xe1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe1\x47\x0b\x2f\x44\xe3\x95\x5d\x2b\x54\x96\x29\x26\x79\xd9\x08\x6b\x7c\x0a\xa9\x4f\x0d\x51\x6e\x53\xa1\x42\xe4\x86\xc6\xde\xd8"[..]);

        ctx.update(b"\x7d\xef\x4d\x43\x9a\x9b\x7a\x6c\x57\x00\xbb\x91\x68\x8b\x1c\xa6\xc4\xec\xdf\xf2\x63\xc6\x64\x64\x46\xab\x4d\x2a\xb9\x47\xdf\x80\xd5\xdb\xaa\xc6\x48\xee\x6c\xd6\x59\x3f\x33\x94\x26\xfd\xc2\xc7\x30\x3a\x36\x27\x1a\x9e\xe9\xb6\xae\x58");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa5\x95\x91\xe2\x8e\xb9\x2d\xfb\xf2\x90\xa0\x67\x72\xbb\x14\x56\x55\x97\x9c\xd0\xfa\x0d\xf1\x5b\xf8\x2a\xa2\x76\x65\xab\xac\x03"[..]);

        ctx.update(b"\x28\x39\x98\xb5\xe2\x39\x4f\xb1\xce\x1c\xee\x35\x5b\x67\x7e\x79\xf8\x9a\x09\xaa\x28\xfa\x52\xd5\xd9\x93\xc3\x07\xf2\x7e\xa7\xb3\x2b\x60\x99\x53\x21\x33\x3c\x52\xaa\x49\x2d\x2a\x65\x8d\x36\xca\x96\x0d\xc5\x3a\x31\xb5\x9c\x81\x35\x5d\xc9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x82\xf2\x7f\x2a\xbb\xe5\xfc\x21\x84\xb4\x28\xeb\x92\x55\x1e\x89\x8a\x51\x77\xe9\x97\xe3\x37\xee\xbd\xc4\x93\x20\x09\x5c\x60\x5e"[..]);

        ctx.update(b"\x09\xf3\x52\xd1\xae\x80\x7b\x25\x8a\x9d\x49\x87\x3a\x4c\x4a\xa4\xc6\x3b\xfb\x52\xbf\x91\x80\x72\x8a\xe7\x98\x1b\xfb\x93\xd1\xf5\x26\x2d\xf8\xe2\x71\x4c\xd4\xe4\xee\x91\x14\x3b\xe8\xe7\x85\xcd\xbc\x08\x88\xcb\xea\x44\xc4\x92\x6c\x7e\x18\xcf");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc3\x0e\xfc\xcc\xac\xda\x26\xde\xd1\x6e\x92\xb0\xf2\xf2\xe6\x67\xdc\x98\xf2\x89\xa7\x54\xca\xc6\x28\xdf\xcb\x42\x1e\xcc\x72\x40"[..]);

        ctx.update(b"\xe6\xc2\xb2\x47\x89\x34\x8d\x64\x6a\x26\x3f\xfc\xdb\x6f\xcd\x51\x1e\x1a\xac\x52\x7b\x22\x7a\x7a\x90\x7a\xec\xf6\xe9\x50\x08\x06\xd0\x93\x94\xa5\x66\xb3\xce\x6c\xf3\xa8\x5d\xb4\x7b\x35\x88\xfb\x92\x6e\xa6\x89\x70\x78\xd5\x52\x48\x90\xef\x1f\xd6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xbf\x9a\xe2\x94\x1d\x2b\xb9\x48\xd3\x3a\xe2\xa8\x22\xe8\x73\x8f\x74\x4b\x5a\x9d\x69\x29\x97\x71\x9b\xd1\xa1\x18\x74\x3a\x07\x31"[..]);

        ctx.update(b"\xb0\x99\xe1\xb5\x23\x48\xf9\x3a\x6c\x22\x89\x6c\x72\x85\x4a\xb0\x4f\x2d\xd3\xd1\xf9\x4c\x5f\xad\x21\xdd\xd1\x2e\x5b\x35\xf5\x7c\x80\xdb\x09\x71\x50\x99\x7c\x4e\x5c\xd3\xf5\xac\x24\x18\x97\xb5\x7c\xf7\x84\x93\x34\x14\x9f\x29\xf7\x7b\x97\x01\x74\x33");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x01\xd5\x82\xc8\x2c\xe6\xe6\xa8\xc0\xcf\xa8\x17\x92\x11\x0b\xc9\x50\xfa\x55\xe3\x1c\x10\x1c\xc6\xf2\xdf\x6b\x2a\xeb\xc1\x89\x25"[..]);

        ctx.update(b"\xac\x40\x65\xc8\xdc\xd5\x80\x31\xb7\x47\x5c\x3d\xeb\xe0\x32\x16\x8f\xa9\xf7\xae\x5a\xd0\x2d\x48\x13\x4d\xfe\x3b\x48\xc5\x8b\xf2\x43\xaf\x82\x38\x3a\xb5\xee\xb2\xc6\x5d\x54\xad\x96\xe8\xfb\x1b\x65\x7b\x8b\xd7\x29\x7c\xda\x31\x8e\x3f\x02\xa6\xa2\xe2\xb7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9f\xb9\xd7\xe6\xb8\x1e\x86\xe4\x41\x69\x5c\xa5\xc7\x62\xcf\x3b\x77\x35\xa7\x5d\xbc\x4b\x2b\x1e\x3e\x56\x67\x84\x8f\x6c\xdd\x73"[..]);

        ctx.update(b"\xd2\xbc\x0c\xe7\x21\x7f\xf2\xe9\x44\xe1\xae\x47\xad\x58\x73\xbf\x39\x1f\x1b\x0c\xc0\x7f\x61\x51\xeb\x4c\x50\xbb\x45\xb2\xfb\x62\x95\x32\x6f\x71\x6c\xe7\xe6\x87\xfa\x0e\x3d\x5d\x25\xc5\xa8\xa8\xdd\x13\xa5\x41\xa9\x29\x2e\x83\x86\xe7\x33\xf4\xf2\xa2\x47\x28");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9c\x1f\xea\x57\x86\x70\x2d\x02\x7b\xb5\xb6\x6b\x3f\xa9\x2d\xe3\x46\x21\xa8\x62\x69\x82\xec\x21\xc0\xec\xf8\xda\xa7\x9d\xea\x05"[..]);

        ctx.update(b"\xa8\x2a\xa0\x79\xad\x87\x8c\xf1\xd6\x99\xe9\x61\x2f\xd3\xae\xf6\x03\xdf\x30\xc3\xe8\x90\x43\x58\xfb\x18\xa2\x97\x6d\x7b\x91\xc3\x3f\x43\xc0\xda\x49\x76\x73\x25\x81\xc5\xe0\x0d\x7a\x06\xd8\x30\x2b\xe4\xaa\xb4\xf2\xd5\x45\xdb\xd9\x4d\x9c\xd8\x17\x32\x4f\xe2\xda");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x46\xeb\x49\xbf\xc5\xb2\xee\x3c\x97\xb9\x11\x02\x52\x5b\x66\x66\xd5\x6b\xff\xaa\xb4\xba\xa0\x30\x59\xb8\x9f\x24\xec\x4b\x56\x22"[..]);

        ctx.update(b"\x72\xfe\xe2\x94\x94\xc2\x5b\x11\x9b\x6c\x5a\x60\x0c\x46\x8e\xc2\x2e\x42\xaf\x5b\xa9\x3e\xb7\x94\x52\x86\x4e\xbe\x46\x9a\x86\xf8\x36\x32\xc8\x52\x01\x80\x0f\x32\x88\xb5\x53\xf7\xbe\xc6\x49\xdd\xfe\x70\x49\x20\xa2\x7a\x8f\x65\xd1\x3a\xa7\x55\x98\x5a\x23\x8b\x3c\xdc");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf7\xee\xdd\xf0\x8d\x3c\x1a\xb5\x24\x16\x7c\x90\x8b\xcd\x86\x13\xd5\x81\xcb\x0d\x03\x52\x0d\x7b\x6b\x64\xd1\x81\xbf\x13\xbf\x1e"[..]);

        ctx.update(b"\xfe\xb7\xa0\x94\xdf\xfb\x5c\x0c\xb6\xf7\x23\xc1\xb5\xe7\xd7\x64\xd4\xab\x5f\x89\x6b\x13\xe4\x09\x00\x59\x40\xa4\x5d\xef\xb3\xeb\xe6\x7b\xd8\x5e\x2a\x21\xd8\xd1\xb8\x00\xde\x14\x34\xff\xcb\xd2\xaa\xc9\x2a\xba\x21\x16\x66\xdd\x74\x2d\x3a\x57\x5d\xa6\x6a\x32\xd3\x75\xb1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x40\xcf\xd8\x8d\x61\x3b\xc7\xd1\x05\xe1\x78\x08\x1d\x2e\x3c\x5a\x09\xcd\x62\x94\x65\x82\x19\x09\xdb\xc1\xe9\xb8\xcd\x04\xa2\x66"[..]);

        ctx.update(b"\x8c\xe3\x1c\xa2\xab\x2e\x7c\xff\x5f\xf5\xbc\xbc\xee\x36\x74\x63\xcd\x5e\xee\x65\xb8\x0b\xdb\x01\x01\x16\x44\x67\x29\xd1\xc3\x1e\xad\x8d\xf2\xa3\x22\x60\x24\x53\x84\x25\x53\x8a\xcc\xc3\x5a\x0d\xc0\xb3\x64\xa5\x0f\xbc\xa5\x0d\x9d\x01\x48\x0e\x1c\x3e\xa8\xe9\xdc\x4f\xc2\xf5");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd6\x1f\xcf\x2c\x31\x78\xcb\xf3\x40\x42\xa0\xfc\x02\x86\x9f\x46\x70\xea\x40\x0e\xa3\x84\xcc\xf0\x4d\x84\x60\x80\x6f\xe1\x1d\xce"[..]);

        ctx.update(b"\x49\x63\xa3\x6a\xf5\x62\xe0\x45\xec\x25\xe3\x9a\x0d\x9c\x2a\x03\x5d\x4f\x6d\x62\x04\x4f\xc9\x75\x98\x6d\x1e\xaa\x9e\xab\xb6\x34\x4e\xf0\x27\x5e\x97\x00\x7a\x1f\x63\xaa\x8c\xef\x1c\x44\xb5\xda\x6f\x54\xa8\xd3\x50\x90\x9a\x4a\x3e\x09\xa7\x01\xc8\xa3\x71\x1b\x50\xc8\x0a\x76\x0f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x69\xdf\x54\xc0\xfb\xb3\x58\x59\xb5\xfb\x24\x7c\x92\xb5\x66\x5e\xd2\x79\x62\xf0\x42\xa7\x90\x51\x9d\xb8\xce\x27\x34\x3b\x95\x90"[..]);

        ctx.update(b"\x06\x63\xac\x24\x6f\x64\x34\x27\xd7\x19\x8f\xeb\x54\x91\x9f\x97\xf9\xe0\x36\x86\x4d\xc3\x2c\x02\xfc\x87\xaf\x50\xcd\x26\xdb\xee\xd4\xb4\x8a\x18\xe9\xf9\x6e\x90\x59\x60\x5d\x9f\xdb\x10\xeb\x57\xc5\x58\x3e\x1c\x7f\x50\x1e\x63\x38\xbd\x4c\xb8\xd9\x36\x15\x06\x37\x85\x26\x13\x8d\x50");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdb\x45\x8a\x32\x7d\xd2\x1b\xb9\xc0\x73\xbd\x57\x36\xa7\x45\xe4\x67\x53\x8d\x94\x76\xb7\x64\x7b\x74\xdc\xd1\x70\x31\xba\xd7\x2d"[..]);

        ctx.update(b"\xe7\x0d\xf9\xce\x76\x33\x4c\xc2\x09\x6c\x0e\x71\x5d\x8c\x96\x65\x24\x92\xb5\xb6\x63\x7d\x3e\x80\x19\x61\xc5\x94\x21\xb9\xa6\x6e\x59\xff\x4a\x68\x04\x7a\xa5\x2d\x6d\x58\x03\x6f\xce\xfc\x05\xdd\x6f\x99\xe6\x4f\x12\x00\x03\x44\x8f\xb6\x9b\x8f\xbc\x4a\xd3\x4d\x8f\xe5\xfd\x1a\x87\x23\x19");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x22\x9b\xe9\xde\xed\xbd\x16\xe9\x48\xee\xbf\x8a\xa5\x83\x8d\x27\x6e\x0e\x6e\x29\x55\x70\xc4\x9e\xad\xc9\xf2\x7c\x0e\x2e\x57\x98"[..]);

        ctx.update(b"\xdd\xda\xd5\x81\x58\x34\xa9\x9d\x79\xab\xf4\xed\x36\xaf\x74\x06\x5b\x2e\x70\xe8\x7c\xb8\x48\x3b\x2e\xe2\x50\xd8\x77\xe8\xde\x8b\x82\x96\xa2\x17\x97\x68\x36\x7b\x4b\xf0\xf6\x74\xf4\x78\x12\x43\x60\xe9\xe2\xb0\xc7\xeb\x79\x4f\xfd\x4e\x7c\x02\xec\xa9\x35\xbf\x42\x40\xe3\x94\x67\x2b\x54\xe8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x13\xb1\x1a\x38\xef\x74\xab\x16\xbd\x65\xb8\x4e\x83\xd3\x48\x48\xbf\x3c\x56\xe6\xca\xb5\xc5\x6c\xc0\x54\x1e\x78\x3d\xf1\xd4\xb7"[..]);

        ctx.update(b"\x84\x61\x7b\xcd\xea\x87\x42\x0c\xb0\xe9\x3f\xee\xbf\xde\x3a\xab\xbf\x7c\xc7\x17\x80\x65\xc8\x1c\x99\xe3\x33\x68\x4f\x38\x15\x63\xb8\x2f\xd7\x64\x20\xf7\xc0\x24\xab\x31\xbc\x73\x3c\x48\x5c\x29\x96\x01\x1d\x60\xc1\x53\xed\xe0\xb0\x78\xeb\xdb\x20\x44\x6f\xdb\xc3\xb6\x4a\xf8\x9e\xcd\x4f\x4e\xaf");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5b\xb5\x1b\xcf\xee\xdc\x61\x8a\x28\xcc\xf3\xbb\xd9\x6b\xe0\x89\x72\xee\xb1\xbc\x22\xd9\xbb\x64\x88\x3d\xab\xdc\x6c\x3c\x6b\x54"[..]);

        ctx.update(b"\x97\x0f\x5a\x93\x08\xf9\x29\xc5\xde\x78\xe8\x11\x9b\x96\xca\xf0\xad\x9c\x4a\x06\xf1\x82\xc7\x3e\x75\xa9\x71\x1c\x82\xcd\x69\x4f\xda\x35\x24\x86\x3e\x2c\x11\x85\x33\xb5\x7b\x78\x06\x7b\xb0\x87\x0e\x07\x63\xaa\x31\x35\x4f\xbf\xa9\xe9\x1d\xda\x08\xfe\x88\xdc\x7a\x7f\xbf\x47\x4b\x1a\x0a\x50\xa9\xd1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x93\x0a\x9c\x33\xe3\x93\xae\x8b\x1c\x39\xe0\x72\xb0\x8f\x97\x2a\xd8\x40\xdb\x54\x01\xa4\xed\xed\x1f\x1f\xd1\x4f\x8a\x0f\x68\xc6"[..]);

        ctx.update(b"\xf2\x9e\xc4\x43\xa7\xc4\xbf\x4f\xcb\xc2\xcc\x4e\x99\x2f\x3b\x1e\x0c\x70\xeb\xf6\x0a\x20\x19\xef\xdc\xb6\x76\xc3\xe2\x61\x7c\xd4\x85\x37\x01\x70\x6c\x8f\x0d\x53\xf3\xab\xe9\x6d\xd9\x9b\x78\xe6\x38\x5d\xca\x60\xbb\x49\xef\x7c\x0b\xad\x09\x9f\x8a\xb4\xa5\xf6\x44\x8c\x4c\xb7\x23\xc7\x54\xaa\x62\x02\xfc");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x36\xd1\xcb\x5d\x93\x0d\x73\xf0\x19\x99\x53\x3b\x03\x53\x13\xa5\x4b\x1e\x2c\xc2\x23\xf0\x7a\xf8\x29\x42\x42\x30\xb8\x40\x51\x77"[..]);

        ctx.update(b"\x49\x13\x70\x9a\x09\xe6\x06\xf0\x03\x49\x26\xf2\x1c\xe0\xfa\xee\xfc\x04\x83\xb5\xc0\xad\xe9\x9f\xef\x95\x0f\xd0\xb0\x9a\x6b\xba\x3a\x64\x15\xe3\x8b\x49\x1d\xe8\x5d\x6d\x22\xc3\x56\x2e\x20\x49\x7f\xbf\x8a\xf6\xf7\x12\x99\xed\xe9\xfc\xe3\x3b\xbb\x94\x2f\x5c\x88\xdc\x88\x00\xd1\xa3\x90\x0f\x6f\xa9\x17\xea");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x09\x48\xd1\xdc\x7a\x0a\x01\x72\xd3\xda\xe4\xae\x40\xd1\x28\x44\x4f\x2f\x63\x09\xb4\x43\xad\x6a\x8a\xb3\x84\x9d\xef\x60\x6a\x7e"[..]);

        ctx.update(b"\x33\x65\xdc\x87\xc7\xf2\x9b\x50\x92\x5f\x2d\x50\xb1\xe4\x37\xa4\x9c\xa9\x95\xca\xac\x35\xfc\xb3\xdb\x29\xdd\x19\xcc\x2e\x8c\x13\x0b\x2c\xa1\xe7\x95\xbd\x9f\x1f\x9b\xe1\xd3\xdf\x83\xf3\x7f\x75\x30\xdb\x38\x32\x83\x86\x8d\x78\x21\x93\xe9\x15\x77\x4f\xbb\xcf\xfc\xe4\xd3\x40\xb6\xcd\x0f\xe5\x00\x8d\x38\x71\x8a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x57\x9d\x6c\x49\x2d\xcf\xb0\x32\xb9\x27\x2a\xa0\xf5\x4d\x13\xec\xe0\xfb\x22\x76\x61\x9d\xd7\xe7\x5f\x86\xae\xf3\xf5\x8b\xfa\xea"[..]);

        ctx.update(b"\x45\xa8\x99\xee\x24\xce\x1e\xb7\xff\xe1\x3a\xa5\xac\xb0\x8c\xab\xe9\x4f\xf0\x03\x95\x71\x53\xb7\x20\x19\xc6\x66\x40\x72\xd0\x45\xe5\xfa\x09\x64\x70\xe6\xe4\x33\x1f\xc3\xcd\x44\xcc\x59\x74\x52\xd8\xd4\x68\x83\x62\x26\xe7\xea\xbc\xa7\xf5\x50\x1f\xfe\xab\x34\x9d\x45\xd2\x10\x58\x68\xb0\xde\xf2\xce\x22\xdb\x44\x3c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5c\x08\x8c\xd0\xc1\x8b\xf6\x99\xed\xf4\x7d\xdb\xbd\x1e\x11\xc9\xca\xa9\x16\x1b\x85\x2a\x60\x24\x49\x2e\x7a\xb7\xdd\x57\x45\xee"[..]);

        ctx.update(b"\x38\x9d\x76\xc1\x8d\x8d\x11\xeb\x5b\x7e\x6e\x80\xd7\x9f\x93\xbe\x52\xc7\x84\xeb\x11\x01\xb7\xf3\x59\x48\xc7\x52\xc9\x8f\x6c\x34\xed\x05\xab\xdb\x0f\x96\x4a\x42\x50\xb5\x87\x00\xf8\x59\x0a\x6e\x0b\x45\x17\x0c\x1d\x8d\x08\x21\x66\x56\xfb\xa2\x19\xdd\xc8\x55\xf2\xbb\x36\x36\x0e\x67\x6e\x1a\x00\xfc\xaf\x92\x21\x3c\x44");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb3\x2d\x1b\x0a\xcc\x09\x7e\xdd\xc1\x3d\xbb\xa9\x1d\x2c\x14\x47\x74\xf4\xbf\x1b\xa2\x27\x57\x70\xb5\x28\x58\x47\x1d\xe2\x6c\xa1"[..]);

        ctx.update(b"\x12\x0b\x80\xb4\x4a\x24\x35\xd4\x78\x37\x66\x41\x6e\xde\x17\x84\xe0\x0d\x34\x67\xc7\xed\x1b\x63\x04\xe7\x5b\x58\x46\xfe\xbb\x81\x2d\x63\x7f\xa6\xb0\x43\xe6\x93\xbc\xe4\xf7\xca\x1d\xf0\x7d\x75\xa2\x11\x51\x9e\x7c\xbd\xd7\xa2\xe8\x54\xcb\xe3\x0e\x68\x3e\xbe\x28\x5b\xdb\xbd\xe6\x30\x6f\x0d\x11\x9b\xa9\x32\x70\x5a\x1e\xf9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x20\x08\x18\xa7\xfa\x96\x34\x23\xd0\xa9\x0b\xd8\x54\x75\xd6\x7b\xef\xdf\xa4\x3b\xb4\x9b\xe7\xc0\x26\xb9\x69\x37\x43\x40\x4d\xcd"[..]);

        ctx.update(b"\x71\x37\x05\x67\x4e\x20\x06\x6d\x61\xbf\xe9\xca\x76\x3a\x87\x05\xcb\x6c\xa1\xe8\xcf\x95\xc6\xd7\x29\x50\x76\x61\xfc\xc6\x87\x15\x69\x22\xc8\x97\x56\x45\xe5\xf3\x6e\xba\x8a\x30\x69\xec\xcb\x29\x8e\x96\xc4\x98\x76\x7c\x7c\x74\x12\x59\x10\x7d\xf0\x18\x0d\x15\xb9\x46\x1d\x51\xff\xc8\xe6\x50\x35\xc9\x2f\xcb\xff\xf7\x44\xee\xad");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x78\x6e\xa3\x68\x15\x64\x77\x61\x6f\xd7\x60\x39\xf9\xb1\xca\xaa\x95\x76\x3a\xdc\xeb\xc1\xcf\xda\xe8\x19\x8f\x82\xa4\x11\x82\xa9"[..]);

        ctx.update(b"\x12\x3c\xca\xec\x52\x2c\x29\x83\xb6\xcc\x2e\x06\xb0\x90\x0c\x33\x1d\xb4\x9d\xc5\xd5\x80\x9c\xdf\x80\xcb\x14\xae\xac\xbe\x18\x4b\x1d\xda\xb5\xde\x64\x3c\x25\x09\xb4\x20\x45\x84\xf0\xa5\x4a\x04\x08\xd0\xaf\x62\x86\xd2\x71\xf9\x64\x8e\x7f\x50\x21\x56\x6c\xf9\x14\xf5\x68\xb8\xff\xf3\x6e\xe8\xe7\x14\xbf\x1b\x23\x22\x8e\x01\xb4\xec");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x91\xb3\xe2\x46\x35\x58\x94\x1d\x06\x26\x93\x6a\x8d\x27\x57\x22\xfe\x2e\x60\x0b\xb2\x4d\xf7\x8d\x7c\x46\x75\x26\x75\xd1\xc4\xda"[..]);

        ctx.update(b"\x75\xd0\x40\xdd\xea\x04\x28\xd8\x23\x1d\xce\x4f\xbc\x98\xcc\x4a\x2f\x57\x8a\x5f\x47\x47\x53\x79\xd7\x40\xae\xd8\xab\x2c\xeb\x59\xbf\x6c\xfc\xf8\xaf\x77\x86\x93\x6f\xc4\x38\xf1\x3c\xeb\x5c\x02\xd2\x07\xab\x68\xe0\x5e\x66\x37\xb9\x44\xa2\xf9\x4c\x3c\xe4\xf6\x56\x61\xed\xb2\xd6\x04\xd7\xb6\xc2\x97\x61\x93\xf8\xc9\x88\x25\x77\xf3\x57");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x74\x9b\xae\x93\x22\xaf\x58\x84\x95\xdb\xc0\xce\x42\x6d\xeb\xd9\xe9\xcc\x4c\x02\xf5\x0c\x52\xca\x24\x6b\x55\x82\x84\x0a\x86\xcb"[..]);

        ctx.update(b"\x7c\x23\x8a\xaf\xde\x6a\xab\xab\x17\x7f\x71\x8e\x0c\xfe\x13\x86\xcb\x14\x21\xb4\x79\x9b\x15\x78\x8b\x86\x2b\xf0\x3a\x80\x72\xbb\x30\xd0\x23\x03\x88\x80\x32\xba\xf5\xbe\x24\x23\x41\x82\x20\x95\xfa\x71\x48\x7c\x68\x02\x4a\xa0\xc1\xdf\x31\xaa\xa0\x8e\x82\xe4\xb4\x50\x84\x40\xb5\x49\xdf\x9c\x9d\xe2\x68\x44\x14\x6c\x3c\xb8\x2c\x06\x0c\xb0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x36\xba\x7b\xf5\x3f\x7d\x25\x37\x38\xf5\x90\xee\x25\xc6\x12\x0e\x0e\x89\xb1\xb8\x9c\x2a\x06\x91\xdb\x6d\x38\xb7\x2c\x20\x05\x5b"[..]);

        ctx.update(b"\x47\x0b\xe1\xab\x59\x5f\x8b\xfc\xbe\x65\x1a\xaf\xbe\x58\xa9\x0c\xd4\xd7\x2d\xba\x54\xf3\x43\x7a\x50\x87\x2c\x16\xbd\xc8\x15\x93\xd2\x1c\x9c\xae\xb1\x30\x91\x87\x49\x44\xf9\x97\x7b\xca\x59\xe7\x7b\x30\x31\x84\x7a\xfa\xc6\x2e\x82\x13\x24\x50\xa1\x65\x64\x35\x49\xa7\x2b\x78\xf7\x76\x73\xbc\x13\xa8\x8e\x9e\x17\xdd\x67\xe8\xf4\x56\x0c\x07\xb2");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8e\xf8\xd6\x47\x2e\x09\xf5\x66\xf7\x0e\x17\xb4\x54\x48\x9c\x19\x3d\x11\xfd\x99\x85\x5f\x7e\x69\xc0\xee\xd3\x89\x51\x5d\xf5\x77"[..]);

        ctx.update(b"\x69\x73\xb2\xd5\xfd\x42\xe7\x8a\x22\x45\xc7\x6f\x25\x68\x1d\x70\x4e\xa1\x94\xe2\xb5\x78\x28\x1e\x58\xc6\xdb\x6f\x6b\xbd\xa9\x2f\xef\x28\xbd\x2c\x86\xf4\xe5\x51\x86\xf0\x44\x82\x4c\xd1\x47\xc7\xd3\x79\xda\xad\x85\x79\xdb\xe9\x50\xd4\xce\x3f\xfd\x06\xd3\xa2\xbd\x44\xd6\x9d\x05\x06\x06\xf1\x9c\xc3\x88\x80\xf3\xfb\x6c\x69\xb4\x7b\xe0\x75\x9d\xdd");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x22\x6f\xa5\x4a\xb3\x1f\x5a\x78\xea\xb4\xf9\x2b\x10\x97\x2f\xcf\xc8\xcc\xad\x25\xe0\xff\xa4\x93\x65\xda\x83\xf2\x72\x20\xf6\xcd"[..]);

        ctx.update(b"\xc4\xd0\xd1\x4c\x93\x05\x88\x65\xd0\xfb\xe2\xe6\xe1\x4e\x99\xf1\x70\xce\xcd\x75\xc1\x83\xf9\x53\x43\xef\x26\xbf\x65\x92\x5f\xe9\xaa\xb4\x64\xdd\x00\x5e\xa3\xa8\x4e\xaa\x63\xbd\x4c\x3c\x43\x97\x97\x11\xe3\x9f\x05\x78\xe1\x4f\x35\x5d\x07\x98\x6b\xcd\xc2\x81\x9b\xb7\xd6\x12\xc4\xf0\x51\x70\x80\x7a\xbd\x49\xd8\x06\x9a\x72\x5c\xb7\x0c\xcf\x74\x29\xd0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x88\xe1\x83\xa7\x94\xf4\x78\xa8\x50\x85\x69\x66\xfd\xed\xd7\x1b\x14\xd8\xab\xa7\x28\x19\xfb\x26\xf1\x4d\x23\x15\xc6\x7f\x2d\xeb"[..]);

        ctx.update(b"\x73\x66\x17\x99\xc7\xd2\xe2\x4e\xa4\x84\x29\x9d\x8c\x8c\x11\x25\x73\x9d\xce\x5c\xc3\x07\xc3\x27\xa3\xe5\xea\xa8\x79\x98\x77\x57\xad\x74\xb8\x55\xb1\xb8\x40\xfd\x3e\xc4\x44\xbb\xe0\x86\x45\x21\xb4\x36\xef\x37\x70\x43\x42\x13\xa5\x8d\xef\xaa\x41\xd4\x05\xe6\x58\x70\xcc\xf9\x24\x43\xf1\xf0\x4a\xed\xe4\x8b\x5e\x3e\x7f\x4c\xaa\xf1\xe8\xce\x2b\x97\xb8\x58");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xad\x1a\xff\xd1\xb1\x94\xbc\x0a\xfc\x38\x9a\x2c\xb2\x32\x81\x24\x4a\x8d\x80\xa5\x24\x26\x2b\x25\x60\xd5\x8f\x7f\x6d\x22\x6f\x51"[..]);

        ctx.update(b"\xd4\x90\x63\x45\x3b\x77\x9b\x18\x93\x3d\x23\x33\x69\x88\x16\x17\xc9\x87\xfe\xf4\xb1\x8e\xea\x6f\xc6\x00\xc1\xa2\x46\x72\xd5\x64\x3e\x0d\xbb\xcb\xfd\x34\x5b\xf0\xc8\x67\x43\xf2\xcf\x68\x41\x3e\x60\x9c\x30\x0d\x5b\xb3\x12\x2e\xe6\x8c\x5f\x05\x61\x52\x9d\xd4\x9d\xec\x31\x39\x48\x94\x75\xf8\xac\xc5\x0a\xfb\x9a\x5b\x70\x97\xa0\xf6\xd3\x11\x82\x9f\x28\x61\xe8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x67\x68\x94\x6a\x3a\x27\xf4\x62\x88\xdd\xfd\x4c\xb7\xb5\xfc\x9e\x27\xcd\x3b\xa4\x6e\xc9\x59\xe6\xf3\xd0\xcb\x5b\xff\x68\xe2\x26"[..]);

        ctx.update(b"\x90\x58\x88\x12\x13\x60\xfe\x46\xf8\x6d\xdd\xd1\x19\x07\x61\x95\x21\x34\x43\xf3\x39\x90\x7f\x87\xe5\xa0\x5b\x12\x28\x64\xa5\xb7\xef\x2b\x3c\x6b\xad\xab\xd2\xa7\x42\x51\xef\x5f\x21\xd6\x39\x38\xb2\x13\xf4\xf6\xf5\x16\xc5\x33\x69\x75\xbb\x57\x2c\xe2\x8f\xc4\x63\x00\x2b\x51\xdc\x76\x6a\xfa\xf2\x41\x1f\xd6\xb1\x4e\x58\x49\x3b\x48\xe4\x7a\x6a\xaf\x12\x07\xb7\x2a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x14\x5f\xf2\x62\xe2\x03\x95\x7a\x88\xa8\x75\xef\x98\x6a\xc4\x84\x20\x4b\x20\xd9\xc4\xbe\x7d\x35\xad\x59\x61\x27\x64\x99\x1d\x49"[..]);

        ctx.update(b"\xb7\x52\xac\x4a\xdc\x97\x01\x21\x19\xd8\x95\xa8\x00\xf3\x18\xdb\x6a\xe7\xe3\xc7\x03\x44\x94\x9d\x09\xe0\xf8\x05\x50\xcc\x2f\xb4\x8a\x86\x0a\x41\xc5\x70\xb8\x34\x35\x81\xa2\x61\x34\x3b\x93\x74\x87\xdd\x1c\xcc\xe3\xd9\x37\x8d\xaf\x88\x2a\xf0\x0c\x1d\xae\xde\x13\xd8\x51\xbe\x5d\x7c\x19\x8b\xc9\x02\x6a\x29\x79\x8f\x29\xf9\x1e\xa4\x22\xab\x76\x94\xa7\x53\x13\x3c\x7a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x2f\x39\x9d\xc5\x53\xda\x60\x7b\x1c\x2d\x49\x0e\x64\x61\x66\x4a\x5a\x2c\x00\x7f\x18\x00\xd7\x6a\x2a\xaf\x77\x42\xa8\x6b\x96\x17"[..]);

        ctx.update(b"\x1e\xf7\x00\x1c\x52\x80\x39\x38\xe9\x37\xd9\x3f\x5c\x85\x3f\x66\x1c\xe4\x83\xb4\x26\x2e\xac\x63\xd8\x1a\x23\x89\x03\x03\x68\xf8\x41\xec\x04\x62\x89\xc2\xbf\xc1\x00\xe0\x12\xf5\x12\xd9\xdf\xaf\xc2\x71\x60\x8f\x3e\xec\x5e\x89\xb7\x8c\xa3\x8d\x6f\x55\xb1\x67\x14\x13\xb4\xe4\x4c\x9e\xb6\x36\x91\x5d\x12\x99\x3f\x09\x2b\x37\xeb\xb3\x5d\x49\xf7\x09\x34\x80\x63\xd5\xdc\xa3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x77\xe3\x71\x34\x9c\x46\x6f\x6c\x33\x1f\xac\x02\x84\x98\x75\x37\xc1\x5d\x04\x30\xe3\xf0\xcc\xa8\xf1\xfe\x2a\xa7\xc7\x94\xef\x09"[..]);

        ctx.update(b"\x28\x4b\xff\x3b\x6a\x68\x3f\xf2\x20\xf5\xcb\xfd\x3c\x2f\xb8\xa7\x2b\xdb\x85\x06\xfc\xdd\x5b\x2d\xc5\x07\x94\xa3\x42\x2f\x7b\x14\xc0\x84\x52\x47\x39\x22\x9a\x50\xc1\xe3\x8e\x07\x20\x78\xf1\xb8\xc1\xb9\x09\x1c\xcb\x3a\xec\x88\x8d\x1f\xfa\x65\x1f\x31\xab\xe0\x46\x31\x8b\xe2\x99\x6c\x82\x43\xe2\x8f\x59\x3b\xc7\xbd\x7a\x19\xaa\xdf\xf7\x37\xcd\x3b\xce\x2d\xba\x49\x6c\x45\xe2");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xbf\xaa\x54\x96\x35\x21\xe6\xe1\xac\x72\xb5\x5e\xe4\xd6\x54\xc0\x41\xf6\x2b\x73\xad\x5a\x53\x05\xc5\x8f\x47\xbb\x25\xdb\x7c\xc2"[..]);

        ctx.update(b"\x08\x1d\x08\x39\x7e\x13\x53\x07\x3f\x8d\xf2\x46\x75\x58\x8a\xd2\x15\xe4\xdc\x36\x15\xa5\x9d\x2a\xd9\xb9\x81\x5a\xee\xcb\x9a\x69\xfa\x37\xe0\x36\xf3\x6f\x11\x5e\x90\x9d\xbb\x02\xfd\x8a\x96\xca\xd3\xbe\x18\x29\x47\xe9\x44\xe3\xa2\x81\xc3\xcd\xf1\xad\x35\xd4\xfd\x62\xc9\x41\x7d\xcb\x0b\x3c\x8b\xef\xfe\x8e\x55\x8e\x6b\xab\x15\x4b\x78\xef\x43\x11\x7c\x28\x08\xaf\x12\x55\xf7\xc5");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xea\x8f\xbe\x27\x3c\xd2\xbb\x3a\xb9\xcc\x7f\x36\x40\xfb\x4e\x31\x53\xa0\x28\xd8\xba\x9a\x88\x0b\x3b\x82\x19\x00\x6c\x28\x28\x1f"[..]);

        ctx.update(b"\x1e\xd6\x24\x4a\x59\x96\xd5\xa4\x42\x3c\x01\x06\xa4\xb8\xe7\x85\xd7\x50\xc7\x31\x38\x43\xf3\xdf\xd6\xac\x27\x2c\x6e\xaa\x1d\xbe\x9a\xbe\x5d\xcd\x45\x74\x53\xe2\x62\x46\xa3\xdf\xdd\xbd\xc1\x40\xd2\xd9\xbc\xa0\xfe\xb3\x5a\x7f\x73\x61\x6a\x42\x8f\x1a\x56\x7b\x2e\x9a\xf8\x6b\x1f\xc8\xae\xc6\xd5\x97\xb1\xb5\x5f\x2a\xa2\x21\x9b\x3b\x66\x2f\xa6\xbd\x34\x07\x96\xc0\xec\xda\x6c\x96\xab");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8f\xc2\x26\x35\x17\xc8\x5d\x7a\xb8\x92\x65\x95\xb8\x04\x7b\x4b\x83\x8a\x07\x5a\x29\xb2\x4d\xa7\x12\x30\x3c\x3e\x6e\x59\x7e\x49"[..]);

        ctx.update(b"\x63\x18\x87\x81\xf4\xe9\xcb\xd1\xe8\x9a\x54\xa6\x5d\xa0\x53\xb9\x37\x22\xe1\x10\x6f\x00\xf0\x24\xad\x15\x82\x42\x1a\xb9\x19\x32\x6f\x8a\x6e\x17\x53\x6d\x65\x96\xe3\xcf\x41\x3a\x92\x31\x14\x17\x33\xe3\x7a\xae\x54\x0f\x87\x11\xce\xfa\xfe\x48\x9a\x87\xc4\xf2\xe6\xfd\x94\x2f\x68\x09\xf3\xbe\xf3\x07\x67\x63\x48\x7d\xe4\x8c\x2e\xe8\x87\x33\xc5\xbc\x87\x06\x17\xa6\x68\xc6\xf0\x14\x71\xed");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x91\xa8\xe2\x85\x02\x90\x85\xe2\x24\x98\x70\x78\x06\x64\x86\xb6\xc6\x05\xcb\xac\x27\xe4\x9e\x84\xf4\x63\x97\x10\xdd\xd0\x5d\x33"[..]);

        ctx.update(b"\x37\x11\x71\xb6\x26\xcc\xea\x8b\x29\x80\xdc\xd3\x3b\xcb\xb8\x49\xfe\x57\xd2\x01\x00\x0b\xc2\x44\x3d\x83\x0a\x9b\xe6\xbe\xfa\x2a\x61\x17\xd1\x3f\x91\x77\x25\x2b\x57\xc8\x85\x51\x59\xad\x5c\x7c\x02\x70\x18\x76\xca\x3c\x8c\xd6\xb7\x8c\x93\x19\x23\xe3\xf7\xf4\xc1\xbe\x91\xe9\xdb\xc6\xb1\x19\xdd\x4b\xc0\x59\x17\xee\x23\x87\xbf\x9d\x63\x1f\x35\xc5\x8b\x09\x8b\xc2\xd2\xb5\x45\x9b\x7a\x6a\x5d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe6\x86\x55\x14\x9d\xc1\x4b\xc5\x75\x9b\x8c\xac\xb7\xc2\xd2\xa8\x50\x45\x6a\x64\xa8\x7a\xf4\x1d\xe8\x4f\x56\x7f\xfe\x9c\x5e\x60"[..]);

        ctx.update(b"\x9b\xf7\xc7\xd9\x90\x59\x24\x85\x89\xb1\x8a\x3f\xd3\xb0\x3f\x4a\x72\xd2\x1d\x2a\xac\xd6\x35\xb5\xcc\x71\xb8\x6d\x20\x43\x18\x0d\x44\x0a\x6a\x62\xa2\x16\x0f\x5c\x3a\xfd\x93\x65\x9f\x8f\xd5\xf1\x42\x0c\x68\x57\xf3\x95\x3a\xdf\xa5\xe1\x73\x1c\x72\x31\xa0\x7c\xdf\x86\x11\x1d\x60\x11\x7f\xe2\x1f\xeb\xf3\xc7\xf9\x06\xe2\x71\xc9\x64\x9d\x0a\xea\x86\xe9\x4b\xf7\x28\xe9\xda\xd5\x2b\x45\xd7\xf4\xd3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x38\x5b\x95\x54\x15\x58\xad\x0a\x9a\x20\xcb\x91\x11\x6e\xda\x03\xc5\xa6\x94\xca\x95\xea\xb1\xca\xf5\xc0\xd1\x50\x18\x99\xd3\xbd"[..]);

        ctx.update(b"\x19\x73\x26\x3a\x26\xd4\xa4\x18\xc6\x91\xdd\x5f\x5c\xcb\x7d\xd7\x24\x1e\xbc\xae\x24\xcc\xeb\xbe\x60\x17\x4c\x0f\x4e\xee\x27\x9d\x89\xcc\x1a\x5b\xc9\xde\xb1\x77\x13\x64\x37\xe0\x2d\x59\xcb\x06\x55\x56\xb6\xe1\x32\xb4\xfb\x34\x0b\xf1\x30\x13\x37\x54\x8d\xa3\xd5\xd4\x43\x21\xbd\x37\x0e\x25\xb6\x77\x0e\x28\xe9\x69\xd3\x44\x95\xa0\xb7\x3c\x05\x90\x70\xbc\xbf\xef\xe5\xe2\xff\x61\x41\xfc\x63\x53\x46");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x39\xd8\x74\x0d\x56\xe5\x8f\x02\x7a\x6a\xf1\x3c\xce\x9b\x64\x84\xb2\x3f\xa6\xb3\x69\x33\x54\x17\x5c\x03\x4a\xc1\x77\x8a\x1a\x75"[..]);

        ctx.update(b"\xec\x8e\xe6\x5c\xef\x05\x45\x2d\xef\xe8\x32\xe0\xa3\xa3\xba\x61\x16\xa8\xd5\xd6\xa8\x63\x8e\x2e\x56\x6f\x63\xfd\x3c\x0a\xef\x87\x58\xc6\xc6\x1a\x91\x82\xa1\xa2\xf8\x99\xbb\x77\x4d\x59\x7a\x79\xc5\xb7\x52\x5f\x0a\x40\xe7\x4c\x84\x53\x71\xa3\xd7\x8e\x65\x1d\xad\x86\x52\x19\xb5\xbb\x08\x33\xec\xcb\x00\x17\x1b\xe5\xf8\x98\xa0\x81\xde\xfc\x72\x4e\xef\x9a\x60\x6c\x20\xa8\x99\xdc\xbd\x4e\x71\x35\x50\xfe");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3e\x7a\x65\x2d\x89\xdf\x9e\xd7\xca\xcf\x53\xd0\x33\x99\x7c\x87\x52\x8b\x8c\xc3\x7d\x59\xb8\xea\xfb\xed\xbe\x4e\x8e\x9c\x5c\x3b"[..]);

        ctx.update(b"\xdf\x41\x0a\x8b\x65\xf1\x71\x39\xfc\x8e\xb8\x49\x5d\xf8\x13\xb1\xf6\xf9\xf1\xb7\x47\x1c\x7e\x62\x8c\xc3\x6d\x2e\x9c\x74\xf6\x03\x11\x74\x1a\x68\xa9\x97\x2e\xf5\xec\x56\x97\x06\x0f\x22\xd3\xb7\x19\xf8\xb9\x0c\x44\x36\x01\x6f\xa6\xad\x06\x60\x10\xf7\xcb\x68\x37\x0a\x3c\x4e\x11\x0e\xb7\x8e\x82\x6c\xcb\x07\x38\x28\x40\xf4\xd4\x70\x57\x1d\x4a\x75\x52\x4f\x8a\x79\x37\x09\x60\x29\xd2\x78\x98\x2e\xf9\xa5\x67");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x29\x4c\x87\x88\x03\x60\x98\xf4\x9c\x23\x01\x65\x34\xd0\x32\x93\x9d\x5e\x96\x38\xe3\xe4\xe8\x47\x45\x33\xf5\x2b\x66\x8f\x2d\xa1"[..]);

        ctx.update(b"\x27\xc2\xee\x5f\x63\xc1\x14\xc1\xf6\x15\x30\x68\x5c\x19\xc2\x1a\x4b\x7d\x47\xb3\x42\x8e\x6b\x82\x65\xb8\x7b\xf0\x88\x49\x19\x62\x8a\x77\x33\x61\xe4\xa5\x36\xf9\x81\xe2\x7e\x5e\xd1\xb3\x12\xd6\x19\x12\x8e\xd2\x0d\xb6\x60\x5e\xa0\x95\x37\xbc\x1b\xa8\xb9\x4c\x6c\xee\x75\xb0\xc1\xb3\x18\x2c\x56\x12\xe3\x53\x8b\xb9\x5c\xe0\x3a\xec\x44\xe2\x47\xcd\xca\x1c\x79\xda\xfe\xab\x88\xb7\x88\xe9\x77\x2e\x9a\x92\x23\x74");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xcf\x02\x7f\x60\x17\x61\x8d\x97\x0d\xa8\x38\x65\xd8\xda\x56\x0f\x45\x4c\xe2\xbd\x86\xaf\x59\x62\x4f\xb8\x7c\x8e\x48\xbe\xb2\x08"[..]);

        ctx.update(b"\x2f\x51\x0b\x72\xad\x1a\x26\xfe\xad\x66\x9c\xca\xfe\x9a\xd0\xfe\x31\xce\x0a\x12\xf3\xd0\x6b\xee\x9a\x6c\x27\x50\xfd\xeb\xee\x6d\x89\xda\xab\xdb\x91\x0d\x71\xae\xe6\xe3\xb2\xe2\xa8\xca\xb3\xf9\x5b\x06\x3c\xe4\x3d\xaf\x94\x27\xd7\xf9\xad\x39\xd3\xb9\x9b\x29\x0d\x24\x4a\x87\xed\xd9\x71\x82\xf3\x22\xc2\x4e\x93\x76\x64\xc9\x44\x43\xa2\x5d\xd4\xeb\xe5\x28\xfe\x0c\xdf\x5a\x3e\x05\x0a\xdf\xe4\xb6\x51\x3f\x68\x87\x0c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x89\x69\x72\x7f\x52\x2a\xb3\x92\xd1\x1c\x3c\xfa\x7d\x02\xfa\x2f\x9b\x45\x7a\x1d\xe2\xaf\xd3\x60\xac\x9b\x2b\x4f\xec\xe8\x4e\x32"[..]);

        ctx.update(b"\x2a\x81\xeb\x0e\x6f\x1d\x13\xdf\x65\x44\xc6\x93\x21\xc5\x61\xa4\x89\xff\x9d\x77\xb9\xae\x11\xd7\xa6\x04\xf8\xa2\x01\xfc\x35\xc2\x5f\xeb\x94\x5f\x96\x57\x53\x53\x6a\x0d\xe1\xdf\x69\x6e\xe5\x9c\x13\x0a\x82\xb4\x3e\x70\xd9\x51\xb9\x43\xd0\x56\x30\x91\x12\x4e\x4f\x61\x58\xa1\x2a\x8b\xe6\x0e\xdd\xcd\x58\x55\x6f\x1a\x9e\x34\xfd\x35\x46\x95\x14\x7f\xf9\x06\x5e\x0e\x49\x69\x81\xfe\xaa\xed\x44\x0c\xbd\x3d\xf4\x60\xd5\xf7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x38\x18\xa1\x4b\x4a\xbd\x90\x0c\x82\x32\x75\xd4\x5a\xfe\xa6\xed\x1d\xf6\xcc\x65\x9e\xb3\xef\x38\x74\x45\xf8\xad\x39\x03\x62\x3f"[..]);

        ctx.update(b"\x9a\xa1\xfa\x54\x0a\xcb\x4d\x2a\x59\xa6\xa3\xaa\x8c\x2f\xdb\xcd\x0a\x4a\x17\xb6\xb5\x55\x08\xe6\x5a\x03\x6c\xb3\x4b\x68\xd4\xf6\x4a\x50\xab\x05\xa9\xd5\x74\xe1\xb0\x31\x53\xb0\x3f\xd0\xcf\x6d\xb4\xaa\xd6\xde\x0f\xcf\x01\xc6\x55\x43\x1a\x5d\x32\x0d\xdc\xfd\xe1\x8b\xf9\x1e\x51\x08\x62\x84\x80\x90\xc2\xb7\x2b\x03\x4b\xb4\xaa\x69\xb6\xe2\x16\x85\x85\x47\xac\xad\x8c\xfc\x76\xd9\xaf\xde\x28\xf9\xed\x87\x48\x8c\x9e\x7d\x91");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf9\x42\x4b\x3d\xd7\x9d\x55\x5e\x2f\x5f\x8a\x18\x59\x33\xf7\x7d\xb9\x58\x64\xf4\xb5\x97\xee\x80\x5d\x5b\x62\xfe\x03\x96\x67\x3e"[..]);

        ctx.update(b"\xcc\xf2\xcc\x0c\xd2\x23\xb8\x0e\xbb\xd4\x9d\xb9\xed\xe7\xa2\x28\x62\x81\x2d\xbc\x4b\x1a\xa6\xc9\xfa\xd0\x7f\x8e\xfe\x01\x71\x6b\x90\x18\x08\x4e\x2e\xa7\x61\x6f\x85\xb7\x33\x3d\x94\x5c\x0c\x97\x0f\x8c\xdd\x40\x01\x30\xb9\x8d\xb6\x7c\xda\x61\xd8\x7c\xe5\x5b\xd8\xf0\x6e\x4e\x8f\x5a\xbf\xf2\x26\x55\xd5\xd9\x33\x28\xac\xa4\x24\x9b\xbd\x76\xe1\x0a\xc4\xa9\xad\xb7\xc6\xc0\x4d\x64\x53\x9d\x6d\x22\x09\xf5\x3f\x64\x59\x00\x65\x57");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x06\xd1\xcd\x54\x4f\x95\x41\x60\xad\xe8\x3a\x10\x2e\xb9\x35\xf7\xf9\x48\x00\x95\x91\xb3\xea\xda\xf4\x76\x58\x68\xc6\x34\x50\xd8"[..]);

        ctx.update(b"\x5a\xa1\xb8\x5c\xa8\x4f\x24\xf4\x87\xf6\x9e\x79\x62\x68\x82\x65\x18\x15\x4b\xd6\xa2\x24\x58\x3e\xf6\xc3\xf1\x5c\x2b\xbb\x85\x6b\xdd\x47\x35\x99\xd3\x26\x97\xba\xdf\xef\xb3\x3e\x06\xa5\x52\x60\xe1\x8b\x79\xa1\x43\x39\x30\x75\x1d\xf0\x87\x3c\x9e\x2b\xdb\x73\x9e\x76\x56\x2c\x3c\x83\xb5\xbb\xc6\x7b\x4e\xb3\xda\x92\x51\x11\xcb\xde\x4e\xb6\x46\x69\xf7\x1b\x36\x85\x28\x40\xe4\x5f\x24\xe5\x63\xd7\x22\xfd\x2b\x8f\x08\x60\x04\x42\x76");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc7\x60\xde\x56\x12\xec\x9d\x04\x78\x48\x72\x7b\x26\x1c\x88\x76\xb6\xc3\x3b\x2c\x57\xee\x1a\xc9\x9d\xa0\x89\x0c\x00\x9b\x95\x40"[..]);

        ctx.update(b"\xb6\x13\xee\xbb\xad\x7a\x18\x60\x92\x8a\xe7\x32\x9a\x38\xcf\xe8\x6f\x11\x74\xa5\x79\x74\x0e\xcc\x5c\xff\x12\x16\xc2\x47\x0f\xe2\x5a\xcc\xd5\xcc\x21\x2c\xdd\x80\xe0\x96\xe5\xd1\xf5\x82\xdb\x05\x57\x9f\xfc\x73\xd5\x0d\x3d\x51\x9e\xcd\xea\xc7\x06\x76\x8d\x47\x87\x29\x11\x21\x06\xc6\x27\x52\x58\x60\xe9\x9d\x05\xef\xbd\xde\x4c\x45\x7d\x9a\x67\x44\xd5\xda\x10\x4d\xec\x3a\x43\xfb\x46\x3d\x46\x12\xe9\xe3\xe7\xb2\xbe\x7e\x05\xd1\x3b\xc1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x01\x24\x99\xf4\xe6\xe2\xaf\x3b\x53\x0d\x26\x4b\xce\x97\xe3\x44\x40\x5f\xfe\x59\xc0\x4b\x31\x43\x2e\x7f\x04\x46\xa6\xef\x33\xae"[..]);

        ctx.update(b"\x82\x3e\xf0\x9a\x0a\x65\x7d\x28\xff\xe8\x17\x3b\x6c\xf0\x4b\xf4\x79\x5a\x96\xb3\x9b\x19\x33\x96\x08\xd5\x32\x3c\x6e\xd8\xca\xcf\xe4\x2b\xf0\x41\x21\xfc\x28\xd5\xeb\x43\x9c\xe2\xd3\xa6\x6b\xd4\x20\x99\x43\x73\x09\xc9\x32\x68\x27\xbb\xb9\xfc\x29\xb0\x23\x24\xca\xcd\x14\xbb\x00\x78\x65\x0e\x76\x05\x2f\x9d\x3d\x92\x26\x20\xbf\x2f\x29\xc9\xf4\x43\x17\xc9\xd8\x2b\xe3\x15\x1c\x23\xeb\x6a\x1b\x6a\x85\x8e\x11\xa1\xdb\xda\x87\x74\xfa\xf5\x12");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x08\x04\x2c\x79\x50\x72\x68\xfd\xce\xbf\x12\x71\x96\x10\x21\x9d\x13\x64\x11\x7e\x66\x91\x6b\x9e\x0c\x1a\x2a\xed\xf5\xb4\xa8\xdf"[..]);

        ctx.update(b"\x96\x07\xcc\xa4\x58\x73\xad\xd1\x9a\x93\xdc\xcf\x3d\x0f\x79\x0e\x85\x6f\xf3\x0b\x84\xc8\x21\x1a\xd6\x9b\x8e\x62\x8f\xfa\x14\x29\x72\xec\xac\x52\x64\x13\x84\x23\x20\x8c\x52\x4c\x2b\x17\xe9\x25\x0b\x78\x09\x38\xb4\x1d\x7c\xff\x43\x00\x5e\xef\xee\xcf\xbd\xb5\x3b\x4b\x62\xbf\x71\xee\xed\xfe\x4c\xd0\x28\xea\xf5\xcb\x95\xc7\x31\xdd\x49\x27\xd9\xcf\xfd\xb1\x8a\x14\x63\x20\x9d\xf4\xb6\x8f\x5a\xea\x95\xf3\x68\x4a\x11\xe9\x88\x26\x05\xb2\x84\x73");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x73\xb4\x08\x6d\x69\x0a\xce\x69\x40\xc0\x19\x12\xac\xb3\xa5\x7b\xfc\x15\xc4\xa1\x6c\x40\xa9\x0b\x43\x29\xf1\xfa\x9f\x30\x85\xd2"[..]);

        ctx.update(b"\x57\x31\xf4\x67\xc5\xb9\x23\xc4\x3a\xf9\xc5\xfa\x84\x9a\xad\x21\xab\x8d\xd7\xdb\x1c\xa1\xa6\x87\x06\x55\x71\xb7\x05\xea\x3e\xe4\xfe\xbd\xcd\x61\x4a\xd4\xd9\x8e\x16\xb7\x9a\x4e\x09\x81\x8e\xbb\x28\x36\x79\x18\xf7\x57\xab\x06\xe1\xb4\x81\xfb\xda\x82\x2e\xf1\x43\xad\xbb\x5b\x0e\x70\x4d\x5d\x22\x22\xa7\x3c\x01\x53\xef\x14\xa8\x17\xb5\xc9\xb7\xa2\x31\x3f\xd1\x15\xcc\xce\x46\x98\xe3\xf0\xef\xa9\xc7\x3d\x5a\xb3\x08\x9a\x27\xe3\xf3\xad\xb2\x37\x59");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf5\x25\x3b\x5c\x69\xdb\x9c\x72\x4a\xeb\xf7\x62\xec\x51\xc2\x21\xf8\xa4\xd4\xe2\x17\x4a\x4b\x7f\x56\xe4\xd6\x9a\xa4\x4a\xdf\xa3"[..]);

        ctx.update(b"\x92\xb2\x3c\x0b\xc4\xd8\xd0\x7d\x22\xe2\x88\x12\x71\x0d\xff\x06\xcb\x9b\xbe\xce\xa2\xc9\x60\xac\x02\x00\xf4\x80\x16\x4f\xa2\xe1\xee\x19\x92\x6c\x7f\x0b\x09\x5c\xec\x51\xd5\x5c\x04\x0a\xec\x99\x0b\xf9\x50\x1a\xbd\x7d\x35\x54\x90\xc3\x66\xf9\x3a\x3a\xe5\x12\x73\x47\xd1\x4d\xfc\x3b\x8d\x98\xe0\x82\x1f\xee\xfa\x1c\xd6\x71\xb7\x52\x30\xba\x1d\xa1\xfa\x6d\x0c\xfb\xb9\x10\xc4\x2f\x49\x1d\xa8\xa5\xc4\x55\x42\x4e\xa6\x58\x86\xdb\x2e\x73\x5b\x2d\x07\xb9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x55\xa0\x59\x7f\x11\xff\x71\xc4\x26\x20\x17\x15\xbe\xb5\x85\xf2\x54\xbb\x31\xc1\xdb\xad\xe5\x33\xf0\x4e\x49\x9c\x33\x91\xff\x79"[..]);

        ctx.update(b"\xbc\xc5\x1e\xa0\xa6\x65\x64\xa1\x71\xda\xbf\xa2\x79\xe3\x84\xf4\xd9\xfd\xcb\xa3\x80\x28\x21\x57\x88\xee\x7d\x78\xc3\xa2\x76\x95\x96\xe6\xb2\x07\x0a\x6f\xa2\xd1\x20\x0d\x6c\xcb\x65\xe5\x29\x00\xc7\x01\x51\x54\xa7\x0c\x73\x6a\x2f\x56\x2a\xc4\xe6\x1f\x4c\x2c\x81\x11\x64\x53\xfd\x0e\x63\xb9\xea\x2c\x92\xcc\x0a\xfe\xcb\x54\x1a\x16\xe9\x0e\xf0\xc7\x7d\x97\xc6\x30\xc3\x8c\xd6\x75\xd4\xf0\x27\x50\x1c\xce\xa6\xc9\x0f\x1f\x78\x41\x18\xed\x8f\xb5\xd2\xb9\x7b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc1\x6c\xa7\x9c\x0a\xb4\x4f\x39\xda\x1c\x65\xe8\x94\x3a\xd2\xe9\x08\x88\xc3\xd8\x0b\x5e\x3b\x3b\x1b\xff\x59\x40\x8b\x59\xd6\xf3"[..]);

        ctx.update(b"\x21\x83\xbf\x54\x65\xb1\x83\x59\xa4\x51\xcf\x30\x30\xba\xb0\x7f\xa7\x61\xd1\x19\x26\x44\x4c\x7d\xba\x88\xc0\xa0\x17\x60\xa4\xec\x8d\x32\xdb\x88\x37\xf5\x88\xda\x76\xde\x23\x4a\x9d\x3b\xcd\x49\xee\x9f\xe0\x92\xc6\x3f\xcb\xb0\x14\x9c\x1d\xc1\x0f\x62\xdf\x79\x29\x90\x27\x6a\x3e\xda\xea\x82\x12\x4c\x12\xd5\xfa\x2f\xd0\x9c\xd9\x30\x34\x7e\x4e\xbf\xfb\x0f\x2c\xb6\xa8\x97\x2b\x3b\xd6\x14\xdd\xe7\xcc\x82\xe3\x28\x52\x59\x60\xe3\xc3\x58\xae\x5e\x64\x81\x50\x5e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7e\x5b\x4b\xd8\x08\xb9\x00\x73\x5c\x0f\xe6\x65\xda\x94\x37\xda\xca\xf8\x52\x49\xdf\x65\x3c\x6e\xe7\x2f\xb2\x96\xe8\xa9\xb8\xbb"[..]);

        ctx.update(b"\x95\x9b\xa9\x4b\x72\x8a\x8a\xdf\x53\x51\x35\x21\x02\x25\x58\x18\x3a\x63\x2e\x24\xad\x82\xa3\x90\x4b\x47\xb9\x66\xc2\x37\x2a\x87\x41\x52\x4f\xfa\x82\xfe\x40\x85\x07\x42\x3d\x7a\xa4\x34\x4d\xd4\x3d\xb5\x24\x82\xd1\x5b\x4a\x03\xb7\xaf\x7a\x33\x28\x99\x49\x65\xfe\x18\x08\xb1\xbb\x36\xea\x90\xcd\x3c\x6a\xa6\x11\x22\xe7\xde\x95\x49\x91\xbc\xa2\x6d\x89\xef\xfa\xd5\x44\x26\x5e\xcc\x31\x25\x53\x55\x9b\x18\x26\x1b\xd0\x27\x2e\xa7\x83\x4a\x51\xd5\x6f\x5f\xc9\x89\x3d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x85\xc2\xf1\x25\xe6\xd1\x1f\x83\x18\x07\x82\x0c\xcf\x0a\x53\x93\x27\x13\xdb\xce\xf3\xe2\x3b\x0e\x3c\x0a\xad\x8f\x89\x13\x9c\xc4"[..]);

        ctx.update(b"\x30\xd6\x2f\xf4\x26\x3a\x73\x1a\x20\x50\x69\xcf\x7c\x0a\x0d\xb0\xb5\x0d\x3d\xb1\x3f\x08\xc0\x0f\xb2\x45\x86\x86\x2b\x7e\x31\xc5\xef\xf1\xd5\x1f\x52\xb6\xfc\x83\xd8\x51\xee\xae\x99\x84\x2f\x9f\xea\x67\x48\x4b\x39\xba\xca\x93\x39\x79\x91\x73\x40\xb1\x46\x7c\xed\x93\x27\x84\x0d\xc6\x0b\x9c\xbf\x02\xa0\x42\xd7\xdb\xf8\xd8\x81\xda\x44\x55\x27\xd2\xdf\x3b\x94\x71\x8e\xc9\x59\x16\x06\x71\x6f\xe2\xae\x8c\x1f\x12\xa7\x4d\x1f\x8c\x0a\xf3\xfe\xb7\x70\x43\xc1\x90\xb8\x36");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x45\x43\x37\x8e\xd5\xa0\x54\x55\x78\x89\xb4\x63\x4d\x4a\xa7\x7b\xd3\x07\x26\xb7\x97\x23\x50\x10\xd6\xa4\x92\x0b\xf9\xcc\x3b\x40"[..]);

        ctx.update(b"\x37\x61\xa4\x86\x12\x49\xc4\xe0\x11\x5b\x99\x7c\x2c\x87\xe8\x5e\x5e\xa2\xbc\xaf\x44\xae\xa3\x1a\xe6\x5c\x19\x75\xd5\xcf\x45\xa4\x76\xa3\xe3\xba\x9d\x2e\x82\x62\x5d\x05\xef\x7e\x1f\xc2\xf1\x0d\xf2\x05\x36\x52\x22\xb9\x8e\x33\x0c\x99\x57\xdf\x30\xf2\x38\xb3\x88\xaa\x13\x5f\x19\x81\x34\x44\xc7\x97\xcf\x21\x6d\x2f\xa9\xef\x2d\xb2\x15\x09\x05\xf7\x22\x90\xa3\xc1\x13\x0e\x7b\x89\x0f\x59\x16\x7e\x1a\xbb\xad\xcb\x44\x4f\xf6\x18\x20\xae\x3e\x53\x78\xcb\xb8\xb0\x3f\x1c\x15");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x08\x81\x67\xf1\x2e\x82\xa4\x56\x98\xca\xd2\x5b\x40\x5d\xa5\xe2\x4a\x0f\x45\x15\xb2\x8b\x91\x5b\xfe\x79\x9d\x09\x90\x72\x35\x98"[..]);

        ctx.update(b"\x6e\x26\xe2\xb4\x1b\x80\xf2\x6b\x87\xcb\xa9\xbf\x5e\x11\xb7\x34\x61\x1e\x89\x11\x0c\x92\x10\x15\x76\x34\xc5\x66\x7a\x89\xea\xca\x14\xa6\x12\x19\x0d\x42\xe3\xf3\xdb\x0f\x1b\xae\xc3\x4b\xd1\x92\x40\x91\xfe\x49\xa8\xf6\x3b\x91\x0d\x51\xcc\x34\x61\xda\xde\x3a\xde\xf7\x0d\x5d\xa5\x52\x29\xbd\xa6\x2f\x3a\xd5\xbf\xfd\xc1\xae\x28\x2a\x41\x52\x0e\x3c\xfe\x84\x7e\x1e\x05\xae\x85\xb7\x6c\x1b\x96\xc1\xdb\x66\x2b\xd9\x4c\x33\xaf\xd2\x0e\x1c\x9c\x4f\x10\xef\xe1\x91\x5c\x16\x86\x8d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xad\x54\x90\x04\xc7\x5f\xa7\x81\x8d\xfa\x6e\x90\x60\x62\xa8\x50\x58\x9f\x7a\x8d\x17\xa1\xc4\x11\xfc\x95\x51\x68\x21\x5e\x45\xcc"[..]);

        ctx.update(b"\xfe\x8b\x3d\xca\x28\x30\x22\xda\xf7\x02\x2a\x88\xdf\xb8\xff\x61\x84\x75\x09\xc4\x1e\x9c\xec\x4f\x22\xe6\x52\x97\x1f\xb2\xea\x1c\xe5\x9c\xb1\x7c\xed\x50\x70\xe6\xf5\xa1\x85\xd2\xeb\xd5\xe7\xef\x0c\x43\x65\x5d\x4d\x21\x2e\x1d\xfa\x97\x96\xc6\x9d\x17\x17\x79\x21\xf1\x96\x16\x37\x22\xe8\xb0\xf9\x6d\x95\x3f\xcd\xae\xdc\x95\x8d\x6a\xf7\x94\xe5\xaa\xa0\x9d\xfc\x0a\xd3\x4b\x56\x0d\xd6\x62\xf9\xb3\x77\x02\xce\x4e\x95\x21\x4b\x91\xe1\x38\x4f\xea\xe6\x3b\x4d\x12\x47\xcf\xad\xe2\x48");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x37\xa8\x9a\x63\xb4\xa3\x0e\x6a\x21\x54\x9a\x24\x5a\x26\x33\xee\x5f\xdb\x61\x8e\x67\x49\xf6\x05\x22\xf5\xad\x03\xa3\x4d\x44\x65"[..]);

        ctx.update(b"\x04\xc4\x8b\x4c\x37\xa0\xda\xfe\x8a\xb2\xc5\x17\xee\xb4\x72\x38\x56\x61\x00\x13\x76\x29\x6a\xc5\xca\x2a\xaf\x67\xdb\x5c\x30\xdf\xbe\x0b\x59\xfc\xdd\x8b\x83\xc3\xfa\x27\x19\x03\x70\x5a\xfa\x30\xbc\x42\x1f\x6d\x3d\x51\x7e\xeb\x38\x46\x2e\x72\x1f\x5b\xcb\x3d\x13\xc0\x1c\x0b\xa7\x49\xf3\x73\xe7\x64\x55\x78\x99\x4d\xdb\x5d\xc2\x85\x58\xbb\x6c\x80\xbc\xd0\xa7\x9d\x39\x1a\xce\x5c\xa4\x24\x0b\xcc\x1d\xa1\x1d\x42\xf2\xb7\x59\x50\xc1\x9d\xef\xf5\xdd\x68\x6f\x12\xf3\x22\x8c\x1d\xee\x1d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x29\xa6\xb0\x43\x53\xea\x82\xf3\xf2\xa2\x85\x03\xa1\x25\x23\x3f\x8a\xfc\x5b\x60\x19\x21\x63\x7a\x0a\x24\xb8\x34\x36\xab\xf9\xb5"[..]);

        ctx.update(b"\xbc\x43\xae\x59\xc1\xbe\xd7\x4f\xd4\xeb\x61\xbe\x97\xb0\x27\x6e\xc5\xad\xda\x96\xc6\x7c\x3b\x46\x80\x37\x8e\x02\x6f\x3b\x4b\xaf\xf8\x2c\x16\xb4\xfa\x3a\xf1\x59\xa7\xa2\xb1\x72\x92\x77\xae\x42\x08\xa2\xbf\x9a\xa4\x28\x66\x41\x1e\xab\x47\x98\x64\x34\xdf\x7e\xec\x2f\x72\x72\x48\xb7\x45\xf7\xfb\x14\x0e\x3d\x8b\x82\xa9\xfd\xd7\x34\x0c\xa5\xab\x99\x7d\xb5\x92\xde\x03\x69\xeb\xb4\x1a\x98\x3f\x53\xaa\xdc\xd2\xbf\x3a\xf3\x78\x4a\x61\x65\xaa\x5a\x7e\x67\xd9\x6c\xa9\x00\x02\x59\xea\x6a\x83");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6e\x12\x44\x58\x71\x79\xef\xf2\x88\x81\xbb\x80\xb0\x2d\x19\x39\x08\xa4\xc4\xd0\x14\xb4\xda\x5b\x7e\x93\xc0\x9e\x2b\x29\x64\xf5"[..]);

        ctx.update(b"\xdd\xad\x78\x97\x28\x58\x10\x44\xff\x76\x0f\x79\x09\x9e\x01\xa2\xcf\xc9\x4b\xa2\x7a\x07\x0e\x22\xde\x71\x4e\x28\xb8\x5f\x54\xeb\x34\xdb\x50\xe6\xb3\x55\x43\x2b\xca\x98\x1c\xfa\x8e\xfc\x21\x92\x0f\xdd\x91\x47\x22\x4d\x88\xd5\x64\x06\x22\xf4\x7b\xa2\xe3\x2c\x9f\x0f\x57\xeb\x5a\x3d\xf5\xd5\x8b\x19\x49\x5b\xb1\xbf\x53\x8f\x66\xeb\x2a\x36\x2a\xb1\xe4\x1f\x62\x6f\xba\xb4\xa0\xff\x68\xcf\x25\x31\xd3\xcf\xa5\x78\xee\xea\x4a\xe3\xde\xc5\xfb\x4f\x5d\x41\xdd\xdc\x5c\x98\x1a\x9d\xec\xaa\xe2\x94");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6e\x3f\xf9\x6f\x4e\x4a\x72\x2c\x91\x23\x92\xb3\xb5\xfa\x20\xae\x76\x2c\xce\xf2\x44\xc0\x11\x26\x87\x20\x96\x21\x84\x38\x3a\x66"[..]);

        ctx.update(b"\x8c\x72\x14\xe1\x4e\xb6\xe0\xee\x14\xde\x3c\xa0\xed\xcb\x51\x0a\xec\x50\x79\xb3\xd1\x43\xff\x48\xe5\xd3\x26\x8d\x26\x1b\x6c\x99\xa4\xf8\x0f\x43\x2b\x18\xd1\xd8\x22\x54\xf8\x0f\xb3\x40\x9a\x3e\x2d\x00\x17\xcd\xa6\x5d\xe7\xeb\xe4\x18\x96\x81\x7e\x7d\x0c\x64\x3e\x3c\x12\xcf\xca\x75\x12\x10\x83\x12\x1e\x81\xae\x59\x34\xba\xd2\xc1\xc1\x05\x60\x05\x63\x13\x99\x78\xf3\xed\xad\x98\xc3\x2c\x20\xe5\x24\x6d\x72\xf4\x2c\x00\xff\xbf\xb6\xe0\x24\x3f\xa9\xe8\xa3\x3d\x7e\x43\x49\x65\xfc\x25\xce\x9c\x46");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb4\xb4\xe5\xed\xda\x4f\xb4\xf7\xc0\xe7\x34\xcb\xcc\xa3\xc7\x8d\xbd\x6f\x2c\x9a\x99\x7a\xe8\x5a\x63\x94\x10\x8c\xbc\x5c\x56\x7d"[..]);

        ctx.update(b"\x3b\x76\x6f\xbb\xa3\x33\x99\x59\xb1\x94\x16\x3c\x36\xb4\xeb\x65\x05\xac\x17\xe7\xb2\x4e\x75\x3e\x99\x16\xe3\xe1\x22\x5d\xa4\x1b\x44\xd0\x03\x16\x48\x21\x39\x39\xd0\xfc\x91\xb3\x17\x3d\xca\x58\x7a\x60\xbd\xad\x38\x38\x40\x3e\x76\xe4\x49\x38\x86\x00\x57\x33\x68\x18\x0d\x50\x91\x62\x20\x84\xec\xac\x74\x33\x1f\x93\x82\x01\x0f\xf6\xfd\xe5\xa6\xaa\xb5\x7f\xd4\x6f\xbb\xab\xd3\x23\xc6\x56\xb0\x36\xd5\x90\x07\x5e\x16\xe3\x0e\x8c\x07\xe3\xd4\x1e\xdb\x07\xcb\x56\x1f\xb7\x85\x0b\x9d\xa6\xbb\x9e\x81\x62");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb4\x50\xe0\x83\xd9\xf3\xf5\x82\x84\xef\x80\x02\x3e\x3b\xac\xb1\x82\x6e\x56\x0a\x22\xf6\xe6\x62\x5d\x14\xad\x13\x48\xdf\x96\xb1"[..]);

        ctx.update(b"\x6e\xd8\xa9\xf9\xf2\xba\xf5\x7e\x65\x9a\xed\x12\x78\x33\xed\xee\x90\x25\xa8\x48\x1c\xf6\x1d\x77\x25\x56\xc7\x58\x26\xa7\x14\xf0\x1f\xbc\x19\x53\xe5\xe1\x84\xc4\xbc\x1c\xd2\x1d\x27\xb3\xc2\x72\xff\xee\x09\xf1\xc3\x1e\x47\x0c\x2f\xde\x3a\xdb\x10\x99\x31\xbb\x56\x4d\xc4\x3c\x6b\x8e\xc3\x3a\xc3\xcd\x6c\x49\xb5\xda\xe1\x57\x89\xa8\xd9\x8c\xd2\x51\x95\x75\xfe\x59\x16\x62\x2b\x37\x8d\x0c\x3c\x11\x13\x7b\x2c\xc8\x9a\x27\xa3\x34\x23\x5e\xc7\xb1\x32\xe5\x55\xda\x7f\xe2\xf0\x94\xef\xf9\xb9\x4f\xc2\xb6\x1f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x17\x31\x2a\x26\xef\x9b\x41\xff\xeb\xe3\x1e\xe7\xee\x51\x16\x27\x04\xa2\xf0\x5a\x59\x05\xff\x8b\x69\x2f\xd6\x03\x94\x2e\x96\x68"[..]);

        ctx.update(b"\xdc\xb4\x58\x19\x3c\x4b\xa8\x2a\x45\xbf\xdf\x02\x80\x00\x84\x1e\x4a\x8c\x67\x1f\x8e\x2b\x84\x1d\xd4\xdc\x2a\x3d\x7b\x0f\x5d\x86\x42\x14\x78\xf0\x0e\xd4\x17\xbe\xe7\xbf\x33\x0e\x43\x05\x8f\xaf\x87\x09\x53\x32\xcb\x13\xf7\x44\xdd\x44\x63\x3b\x89\xd0\x55\xee\x18\x67\x02\xca\x17\xf4\x00\x22\x5a\x8a\x63\xc7\x40\x9b\x27\xd9\x37\x6c\xce\xdc\x83\xc5\x49\x0e\x3c\xed\x1c\xf8\x96\xd6\xbe\x44\x66\x94\x3e\xce\xe1\x13\x2c\xfe\xa6\x0a\xb7\x54\xff\x2b\x13\x44\x63\x9d\x1e\x45\x40\xab\x42\x25\xf8\xbe\x9d\x9f\x29\x88");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x93\x92\xce\x14\xd3\x44\xde\x05\xe4\xf8\x7f\x63\x4b\x45\x70\x79\x34\x3e\x27\x0f\xff\x5b\x5c\x7b\x73\x1e\x9f\xef\x13\x71\x15\xdd"[..]);

        ctx.update(b"\x3e\x3a\x52\xd3\x26\x1e\x11\x94\x24\x97\x86\xd6\xc0\xe1\x8d\x52\xd9\x2f\x1c\x76\x39\xf0\x79\xc2\x6c\x51\xaa\x72\xd1\x03\x2e\x5d\xf1\x3e\xea\x1d\x10\x06\x66\x70\x02\xad\x39\xde\x40\x99\xc2\x9c\x3b\x47\x19\xb1\xf0\x90\x45\x57\xbd\x2b\xb0\xa4\x73\x74\xd8\x69\xac\x6b\x46\x5b\x5f\x00\xc4\x70\xb1\x8e\xcb\x8c\x0e\xa5\x3b\x5d\x79\x0c\x4e\x83\x20\x06\xcf\xf5\x34\xd5\x87\xa0\xf7\x7d\xf9\x51\x17\xca\x4f\xd4\x3a\x94\x93\x5e\xda\x42\x22\x28\x53\x8d\x5e\x5d\x3a\x87\xa4\x36\xf1\xdb\x7e\x63\x78\x56\x19\xae\x86\xa6\xf9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb3\x4e\x72\xce\xfe\xfb\x63\xd6\xe3\x09\xbc\xfb\x4f\x0b\x1d\x35\x0f\x2c\x5c\x58\x2d\xe3\xb9\x3a\xd1\x37\xf9\x21\xa9\x2a\x7e\x79"[..]);

        ctx.update(b"\xbc\x81\x73\xc8\x78\xca\x60\xe9\xa0\xf8\x23\xf9\xa5\x89\xd4\xff\x84\x54\x7b\x38\x9b\x11\x7f\xb6\xbb\x1b\x61\x4e\x7e\x75\xa9\xb1\xdb\x0b\x21\xd9\xf7\x3b\x42\xa7\x3e\x94\xec\xca\xb3\xde\x5a\xe2\x84\x5a\x54\xe5\xe2\x4b\xa6\xc2\x0f\xb4\xd2\x45\xb9\x64\x02\x3b\x86\x30\x40\xd6\xf0\x80\xe9\x53\x53\x0d\x5f\xd9\x44\xe8\xff\xa5\x25\xbf\x53\x64\xf6\x5c\x88\xe0\x6e\x6e\x22\xdf\x4b\x8c\xee\x48\xe6\x77\x38\x88\x0a\x9f\x3f\x34\x06\xe9\xe6\xf0\x01\xb0\xac\x8f\x8e\x0a\xde\x7c\x81\x4c\x0c\x58\x00\xd0\xb9\xe4\xdd\xf5\x56\x22");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf6\x91\xd0\x1e\xe9\xab\x67\x5f\x38\x72\x31\x3b\x77\xe6\xa4\x54\x3c\x71\xe3\xe8\x9a\xa9\x4c\x48\xf9\x1d\x6e\xe7\xfa\x1a\xb4\xfb"[..]);
    }
}
