//**************************************************************************************************
//
//     OSSIM Open Source Geospatial Data Processing Library
//     See top level LICENSE.txt file for license information
//
//**************************************************************************************************

#ifndef ossimToolClient_HEADER
#define ossimToolClient_HEADER 1

#include <ossim/base/ossimConstants.h>
#include <ossim/base/ossimFilename.h>

/**
 * Utility class providing an interface to ossimTool-derived functionality via TCP sockets
 * Results are returned either as streamed text (for non-image responses such as image info) or
 * streamed binary file representing imagery or vector products. Code interfacing to this class
 * should know the commands available (or execute the command "help" and view the text response).
 */
class OSSIM_DLL ossimToolClient
{
public:

   ossimToolClient();

   /**
    * Closes any open connections.
    */
   ~ossimToolClient();

   /**
    * Accepts hostname and portname as a string. Returns socket file descriptor or -1 if error.
    */
   int connectToServer(char* hostname, char* portname=NULL);

   /**
    * Sets the local destination directory for file product. The product file name used
    * will match the server-side's file name. This method must be called prior to execute() for
    * operations generating file products, otherwise the products are written to the client's temp
    * directory. If the path directory specified does not exist, it will be created.
    * Returns true if successful.
    */
   bool setProductFilePath(const char* filepath);

    /**
    * Returns the filename (including full client-side path) of the local product file waiting
    * after execute(), or NULL if none.
    */
   const char* getProductFilePath() const { return m_prodFilePath; }

   /**
    * Executes OSSIM tool command with options. Examples:
    *
    *    "info -p <server-side-filename>" -- Fetches projection info for server side file specified
    *    "shoreline -i <server/path/band3.tif>, <server/path/band6.tif> <server/path/output.json>"
    *
    * Returns TRUE if product generated and received. setProductFilePath() must be called prior to
    * execute() for operations generating file products, otherwise the products are written to the
    * client's temp directory.
    */
   bool execute(const char* command_spec);

   /**
    * Returns the server's text response generated by the execute() call, or NULL if no response
    * available. Some operations generate both text and file products.
    */
   const char* getTextResponse() const { return m_textResponse.chars(); }

   /**
    * Closes connection to server and ready to connect again.
    */
   bool disconnect();

protected:

   void error(const char *msg);

   bool receiveText();
   bool receiveFile();
   bool acknowledgeRcv();

   int m_svrsockfd;
   char* m_buffer;
   ossimFilename m_prodFilePath;
   ossimString m_textResponse;
};

#endif
