## Copyright (C) 2024 David Legland
## All rights reserved.
## 
## Redistribution and use in source and binary forms, with or without
## modification, are permitted provided that the following conditions are met:
## 
##     1 Redistributions of source code must retain the above copyright notice,
##       this list of conditions and the following disclaimer.
##     2 Redistributions in binary form must reproduce the above copyright
##       notice, this list of conditions and the following disclaimer in the
##       documentation and/or other materials provided with the distribution.
## 
## THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ''AS IS''
## AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
## IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
## ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
## ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
## DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
## SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
## CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
## OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
## OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
## 
## The views and conclusions contained in the software and documentation are
## those of the authors and should not be interpreted as representing official
## policies, either expressed or implied, of the copyright holders.

function edge2 = clipEdge(edge, box)
%CLIPEDGE Clip an edge with a rectangular box.
%
%   EDGE2 = clipEdge(EDGE, BOX);
%   EDGE: [x1 y1 x2 y2],
%   BOX : [xmin xmax ymin ymax], or [xmin xmax ; ymin ymax].
%   return :
%   EDGE2 = [xc1 yc1 xc2 yc2];
%
%   If clipping is null, return [0 0 0 0];
%
%   if EDGE is a N-by-4 array, return an N-by-4 array, corresponding to
%   each clipped edge.
%
%   See also 
%   edges2d, boxes2d, clipLine
%

% ------
% Author: David Legland 
% E-mail: david.legland@inrae.fr
% Created: 2005-05-14
% Copyright 2005-2023 INRA - Cepia Software Platform

% process data input
if size(box, 1) == 2
    box = box';
end

% get limits of window
xmin = box(1);
xmax = box(2);
ymin = box(3);
ymax = box(4);


% convert window limits into lines
lineX0 = [xmin ymin xmax-xmin 0];
lineX1 = [xmin ymax xmax-xmin 0];
lineY0 = [xmin ymin 0 ymax-ymin];
lineY1 = [xmax ymin 0 ymax-ymin];


% compute outcodes of each vertex
p11 = edge(:,1) < xmin; p21 = edge(:,3) < xmin;
p12 = edge(:,1) > xmax; p22 = edge(:,3) > xmax;
p13 = edge(:,2) < ymin; p23 = edge(:,4) < ymin;
p14 = edge(:,2) > ymax; p24 = edge(:,4) > ymax;
out1 = [p11 p12 p13 p14];
out2 = [p21 p22 p23 p24];

% detect edges totally inside window -> no clip.
inside = sum(out1 | out2, 2) == 0;

% detect edges totally outside window
outside = sum(out1 & out2, 2) > 0;

% select edges not totally outside, and process separately edges totally
% inside window
ind = find(~(inside | outside));

% allocate memroty for all clipped edges
edge2 = zeros(size(edge));

% copy result of edges totally inside clipping box
edge2(inside, :) = edge(inside, :);

% iterate over edges
for i = 1:length(ind)
    % current edge
    iedge = edge(ind(i), :);
        
    % compute intersection points with each line of bounding window
    px0 = intersectLineEdge(lineX0, iedge);
    px1 = intersectLineEdge(lineX1, iedge);
    py0 = intersectLineEdge(lineY0, iedge);
    py1 = intersectLineEdge(lineY1, iedge);
         
    % create array of points
    points  = [px0; px1; py0; py1; iedge(1:2); iedge(3:4)];
    
    % remove infinite points (edges parallel to box edges)
	points  = points(all(isfinite(points), 2), :);
    
    % sort points by x then y
    points = sortrows(points);
    
    % get center positions between consecutive points
    centers = (points(2:end,:) + points(1:end-1,:))/2;
    
    % find the centers (if any) inside window
    inside = find(  centers(:,1) >= xmin & centers(:,2) >= ymin & ...
                    centers(:,1) <= xmax & centers(:,2) <= ymax);

    % if multiple segments are inside box, which can happen due to finite
    % resolution, only take the longest segment
    if length(inside) > 1
        % compute delta vectors of the segments
        dv = points(inside+1,:) - points(inside,:); 
        % compute lengths of segments
        len = hypot(dv(:,1), dv(:,2));
        % find index of longest segment
        [a, I] = max(len); %#ok<ASGLU>
        inside = inside(I);
    end
    
    % if one of the center points is inside box, then the according edge
    % segment is indide box
    if length(inside) == 1
         % restore same direction of edge
        if iedge(1) > iedge(3) || (iedge(1) == iedge(3) && iedge(2) > iedge(4))
            edge2(ind(i), :) = [points(inside+1,:) points(inside,:)];
        else
            edge2(ind(i), :) = [points(inside,:) points(inside+1,:)];
        end
    end
    
end % end of loop over edges

