#!/bin/sh
# SPDX-License-Identifier: LGPL-2.1+

set -eu
[ -n "${DEBUG:-}" ] && set -x

[ $(id -u) -eq 0 ]

# Run lxcfs testsuite
export LXCFSDIR=$(mktemp -d)
pidfile=$(mktemp)
export LXCFSPID=-1

cmdline=$(realpath $0)
dirname=$(dirname ${cmdline})
topdir=$(dirname ${dirname})

FAILED=1
UNSHARE=1
cleanup() {
	echo "=> Cleaning up"
	set +e
	if [ $LXCFSPID -ne -1 ]; then
		kill -9 $LXCFSPID
	fi
	if [ ${LXCFSDIR} != "/var/lib/lxcfs" ]; then
		umount -l ${LXCFSDIR}
		rmdir ${LXCFSDIR}
	fi
	rm -f ${pidfile}
	if [ ${FAILED} -eq 1 ]; then
		echo "=> FAILED at $TESTCASE"
		exit 1
	fi
	echo "=> PASSED"
	exit 0
}

TESTCASE="setup"
lxcfs=${topdir}/src/lxcfs

if [ -x ${lxcfs} ]; then
	if [ -n "${LD_LIBRARY_PATH:-}" ]; then
		export LD_LIBRARY_PATH="${topdir}/src/.libs/:${LD_LIBRARY_PATH}"
	else
		export LD_LIBRARY_PATH="${topdir}/src/.libs/"
	fi
	echo "=> Spawning ${lxcfs} ${LXCFSDIR}"
	${lxcfs} -p ${pidfile} ${LXCFSDIR} &
	LXCFSPID=$!
else
	UNSHARE=0
	LXCFSPID=$(cat "/run/lxcfs.pid")
	echo "=> Re-using host lxcfs"
	rmdir $LXCFSDIR
	export LXCFSDIR=/var/lib/lxcfs
fi

trap cleanup EXIT HUP INT TERM

count=1
while ! mountpoint -q $LXCFSDIR; do
	sleep 1s
	if [ $count -gt 5 ]; then
		echo "lxcfs failed to start"
		false
	fi
	count=$((count+1))
done

RUNTEST() {
	echo ""
	echo "=> Running ${TESTCASE}"

	if [ "${UNSHARE:-1}" != "0" ]; then
		unshare -fmp --mount-proc $*
	else
		$*
	fi
}

TESTCASE="Stress readdir"
RUNTEST ${dirname}/test_readdir
TESTCASE="test_proc"
RUNTEST ${dirname}/test_proc
TESTCASE="test_cgroup"
RUNTEST ${dirname}/test_cgroup
TESTCASE="test_read_proc.sh"
RUNTEST ${dirname}/test_read_proc.sh
TESTCASE="cpusetrange"
RUNTEST ${dirname}/cpusetrange
TESTCASE="meminfo hierarchy"
RUNTEST ${dirname}/test_meminfo_hierarchy.sh
TESTCASE="liblxcfs reloading"
UNSHARE=0 RUNTEST ${dirname}/test_reload.sh
TESTCASE="SIGUSR2 virtualization mode switching"
echo "==> Switching to non-virtualization mode"
kill -USR2 $LXCFSPID
RUNTEST ${dirname}/test_sigusr2.sh
echo "==> Switching to virtualization mode"
kill -USR2 $LXCFSPID

# Check for any defunct processes - children we didn't reap
n=`ps -ef | grep lxcfs | grep defunct | wc -l`
[ $n = 0 ]

FAILED=0
