/*
** nsw_passwd.c           Passwd cover routines for the Name Service Switch
**
** Copyright (c) 1993 Signum Support AB, Sweden
**
** This file is part of the NYS Library.
**
** The NYS Library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS Library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
** 
** You should have received a copy of the GNU Library General Public
** License along with the NYS Library; see the file COPYING.LIB.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
**
** Author: Peter Eriksson <pen@signum.se>
*/

#include "config.h"

#include <stdio.h>
#include <errno.h>
#include "nsswitch.h"
#include "nsw_passwd.h"

#ifdef ENABLE_NIS
#include "../../nis/src/nis_passwd.h"
#endif

#ifdef ENABLE_YP
#include "../../yp/src/yp_passwd.h"
#endif


#ifndef FINALOUT
#  define _setpwent setpwent
#  define _endpwent endpwent
#  define _getpwent getpwent
#  define _getpwuid getpwuid
#  define _getpwnam getpwnam
#else
#  define nsw_setpwent setpwent
#  define nsw_endpwent endpwent
#  define nsw_getpwent getpwent
#  define nsw_getpwuid getpwuid
#  define nsw_getpwnam getpwnam
#endif


static struct nsw *nswp = NULL;
static int setptr = 0;


void nsw_setpwent(void)
{
    int i;


    if (nswp == NULL)
    {
	nswp = getnswbyname("passwd");
	setptr = 0;
    }
    
    if (nswp == NULL)
    {
	_setpwent();
	return;
    }

    for (i = 0; i < nswp->orderc; i++)
	switch (nswp->orderl[i])
	{
	  case NSWO_FILES:
	    _setpwent();
	    break;

#ifdef ENABLE_NIS
	  case NSWO_NISPLUS:
	    nis_setpwent();
	    break;
#endif
#ifdef ENABLE_YP	    
	  case NSWO_NIS:
	    yp_setpwent();
	    break;
#endif
	}
}


void nsw_endpwent(void)
{
    int i;


    if (nswp == NULL)
    {
	nswp = getnswbyname("passwd");
	setptr = 0;
    }
    
    if (nswp == NULL)
    {
	_endpwent();
	return;
    }

    for (i = 0; i < nswp->orderc; i++)
	switch (nswp->orderl[i])
	{
	  case NSWO_FILES:
	    _endpwent();
	    break;
	    
#ifdef ENABLE_NIS
	  case NSWO_NISPLUS:
	    nis_endpwent();
	    break;
#endif
#ifdef ENABLE_YP
	  case NSWO_NIS:
	    yp_endpwent();
	    break;
#endif
	}
}


struct passwd *nsw_getpwent(void)
{
    struct passwd *pp = NULL;
    int i;

    
    if (nswp == NULL)
    {
	nswp = getnswbyname("passwd");
	setptr = 0;
    }

    if (nswp == NULL)
	return _getpwent();

    errno = 0;
    do
    {
	switch (nswp->orderl[setptr])
	{
	  case NSWO_RETURN:
	    if (errno == 0)
		return NULL;
	    break;
	    
	  case NSWO_FILES:
	    pp = _getpwent();
	    break;

#ifdef ENABLE_NIS
	  case NSWO_NISPLUS:
	    pp = nis_getpwent();
	    break;
#endif
#ifdef ENABLE_YP
	  case NSWO_NIS:
	    pp = yp_getpwent();
	    break;
#endif
	}
    } while (pp == NULL && setptr++ < nswp->orderc);

    return pp;
}


struct passwd *nsw_getpwuid(uid_t uid)
{
    struct passwd *pp = NULL;
    int i;


    if (nswp == NULL)
    {
	nswp = getnswbyname("passwd");
	setptr = 0;
    }
    
    if (nswp == NULL)
	return _getpwuid(uid);
    
    errno = 0;
    for (i = 0; i < nswp->orderc && pp == NULL; i++)
	switch (nswp->orderl[i])
	{
	  case NSWO_RETURN:
	    if (errno == 0)
		return NULL;
	    break;
	    
	  case NSWO_FILES:
	    pp = getpwuid(uid);
	    break;
	    
#ifdef ENABLE_NIS
	  case NSWO_NISPLUS:
	    pp = nis_getpwuid(uid);
	    break;
#endif
#ifdef ENABLE_YP
	  case NSWO_NIS:
	    pp = yp_getpwuid(uid);
	    break;
#endif
	}

    return pp;
}


struct passwd *nsw_getpwnam(char *name)
{
    struct passwd *pp = NULL;
    int i;


    if (nswp == NULL)
    {
	nswp = getnswbyname("passwd");
	setptr = 0;
    }
    
    if (nswp == NULL)
	return _getpwnam(name);
    
    errno = 0;
    for (i = 0; i < nswp->orderc && pp == NULL; i++)
	switch (nswp->orderl[i])
	{
	  case NSWO_RETURN:
	    if (errno == 0)
		return NULL;
	    break;
	    
	  case NSWO_FILES:
	    pp = getpwnam(name);
	    break;

#ifdef ENABLE_NIS
	  case NSWO_NISPLUS:
	    pp = nis_getpwnam(name);
	    break;
#endif
	    
#ifdef ENABLE_YP
	  case NSWO_NIS:
	    pp = yp_getpwnam(name);
	    break;
#endif
	}

    return pp;
}
