#############################################################################
#                                                                           #
# AWK SCRIPT           M.J.E. Mol    (C) 1994, 1998                         #
#                      marcel@mesa.nl                                       #
#                                                                           #
# Date:     16apr1998                                                       #
# File:     cpu_week.awk                                                    #
# Version:  1.3                                                             #
#                                                                           #
# Description                                                               #
#    Take output from cpu_extract.awk and generate a gnuplot script that    #
#    creates postcript graphs of cpubusy per interval, covering one week.   #
#    Sample records are ignored.                                            #
#                                                                           #
# Version Date      Name             Comments                               #
# ------- --------- --------------   -------------------------------------- #
#  1.0    02sep94   M.J.E. Mol       Initial version                        #
#  1.1    05sep94   M.J.E. Mol       Map daynum 0 to daynum 7               #
#                                    Reset minday/maxday after each plot    #
#  1.2    13sep94   M.J.E. Mol       Undo daynum remap, use %U on logfiles. #
#  1.3    16apr1998 M.J.E. Mol       Rename script. Allow comments in input #
#                                                                           #
#############################################################################
#############################################################################
#
# Expected input:
#
#     <hostname> <weeknumber_of_year> <daynumber_of_month><monthname><year>
#         <daynumber_of_week> <interval_start_time> <interval_length>
#         <cpu_system> <cpu_iowait> <cpu_user> <cpu_idle> 
#
# Outputs:
#
#    A file specified by the 'gpf' variable containing a gnuplot program.
#    Datafiles per host and per week and used by the gnuplot program 
#    containing records of the following layout: 
#
#	<hostname> <daynumber_of_week>.<frac>
#              <cpu_sys> <cpu_sys+user> <cpu_sys+user+wait>
#
#    where <frac> indicates a fraction of the day, e.g. 1.5 means monday noon.
#    <cpu...> are the values for the interval at that part of that day.
#
#############################################################################

BEGIN {
        firsttime = 1
        minday = 100
        maxday = 0
    
    printf "set terminal postscript\n" > gpf
    printf "set time\n" >> gpf
    printf "set grid\n" >> gpf
    printf "set xlabel \"Time\"\n" >> gpf
    printf "set ylabel \"Busy %%\"\n" >> gpf
    printf "set tics out\n" >> gpf
    printf "set xrange[0:7]\n" >> gpf
    printf "set yrange[0:100]\n" >> gpf
    printf "set xtics 0,1,7\n" >> gpf
    printf "set ytics 0,10,100\n" >> gpf
    printf "set data style lines\n" >> gpf

}

#
# Skip comments
#
/^#/ { next }

{
        if (host != $1 || week != $2) {
            if (firsttime)
                firsttime = 0
            else
	        dump_plot()
            datafile = dataf "." $1 "." $2
        }

	host = $1
        week = $2
        date = $3
        daynum = $4
        time = $5
        interval = $6

        if (daynum > maxday) {
            maxday = daynum
            maxdate = date
        }
        if (daynum < minday) {
            minday = daynum
            mindate = date
        }

	cpusys[daynum, time, host]  += $7
	cpuwait[daynum, time, host] += $8
	cpuuser[daynum, time, host] += $9
	cpuidle[daynum, time, host] += $10

}

END {
    dump_plot()
}

function dump_plot () {
    system("rm -f " datafile)   
    sortcomm = "sort +1n >>" datafile
    printf "set title \"CPU Busy Figures for %s in week %s - %s-%s\"\n",
           host, week, mindate, maxdate >> gpf

    printf "plot '%s' using 2:5 title \"CPU wait\",\\\n", datafile >> gpf
    printf "     '%s' using 2:4 title \"CPU user\",\\\n", datafile >> gpf
    printf "     '%s' using 2:3 title \"CPU sys\"\n", datafile >> gpf

    printf "#pause -1 \"bye\"\n" >> gpf

    for (combined in cpusys) {
        split(combined, sep, SUBSEP)
        dayn = sep[1]
        time = sep[2]
	host = sep[3]
        min  = substr(time,1,2) * 60 + substr(time,4,2) 
        daypart = dayn + min / (60*24)
        sys  = cpusys[combined]
        wait = cpuwait[combined]
        user = cpuuser[combined]
	printf "%s %f %f %f %f\n",
               host, daypart, sys, sys+user, sys+user+wait | sortcomm
        delete cpusys[combined]
        delete cpuwait[combined]
        delete cpuuser[combined]
        delete cpuidle[combined]
    }
    close (sortcomm)

    minday = 100
    maxday = 0
}
