/* xscreensaver, Copyright (c) 1992 Jamie Zawinski <jwz@lucid.com>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#include "version.h"

/*   ========================================================================
 *   First we wait until the keyboard and mouse become idle for the specified
 *   amount of time.  We do this by periodically checking with the XIdle
 *   server extension.
 *
 *   Then, we map a full screen black window.  
 *
 *   We place a __SWM_VROOT property on this window, so that newly-started
 *   clients will think that this window is a "virtual root" window.
 *
 *   If there is an existing "virtual root" window (one that already had
 *   an __SWM_VROOT property) then we remove that property from that window.
 *   Otherwise, clients would see that window (the real virtual root) instead
 *   of ours (the impostor.)
 *
 *   Then we pick a random program to run, and start it.  Two assumptions 
 *   are made about this program: that it has been specified with whatever
 *   command-line options are necessary to make it run on the root window;
 *   and that it has been compiled with vroot.h, so that it is able to find
 *   the root window when a virtual-root window manager (or this program) is
 *   running.
 *
 *   Then, we wait for keyboard or mouse events to be generated on the window.
 *   When they are, we kill the inferior process, unmap the window, and restore
 *   the __SWM_VROOT property to the real virtual root window if there was one.
 *
 *   While we are waiting, we also set up timers so that, after a certain 
 *   amount of time has passed, we can start a different screenhack.  We do
 *   this by killing the running child process with SIGTERM, and then starting
 *   a new one in the same way.
 *
 *   If there was a real virtual root, meaning that we removed the __SWM_VROOT
 *   property from it, meaning we must (absolutely must) restore it before we
 *   exit, then we set up signal handlers for most signals (SIGINT, SIGTERM,
 *   etc.) that do this.  Most Xlib and Xt routines are not reentrant, so it
 *   is not generally safe to call them from signal handlers; however, this
 *   program spends most of its time waiting, so the window of opportunity 
 *   when code could be called reentrantly is fairly small; and also, the worst
 *   that could happen is that the call would fail.  If we've gotten one of
 *   these signals, then we're on our way out anyway.  If we didn't restore the
 *   __SWM_VROOT property, that would be very bad, so it's worth a shot.  Note
 *   that this means that, if you're using a virtual-root window manager, you
 *   can really fuck up the world by killing this process with "kill -9".
 *
 *   This program accepts ClientMessages of type SCREENSAVER; these messages
 *   may contain the atom ACTIVATE or DEACTIVATE, meaning to turn the 
 *   screensaver on or off now, regardless of the idleness of the user,
 *   and a few other things.  The included "xscreensaver_command" program
 *   sends these messsages.
 *
 *   If we don't have the XIdle extension, then we do the XAutoLock trick:
 *   notice every window that gets created, and wait 30 seconds or so until
 *   its creating process has settled down, and then select KeyPress events on
 *   those windows which already select for KeyPress events.  It's important
 *   that we not select KeyPress on windows which don't select them, because
 *   that would interfere with event propagation.  This will break if any
 *   program changes its event mask to contain KeyRelease or PointerMotion
 *   more than 30 seconds after creating the window, but that's probably
 *   pretty rare.
 *   
 *   The reason that we can't select KeyPresses on windows that don't have
 *   them already is that, when dispatching a KeyPress event, X finds the
 *   lowest (leafmost) window in the hierarchy on which *any* client selects
 *   for KeyPress, and sends the event to that window.  This means that if a
 *   client had a window with subwindows, and expected to receive KeyPress
 *   events on the parent window instead of the subwindows, then that client
 *   would malfunction if some other client selected KeyPress events on the
 *   subwindows.  It is an incredible misdesign that one client can make
 *   another client malfunction in this way.
 *
 *   To detect mouse motion, we periodically wake up and poll the mouse
 *   position and button/modifier state, and notice when something has
 *   changed.  We make this check every five seconds by default, and since the
 *   screensaver timeout has a granularity of one minute, this makes the
 *   chance of a false positive very small.  We could detect mouse motion in
 *   the same way as keyboard activity, but that would suffer from the same
 *   "client changing event mask" problem that the KeyPress events hack does.
 *   I think polling is more reliable.
 *
 *   None of this crap happens if we're using the XIdle extension, so install
 *   it if the description above sounds just too flaky to live.  It is, but
 *   those are your choices.
 *
 *   Perhaps adding a locking capability to this would be useful.  I don't
 *   ever use screen lockers myself, but lots of other people do.
 *
 *   Debugging hints:
 *     - Have a second terminal handy.
 *     - Be careful where you set your breakpoints, you don't want this to
 *       stop under the debugger with the keyboard grabbed or the blackout
 *       window exposed.
 *     - you probably can't set breakpoints in functions that are called on
 *       the other side of a call to fork() -- if your clients are dying 
 *       with signal 5, Trace/BPT Trap, you're losing in this way.
 *     - If you aren't using XIdle, don't leave this stopped under the
 *       debugger for very long, or the X input buffer will get huge because
 *       of the keypress events it's selecting for.  This can make your X
 *       server wedge with "no more input buffers."
 *       
 *   ========================================================================
 */

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/Xos.h>
#ifndef ESRCH
#include <errno.h>
#endif

#include <sys/resource.h>	/* for setpriority() and PRIO_PROCESS */
#include <sys/wait.h>		/* for waitpid() and associated macros */
#include <signal.h>		/* for the signal names */

extern char **environ;		/* why isn't this in some header file? */

#if !defined(SIGCHLD) && defined(SIGCLD)
#define SIGCHLD SIGCLD
#endif

extern char *get_string_resource ();
extern Bool get_boolean_resource ();
extern int get_integer_resource ();

char *progname;
char *progclass;
XrmDatabase db;

static Display *dpy;
static Screen *screen;
static Window window;
static Cursor cursor;
static Colormap cmap, cmap2;
static Window real_vroot, real_vroot_value;
static XtAppContext app;
static Time timeout;
static Time cycle;
static Time pointer_timeout;
static Time notice_events_timeout;
static time_t initial_delay;
static time_t last_activity_time; /* for non-XIdle mode */
static int nice_inferior;
static int fade_seconds, fade_ticks;
static pid_t pid;
static XtIntervalId timer_id, check_pointer_timer_id, cycle_id;
static Atom XA_VROOT;
static Atom XA_SCREENSAVER, XA_SCREENSAVER_VERSION, XA_SCREENSAVER_ID;
static Atom XA_ACTIVATE, XA_DEACTIVATE, XA_CYCLE, XA_NEXT, XA_PREV, XA_EXIT;
static Boolean use_xidle;
static Boolean verbose_p;
static Boolean install_cmap_p;
static Boolean fade_p, unfade_p;
static unsigned long black_pixel;

static char **screenhacks;
static int screenhacks_count;
static int current_hack;
static int next_mode_p;

/* this must be `sh', not whatever $SHELL happens to be. */
static char *shell;

static void spawn_screenhack ();
static void kill_screenhack ();
static void save_real_vroot ();
static void restore_real_vroot ();
static void handle_signals ();
static void unblank_screen ();
static Boolean handle_clientmessage ();
static char *timestring ();
static char *current_hack_name ();

static void
store_vroot_property (win, value)
     Window win, value;
{
#if 0
  printf ("%s: storing XA_VROOT = 0x%x (%s) = 0x%x (%s)\n", progname, 
	  win,
	  (win == window ? "ScreenSaver" :
	   (win == real_vroot ? "VRoot" :
	    (win == real_vroot_value ? "Vroot_value" : "???"))),
	  value,
	  (value == window ? "ScreenSaver" :
	   (value == real_vroot ? "VRoot" :
	    (value == real_vroot_value ? "Vroot_value" : "???"))));
#endif
  XChangeProperty (dpy, win, XA_VROOT, XA_WINDOW, 32, PropModeReplace,
		   (unsigned char *) &value, 1);
}

static void
remove_vroot_property (win)
{
#if 0
  printf ("%s: removing XA_VROOT from 0x%x (%s)\n", progname, win, 
	  (win == window ? "ScreenSaver" :
	   (win == real_vroot ? "VRoot" :
	    (win == real_vroot_value ? "Vroot_value" : "???"))));
#endif
  XDeleteProperty (dpy, win, XA_VROOT);
}


static void
make_screensaver_window ()
{
  XClassHint class_hints;
  XSetWindowAttributes attrs;
  unsigned long attrmask;
  int width = WidthOfScreen (screen);
  int height = HeightOfScreen (screen);
  XColor black;
  Pixmap bit;
  char id [2048];

  attrmask = CWOverrideRedirect | CWEventMask | CWBackPixel | CWBackingStore;
  attrs.override_redirect = True;
  attrs.event_mask = (KeyPressMask | KeyReleaseMask |
		      ButtonPressMask | ButtonReleaseMask |
		      PointerMotionMask);
  attrs.background_pixel = BlackPixelOfScreen (screen);
  attrs.backing_store = NotUseful;
  window = XCreateWindow (dpy,
			  RootWindowOfScreen (screen),
			  0, 0, width, height, 0,
			  CopyFromParent, CopyFromParent,
			  DefaultVisualOfScreen (screen),
			  attrmask, &attrs);
  class_hints.res_name = progname;
  class_hints.res_class = progclass;
  XSetClassHint (dpy, window, &class_hints);
  XStoreName (dpy, window, "screensaver");
  XChangeProperty (dpy, window, XA_SCREENSAVER_VERSION, XA_STRING, 8,
		   PropModeReplace, (unsigned char *) screensaver_version,
		   strlen (screensaver_version));

  sprintf (id, "%d on host ", getpid ());
  if (! XmuGetHostname (id + strlen (id), sizeof (id) - strlen (id) - 1))
    strcat (id, "???");
  XChangeProperty (dpy, window, XA_SCREENSAVER_ID, XA_STRING, 8,
		   PropModeReplace, (unsigned char *) id, strlen (id));

  bit = XCreatePixmapFromBitmapData (dpy, window, "\000", 1, 1,
				     BlackPixelOfScreen (screen),
				     BlackPixelOfScreen (screen), 1);
  black.red = black.green = black.blue = 0;
  cursor = XCreatePixmapCursor (dpy, bit, bit, &black, &black, 0, 0);
  XFreePixmap (dpy, bit);
  if (install_cmap_p)
    {
      cmap = XCreateColormap (dpy, window, DefaultVisualOfScreen (screen),
			      AllocNone);
      if (! XAllocColor (dpy, cmap, &black)) abort ();
      black_pixel = black.pixel;
    }
  else
    {
      cmap = DefaultColormapOfScreen (screen);
      black_pixel = BlackPixelOfScreen (screen);
    }

  if (fade_p)
    {
      cmap2 = copy_colormap (dpy, cmap, 0);
      if (! cmap2)
	fade_p = unfade_p = 0;
    }
  else
    cmap2 = 0;
}

static void
idle_timer (junk1, junk2)
     void *junk1;
     XtPointer junk2;
{
  /* What an amazingly shitty design.  Not only does Xt execute timeout
     events from XtAppNextEvent() instead of from XtDispatchEvent(), but
     there is no way to tell Xt to block until there is an X event OR a
     timeout happens.  Once your timeout proc is called, XtAppNextEvent()
     still won't return until a "real" X event comes in.

     So this function pushes a stupid, gratuitous, unnecessary event back
     on the event queue to force XtAppNextEvent to return Right Fucking Now.
     When the code in sleep_until_idle() sees an event of type XAnyEvent,
     which the server never generates, it knows that a timeout has occurred.
   */
  XEvent fake_event;
  fake_event.type = 0;	/* XAnyEvent type, ignored. */
  fake_event.xany.display = dpy;
  fake_event.xany.window  = 0;
  XPutBackEvent (dpy, &fake_event);
}

static void
notice_events (window, top_p)
     Window window;
     Bool top_p;
{
  unsigned long events;
  Window root, parent, *kids;
  unsigned int nkids;
  if (!XQueryTree (dpy, window, &root, &parent, &kids, &nkids))
    return;
  if (window == root)
    top_p = False;
  if (parent == root || parent == None)
    events = KeyPressMask;
  else
    {
      XWindowAttributes attrs;
      XGetWindowAttributes (dpy, window, &attrs);
      events = ((attrs.all_event_masks | attrs.do_not_propagate_mask)
		& KeyPressMask);
    }
  /* Select for SubstructureNotify on all windows.
     Select for KeyPress on all windows that already have it selected.
     Do we need to select for ButtonRelease?  I don't think so.
   */
  XSelectInput (dpy, window, SubstructureNotifyMask | events);

  if (top_p && verbose_p && (events & KeyPressMask))
    {
      /* Only mention one window per tree (hack hack). */
      printf ("%s: selected KeyPress on 0x%X\n", progname, window);
      top_p = False;
    }

  if (kids)
    {
      while (nkids)
	notice_events (kids [--nkids], top_p);
      XFree ((char *) kids);
    }
}


static int
notice_events_ehandler (dpy, error)
     Display *dpy;
     XErrorEvent *error;
{
  /* When we notice a window being created, we spawn a timer that waits
     30 seconds or so, and then selects events on that window.  This error
     handler is used so that we can cope with the fact that the window
     may have been destroyed <30 seconds after it was created.
   */
  if (error->error_code == BadWindow ||
      error->error_code == BadDrawable)
    return 0;
  XmuPrintDefaultErrorMessage (dpy, error, stderr);
  exit (-1);
}

static void
notice_events_timer (closure, timer)
     XtPointer closure;
     void *timer;
{
  Window window = (Window) closure;
  int (*old_handler) ();
  old_handler = XSetErrorHandler (notice_events_ehandler);
  notice_events (window, True);
  XSync (dpy, False);
  XSetErrorHandler (old_handler);
}


/* When the screensaver is active, this timer will periodically change
   the running program.
 */
static void
cycle_timer (junk1, junk2)
     void *junk1;
     XtPointer junk2;
{
  if (verbose_p)
    printf ("%s: changing graphics hacks.\n", progname);
  kill_screenhack ();
  spawn_screenhack ();
  cycle_id = XtAppAddTimeOut (app, cycle, cycle_timer, 0);
}


/* Call this when user activity (or "simulated" activity) has been noticed.
 */
static void
reset_timers ()
{
#if 0
  if (verbose_p)
    printf ("%s: restarting timer (%d, %d)\n", progname, timeout, timer_id);
#endif
  XtRemoveTimeOut (timer_id);
  timer_id = XtAppAddTimeOut (app, timeout, idle_timer, 0);
  if (cycle_id) abort ();

  last_activity_time = time ((time_t *) 0);
}

/* When we aren't using XIdle, this timer is used to periodically wake up
   and poll the mouse position, which is possibly more reliable than
   selecting motion events on every window.
 */
static void
check_pointer_timer (closure, this_timer)
     void *closure;
     XtPointer this_timer;
{
  static int last_root_x = -1;
  static int last_root_y = -1;
  static Window last_child = (Window) -1;
  static unsigned int last_mask = 0;
  Window root, child;
  int root_x, root_y, x, y;
  unsigned int mask;
  XtIntervalId *timerP = (XtIntervalId *) closure;
#ifdef HAVE_XIDLE
  if (use_xidle)
    abort ();
#endif

  *timerP = XtAppAddTimeOut (app, pointer_timeout, check_pointer_timer,
			     closure);

  XQueryPointer (dpy, window, &root, &child, &root_x, &root_y, &x, &y, &mask);
  if (root_x == last_root_x && root_y == last_root_y &&
      child == last_child && mask == last_mask)
    return;
#if 0
  if (verbose_p && this_timer)
    if (root_x == last_root_x && root_y == last_root_y && child == last_child)
      printf ("%s: modifiers changed at %s.\n", progname, timestring ());
    else
      printf ("%s: pointer moved at %s.\n", progname, timestring ());
#endif
  last_root_x = root_x;
  last_root_y = root_y;
  last_child = child;
  last_mask = mask;

  reset_timers ();
}


static void
sleep_until_idle (until_idle_p)
     Boolean until_idle_p;
{
  XEvent event;

  if (until_idle_p)
    {
      timer_id = XtAppAddTimeOut (app, timeout, idle_timer, 0);
#ifdef HAVE_XIDLE
      if (! use_xidle)
#endif
	/* start polling the mouse position */
	check_pointer_timer (&check_pointer_timer_id, 0);
    }

  while (1)
    {

      time_t activity_time = last_activity_time;

      XtAppNextEvent (app, &event);

      switch (event.xany.type) {
      case 0:		/* our synthetic "timeout" event has been signalled */
	if (until_idle_p)
	  {
	    Time idle;
#ifdef HAVE_XIDLE
	    if (use_xidle)
	      {
		if (! XGetIdleTime (dpy, &idle))
		  {
		    fprintf (stderr, "%s: %sXGetIdleTime() failed.\n",
			     progname, (verbose_p ? "## " : ""));
		    exit (1);
		  }
	      }
	    else
#endif /* HAVE_XIDLE */
	      idle = 1000 * (last_activity_time - time ((time_t *) 0));
	    
	    if (idle >= timeout)
	      goto DONE;
	    else
	      timer_id = XtAppAddTimeOut (app, timeout - idle,
					  idle_timer, 0);
	  }
	break;

      case ClientMessage:
	if (handle_clientmessage (&event, until_idle_p))
	  goto DONE;
	break;

      case CreateNotify:
#ifdef HAVE_XIDLE
	if (! use_xidle)
#endif
	  XtAppAddTimeOut (app, notice_events_timeout, notice_events_timer,
			   (XtPointer) event.xcreatewindow.window);
	break;

      case KeyPress:
      case KeyRelease:
      case ButtonPress:
      case ButtonRelease:
      case MotionNotify:

#if 0
	if (verbose_p)
	  {
	    if (event.xany.type == MotionNotify)
	      printf ("%s: MotionNotify at %s\n", progname, timestring ());
	    else if (event.xany.type == KeyPress)
	      printf ("%s: KeyPress seen on 0x%X at %s\n", progname,
		      event.xkey.window, timestring ());
	  }
#endif

	/* We got a user event */
	if (!until_idle_p)
	  goto DONE;
	else
	  reset_timers ();
	break;

      default:
	XtDispatchEvent (&event);
      }
    }
 DONE:

  if (check_pointer_timer_id)
    {
      XtRemoveTimeOut (check_pointer_timer_id);
      check_pointer_timer_id = 0;
    }
  if (timer_id)
    {
      XtRemoveTimeOut (timer_id);
      timer_id = 0;
    }

  if (until_idle_p && cycle_id)
    abort ();

  return;
}


static Boolean
handle_clientmessage (event, until_idle_p)
     XEvent *event;
     Boolean until_idle_p;
{
  Atom type;
  if (event->xclient.message_type != XA_SCREENSAVER)
    {
      char *str;
      str = XGetAtomName (dpy, event->xclient.message_type);
      fprintf (stderr, "%s: %sunrecognised ClientMessage type %s received\n",
	       progname, (verbose_p ? "## " : ""),
	       (str ? str : "(null)"));
      if (str) XFree (str);
    }
  if (event->xclient.format != 32)
    {
      fprintf (stderr, "%s: %sClientMessage of format %d received, not 32\n",
	       progname, (verbose_p ? "## " : ""), event->xclient.format);
    }
  type = event->xclient.data.l[0];
  if (type == XA_ACTIVATE)
    {
      if (until_idle_p)
	{
	  if (verbose_p)
	    printf ("%s: ACTIVATE ClientMessage received.\n", progname);
	  return True;
	}
      fprintf (stderr,
	       "%s: %sClientMessage ACTIVATE received while already active.\n",
	       progname, (verbose_p ? "## " : ""));
    }
  else if (type == XA_DEACTIVATE)
    {
      if (! until_idle_p)
	{
	  if (verbose_p)
	    printf ("%s: DEACTIVATE ClientMessage received.\n", progname);
	  return True;
	}
      fprintf (stderr,
	       "%s: %sClientMessage DEACTIVATE received while inactive.\n",
	       progname, (verbose_p ? "## " : ""));
    }
  else if (type == XA_CYCLE)
    {
      if (! until_idle_p)
	{
	  if (verbose_p)
	    printf ("%s: CYCLE ClientMessage received.\n", progname);
	  if (cycle_id)
	    XtRemoveTimeOut (cycle_id);
	  cycle_id = 0;
	  cycle_timer (0, 0);
	  return False;
	}
      fprintf (stderr,
	       "%s: %sClientMessage CYCLE received while inactive.\n",
	       progname, (verbose_p ? "## " : ""));
    }
  else if (type == XA_NEXT || type == XA_PREV)
    {
      if (verbose_p)
	printf ("%s: %s ClientMessage received.\n", progname,
		(type == XA_NEXT ? "NEXT" : "PREV"));
      next_mode_p = 1 + (type == XA_PREV);

      if (! until_idle_p)
	{
	  if (cycle_id)
	    XtRemoveTimeOut (cycle_id);
	  cycle_id = 0;
	  cycle_timer (0, 0);
	}
      else
	return True;
    }
  else if (type == XA_EXIT)
    {
      printf ("%s: EXIT ClientMessage received.\n", progname);
      if (! until_idle_p)
	{
	  unblank_screen ();
	  kill_screenhack ();
	  XSync (dpy, False);
	}
      exit (0);
    }
  else
    {
      char *str;
      str = XGetAtomName (dpy, type);
      if (str)
	fprintf (stderr,
		 "%s: %sunrecognised screensaver ClientMessage %s received\n",
		 progname, (verbose_p ? "## " : ""), str);
      else
	fprintf (stderr,
		 "%s: %sunrecognised screensaver ClientMessage 0x%x received\n",
		 progname, (verbose_p ? "## " : ""),
		 event->xclient.data.l[0]);
      if (str) XFree (str);
    }
  return False;
}

static void
hack_environment ()
{
  /* Store $DISPLAY into the environment, so that the $DISPLAY variable that
     the spawned processes inherit is the same as the value of -display passed
     in on our command line (which is not necessarily the same as what our
     $DISPLAY variable is.)
   */
  char *s, buf [2048];
  int i;
  sprintf (buf, "DISPLAY=%s", DisplayString (dpy));
  i = strlen (buf);
  s = (char *) malloc (i+1);
  strncpy (s, buf, i+1);
  if (putenv (s))
    abort ();
}

static void
exec_screenhack (command)
     char *command;
{
  char *tmp;
  char buf [512];
  char *av [5];
  int ac = 0;

  /* Close this fork's version of the display's fd.  It will open its own. */
  close (ConnectionNumber (dpy));
  
  /* I don't believe what a sorry excuse for an operating system UNIX is!

     - I want to spawn a process.
     - I want to know it's pid so that I can kill it.
     - I would like to receive a message when it dies of natural causes.
     - I want the spawned process to have user-specified arguments.

     The *only way* to parse arguments the way the shells do is to run a
     shell (or duplicate what they do, which would be a *lot* of code.)

     The *only way* to know the pid of the process is to fork() and exec()
     it in the spawned side of the fork.

     But if you're running a shell to parse your arguments, this gives you
     the pid of the SHELL, not the pid of the PROCESS that you're actually
     interested in, which is an *inferior* of the shell.  This also means
     that the SIGCHLD you get applies to the shell, not its inferior.

     So, the only solution other than implementing an argument parser here
     is to force the shell to exec() its inferior.  What a fucking hack!
     We prepend "exec " to the command string.
   */
  tmp = command;
  command = (char *) malloc (strlen (tmp) + 6);
  bcopy ("exec ", command, 5);
  bcopy (tmp, command + 5, strlen (tmp) + 1);

  /* Invoke the shell as "/bin/sh -c 'exec prog -arg -arg ...'" */
  av [ac++] = shell;
  av [ac++] = "-c";
  av [ac++] = command;
  av [ac++] = 0;
  
  if (verbose_p)
    printf ("%s: spawning \"%s\" in pid %d.\n", progname, command, getpid ());

  if (setpriority (PRIO_PROCESS, getpid(), nice_inferior) != 0)
    {
      sprintf (buf, "%s: setpriority(PRIO_PROCESS, %d, %d) failed",
	       progname, getpid(), nice_inferior);
      perror (buf);
    }

  /* Now overlay the current process with /bin/sh running the command.
     If this returns, it's an error.
   */
  execve (av [0], av, environ);

  sprintf (buf, "%s: execve() failed", progname);
  perror (buf);
  exit (1);	/* Note this this only exits a child fork.  */
}

/* to avoid a race between the main thread and the SIGCHLD handler */
static int killing = 0;

static void
await_child_death (killed)
     Boolean killed;
{
  int status;
  pid_t kid;
  killing = 1;
  if (! pid)
    return;
  kid = waitpid (pid, &status, WUNTRACED);
  if (kid == pid)
    {
      if (WIFEXITED (status))
	{
	  int exit_status = WEXITSTATUS (status);
	  if (exit_status & 0x80)
	    exit_status |= ~0xFF;
	  if (exit_status != 0 && verbose_p)
	    printf ("%s: child pid %d exited abnormally (code %d).\n",
		    progname, pid, exit_status);
	  else if (verbose_p)
	    printf ("%s: child pid %d exited normally.\n", progname, pid);
	}
      else if (WIFSIGNALED (status))
	{
	  if (!killed || WTERMSIG (status) != SIGTERM)
	    fprintf (stderr,
		     "%s: %schild pid %d (%s) terminated with signal %d!\n",
		     progname, (verbose_p ? "## " : ""),
		     pid, current_hack_name (), WTERMSIG (status));
	  else if (verbose_p)
	    printf ("%s: child pid %d (%s) terminated with SIGTERM.\n",
		    progname, pid, current_hack_name ());
	}
      else if (WIFSTOPPED (status))
	fprintf (stderr, "%s: %schild pid %d (%s) stopped with signal %d!\n",
		 progname, (verbose_p ? "## " : ""), pid,
		 current_hack_name (), WSTOPSIG (status));
      else
	fprintf (stderr, "%s: %schild pid %d (%s) died in a mysterious way!",
		 progname, (verbose_p ? "## " : ""), pid, current_hack_name ());
    }
  else if (kid <= 0)
    fprintf (stderr, "%s: %swaitpid(%d, ...) says there are no kids?  (%d)\n",
	     progname, (verbose_p ? "## " : ""), pid, kid);
  else
    fprintf (stderr, "%s: %swaitpid(%d, ...) says proc %d died, not %d?\n",
	     progname, (verbose_p ? "## " : ""), pid, kid, pid);
  killing = 0;
  pid = 0;
}


#ifdef SIGCHLD
static void
sigchld_handler (sig)
     int sig;
{
  if (killing)
    return;
  if (! pid)
    abort ();
  await_child_death (False);
}
#endif


static void
grab_keyboard_and_mouse ()
{
  Status status;
  XSync (dpy, False);
  status = XGrabKeyboard (dpy, window, True, GrabModeAsync, GrabModeAsync,
			  CurrentTime);
  if (status != GrabSuccess)
    {	/* try again in a second */
      sleep (1);
      status = XGrabKeyboard (dpy, window, True, GrabModeAsync, GrabModeAsync,
			      CurrentTime);
      if (status != GrabSuccess)
	{
	  fprintf (stderr, "%s: %scouldn't grab keyboard!  (%d)\n",
		   progname, (verbose_p ? "## " : ""), status);
	  exit (1);
	}
    }
#define ALL_POINTER_EVENTS \
	(ButtonPressMask | ButtonReleaseMask | EnterWindowMask | \
	 LeaveWindowMask | PointerMotionMask | PointerMotionHintMask | \
	 Button1MotionMask | Button2MotionMask | Button3MotionMask | \
	 Button4MotionMask | Button5MotionMask | ButtonMotionMask)

  status = XGrabPointer (dpy, window, True, ALL_POINTER_EVENTS,
			 GrabModeAsync, GrabModeAsync, None, cursor,
			 CurrentTime);
  if (status != GrabSuccess)
    {	/* try again in a second */
      sleep (1);
      status = XGrabPointer (dpy, window, True, ALL_POINTER_EVENTS,
			     GrabModeAsync, GrabModeAsync, None, cursor,
			     CurrentTime);
      if (status != GrabSuccess)
	{
	  fprintf (stderr, "%s: %scouldn't grab pointer!  (%d)\n",
		   progname, (verbose_p ? "## " : ""), status);
	  exit (1);
	}
    }
}

static void
ungrab_keyboard_and_mouse ()
{
  XUngrabPointer (dpy, CurrentTime);
  XUngrabKeyboard (dpy, CurrentTime);
}

static void 
raise_window (first_time_p)
     Bool first_time_p;
{
  if (install_cmap_p)
    XSetWindowColormap (dpy, window, cmap);
  XSetWindowBackground (dpy, window, black_pixel);
  XDefineCursor (dpy, window, cursor);

  if (fade_p)
    {
      copy_colormap (dpy, cmap, cmap2);
      XGrabServer (dpy);
      XInstallColormap (dpy, cmap2);
      fade_colormap (dpy, cmap, cmap2, fade_seconds, fade_ticks, True);
      XClearWindow (dpy, window);
      XMapRaised (dpy, window);
      XInstallColormap (dpy, cmap);
      XUngrabServer (dpy);
    }
  else
    {
      XClearWindow (dpy, window);
      XMapRaised (dpy, window);
    }

  if (install_cmap_p)	/* does not work... */
    XInstallColormap (dpy, cmap);
}

static void
blank_screen ()
{
  save_real_vroot ();
  store_vroot_property (window, window);
  raise_window (True);
  grab_keyboard_and_mouse ();
}

static void
unblank_screen ()
{
  if (unfade_p)
    {
      blacken_colormap (dpy, cmap2);
      XGrabServer (dpy);
      XInstallColormap (dpy, cmap2);
      XUnmapWindow (dpy, window);
      fade_colormap (dpy, cmap, cmap2, fade_seconds, fade_ticks, False);
      XInstallColormap (dpy, cmap);
      XUngrabServer (dpy);
    }
  else
    {
      XUnmapWindow (dpy, window);
    }
  ungrab_keyboard_and_mouse ();
  restore_real_vroot ();
}


static void
spawn_screenhack ()
{
  raise_window (False);
  XFlush (dpy);

  if (screenhacks_count)
    {
      char *hack;
      pid_t forked;
      char buf [255];
      int new_hack;
      if (screenhacks_count == 1)
	new_hack = 0;
      else if (next_mode_p == 1)
	new_hack = (current_hack + 1) % screenhacks_count,
	next_mode_p = 0;
      else if (next_mode_p == 2)
	new_hack = (current_hack + screenhacks_count - 1) % screenhacks_count,
	next_mode_p = 0;
      else
	while ((new_hack = random () % screenhacks_count) == current_hack)
	  ;
      current_hack = new_hack;
      hack = screenhacks[current_hack];

      switch (forked = fork ())
	{
	case -1:
	  sprintf (buf, "%s: couldn't fork", progname);
	  perror (buf);
	  restore_real_vroot ();
	  exit (1);
	case 0:
	  exec_screenhack (hack); /* this does not return */
	  break;
	default:
	  pid = forked;
	  break;
	}
    }
}

static void
kill_screenhack ()
{
  killing = 1;
  if (! pid)
    return;
  if (kill (pid, SIGTERM) < 0)
    {
      if (errno == ESRCH)
	{
	  /* Sometimes we don't get a SIGCHLD at all!  WTF?
	     It's a race condition.  It looks to me like what's happening is
	     something like: a subprocess dies of natural causes.  There is a
	     small window between when the process dies and when the SIGCHLD
	     is (would have been) delivered.  If we happen to try to kill()
	     the process during that time, the kill() fails, because the
	     process is already dead.  But! no SIGCHLD is delivered (perhaps
	     because the failed kill() has reset some state in the kernel?)
	     Anyway, if kill() says "No such process", then we have to wait()
	     for it anyway, because the process has already become a zombie.
	     I love Unix.
	   */
	  await_child_death (False);
	}
      else
	{
	  char buf [255];
	  sprintf (buf, "%s: couldn't kill child process %d", progname, pid);
	  perror (buf);
	}
    }
  else
    {
      if (verbose_p)
	printf ("%s: killing pid %d.\n", progname, pid);
      await_child_death (True);
    }
}


static char *
current_hack_name ()
{
  static char chn [1024];
  char *hack = screenhacks [current_hack];
  int i;
  for (i = 0; hack [i] != 0 && hack [i] != ' ' && hack [i] != '\t'; i++)
    chn [i] = hack [i];
  chn [i] = 0;
  return chn;
}


#ifdef _VROOT_H_
ERROR!  You must not include vroot.h in this file.
#endif

static void
ensure_no_screensaver_running ()
{
  int i;
  Window root = RootWindowOfScreen (screen);
  Window root2, parent, *kids;
  unsigned int nkids;

  window = 0;
  if (! XQueryTree (dpy, root, &root2, &parent, &kids, &nkids))
    abort ();
  if (root != root2)
    abort ();
  if (parent)
    abort ();
  if (! (kids && nkids))
    abort ();
  for (i = 0; i < nkids; i++)
    {
      Atom type;
      int format;
      unsigned long nitems, bytesafter;
      char *version;

      if (XGetWindowProperty (dpy, kids[i], XA_SCREENSAVER_VERSION, 0, 1,
			      False, XA_STRING, &type, &format, &nitems,
			      &bytesafter, (unsigned char **) &version)
	  == Success
	  && type != None)
	{
	  char *id;
	  if (!XGetWindowProperty (dpy, kids[i], XA_SCREENSAVER_ID, 0, 512,
				   False, XA_STRING, &type, &format, &nitems,
				   &bytesafter, (unsigned char **) &id)
	      == Success
	      || type == None)
	    id = "???";

	  fprintf (stderr,
       "%s: %salready running on display %s (window 0x%x)\n from process %s.\n",
		   progname, (verbose_p ? "## " : ""), DisplayString (dpy),
		   (int) kids [i], id);
	  exit (1);
	}
    }
}

static void
save_real_vroot ()
{
  int i;
  Window root = RootWindowOfScreen (screen);
  Window root2, parent, *kids;
  unsigned int nkids;

  real_vroot = 0;
  real_vroot_value = 0;
  if (! XQueryTree (dpy, root, &root2, &parent, &kids, &nkids))
    abort ();
  if (root != root2)
    abort ();
  if (parent)
    abort ();
  if (! (kids && nkids))
    abort ();
  for (i = 0; i < nkids; i++)
    {
      Atom type;
      int format;
      unsigned long nitems, bytesafter;
      Window *vrootP = 0;

      if (XGetWindowProperty (dpy, kids[i], XA_VROOT, 0, 1, False, XA_WINDOW,
			      &type, &format, &nitems, &bytesafter,
			      (unsigned char **) &vrootP)
	  != Success)
	continue;
      if (! vrootP)
	continue;
      if (real_vroot)
	{
	  if (*vrootP == window) abort ();
	  fprintf (stderr,
	    "%s: %smore than one virtual root window found (0x%x and 0x%x).\n",
		   progname, (verbose_p ? "## " : ""),
		   (int) real_vroot, (int) kids [i]);
	  exit (1);
	}
      real_vroot = kids [i];
      real_vroot_value = *vrootP;
    }

  if (real_vroot)
    {
      handle_signals (True);
      remove_vroot_property (real_vroot);
      XSync (dpy, False);
    }

  XFree ((char *) kids);
}

static Boolean
restore_real_vroot_1 ()
{
  if (verbose_p && real_vroot)
    printf ("%s: restoring __SWM_VROOT property on the real vroot (0x%x).\n",
	    progname, real_vroot);
  remove_vroot_property (window);
  if (real_vroot)
    {
      store_vroot_property (real_vroot, real_vroot_value);
      real_vroot = 0;
      real_vroot_value = 0;
      /* make sure the property change gets there before this process
	 terminates!  We might be doing this because we have intercepted
	 SIGTERM or something. */
      XSync (dpy, False);
      return True;
    }
  return False;
}

static void
restore_real_vroot ()
{
  if (restore_real_vroot_1 ())
    handle_signals (False);
}

static char *sig_names [255] = { 0 };

static void
restore_real_vroot_handler (sig)
     int sig;
{
  signal (sig, SIG_DFL);
  if (restore_real_vroot_1 ())
    fprintf (stderr, "\n%s: %s%s (%d) intercepted, vroot restored.\n",
	     progname, (verbose_p ? "## " : ""),
	     ((sig < sizeof(sig_names) && sig >= 0 && sig_names [sig])
	      ? sig_names [sig] : "unknown signal"),
	     sig);
  kill (getpid (), sig);
}


static void
catch_signal (sig, signame, on_p)
     int sig;
     char *signame;
     Boolean on_p;
{
  if (! on_p)
    signal (sig, SIG_DFL);
  else
    {
      sig_names [sig] = signame;
      if (((int) signal (sig, restore_real_vroot_handler)) == -1)
	{
	  char buf [255];
	  sprintf (buf, "%s: couldn't catch %s (%d)", progname, signame, sig);
	  perror (buf);
	  restore_real_vroot ();
	  exit (1);
	}
    }
}

static void
handle_signals (on_p)
     Boolean on_p;
{
#if 0
  if (on_p) printf ("handling signals\n");
  else printf ("unhandling signals\n");
#endif

  catch_signal (SIGHUP,  "SIGHUP",  on_p);
  catch_signal (SIGINT,  "SIGINT",  on_p);
  catch_signal (SIGQUIT, "SIGQUIT", on_p);
  catch_signal (SIGILL,  "SIGILL",  on_p);
  catch_signal (SIGTRAP, "SIGTRAP", on_p);
  catch_signal (SIGIOT,  "SIGIOT",  on_p);
  catch_signal (SIGABRT, "SIGABRT", on_p);
#ifdef SIGEMT
  catch_signal (SIGEMT,  "SIGEMT",  on_p);
#endif
  catch_signal (SIGFPE,  "SIGFPE",  on_p);
  catch_signal (SIGBUS,  "SIGBUS",  on_p);
  catch_signal (SIGSEGV, "SIGSEGV", on_p);
  catch_signal (SIGSYS,  "SIGSYS",  on_p);
  catch_signal (SIGTERM, "SIGTERM", on_p);
#ifdef SIGXCPU
  catch_signal (SIGXCPU, "SIGXCPU", on_p);
#endif
#ifdef SIGXFSZ
  catch_signal (SIGXFSZ, "SIGXFSZ", on_p);
#endif
#ifdef SIGDANGER
  catch_signal (SIGDANGER, "SIGDANGER", on_p);
#endif
}


static void
disable_builtin_screensaver ()
{
  int timeout, interval, prefer_blank, allow_exp;
  XForceScreenSaver (dpy, ScreenSaverReset);
  XGetScreenSaver (dpy, &timeout, &interval, &prefer_blank, &allow_exp);
  if (timeout != 0)
    {
      XSetScreenSaver (dpy, 0, interval, prefer_blank, allow_exp);
      printf ("%s%sisabling server builtin screensaver.\n\
	You can re-enable it with \"xset s on\".\n",
	      (verbose_p ? "" : progname), (verbose_p ? "\n\tD" : ": d"));
    }
}


static XrmOptionDescRec options [] = {
  { "-timeout",	".timeout",	XrmoptionSepArg, 0 },
  { "-idelay",	".initialDelay",XrmoptionSepArg, 0 },
  { "-cycle",	".cycle",	XrmoptionSepArg, 0 },
  { "-verbose",	".verbose",	XrmoptionNoArg, "on" },
  { "-silent",	".verbose",	XrmoptionNoArg, "off" },
  { "-xidle",	".xidle",	XrmoptionNoArg, "on" },
  { "-no-xidle",".xidle",	XrmoptionNoArg, "off" }
};

static char *defaults[] = {
#include "XScreenSaver.ad.h"
 0
};


static void
get_resources ()
{
  timeout         = 60000 * get_integer_resource ("timeout", "Time");
  cycle           = 60000 * get_integer_resource ("cycle",   "Time");
  nice_inferior   = get_integer_resource ("nice", "Nice");
  verbose_p       = get_boolean_resource ("verbose", "Boolean");
  install_cmap_p  = get_boolean_resource ("installColormap", "Boolean");
  fade_p	  = get_boolean_resource ("fade", "Boolean");
  unfade_p	  = get_boolean_resource ("unfade", "Boolean");
  fade_seconds	  = get_integer_resource ("fadeSeconds", "Time");
  fade_ticks	  = get_integer_resource ("fadeTicks", "Integer");
  shell           = get_string_resource ("bourneShell", "BourneShell");
  initial_delay   = get_integer_resource ("initialDelay","Time");
  pointer_timeout = 1000 * get_integer_resource ("pointerPollTime","Time");
  notice_events_timeout = 1000 * get_integer_resource ("windowCreationTimeout",
						       "Time");
  if (timeout < 1) timeout = 1;
  if (cycle < 1) cycle = 1;
  if (initial_delay < 0) initial_delay = 30;
  if (pointer_timeout <= 0) pointer_timeout = 5;
  if (notice_events_timeout <= 0) notice_events_timeout = 10;
  if (fade_seconds <= 0 || fade_ticks <= 0) fade_p = False;
  if (! fade_p) unfade_p = False;

  /* don't set use_xidle unless it is explicitly specified */
  if (get_string_resource ("xidle", "Boolean"))
    use_xidle = get_boolean_resource ("xidle", "Boolean");
  else
#ifdef HAVE_XIDLE	/* pick a default */
    use_xidle = True;
#else
    use_xidle = False;
#endif

  {
    char *data = get_string_resource ("programs", "Programs");
    int size = strlen (data);
    int i = 0, hacks_size = 10;
    screenhacks = (char **) malloc (sizeof (char *) * hacks_size);
    screenhacks_count = 0;
    while (i < size)
      {
	int end, start = i;
	if (data[i] == ' ' || data[i] == '\t' || data[i] == '\n' ||
	    data[i] == 0)
	  {
	    i++;
	    continue;
	  }
	if (hacks_size <= screenhacks_count)
	  screenhacks = (char **) realloc (screenhacks,
					   (hacks_size = hacks_size * 2) *
					   sizeof (char *));
	screenhacks [screenhacks_count++] = data + i;
	while (data[i] != 0 && data[i] != '\n')
	  i++;
	end = i;
	while (i > start && (data[i-1] == ' ' || data[i-1] == '\t'))
	  i--;
	data [i] = 0;
	i = end + 1;
      }
    if (screenhacks_count)
      screenhacks = (char **)
	realloc (screenhacks, (screenhacks_count * sizeof(char *)));
    else
      free (screenhacks), free (data), screenhacks = 0;
  }
}

static char *
timestring ()
{
  long now = time ((time_t *) 0);
  char *str = (char *) ctime (&now);
  char *nl = (char *) strchr (str, '\n');
  if (nl) *nl = 0; /* take off that dang newline */
  return str;
}

main (argc, argv)
     int argc;
     char **argv;
{
  Widget toplevel;
  progclass = "XScreenSaver";
  toplevel = XtAppInitialize (&app, progclass, options, XtNumber (options),
			      &argc, argv, defaults, 0, 0);
  dpy = XtDisplay (toplevel);
  screen = XtScreen (toplevel);
  db = XtDatabase (dpy);
  current_hack = -1;
  next_mode_p = 0;
  XtGetApplicationNameAndClass (dpy, &progname, &progclass);
  if (argc > 1)
    {
      fprintf (stderr, "%s: unknown option %s\n", progname, argv [1]);
      exit (1);
    }
  get_resources ();
  hack_environment ();
  XA_VROOT = XInternAtom (dpy, "__SWM_VROOT", False);
  XA_SCREENSAVER = XInternAtom (dpy, "SCREENSAVER", False);
  XA_SCREENSAVER_VERSION = XInternAtom (dpy, "_SCREENSAVER_VERSION", False);
  XA_SCREENSAVER_ID = XInternAtom (dpy, "_SCREENSAVER_ID", False);
  XA_ACTIVATE = XInternAtom (dpy, "ACTIVATE", False);
  XA_DEACTIVATE = XInternAtom (dpy, "DEACTIVATE", False);
  XA_CYCLE = XInternAtom (dpy, "CYCLE", False);
  XA_NEXT = XInternAtom (dpy, "NEXT", False);
  XA_PREV = XInternAtom (dpy, "PREV", False);
  XA_EXIT = XInternAtom (dpy, "EXIT", False);
  ensure_no_screensaver_running ();
  make_screensaver_window ();
  srandom ((int) time ((time_t *) 0));
  pid = 0;
  cycle_id = 0;
  killing = 0;
  if (use_xidle)
    {
#ifdef HAVE_XIDLE
      int first_event, first_error;
      if (! XidleQueryExtension (dpy, &first_event, &first_error))
	{
	  fprintf (stderr,
		   "%s: display %s does not support the XIdle extension.\n",
		   progname, DisplayString (dpy));
	  use_xidle = 0;
	}
#else
      fprintf (stderr, "%s: not compiled with support for XIdle.\n",
	       progname);
      use_xidle = 0;
#endif
    }

#ifdef SIGCHLD
  if (((int) signal (SIGCHLD, sigchld_handler)) == -1)
    {
      char buf [255];
      sprintf (buf, "%s: couldn't catch SIGCHLD", progname);
      perror (buf);
    }
#endif

  if (verbose_p)
    printf ("%s %s started, pid = %d.\n\
	Copyright (c) 1992 by Jamie Zawinski <jwz@lucid.com>\n",
	    progname, screensaver_version, getpid ());

  disable_builtin_screensaver ();

#ifdef HAVE_XIDLE
  if (! use_xidle)
#endif
    {
      if (initial_delay)
	{
	  if (verbose_p)
	    printf ("%s: waiting for a while...\n", progname);
	  sleep (initial_delay);
	}
      if (verbose_p)
	printf ("%s: selecting events on extant windows...", progname);
      notice_events_timer ((XtPointer)
			   RootWindowOfScreen (XtScreen (toplevel)),
			   0);
      if (verbose_p) printf (" done.\n");
    }

  while (1)
    {
      sleep_until_idle (True);
      if (verbose_p)
	printf ("%s: user is idle; waking up at %s.\n", progname, timestring());
      blank_screen ();
      spawn_screenhack ();
      if (cycle)
	cycle_id = XtAppAddTimeOut (app, cycle, cycle_timer, 0);
      sleep_until_idle (False); /* until not idle */
      unblank_screen ();
      kill_screenhack ();
      if (cycle_id)
	{
	  XtRemoveTimeOut (cycle_id);
	  cycle_id = 0;
	}
      if (verbose_p)
	printf ("%s: user is active; going to sleep at %s.\n", progname,
		timestring ());
    }
}
