/* xscreensaver, Copyright (c) 1992 Jamie Zawinski <jwz@lucid.com>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#include "screenhack.h"

struct qpoint {
  int x, y;
  int dx, dy;
};

struct qline {
  struct qpoint p1, p2;
  XColor color;
};

struct qix {
  int fp;
  int nlines;
  struct qline *lines;
};

static GC draw_gc, erase_gc;
static unsigned int default_fg_pixel;
static int maxx, maxy, max_spread, color_shift;
static Bool random_p, solid_p;
static int delay;
static Colormap cmap;

static void
get_geom (dpy, window)
     Display *dpy;
     Window window;
{
  XWindowAttributes xgwa;
  XGetWindowAttributes (dpy, window, &xgwa);
  maxx = xgwa.width;
  maxy = xgwa.height;
};

static struct qix *
init_qix (dpy, window)
     Display *dpy;
     Window window;
{
  int i;
  XGCValues gcv;
  XWindowAttributes xgwa;
  struct qix *qix;
  XGetWindowAttributes (dpy, window, &xgwa);
  cmap = xgwa.colormap;
  qix = (struct qix *) calloc (1, sizeof (struct qix));
  qix->nlines = get_integer_resource ("segments", "Integer");
  if (qix->nlines <= 0) qix->nlines = 20;
  qix->lines = (struct qline *) calloc (qix->nlines, sizeof (struct qline));
  get_geom (dpy, window);
  max_spread = get_integer_resource ("spread", "Integer");
  if (max_spread <= 0) max_spread = 10;
  random_p = get_boolean_resource ("random", "Boolean");
  solid_p = get_boolean_resource ("solid", "Boolean");
  delay = get_integer_resource ("delay", "Integer");
  color_shift = get_integer_resource ("colorShift", "Integer");
  if (color_shift < 0 || color_shift >= 360) color_shift = 5;
  if (delay < 0) delay = 0;

  if (!mono_p)
    {
      hsv_to_rgb (random () % 360, frand (1.0), frand (0.5) + 0.5,
		  &qix->lines[0].color.red, &qix->lines[0].color.green,
		  &qix->lines[0].color.blue);
      if (!XAllocColor (dpy, cmap, &qix->lines[0].color))
	{
	  qix->lines[0].color.pixel = default_fg_pixel;
	  if (!XQueryColor (dpy, cmap, &qix->lines[0].color))
	    abort ();
	  if (!XAllocColor (dpy, cmap, &qix->lines[0].color))
	    abort ();
	}
    }
  qix->lines[0].p1.x = random () % maxx;
  qix->lines[0].p1.y = random () % maxy;
  qix->lines[0].p2.x = random () % maxx;
  qix->lines[0].p2.y = random () % maxy;
  qix->lines[0].p1.dx = (random () % (max_spread + 1)) - (max_spread / 2);
  qix->lines[0].p1.dy = (random () % (max_spread + 1)) - (max_spread / 2);
  qix->lines[0].p2.dx = (random () % (max_spread + 1)) - (max_spread / 2);
  qix->lines[0].p2.dy = (random () % (max_spread + 1)) - (max_spread / 2);
  for (i = 1; i < qix->nlines; i++)
    {
      qix->lines[i] = qix->lines[0];
      if ((!mono_p) && (!XAllocColor (dpy, cmap, &qix->lines[i].color)))
	abort ();
    }
  gcv.foreground = default_fg_pixel =
    get_pixel_resource ("foreground", "Foreground", dpy, cmap);
  draw_gc = XCreateGC (dpy, window, GCForeground, &gcv);
  gcv.foreground = get_pixel_resource ("background", "Background", dpy, cmap);
  erase_gc = XCreateGC (dpy, window, GCForeground, &gcv);
  return qix;
}

static void
free_qline (dpy, window, cmap, qline)
     Display *dpy;
     Window window;
     Colormap cmap;
     struct qline *qline;
{
  if (solid_p)
    {
      static struct qline prev = { 0 };
      XPoint points [4];
      points [0].x = qline->p1.x; points [0].y = qline->p1.y;
      points [1].x = qline->p2.x; points [1].y = qline->p2.y;
      points [2].x = prev.p2.x; points [2].y = prev.p2.y;
      points [3].x = prev.p1.x; points [3].y = prev.p1.y;
      XFillPolygon (dpy, window, erase_gc, points, 4,
		    Complex, CoordModeOrigin);
      prev = *qline;
    }
  else
    XDrawLine (dpy, window, erase_gc, qline->p1.x, qline->p1.y,
	       qline->p2.x, qline->p2.y);

  if (! mono_p) XFreeColors (dpy, cmap, &qline->color.pixel, 1, 0);
}

static void
add_qline (dpy, window, cmap, qline, prev_qline)
     Display *dpy;
     Window window;
     Colormap cmap;
     struct qline *qline, *prev_qline;
{
  *qline = *prev_qline;

#define wiggle(point,delta,max)						\
  if (random_p) delta += (random () % 3) - 1;				\
  if (delta > max_spread) delta = max_spread;				\
  else if (delta < -max_spread) delta = -max_spread;			\
  point += delta;							\
  if (point < 0) point = 0, delta = -delta, point += delta<<1;		\
  else if (point > max) point = max, delta = -delta, point += delta<<1;
  
  wiggle (qline->p1.x, qline->p1.dx, maxx);
  wiggle (qline->p1.y, qline->p1.dy, maxy);
  wiggle (qline->p2.x, qline->p2.dx, maxx);
  wiggle (qline->p2.y, qline->p2.dy, maxy);
  if (! mono_p)
    {
      cycle_hue (&qline->color, color_shift);
      qline->color.flags = DoRed | DoGreen | DoBlue;
      if (!XAllocColor (dpy, cmap, &qline->color))
	{
	  qline->color = prev_qline->color;
	  if (!XAllocColor (dpy, cmap, &qline->color))
	    abort (); /* same color should work */
	}
      XSetForeground (dpy, draw_gc, qline->color.pixel);
    }
  if (solid_p)
    {
      XPoint points [4];
      points [0].x = qline->p1.x; points [0].y = qline->p1.y;
      points [1].x = qline->p2.x; points [1].y = qline->p2.y;
      points [2].x = prev_qline->p2.x; points [2].y = prev_qline->p2.y;
      points [3].x = prev_qline->p1.x; points [3].y = prev_qline->p1.y;
      XFillPolygon (dpy, window, draw_gc, points, 4,
		    Complex, CoordModeOrigin);
    }
  else
    XDrawLine (dpy, window, draw_gc, qline->p1.x, qline->p1.y,
	       qline->p2.x, qline->p2.y);
}

static void
qix1 (dpy, window, qix)
     Display *dpy;
     Window window;
     struct qix *qix;
{
  int ofp = qix->fp - 1;
  static int gtick = 0;
  if (gtick++ == 500)
    get_geom (dpy, window), gtick = 0;
  if (ofp < 0) ofp = qix->nlines - 1;
  free_qline (dpy, window, cmap, &qix->lines [qix->fp]);
  add_qline (dpy, window, cmap, &qix->lines[qix->fp], &qix->lines[ofp]);
  if ((++qix->fp) >= qix->nlines)
    qix->fp = 0;
  XSync (dpy, True);
  if (delay) usleep (delay);
}


char *progclass = "Qix";

char *defaults [] = {
  "*background:	black",
  "*foreground:	white",
  "*segments:	50",
  "*spread:	8",
  "*colorShift:	3",
  "*solid:	false",
  "*delay:	25000",
  "*random:	true",
  0
};

XrmOptionDescRec options [] = {
  { "-segments",	".segments",	XrmoptionSepArg, 0 },
  { "-spread",		".spread",	XrmoptionSepArg, 0 },
  { "-delay",		".delay",	XrmoptionSepArg, 0 },
  { "-color-shift",	".colorShift",	XrmoptionSepArg, 0 },
  { "-random",		".random",	XrmoptionNoArg, "true" },
  { "-linear",		".random",	XrmoptionNoArg, "false" },
  { "-solid",		".solid",	XrmoptionNoArg, "true" },
  { "-hollow",		".solid",	XrmoptionNoArg, "false" },
};
int options_size = (sizeof (options) / sizeof (options[0]));

void
screenhack (dpy, window)
     Display *dpy;
     Window window;
{
  struct qix *q = init_qix (dpy, window);
  while (1)
    qix1 (dpy, window, q);
}
