/*
 * Copyright (c) 1994, 1995.  Netscape Communications Corporation.  All
 * rights reserved.
 * 
 * Use of this software is governed by the terms of the license agreement for
 * the Netscape FastTrack or Netscape Enterprise Server between the
 * parties.
 */


/* ------------------------------------------------------------------------ */


/*
 * req.h: Request-specific data structures and functions
 * 
 * Rob McCool
 */


#ifndef REQ_H
#define REQ_H


#include "netsite.h"
#include "base/pblock.h"
#include "base/session.h"
#include "frame/objset.h"

#ifndef XP_UNIX
#include <sys/types.h>
#endif /* XP_UNIX */

#include "base/file.h"   /* struct stat */



/* ------------------------------ Constants ------------------------------- */


#define REQ_HASHSIZE 10
#define REQ_MAX_LINE 4096

    
/*
 * The REQ_ return codes. These codes are used to determine what the server
 * should do after a particular module completes its task.
 *
 * Func type functions return these as do many internal functions.
 */

/* The function performed its task, proceed with the request */
#define REQ_PROCEED 0
/* The entire request should be aborted: An error occurred */
#define REQ_ABORTED -1
/* The function performed no task, but proceed anyway. */
#define REQ_NOACTION -2
/* Tear down the session and exit */
#define REQ_EXIT -3
/* Restart the entire request-response process */
#define REQ_RESTART -4


/* ------------------------------ Structures ------------------------------ */


typedef struct {
    /* Server working variables */
    pblock *vars;

    /* The method, URI, and protocol revision of this request */
    pblock *reqpb;
    /* Protocol specific headers */
    int loadhdrs;
    pblock *headers;

    /* Server's response headers */
    int senthdrs;
    pblock *srvhdrs;

    /* The object set constructed to fulfill this request */
    httpd_objset *os;
    /* Array of objects that were created from .nsconfig files */
    httpd_objset *tmpos;

    /* The stat last returned by request_stat_path */
    char *statpath;
    char *staterr;
    struct stat *finfo;

    /* access control state */
    int aclstate;		/* ACL decision state */
    int acldirno;		/* deciding ACL directive number */
    char * aclname;		/* name of deciding ACL */
    pblock * aclpb;		/* parameter block for ACL PathCheck */

#ifdef MCC_PROXY
    struct hostent *hp;	/* proxy NSAPI: DNS resolution result */
    char * host;	/* proxy NSAPI: host to resolve/connect to */
    int    port;	/* proxy NSAPI: port to connect to */

    void * socks_rq;	/* SOCKS request data */
#endif

    int request_is_cacheable;  /* default TRUE */
    int directive_is_cacheable; /* default FALSE */

    char *cached_headers;
    int cached_headers_len;	/* length of the valid headers */
    char *cached_date_header;
} Request;


/* ------------------------------ Prototypes ------------------------------ */


/*
 * request_create creates a new request structure.
 */

NSAPI_PUBLIC Request *request_create(void);

/*
 * request_free destroys a request structure.
 */

NSAPI_PUBLIC void request_free(Request *req);


/*
 * Restarts a request for a given URI internally. If rq is non-NULL, the
 * function will keep the old request's headers and protocol, but with a new 
 * URI and method of GET. If the previous method was HEAD, this is preserved.
 * Any other method becomes GET. You may assume that if you give it a request
 * structure that it will use the same structure.
 *
 * Once you have this new Request, you must then do what you want with
 * it (e.g. send the object back, perform uri2path translation, etc.)
 */

NSAPI_PUBLIC Request *request_restart_internal(char *uri, Request *rq);


/*
 * request_header finds the named header depending on the requesting 
 * protocol. If possible, it will not load headers until the first is 
 * requested. You have to watch out because this can return REQ_ABORTED.
 */

NSAPI_PUBLIC int request_header(char *name, char **value, 
                                Session *sn, Request *rq);

/*
 * request_loadheaders just makes sure the headers have been loaded.
 */

#define request_loadheaders(sn,rq) (int)(REQ_PROCEED)


/*
 * request_stat_path tries to stat path. If path is NULL, it will look in
 * the vars pblock for "path". If the stat is successful, it returns the stat 
 * structure. If not, returns NULL and leaves a message in rq->staterr. If a 
 * previous call to this function was successful, and path is the same, the 
 * function will simply return the previously found value.
 *
 * User functions should not free this structure.
 */

NSAPI_PUBLIC struct stat *request_stat_path(char *path, Request *rq);


/*
 * request_handle handles one request from the session's inbuf.
 */

NSAPI_PUBLIC void request_handle(Session *sn);


/* XXXrobm temporary compatibility */
#define request_uri2path servact_uri2path
#define request_pathchecks servact_pathchecks
#define request_fileinfo servact_fileinfo
#define request_service servact_service

#define request_handle_processed servact_handle_processed
#define request_translate_uri servact_translate_uri
#define request_finderror servact_finderror

#endif
