/*
 * Copyright (c) 1994, 1995.  Netscape Communications Corporation.  All
 * rights reserved.
 * 
 * Use of this software is governed by the terms of the license agreement for
 * the Netscape FastTrack or Netscape Enterprise Server between the
 * parties.
 */


/* ------------------------------------------------------------------------ */


/*
 * util.h: A hodge podge of utility functions and standard functions which 
 *         are unavailable on certain systems
 * 
 * Rob McCool
 */


#ifndef HTTPD_UTIL_H
#define HTTPD_UTIL_H

#include "buffer.h"    /* filebuf for getline */

#include <time.h>      /* struct tm */


/* ------------------------------ Prototypes ------------------------------ */


/*
 * getline scans in buf until it finds a LF or CRLF, storing the string in
 * l. It will terminate the string and return:
 * 
 *  0 when done, with the scanned line (minus CR or LF) in l
 *  1 upon EOF, with the scanned line (minus CR or LF) in l
 * -1 on error with the error description in l (uses lineno for information)
 */

NSAPI_PUBLIC
int util_getline(filebuffer *buf, int lineno, int maxlen, char *l);


/*
 * can_exec returns 1 if you can execute the file described by finfo, and 
 * 0 if you can't.
 */

#ifdef XP_UNIX
#include <sys/stat.h>
#include <sys/types.h>

NSAPI_PUBLIC int util_can_exec(struct stat *finfo, uid_t uid, gid_t gid);

#endif /* XP_UNIX */
/*
 * env_create creates a new environment with the given env, with n new
 * entries, and places the current position that you should add your
 * entries with at pos.
 * 
 * If env is NULL, it will allocate a new one. If not, it will reallocate
 * that one.
 */

NSAPI_PUBLIC char **util_env_create(char **env, int n, int *pos);

/*
 * util_env_str allocates a string from the given name and value and
 * returns it. It does not check for things like = signs in name.
 */

NSAPI_PUBLIC char *util_env_str(char *name, char *value);

/*
 * env_replace replaces the occurrence of the given variable with the 
 * value you give.
 */

NSAPI_PUBLIC void util_env_replace(char **env, char *name, char *value);

/*
 * util_env_free frees an environment.
 */

NSAPI_PUBLIC void util_env_free(char **env);

/*
 * util_env_copy copies an env
 */
NSAPI_PUBLIC char **util_env_copy(char **src, char **dst);


/*
 * util_env_find looks through env for the named string. Returns the
 * corresponding value if the named string is found, or NULL if not.
 */
NSAPI_PUBLIC char *util_env_find(char **env, char *name);


/*
 * hostname gets the local hostname. Returns NULL if it can't find a FQDN.
 * You are free to realloc or free this string.
 */

NSAPI_PUBLIC char *util_hostname(void);


/*
 * chdir2path changes the current directory to the one that the file
 * path is in. path should point to a file. Caveat: path must be a writable
 * string. It won't get modified permanently.
 */

NSAPI_PUBLIC int util_chdir2path(char *path);

/*
 * is_mozilla checks if the given user-agent is mozilla, of at least
 * the given major and minor revisions. These are strings to avoid 
 * ambiguities like 1.56 > 1.5
 */

NSAPI_PUBLIC int util_is_mozilla(char *ua, char *major, char *minor);

/*
 * is_url will return 1 if the given string seems to be a URL, or will 
 * return 0 otherwise. 
 * 
 * Because of stupid news URLs, this will return 1 if the string has 
 * all alphabetic characters up to the first colon and will not check for 
 * the double slash.
 */

NSAPI_PUBLIC int util_is_url(char *url);

/*
 * util_later_than checks the date in the string ims, and if that date is 
 * later than or equal to the one in the tm struct lms, then it returns 1.
 *
 * util_time_equal is above, but checks for exact equality.
 *
 * Handles RFC 822, 850, and ctime formats.
 */

NSAPI_PUBLIC int util_later_than(struct tm *lms, char *ims);
NSAPI_PUBLIC int util_time_equal(struct tm *lms, char *ims);

/* 
 * util_str_time_equal checks the character-string dates are equal.
 * Supports rfc1123 and rfc850 formats.  t1 must be rfc1123
 * Returns 0 if equal, -1 otherwise
 */
NSAPI_PUBLIC int util_str_time_equal(char *t1, char *t2);

/*
 * util_uri_is_evil returns 1 if a URL has ../ or // in it.
 */
NSAPI_PUBLIC int util_uri_is_evil(char *t);

/*
 * util_uri_parse gets rid of /../, /./, and //.
 * 
 * Assumes that either the string starts with a /, or the string will
 * not .. right off of its beginning.  As such, ../foo.gif will
 * not be changed, although /../foo.gif will become /foo.gif.
 */

NSAPI_PUBLIC void util_uri_parse(char *uri);

/*
 * util_uri_unescape unescapes the given URI in place (% conversions only).
 */

NSAPI_PUBLIC void util_uri_unescape(char *s);

/*
 * util_uri_escape escapes any nasty chars in s and copies the string into d.
 * If d is NULL, it will allocate and return a properly sized string.
 * Warning: does not check bounds on a given d.
 *
 * util_url_escape does the same thing but does it for a url, i.e. ?:+ is 
 * not escaped.
 */

NSAPI_PUBLIC char *util_uri_escape(char *d, char *s);
NSAPI_PUBLIC char *util_url_escape(char *d, char *s);

/*
 * util_sh_escape places a \ in front of any shell-special characters.
 * Returns a newly-allocated copy of the string.
 */

NSAPI_PUBLIC char *util_sh_escape(char *s);


/*
 * util_mime_separator generates a new MIME separator into the given buffer.
 * The buffer should be more than 4 + 3*10 + 1 bytes long. A CRLF is prepended
 * to the beginning of the string, along with two dashes. The string is null
 * terminated, with no CRLF. The intent is that you create your content-type
 * header by accessing &sep[4], and afterwards print sep followed by CRLF
 * for message boundaries.
 *
 * Returns the length of the string.
 */
NSAPI_PUBLIC int util_mime_separator(char *sep);

/*
 * util_itoa converts the given integer to a string into a.
 */

NSAPI_PUBLIC int util_itoa(int i, char *a);

/*
 * util_vsprintf and util_sprintf are simplified clones of the System V 
 * vsprintf and sprintf routines.
 * 
 * Returns the number of characters printed. Only handles %d and %s,
 * does not handle any width or precision.
 */

#include <stdarg.h>

NSAPI_PUBLIC int util_vsprintf(char *s, register char *fmt, va_list args);
NSAPI_PUBLIC int util_sprintf(char *s, char *fmt, ...);

/* These routines perform bounds checks. */
NSAPI_PUBLIC int util_vsnprintf(char *s, int n, register char *fmt, 
                                va_list args);
NSAPI_PUBLIC int util_snprintf(char *s, int n, char *fmt, ...);

#ifdef XP_WIN32
/* util_delete_directory()
 * This routine deletes all the files in a directory.  If delete_directory is
 * TRUE it will also delete the directory itself.
 */
NSAPI_PUBLIC VOID util_delete_directory(char *FileName, BOOL delete_directory);
#endif

/* util_strftime()
 * Thread safe version of strftime.
 * No bounds checking is done s.  t must be a valid tm structure.
 */
NSAPI_PUBLIC int util_strftime(char *s, const char *format, const struct tm *t);

#endif
