/*
 * Copyright (c) 1994, 1995.  Netscape Communications Corporation.  All
 * rights reserved.
 * 
 * Use of this software is governed by the terms of the license agreement for
 * the Netscape FastTrack or Netscape Enterprise Server between the
 * parties.
 */


/* ------------------------------------------------------------------------ */


/*
 * crit.h: Critical section abstraction. Used in threaded servers to protect
 *         areas where two threads can interfere with each other.
 *
 *         Condvars are condition variables that are used for thread-thread 
 *         synchronization.
 * 
 * Rob McCool
 */

#ifndef CRIT_H
#define CRIT_H

#ifdef __cplusplus
extern "C" {
#endif


typedef void* CRITICAL;


/*
 * crit_init creates and returns a new critical section variable. At the 
 * time of creation no one has entered it.
 */
NSAPI_PUBLIC CRITICAL crit_init(void);

/*
 * crit_enter enters a critical section. If someone is already in the
 * section, the calling thread is blocked until that thread exits.
 */
NSAPI_PUBLIC void crit_enter(CRITICAL id);


/*
 * crit_exit exits a critical section. If another thread is blocked waiting
 * to enter, it will be unblocked and given ownership of the section.
 */
NSAPI_PUBLIC void crit_exit(CRITICAL id);


/*
 * crit_terminate removes a previously allocated critical section variable.
 */
NSAPI_PUBLIC void crit_terminate(CRITICAL id);




typedef void* CONDVAR;


/*
 * condvar_init initializes and returns a new condition variable. You 
 * must provide a critical section to be associated with this condition 
 * variable.
 */
NSAPI_PUBLIC CONDVAR condvar_init(CRITICAL id);


/*
 * condvar_wait blocks on the given condition variable. The calling thread
 * will be blocked until another thread calls condvar_notify on this variable.
 * The caller must have entered the critical section associated with this
 * condition variable prior to waiting for it.
 */
NSAPI_PUBLIC void condvar_wait(CONDVAR cv);


/*
 * condvar_notify awakens any threads blocked on the given condition
 * variable. The caller must have entered the critical section associated
 * with this variable first.
 */
NSAPI_PUBLIC void condvar_notify(CONDVAR cv);


/*
 * condvar_terminate frees the given previously allocated condition variable
 */
NSAPI_PUBLIC void condvar_terminate(CONDVAR cv);




typedef void *COUNTING_SEMAPHORE;

/*
 * Create a counting semaphore.  
 * Return non-zero on success, 0 on failure.
 */
NSAPI_PUBLIC COUNTING_SEMAPHORE cs_init(int initial_count);

/*
 * Destroy a counting semaphore 
 */
NSAPI_PUBLIC void cs_terminate(COUNTING_SEMAPHORE csp);

/*
 * Wait to "enter" the semaphore.
 * Return 0 on success, -1 on failure.
 */
NSAPI_PUBLIC int cs_wait(COUNTING_SEMAPHORE csp);

/*
 * Enter the semaphore if the count is > 0.  Otherwise return -1.
 *
 */
NSAPI_PUBLIC int cs_trywait(COUNTING_SEMAPHORE csp);

/*
 * Release the semaphore- allowing a thread to enter.
 * Return 0 on success, -1 on failure.
 */
NSAPI_PUBLIC int cs_release(COUNTING_SEMAPHORE csp);

#ifdef __cplusplus
}
#endif

#endif
