static const char file_id[] = "Thoranalyzer.cc";

////////////////////////////////////////////////////////////////////////////
// Version identification:
// @(#)Thoranalyzer.cc	2.10	11/26/92
//
// Copyright (c) 1990, 1991, 1992 The Regents of the University of California.
// All rights reserved.
// See the file ~ptolemy/copyright for copyright notice,
// limitation of liability, and disclaimer of warranty provisions.
//
// Programmer:  Anders Wass
// Date of creation: 2/11/91
// Description:
//	This is the star used by the thor domain to display values in
// different nodes selected by the user. It is not a standard model star so
// it should go with the kernel files.
//
////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////
//
//  NOTE: This file is edited after the conversion from chdl to C++.
//  Every changes made to the analyzer star should be made to this C++ file
//  from now on. This file is no longer compatible with its predecessor
//  file. Among major changes are new states that makes it possible to
//  control behavior of old analyzer windows, and adding signal group names.
//	/AW
//
////////////////////////////////////////////////////////////////////////////


#include "ThorStar.h"
#include "ThorState.h"
#include "libarp.h"
#include "KnownBlock.h"
#include "StringState.h"
#include "miscFuncs.h"
#include "StrBuf.h"
#include <signal.h>

// The hack below was done to be able to debug this module
// without building a complete pTolemy kernel (including pThor)
// #ifdef current_time
// #undef current_time
// #endif
// #define current_time (getCurTime())
// end hack !

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <ctype.h>
extern "C"
{
#include "history.h"
#include "ipc.h"
int socketpair(int,int,int,int[]);
}

// this must be done this way since the vfork doesn't saves the registers
// which mess it up for the parent. The `#pragma unknown_control_flow(vfork)'
// doesn't work since g++ doesn't recognize this sun4 compiler key.
#ifdef sparc
#define vfork fork
#endif

// these two functions are converted to C++ ...
extern void csimenv(char* &lib, char* &bin, char* &inc);
extern const char* sperror(const char* =0, const char* =0, const char* =0);
extern void clrErrNo();
extern int expandSig(const char*,char*&);

#define ANALYZER_PROG "ANALYZER_X11"	/* child process */


	/* Logic analyzer monitor module */

const char *star_nm_Thoranalyzer = "Thoranalyzer";

class Thoranalyzer : public ThorStar
{
private:
    MultiInThorPort	Inputs;
    int			pid; // This shouldn't be touched from the outside
    int			commLink; // This shouldn't be touched from the outside
    IntState		stop_run;
    IntState		quit;
    IntState		ThorEnv;
    StringState		busses;	// holds the bus definitions
    const char*		bus_str; // private string copy of above

public:
    int			unknown_names;
    int			isItMonitor() { return TRUE; }
    Thoranalyzer();		// Constructor
    ~Thoranalyzer();		// Destructor
    void		go();
    void		setup();
    void		wrapup();
    Block*	makeNew() const { LOG_NEW; return new Thoranalyzer; }
//
//	These functions should be deleted when they are properly submitted
//	in the final thor system.
//
    const char*		iobname(int porttype, int index);
    const char*		mname() { return name(); }
};

Thoranalyzer :: Thoranalyzer ()	// Constructor
{
    setDescriptor(
"    This star communicates with the analyzer X window. The values of this\n"
"    star's inputs is sent to the X analyzer window for display. Each trace\n"
"    will be labeled with with one of the star's port names that is connected\n"
"    to this node like `starname.portname'. The behavior of the analyzer is\n"
"    controlled by commands in the analyzer window.\n\n"
"    State descriptions:\n\n"
"    stop_run:  This is a flag which stops and starts the scheduler. It\n"
"	is controled from the analyzer window with the stop/cont commands.\n\n"
"    quit:  This is a flag that determines the behavior of the analyzer window\n"
"	when the scheduler is restarted (= new run command) or when the\n"
"	wrapup command is executed. quit = 0 means let the window live.\n"
"	quit != 0 means kill the window when restarted or at wrapup.\n"
"    ThorEnv:  This flag controls how the executable window program is\n"
"	searched. If ThorEnv == 0 the PATH will be searched before the THOR\n"
"	environmen. If ThorEnv != 0, PATH will be searched if THOR environment\n"
"	fails.\n"
"    busses:  This state contains a list of names of all the signals to be\n"
"	shown in the analyzer window. Its purpose is to be able to group signals\n"
"	into busses in the analyzer window. Signal names are separated by white\n"
"	spaces. Each signal that is supposed to be grouped must have an index\n"
"	within `[ ]'. Index ranging is allowed and index must be a positive\n"
"	number. If only one index in a range is specified, the other will be\n"
"	defaulted to 0. The total number of individual signals in the list must\n"
"	equal the number of inputs connected in order to prevent an error from\n"
"	the analyzer window.\n"
"	The order of how the input nodes of the analyzer is connected is.\n"
"	significant. First analyzer input node connected will have the first\n"
"	name in the list. (Numbered nodes will make an exception.)\n"
"	Example: Analyzer has 5 inputs and busses defined to:\n"
"		\"clk out[0] out[1] out[2] out[3]\", the analyzer window will\n"
"	hold two waveforms, clk on top and the bus value of out[0-3] below.\n"
"	The name list in the example is identical to the following list:\n"
"	\"clk out[0-3]\"  or  \"clk out[-3]\".\n"
"	If busses is set, then the default naming of signals is prevented.\n");

    //	Ports:

    addPort(Inputs.setPort("Inputs",this));

    //	States:

    addState(stop_run.setState("stop_run",this,"0"));
    addState(quit.setState("quit",this,"0"));
    addState(ThorEnv.setState("ThorEnv",this,"0"));
    addState(busses.setState("busses",this,""));
    unknown_names = 0;
    pid = 0;
    commLink = 0;
    bus_str = NULL;
}

Thoranalyzer :: ~Thoranalyzer () { wrapup(); } // Destructor

char ThorAnalyzerBuf[BUFSIZ];	// used by all analyzer/banalyzer stars ...

const char* Thoranalyzer :: iobname(int /* porttype */, int index)
{
    if (index < 1 || index > numberPorts())
	sprintf(ThorAnalyzerBuf,"Unkown_%d",unknown_names++);
    else
	sprintf(ThorAnalyzerBuf,"%s.%s",
		Inputs[index-1].far()->parent()->name(),
		Inputs[index-1].far()->name());
    return ThorAnalyzerBuf;
}

const char* getAnalyzerPath(const char* pname, int envFirst = 0)
{
    if (!pname) return NULL;
    char cmd[512];
    char *csim_lib, *csim_bin, *csim_inc;

    clrErrNo();
    if (envFirst) {
	csimenv(csim_lib,csim_bin,csim_inc);
	sprintf(cmd, "%s/%s", csim_bin, pname);
	if (!access(cmd,1))
	    return savestring(cmd);
    }

    char *s = getenv("PATH");
    if (s) {
	char *path = savestring(s);
	for (s=path; *s; s++)
	    if (*s == ':') *s = ' ';
	StrBuf sbuf = path;
	const char* spath;
	while (spath = sbuf.getword()) {
	    sprintf(cmd, "%s/%s", spath, pname);
	    if (!access(cmd,1))
		return savestring(cmd);
	}
    }

    if (!envFirst) {
	// no path or not in path, try thor environment ...
	csimenv(csim_lib,csim_bin,csim_inc);
	sprintf(cmd, "%s/%s", csim_bin, pname);
	if (!access(cmd,1))
	    return savestring(cmd);
    }
    return NULL;	// can't find executeable path to pname
}

void Thoranalyzer :: setup()
{
    clrErrNo();
    if (pid != 0)
	wrapup(); // close commLink and terminate old window if quit != 0
    if (pid == 0) {			/* Initialization Procedure */
        int		channel[2];	/* IPC channel with child process */
	int		len;
	register int	i;

	if (IN_COUNT > MAXINPUTS) {
	    StringList msg = "analyzer: Too many inputs (max. ";
	    msg += MAXINPUTS;
	    msg += ")";
	    Error :: abortRun(*this,msg);
	    return;
	}

	if (socketpair(AF_UNIX, SOCK_STREAM, 0, channel) < 0) {
	    Error :: abortRun(*this,sperror("analyzer_socket",""));
	    return;
	}

	if ((pid = vfork()) == -1) {
	    Error :: abortRun(*this,sperror("analyzer vfork",""));
	    return;
	}

	if (pid == 0) {		/* child process */
	    const char *cmd, *arg0, *arg1, *arg2;
	    char arg3[8], arg4[8];

	    close(channel[1]);	/* child will use socket 0 */
#ifndef DEBUG_AN
	    close(0);			/* close stdin */
	    close(1);			/* close stdout */
#endif

	    arg0 = ANALYZER_PROG;
	    arg1 = mname();
	    arg2 = "Ptolemy/Thor";
	    sprintf(arg3, "%d", IN_COUNT);
	    sprintf(arg4, "%d", channel[0]);
	    if (!(cmd = getAnalyzerPath(arg0,ThorEnv))) {
		Error :: warn(*this, "Can't find executable \"", arg0,
			      "\"\nPlease, check your path (or environment).");
		goto exec_failed;
	    }
	    clrErrNo();
#ifdef DEBUG_AN
	    fprintf(stderr,"%s%s\n%s %s %s %s %s %s\n", name(),
		    "*child: ready to execute command:",
		    cmd, arg0, arg1, arg2, arg3, arg4);
#endif
	    execl(cmd, arg0, arg1, arg2, arg3, arg4, NULL);
	    Error :: warn(*this, arg0, ": Execution failed: ", sperror());
	exec_failed:
	    close(channel[0]);	/* let the parent die */
	    _exit(1);
	}

	/* parent process */

	
	close(channel[0]);
	commLink = channel[1];
	stop_run = 0;
	bus_str = busses;
	while (bus_str && isspace(*bus_str)) bus_str++;

	if (bus_str && *bus_str) {
	    char *exp_str;
	    switch (expandSig(bus_str,exp_str)) {
	    case -3:
		Error :: abortRun(*this,"Syntax error in string state busses.\n",
				  "Can't do character match. Need a full name.");
		return;
	    case -2:
		Error :: abortRun(*this,"Syntax error in string state busses.");
		return;
	    case -1:
		Error :: abortRun(*this,"Bad contents in string state busses.");
		return;
	    case 0:
		break;
	    }
	    StrBuf busbuffer = exp_str;
	    const char* str;
	    while (str = busbuffer.getword()) {
		len = strlen(str) + 1;
		if (ipc_Send(commLink, str, len, ADDSIGNAL) != IPC_OK) {
		    Error :: abortRun(*this,sperror(mname(),":can't send signal."));
		    return;
		}
	    }
	}
	else
	    for (i = 0; i < IN_COUNT; i++) {
		const char* signame = iobname(CINPUT, i + 1);
		len = strlen(signame) + 1;
		if (ipc_Send(commLink, signame, len, ADDSIGNAL) != IPC_OK) {
		    Error :: abortRun(*this,
				      sperror("analyzer",": can't send signal."));
		    return;
		}
	    }
    }				// end initialization procedure ...
}


void Thoranalyzer :: go()
{
    CommHist	 newHist;
    MsgType	 msg;
    int		 len;
    char*	 rcvMsgP;
    register int i;

    clrErrNo();

    /* sleep until we get another signal from ANALYZER to wake us */
    while (stop_run) {
        msg = ipc_Receive(commLink, &rcvMsgP, &len);
	if (msg == CONT)
	    stop_run = 0;
	else if (msg == IPC_ERR) {
	    Error :: warn(*this,"analyzer window is gone, continuing ...");
	    stop_run = 0;
	    commLink = -1;
	}
    }

    /* Send current inputs to child process, encoded as chars 0..4 */

    if (commLink >= 0) {		/* child still alive */
	newHist.time = current_time;
	int ival;
	for (i = 0; i < IN_COUNT; i++) {
	    ival = newHist.levels[i] = Inputs[i];
#ifdef DEBUG_AN
	    char* cval;
	    switch (ival) {
	    case ONE:
		cval = "ONE"; break;
	    case ZERO:
		cval = "ZERO"; break;
	    case FLOAT:
		cval = "FLOAT"; break;
	    default:
		fprintf(stderr,"*> %s*parent: Unexpected value %d.\n",
			name(),ival);
	    case UNDEF:
		cval = "UNDEF"; break;
	    }
	    fprintf(stderr,"*> %s*parent: putting %s into newHist.levels[%d].\n",
		    name(), cval, i );
#endif
	}

        i = sizeof(CommHist) - MAXINPUTS + IN_COUNT;
	if (ipc_Send(commLink, (char*) &newHist, i, SIGVALUES) != IPC_OK) {
	    Error :: warn(*this,"analyzer window is gone, continuing ...");
	    commLink = -1;		/* don't try again */
	    return;
	}
	msg = ipc_Receive(commLink, &rcvMsgP, &len);
        if (msg == STOP)
	    stop_run = 1;
	else if (msg == IPC_ERR) {
	    Error :: error(*this,sperror(mname(),"\n"),
			   "Communication error, continue without analyzer ...");
	    commLink = -1;
	}
    }

    return;
}

void Thoranalyzer :: wrapup ()
{
    if (pid) {
	close(commLink);
	commLink = -1;
	if (quit)
	    kill(pid,SIGTERM);
	pid = 0;
    }
    clrErrNo();			// ignore error message
}

static Thoranalyzer proto;
static KnownBlock entry(proto,"analyzer");

int Force_anaLink()		// if referenced outside this
{ return 1; }			// module, it forces this module's
				// code to be linked in ...

//	<EOF: Thoranalyze.cc>
