/**************************************************************************
 * Version identification:
 * @(#)pass1Syn.y	1.4	7/16/92
 *
 *  Copyright (c) 1990,1991 The Regents of the University of California.
 *                        All Rights Reserved.
 *
 *  Programmer:  Anders Wass
 *  Date of creation: 11/02/91
 *
 *
 *  Description:
 *
 * This is a syntactic analyzer of Thor model files. The result of the
 * analyze will be a C++-file, ready to compile and link into pTolemy and its
 * pThor domain. Since a syntactic analyze is done, it will be more fretful
 * about the way the models are declared than epp used to be. Only one model
 * per file is allowed. The behaviour of the analyzer is this:
 *
 * Everything appering before `MODEL()' is copied as is to ccfile.
 * Everything appering after `MODEL()' is copied to file cctmp with one
 * exception: Everything that is declared as an IN_LIST, OUT_LIST, BI_LIST,
 * ST_LIST, and INIT -- ENDINIT is extracted and temporary stored for later
 * use. When all the declarations are read, they are written to the ccfile
 * as C++ class declaration and constructor. Then the code within the
 * INIT/ENDINIT marks are written to the `start()' function. And at the end,
 * the content of the cctmp is appended to the ccfile to form the `go()'
 * function.
 *
 *	-- Anders Wass  1990/11/02
 *
 *******************************************************************************/


%{
#ifdef _IBMR2
#pragma alloca
#endif

#include <stdio.h>
#include <string.h>
#include "comdefs.h"

extern char yytext[];
extern int  yylineno;
extern int  yylex();

DECSTRUCT *DeclHead=NULL;
SIGSTRUCT *TmpSigs=NULL;
VARLISTSTRUCT VarList = {NULL,NULL,NULL};
extern ICL *iclptr;
extern char *VGrpSize;

FILE *ccfile = NULL, *modfile = NULL, *cctmp = NULL;
FILE *p1lexout=NULL;
int NewLine, ModelType, IsThorModel=0;
NAMETABLE *vectorTable = 0;
%}

%union {
    char	*str;
    int		 ival;
    SIGSTRUCT	*sprop;
    DECSTRUCT	*dprop;
}


%token	MODEL WORD IN_LIST OUT_LIST BI_LIST ST_LIST ENDLIST
	INT SIG TSIG GRP TGRP VGRP BIGEND LITTLEEND CVAR
	PRIVATE PROTECTED PUBLIC

    /* GRP is returned when either GRP, BUS, or REG is encounted.
     * VGRP is returned when either VGRP, VBUS or VREG is
     * encounted. Exists TBUS and TREG ? If so, the same rule
     * should be valid for those too. CVAR is used to declare
     * variables in the C++ class generated from the model.
     * The property of CVAR can be either PRIVATE, PROTECTED,
     * or PUBLIC. PRIVATE is the default.
     */

%type <ival>	INT IN_LIST OUT_LIST BI_LIST ST_LIST ENDLIST
%type <ival>	SIG TSIG GRP TGRP VGRP BIGEND LITTLEEND CVAR
%type <ival>	PRIVATE PROTECTED PUBLIC
%type <str>	WORD
%type <dprop>	list_decls list_decl
%type <sprop>	signals signal temp_sigs classvar

%start begin

%%

begin:			/* empty */
		|	mod_id mod_body
				{   make_Gofcn(cctmp,ccfile);   }
		;

mod_id:			MODEL '(' WORD ')' '{'
				{ /* prepend `Thor' to modelname ... */
				    ClassName = (char*) malloc(strlen($3) + 5);
				    sprintf(ClassName,"Thor%s",$3);
				    /* Make ModelName a substring of ClassName */
				    ModelName = ClassName + 4;
				    free($3); /* string from lex no longer needed */
				    IsThorModel=1;
				    whereIam(cctmp,yylineno,FileIParse);
				}
		;

mod_body:		/* empty */
				{
				    printHeader(DeclHead,TmpSigs,&VarList,
						iclptr,ccfile,ModelType);
				}
		|	list_decls
				{
				    printHeader(DeclHead,TmpSigs,&VarList,
						iclptr,ccfile,ModelType);
				}
		;

list_decls:		list_decl
				{   $$ = DeclHead = $1;   }
		|	list_decls list_decl
				{   $$ = DeclHead = app_declists($2,$1);   }
		;

list_decl:		IN_LIST  signals ENDLIST ';'
				{   $$ = newdeclist($2,IN_LIST,$1,$3);   }
		|	OUT_LIST signals ENDLIST ';'
				{   $$ = newdeclist($2,OUT_LIST,$1,$3);   }
		|	BI_LIST  signals ENDLIST ';'
				{   $$ = newdeclist($2,BI_LIST,$1,$3);   }
		|	ST_LIST  signals ENDLIST ';'
				{   $$ = newdeclist($2,ST_LIST,$1,$3);   }
		|	temp_sigs
				{   TmpSigs = append_sigs($1,TmpSigs);
				    $$ = NULL;
				}
		|       classvar {  $$ = NULL;  }
		;

temp_sigs:		TSIG '(' WORD ')' ';'
				{   $$ = newsig($3,TSIG,"1",-1,$1);
				    free($3);
				}
		|	TGRP '(' WORD ',' WORD ')' ';'
				{   $$ = newsig($3,TGRP,$5,BIGEND,$1);
				    vectorTable = insertname($3,$5,BIG_END,vectorTable,$1);
				    free($3); free($5);
				}
		|	TGRP '(' WORD ',' WORD ',' BIGEND ')' ';'
				{   $$ = newsig($3,TGRP,$5,BIGEND,$1);
				    vectorTable = insertname($3,$5,BIG_END,vectorTable,$1);
				    free($3); free($5);
				}
		|	TGRP '(' WORD ',' WORD ',' LITTLEEND ')' ';'
				{   $$ = newsig($3,TGRP,$5,LITTLEEND,$1);
				    vectorTable = insertname($3,$5,LIT_END,vectorTable,$1);
				    free($3); free($5);
				}
		;

signals:		signal
		|	signals signal
				{   append_sigs($2,$1);   }
		;

signal:			SIG '(' WORD ')' ';'
				{   $$ = newsig($3,SIG,"1",-1,$1);
				    free($3);
				}
		|	TSIG '(' WORD ')' ';'
				{   $$ = NULL;
				    TmpSigs = append_sigs(newsig($3,TSIG,"1",-1,$1),TmpSigs);
				    free($3);
				}
		|	GRP '(' WORD ',' WORD ')' ';'
				{   $$ = newsig($3,GRP,$5,BIGEND,$1);
				    vectorTable = insertname($3,$5,BIG_END,vectorTable,$1);
				    free($3); free($5);
				}
		|	GRP '(' WORD ',' WORD ',' BIGEND ')' ';'
				{   $$ = newsig($3,GRP,$5,BIGEND,$1);
				    vectorTable = insertname($3,$5,BIG_END,vectorTable,$1);
				    free($3); free($5);
				}
		|	GRP '(' WORD ',' WORD ',' LITTLEEND ')' ';'
				{   $$ = newsig($3,GRP,$5,LITTLEEND,$1);
				    vectorTable = insertname($3,$5,LIT_END,vectorTable,$1);
				    free($3); free($5);
				}
		|	TGRP '(' WORD ',' WORD ')' ';'
				{   $$ = NULL;
				    TmpSigs = append_sigs(newsig($3,TGRP,$5,BIGEND,$1),TmpSigs);
				    vectorTable = insertname($3,$5,BIG_END,vectorTable,$1);
				    free($3); free($5);
				}
		|	TGRP '(' WORD ',' WORD ',' BIGEND ')' ';'
				{   $$ = NULL;
				    TmpSigs = append_sigs(newsig($3,TGRP,$5,BIGEND,$1),TmpSigs);
				    vectorTable = insertname($3,$5,BIG_END,vectorTable,$1);
				    free($3); free($5);
				}
		|	TGRP '(' WORD ',' WORD ',' LITTLEEND ')' ';'
				{   $$ = NULL;
				    TmpSigs = append_sigs(newsig($3,TGRP,$5,LITTLEEND,$1),TmpSigs);
				    vectorTable = insertname($3,$5,LIT_END,vectorTable,$1);
				    free($3); free($5);
				}
		|	VGRP '(' WORD ')' ';'
				{   char *sz = (char *)
					malloc(strlen($3)+strlen(VGrpSize)+1);
				    sprintf(sz,VGrpSize,$3);
				    $$ = newsig($3,VGRP,sz,BIGEND,$1);
				    vectorTable = insertname($3,sz,BIG_END,vectorTable,$1);
				    free($3); free(sz);
				}
		|	VGRP '(' WORD ',' BIGEND ')' ';'
				{   char *sz = (char *)
					malloc(strlen($3)+strlen(VGrpSize)+1);
				    sprintf(sz,VGrpSize,$3);
				    $$ = newsig($3,VGRP,sz,BIGEND,$1);
				    vectorTable = insertname($3,sz,BIG_END,vectorTable,$1);
				    free($3); free(sz);
				}
		|	VGRP '(' WORD ',' LITTLEEND ')' ';'
				{   char *sz = (char *)
					malloc(strlen($3)+strlen(VGrpSize)+1);
				    sprintf(sz,VGrpSize,$3);
				    $$ = newsig($3,VGRP,sz,LITTLEEND,$1);
				    vectorTable = insertname($3,sz,LIT_END,vectorTable,$1);
				    free($3); free(sz);
				}
		;


classvar:		CVAR '(' WORD ',' WORD ')' ';'
				{
				    VarList.private =
					append_vars(newvar($3,$5,$1),
						    VarList.private);
				}
		|
			CVAR '(' WORD ',' WORD ',' PRIVATE ')' ';'
				{
				    VarList.private =
					append_vars(newvar($3,$5,$1),
						    VarList.private);
				}
		|
			CVAR '(' WORD ',' WORD ',' PROTECTED ')' ';'
				{
				    VarList.protected =
					append_vars(newvar($3,$5,$1),
						    VarList.protected);
				}
		|
			CVAR '(' WORD ',' WORD ',' PUBLIC ')' ';'
				{
				    VarList.public =
					append_vars(newvar($3,$5,$1),
						    VarList.public);
				}
		;

%%

static char identification[] = "@(#)pass1Syn.y	1.4\t7/16/92  (c) UC Berkeley";

yyerror(s)
    char *s;
{
    errmsg("s","\"%s\", line %3d: %s at or near \"%s\".\n",
	   FileIParse, yylineno, s, yytext);
}

	/*
	 * This function is called to parse the thor model file and create a
	 * C++ file when done. The parameters are;
	 *	infile:      file pointer to file to be parsed.
	 *	infilename:  name of file to be parsed.
	 *	outfile:     file pointer to file to be parsed.
	 *	tfile:       file pointer to temporary file.
	 *	TypeOfModel: The kind of model it should response as.
	 *		     If passed value = 0, then the combination of
	 *		     in and outputs will determine the kind of
	 *		     model to use.
	 *	vecTbl:      the address to a pointer to a table of all
	 *		     declared thor vectors and their sizes.
	 *
	 */

int pass1(infile,infilename,outfile,tfile,TypeOfModel,vecTbl)
    FILE *infile, *outfile, *tfile;
    char *infilename;
    int TypeOfModel;
    NAMETABLE **vecTbl;
{
    int rval = 0;
    vectorTable = 0;
    IsThorModel=0;
    VarList.private = VarList.protected = VarList.public = NULL;
    if (!infile) {
	errmsg("f","pass1: no infile opened!\n");
	return 3;
    }
    if (!outfile) {
	errmsg("f","pass1: no outfile opened!\n");
	return 4;
    }
    if (!tfile) {
	errmsg("f","pass1: no tempfile opened!\n");
	return 5;
    }
    modfile = infile; ccfile = outfile; cctmp = tfile;
    p1lexout = ccfile;
    if (infilename)
	FileIParse = infilename;
    else
	FileIParse = "(stdin)";
    ModelType = TypeOfModel;
    if (yyparse())
	rval = 2;
    if (!IsThorModel) {
	errmsg("s","\"%s\", line %3d: warning: No `MODEL()' declaration found.\n",
	       FileIParse, yylineno-1);
	rval = 1;
    }
    ccfile = modfile = cctmp = NULL; FileIParse = NULL;
    iclptr = freeIclList(iclptr);
    if (debug) {
	char *descbuf;
	errmsg("s","Vector declaration list:\n");
	printDecList(DeclHead);
	errmsg("s","Name table contents:\n");
	printNameTbl(vectorTable);
	errmsg("s","\nDeclared C++ variable:\n");
	printVarList(&VarList);
	errmsg("s","\n");
	if (descbuf = getdescriptor()) {
	    errmsg("s","Descriptor string: (len = %d, bufsize = %d)\n",
		   strlen(descbuf), 2*BUFSIZ);
	    errmsg("s",descbuf);
	}
    }
    DeclHead = freeDecList(DeclHead);
    TmpSigs = freeSigList(TmpSigs);
    VarList.private = freeVarList(VarList.private);
    VarList.protected = freeVarList(VarList.protected);
    VarList.public = freeVarList(VarList.public);
    *vecTbl = vectorTable;
    return rval;
}
