static const char file_id[] = "ThorScheduler.cc";
/**************************************************************************
Version identification:
@(#)ThorScheduler.cc	1.25 11/25/92

Copyright (c) 1990, 1991, 1992 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY 
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES 
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF 
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF 
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.
							COPYRIGHTENDKEY

 Programmer:  Seungjun Lee
 Date of creation: 6/20/90

**************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif
#include "EventHorizon.h"
#include "ThorScheduler.h"
#include "GalIter.h"

extern const char ThordomainName[];

// currentElem is declared as static to support mname()
static ThorStar* currentElem = 0;


void
TimeSlot :: stimOutputs(ElemList* nodeList)
{

     // Fetch output changes scheduled earlier
     for (OutChange* change = (OutChange*) changes.takeFromFront(); change;
               change = (OutChange*) changes.takeFromFront()){
          ThorPortHole& p = *(ThorPortHole*)change->pin;
          p.getBuffParticle() << change->val;
          
          // Schedule the node?
          ThorNode* np = p.node();

          if (np->isItPersistent()   // Not on wormhole boundary?
          && !np->isNodeScheduled()  // Not scheduled yet?
          && notEqual(p, *np)){      // Is it getting new data?
               nodeList->append(np);
               np->setScheduled();
          }

          // Free memory
          LOG_DEL; delete change;
     }
}

void
TimeSlot :: callMonitors()
{
     // Fire monitors that have been scheduled at the time-slot
     for (currentElem = (ThorStar*) mon.takeFromFront(); currentElem;
               currentElem = (ThorStar*) mon.takeFromFront()){
	  if (!currentElem->run()) return;
          currentElem->resetScheduled();
     }
}


TimeWheel :: TimeWheel(int i)
{
     size = i;
     current = -1;
     LOG_NEW; buffer = new DoubleLinkList*[size];
     for (int j = 0; j < size; j++){
          LOG_NEW; buffer[j] = new DoubleLinkList;
          buffer[j]->initialize();
     }
}

void
TimeWheel :: reset()
{
     current = -1;
     for (int j = 0; j < size; j++)
          buffer[j]->initialize();
}

#define INFINITY 1000000

TimeSlot*
TimeWheel :: get(TimeType& currTime, TimeType maxTime)
{
     // Stop turning the time-wheel if the slot has newer data than upper
     TimeType upper = currTime + size;
     TimeType time, nextTime;

     current = currTime % size;
     time = nextTime = INFINITY;

     for (int i = 0; i < size; i++){
	  DoubleLinkList* tempB = buffer[(current+i)%size];
          TimeSlot* slot;
	  if (tempB->head()) slot = (TimeSlot*) tempB->head()->content();
	  else slot = NULL;

          // Time-slot empty?
          if (slot == NULL)
               continue;

          // The first event to come?
          if ((time = slot->getTime()) < upper){
               nextTime = time;
               break;
          }
          else if (time < nextTime)
               nextTime = time;
     }

     // Timewheel empty?
     if (nextTime == INFINITY){
          current = 0;
          currTime = INFINITY;
       
          return NULL;
     }
     else{  
          current = nextTime % size;
          currTime = nextTime;
          
          // Scheduled event before maxTime?
          if (nextTime <= maxTime)
               return (TimeSlot*)buffer[current]->takeFromFront();
          else
               return NULL;
     }
}

TimeSlot*
TimeWheel :: findSlot(TimeType nextTime)
{
     TimeSlot* newSlot;

     DoubleLinkList* slotList = buffer[nextTime%size];

     // Create a time-slot if the list is empty
     if (slotList->size() <= 0){
          LOG_NEW; newSlot = new TimeSlot(nextTime);
          slotList->append(newSlot);
          return newSlot;
     }

     DoubleLinkIter nextSlot(*slotList);   
     TimeSlot* slot;
     while ((slot = (TimeSlot*) nextSlot++) != 0) {
          if (nextTime == slot->getTime())
               return slot;
          else if (nextTime > slot->getTime())
               continue;
          else
               break;
     }

     // Create a new time-slot and append it to the list
     LOG_NEW; newSlot = new TimeSlot(nextTime);
     slotList->insert(newSlot);
     return newSlot;
}
          

/*******************************************************************
          Main Thor scheduler routines
*******************************************************************/



     ////////////////////////////
     // displaySchedule
     ////////////////////////////

StringList ThorScheduler :: displaySchedule () {
     return "Thor schedule is computed at run-time";
}



     ////////////////////////////
     // setup
     ////////////////////////////

void ThorScheduler :: setup () {

     if (!galaxy()) {
	     Error::abortRun("ThorScheduler: no galaxy!");
	     return;
     }
     // Initialize the iterator
     GalStarIter nexts(*galaxy());

     // Initialize the requestHaltFlag
     clearHalt();

     // Initialize the global clock
     thorTime = 0;
     stopTime = 40;

     // Initialize the synchronizing clock
     currentTime = 0.0;

     // Initialize the timeWheel
     timeWheel->reset();

     // Initialize the element list and node list
     nodeList.initialize();
     elemList.initialize();

     // Run INIT section of all the atomic blocks,
     // and schedule generators and monitors at time zero
     nexts.reset();
     Star* st;
     
     // Get the next atomic block, and initialize states...
     while ((st = nexts++) != 0) {
	  if (!st->isA("ThorStar")) {
	      Error::abortRun(*st, 
				  " is not a Thor star: domain = ",
				  st->domain());
	      return;
          }
          ThorStar& star = *(ThorStar*)st;

          // set reference to currentTime ... (awa)
          star.setCurTime(thorTime);

          // Set reference to timeWheel
          star.timeWheel = timeWheel;

//**// following code inserted by awa (from DEScheduler.cc)
//**// further modified by S.Lee to count the number of connections
          // detect if any porthole is not connected.

          star.setInCount(0);
          star.setOutCount(0);
          star.setBiCount(0);
          star.setStCount(star.numberStates());

          // check out the connectivity of each port
          BlockPortIter nextp(star);
          PortHole *pt;
          while ((pt = nextp++) != 0) {
               PortHole& port = *pt;
               if (port.far() == NULL) {
                    StringList msg = port.fullName();
                    msg += " is not connected!";
                    Error::abortRun(msg);
                    return;
               }
               else{
                  // Increase the number of inputs or outputs of 
               // ThorStar accordingly
                    if (port.isItInput())
                         if (port.isItOutput())
                              ((ThorStar*)port.parent())->incBiCount();
                         else
                              ((ThorStar*)port.parent())->incInCount();
                    else
                         ((ThorStar*)port.parent())->incOutCount();
               }
          }
//**// until here

          // Call initialization
          star.initialize();
	  if (haltRequested()) return;

	  // Mark star as unscheduled
	  star.resetScheduled();

          // Schedule if generator or monitor
          if (star.isItGenerator())
               selfSched(0, 0, &star);
          if (star.isItMonitor())
               selfSched(0, 1, &star);
          

     }

     return;

}

     ////////////////////////////
     // run
     ////////////////////////////

// Turn the time-wheel and fire events scheduled at current time slot
// at each time step, elements are fired and nodes are evaluated
// the loop continues until all the elements reach steady states
int
ThorScheduler :: run () {

     if (haltRequested()){
	  Error::abortRun(*galaxy(),": Can't continue after run-time error");
	  return FALSE;
     }

     // Synchronize with outer domain if used as a wormhole
     stopBeforeDeadFlag = TRUE;
     thorTime = (int)currentTime;

     while ((thorTime <= stopTime) && !haltRequested()) {

          // Turn the time-wheel till there's a scheduled event
          TimeSlot *slot = timeWheel->get((TimeType&)thorTime, stopTime);

          // Return if the time-wheel is empty
          if (slot == NULL){
               if (thorTime == INFINITY)
                    stopBeforeDeadFlag = FALSE;
               break;
          }

          // Fire generators scheduled at current time slot
          stimElements(slot->callGenerators());
	  if (haltRequested()) return FALSE;

          // Fire delayed outputs scheduled earlier
          slot->stimOutputs(&nodeList);

          // Resolve node values and get the list of elements affected
          evaluateNodes();

          // Do loop till everything gets stable
          while (elemList.size()){
	       stimElements(elemList);
	       if (haltRequested()) return FALSE;
               evaluateNodes();
          }

          // Fire monitors scheduled at current time slot
          slot->callMonitors();

     } // end of while

     if (haltRequested())
	  return FALSE;

     // Synchronize with outer domain if used as a wormhole
     currentTime = (float)thorTime;
     
     return TRUE;
}

void
ThorScheduler :: evaluateNodes() {
     
     // Evaluate nodes in the list by calling its own evaluating function
     for (ThorNode* node = (ThorNode*) nodeList.takeFromFront(); node; 
          node = (ThorNode*) nodeList.takeFromFront()){

          // Evaluate the value of the node
          node->resolve();
          node->resetScheduled();

          // Get the list of elements to be fired if the node value
          // is changed
          if (node->nodeHasChanged()){
               ListIter nextl(node->outputs);
               PortHole *p;
               while ((p = (PortHole*) nextl++) != 0){
                    ThorStar* sp = (ThorStar*) p->parent();
                    if (!sp->isStarScheduled()){
                         elemList.append(sp);
                         sp->setScheduled();
                    }
               }
          }
     }
}

void
ThorScheduler :: stimElements(ElemList& list) {

     // Fire generators that have been scheduled at the time-slot
     for (currentElem = (ThorStar*) list.takeFromFront(); currentElem; 
          currentElem = (ThorStar*) list.takeFromFront()){

          // Get new inputs
          //for (int k = elem->numberPorts(); k > 0; k--) {
          //     ThorPortHole& p = (ThorPortHole&) elem->nextPort();
          //     p.grabData();
          //}

	  if (!currentElem->run()) return;
          currentElem->resetScheduled();

          // Get the list of nodes whose value is to be chaged
          BlockPortIter nexti(*currentElem);
          PortHole *ph;
          while ((ph = nexti++) != 0){
               ThorPortHole& p = *(ThorPortHole*) ph;

               // Continue if not ouput
               if (!p.isItOutput())
                    continue;

               // Schedule into the time-wheel if it is a delayed output
               if (p.delay > 0){
                    schedOut(p.delay, int(p), &p);
                    continue;
               }

               // Schedule the node
               ThorNode* np = p.node();

               // Is it on wormhole boundary?
               if (p.far()->isItOutput() && notEqual(p, *np)){
	            np->setNodeHasChanged();     // notify the scheduler ...
		    np->setNodeValue((int)p.getBuffParticle());
                    EventHorizon* eh = p.far()->asEH();
                    eh->setTimeMark((double)thorTime);
                    continue;
               }

               // Is the node getting new data and not scheduled yet?
               if (!np->isNodeScheduled() && notEqual(p, *np)){
                     nodeList.append(np);
                     np->setScheduled();
               }
          }
     }
}

void 
schedStar(int time, int priority, ThorStar* elem) {

     // Find a corresponding time-slot in the time-wheel
     // Create one if it doesn't exist
     TimeSlot* slot = elem->timeWheel->findSlot(time);

     if (priority) // Monitor
          slot->putMon(elem);
     else          // Generator
          slot->putGen(elem);
}

void 
selfSched(int delta_time, int priority, ThorStar* elem) {

     int nextTime = elem->getCurTime() + delta_time;
     schedStar(nextTime, priority, elem);
}

void 
selfUnsched(int time, ThorStar* elem) {

     // Find a corresponding time-slot in the time-wheel
     TimeSlot* slot = elem->timeWheel->findSlot(elem->getCurTime() + time);

     // Delete elem from the scheduled element list
     slot->deleteGen(elem);
     slot->deleteMon(elem);
}

void 
schedOut(TimeType time, int value, ThorPortHole* pin) {

     // Find a corresponding time-slot in the time-wheel
     // Create one if it doesn't exist
     ThorStar *elem = (ThorStar*)pin->parent();
     TimeSlot* slot = elem->timeWheel->findSlot(elem->getCurTime() + time);

     LOG_NEW; slot->putOut(new OutChange(pin, value));
}

// called by a model to get the current time.
TimeType getCurrentTime() {return currentElem->getCurTime();}

// called by a model to get the user model name. returns a string
const char* mname() {return currentElem->name();}

