
/*
 *
 * Version identification:
 * @(#)postscript.c	1.2 11/23/92
 *
 */

#ifndef lint
static char rcsid[]= "$Header: postscript.c,v 1.2 87/01/28 00:20:06 salz Exp $";
#endif
/*
 * Postscript interface routines 
 */

#include <stdio.h>
#include <sys/time.h>
#include <string.h>
#include "defs.h"

#define	PS_UNITS	72			/* PostScript unit (1/inch) */
#define	ps_YSIZE	(15 * PS_UNITS / 2)	/* y page size = 7.5 inch */
#define	ps_XSIZE	(10 * PS_UNITS)		/* x page size = 10 inch */
#define	XMARGIN		(PS_UNITS / 2)		/* margins 1/2 inch */
#define	YMARGIN		(PS_UNITS / 2)
#define	FSIZE		9			/* default font size */
#define	MINFSIZE	4			/* minimum font size */

#define	YTRACE		(trace.top - trace.bot)	/* vertical size of traces */
#define	YTIMES		20			/* in PS units */
#define	YBANNER		12			/* in PS units */


#define	BANN_TOP	(ps_YSIZE - YTIMES + PSY( trace.bot ) )
#define	BANN_BOT	(BANN_TOP - YBANNER)
#define	BANN_POS	(BANN_BOT + 2)
#define	TIM_TOP		PSY( trace.bot )
#define	TIM_BOT		(TIM_TOP - YTIMES)
#define	TIM_POS		(TIM_BOT + 4)
#define	TIM_POS1	(TIM_BOT + 16)

		/* scale conversion macros */
#define	PSX( x )	( (x) * (ps_XSIZE - 2) / trace.right )
#define	PSY( y )	( (y) * (ps_YSIZE - YTIMES - YBANNER) / YTRACE )
#define	PS( x, y )	PSX( x ), PSY( y )

#define	CHARHEIGHT	(metrics.charHeight)
#define	CHARWIDTH	(metrics.charWidth)
#define	min( a, b )	( ( (a) < (b) ) ? (a) : (b) )


static FILE *psout;

void CmdPrint( narg, args )
  int   narg;
  char  *args[];
  {
    TimeType  t1, t2, time, oldStart;
    char      fname[ 256 ];
    char      *date;
    struct timeval theTime;
    
    if( (trace.bot >= trace.top) || (traces.dispTraces == 0) )
      {
	PRINT( "There are no traces to print\n" );
	return;
      }
    switch( narg )
      {
	case 0 :
	    t1 = times.dispStart;
	    t2 = times.dispStart + times.dispSteps;
	    sprintf( fname, "%s%s", names.model, PS_EXT );
	    break;
	case 1 :
	    if( sscanf( args[0], "%d-%d", &t1, &t2 ) == 2 )
	      {
		if( (t1 < times.first) || (t2 > times.last) || (t1 >= t2) )
		  {
		    PRINT( "Error: times out of bounds\n" );
		    return;
		  }
		sprintf( fname, "%s%s", names.model, PS_EXT );
	      }
	    else		/* arg must be filename */
	      {
		strcpy( fname, args[0] );
		t1 = times.dispStart;
		t2 = times.dispStart + times.dispSteps;
	      }
	    break;
	case 2 :
	    if( sscanf( args[1], "%d-%d", &t1, &t2 ) != 2 )
	      {
		PRINT( "Syntax Error in time spec.\n" );
		return;
	      }
	    if( (t1 < times.first) || (t2 > times.last) || (t1 >= t2) )
	      {
		PRINT( "Error: times out of bounds\n" );
		return;
	      }
	    strcpy( fname, args[0] );
	    break;
	default :
	    PRINT( "Too many arguments\n" );
	    return;
      }
    psout = fopen( fname, "w" );
    if( psout == NULL )
      {
	PRINT( "Can't open file: " );
	PRINT( fname );
	PUTC( '\n' );
	return;
      }
    PRINT( "Writing file..." );
    FLUSH();
    WritePreamble();
    gettimeofday( &theTime, (struct timezone*)0 );
    date = ctime( &theTime.tv_sec );
    oldStart = times.dispStart;
    for( time = t1; time < t2; time += (times.dispSteps + 1) )
      {
	fprintf( psout, "MSAVE\n" );
	times.dispStart = time;
	DrawOutline( date );
	PrintSignals();
	PrintTraces( time, min( t2, time + times.dispSteps ) );
	PrintTimes( time, min( t2, time + times.dispSteps ) );
	fprintf( psout, "showpage MRESTORE\n" );
      }
    if( SigGroups() )
      {
	DrawOutline( date );
	PrintLegend();
	fprintf( psout, "showpage\n" );
      }
    fclose( psout );
    times.dispStart = oldStart;
    PRINT( "done\n" );
  }


int SigGroups()
  {
    TraceEnt  *t, *lastT;

    lastT = &(traces.traces[ traces.dispTraces ]);
    for( t = traces.traces; t != lastT; t++ )
      {
	if( t->nsig > 1 )
	    return( 1 );
      }
    return( 0 );
  }



DrawOutline( date )
  char  *date;
  {
    fprintf( psout, "0.8 setgray\n" );
    fprintf( psout,"%d %d %d %d BOX fill\n", 0, BANN_BOT, ps_XSIZE, BANN_TOP);
    fprintf( psout, "0 setgray\n" );
    fprintf( psout, "%d %d %d %d L\n", 0, BANN_BOT, ps_XSIZE, BANN_BOT );
    fprintf( psout,"%d %d %d %d BOX stroke\n", 0, TIM_BOT,ps_XSIZE,BANN_TOP );
    PrintStr( banner.left );
    fprintf( psout, " %d %d SL\n", 5, BANN_POS );
    fprintf( psout, "%d %d (%s) SR\n", BANN_POS , ps_XSIZE - 4, date );
  }


int PrintSignals()
  {
    int       nBus, rlen;
    char      s[100];
    XCoord    x;
    YCoord    y;
    TraceEnt  *t, *lastT;
    char      *r, *Root();

    ps_Line( trace.left, trace.bot, trace.left, trace.top );

    x = sigNames.right - 1;
    nBus = 0;
    lastT = &(traces.traces[ traces.dispTraces ]);
    for( t = traces.traces; t != lastT; t++ )
      {
	y = (t->bot + t->top) / 2;
	if( t->nsig == 1 )
	    ps_StrRightV( inputs.names[ t->signals[0] ], x, y );
	else
	  {
	    if( t->sameRoot )
	      {
		int  n1, n2;

		r = Root( inputs.names[ t->signals[0] ]);
		rlen = strlen( r );
		if( inputs.names[ t->signals[0] ][ rlen ] == '[' )
		    rlen++;
		n1 = atoi( &(inputs.names[ t->signals[0] ][ rlen ]) );
		n2 = atoi( &(inputs.names[ t->signals[ t->nsig-1 ] ][rlen]) );
		(void) sprintf( s, "%s[%d-%d]", r, n1, n2 );
	      }
	    else
	      {
		sprintf( s, "GRP %d [%d]", nBus++, t->nsig );
	      }
	    ps_StrRightV( s, x, y );
	  }
      }
  }


PrintTimes( t1, t2 )
  TimeType t1, t2;
  {
    TimeType  step, i;
    int       x, x2;
    char      s[ 25 ];
    
    step = ( (times.dispSteps >= 10) ? round(times.dispSteps / 10.0) : 1 );
    for( i = t1; i <= t2; i += step )
      {
	x = PSX( TimeToX( i ) );
	fprintf( psout, "%d %d %d %d L\n", x, TIM_POS1, x, TIM_TOP );
	sprintf( s, "%d", i );
	if( i == t2 )
	    fprintf( psout, "%d %d (%s) SR\n", TIM_POS, x, s );
	else
	    fprintf( psout, "%d %d (%s) SC\n", TIM_POS, x, s );
      }
    x2 = PSX( TimeToX( t2 ) );
    fprintf( psout, "%d %d %d %d L\n", x2, TIM_POS1, x2, TIM_TOP );
    x = PSX( TimeToX( t1 ) );
    fprintf( psout, "%d %d %d %d L\n", x, TIM_TOP, x2, TIM_TOP );
  }


/*
 * Same procedure for displaying traces on the screen, but the output
 * is sent to the postscript file and bus changes are drawn with a slant.
 */
PrintTraces( time1, time2 )
  TimeType  time1, time2;
  {
    int       j, change;
    TraceEnt  *t, *lastT;
    HistRef   endH, startH, begH;
    TimeType  startT, endT;
    YCoord    base, top, mid;
    XCoord    x1, x2;


    if( times.first < 0 )
	return;			/* done, no traces yet */

    if( time2 > times.last )
	time2 = times.last;

    begH = GetHist( time1 );
    if( begH == NULL )
      return;

    lastT = &(traces.traces[ traces.dispTraces ]);
    for( t = traces.traces; t != lastT; t++ )
      {
	base = t->bot;
	top = t->top;
	mid = (base + top) / 2;
	startT = time1;
	startH = begH;
	
	while( startT < time2 )			/* Draw a single trace */
	  {
	    x1 = TimeToX( startT );
	    endH = FindChanged( t, startH, time2 );

	    if( endH == startH )	/* No change before time2 */
	      {
		endT = time2;
		change = 0;
	      }
	    else
	      {
		endT = endH->time;
		change = 1;
	      }

	    x2 = TimeToX( endT );
	    if( t->nsig == 1 )		/* not a bus */
	      {
		j = ( (startH->levels[ t->signals[0] ]) << 2 ) |
		      (endH->levels[ t->signals[0] ]);
		switch( j )
		  {
		    case 0:				/* 0 -> 0 */
		      ps_Line( x1, base, x2, base );
		      break;
		    case 1:				/* 0 -> 1 */
		    case 2:				/* 0 -> U */
		      ps_Line( x1, base, x2, base );
		      ps_Line( x2, base, x2, top );
		      break;
		    case 3:				/* 0 -> Z */
		      ps_Line( x1, base, x2, base );
		      ps_Line( x2, base, x2, mid );
		      break;
		    case 4:				/* 1 -> 0 */
		    case 6:				/* 1 -> U */
		      ps_Line( x1, top, x2, top );
		      ps_Line( x2, base, x2, top );
		      break;
		    case 5:				/* 1 -> 1 */
		      ps_Line( x1, top, x2, top );
		      break;
		      break;
		    case 7:				/* 1 -> Z */
		      ps_Line( x1, top, x2, top );
		      ps_Line( x2, top, x2, mid );
		      break;
		    case 8:				/* U -> 0 */
		    case 9:				/* U -> 1 */
		    case 11:				/* U -> Z */
		      ps_Line( x2, base, x2, top );
		    case 10:				/* U -> U */
		      ps_SetPattern( 0.78 );
		      ps_FilledRectangle( x1, base, x2, top );
		      ps_SetPattern( 0.0 );
		      break;
		    case 12:				/* Z -> 0 */
		      ps_Line( x1, mid, x2, mid );
		      ps_Line( x2, mid, x2, base );
		      break;
		    case 13:				/* Z -> 1 */
		      ps_Line( x1, mid, x2, mid );
		      ps_Line( x2, mid, x2, top );
		      break;
		    case 14:				/* Z -> U */
		      ps_Line( x2, base, x2, top );
		    case 15:				/* Z -> Z */
		      ps_Line( x1, mid, x2, mid );
		      break;
		  }
	      }
	    else			/* bus */
	      {
		if( IsTriState( startH->levels, t->signals, t->nsig ) )
		  {
		    if( change )
		      {
			ps_Line( x1+2, mid, x2-2, mid );
			ps_Line( x2-2, mid, x2+2, top );
			ps_Line( x2-2, mid, x2+2, base );
		      }
		    else
			ps_Line( x1+2, mid, x2, mid );
		  }
		else
		  {
		    BusValue(   Convert( startH->levels, t ),
				x1, x2, mid );
		    if( change )
		      {
			ps_Line( x1+2, base, x2-2, base );
			ps_Line( x1+2, top, x2-2, top );
			if( IsTriState( endH->levels, t->signals, t->nsig ) )
			  {
			    ps_Line( x2-2, base, x2+2, mid );
			    ps_Line( x2-2, top, x2+2, mid );
			  }
			else
			  {
			    ps_Line( x2-2, base, x2+2, top );
			    ps_Line( x2-2, top, x2+2, base );
			  }
		      }
		    else
		      {
			ps_Line( x1+2, base, x2, base );
			ps_Line( x1+2, top, x2, top );
		      }
		  }
	      }

	    startT = endT;
	    startH = endH;
	  }			/* end of while */
      }				/* end of for all traces */
  }


PrintLegend()
  {
    int       i, j, nBus, rlen;
    char      s[256];
    TraceEnt  *t;
    char      *r, *Root();

    nBus = 0;
    fprintf( psout, "/GX %d  def\n", PSX( sigNames.right + 25 ) );
    fprintf( psout, "/GY %d  def\n", BANN_BOT - 2 * (FSIZE + 3)  );
    fprintf( psout, "(Legend:) 4 %d SL\n", BANN_BOT - FSIZE - 2 );
    for( i = 0, t = traces.traces; i < traces.dispTraces; i++, t++ )
      {
	if( t->nsig > 1 )
	  {
	    if( t->sameRoot )
	      {
		int  n1, n2;

		r = Root( inputs.names[ t->signals[0] ]);
		rlen = strlen( r );
		if( inputs.names[ t->signals[0] ][ rlen ] == '[' )
		    rlen++;
		n1 = atoi( &(inputs.names[ t->signals[0] ][ rlen ]) );
		n2 = atoi( &(inputs.names[ t->signals[ t->nsig-1 ] ][rlen]) );
		(void) sprintf( s, "%s[%d-%d]", r, n1, n2 );
	      }
	    else
	      {
		sprintf( s, "GRP %d [%d]  =>", nBus++, t->nsig );
	      }
	    for( j = t->nsig-1; j >= 0; j-- )
	      {
		PrintStr( inputs.names[ t->signals[ j ] ] );
	      }
	    PrintStr( s );
	    fprintf( psout, "%d LE\n", t->nsig );
	  }
      }
  }



WritePreamble()
  {
    static char defs[] = "%!\n\
/MSAVE { /mStat save def } def\n\
/MRESTORE { mStat restore } def\n\
/SET { exch def } def\n\
/SF { /wi SET theFont [wi 0 0 FSIZE 0 0] makefont setfont } def\n\
/L { newpath moveto lineto stroke } def\n\
/BOX { /@yb SET /@xb SET /@yt SET /@xt SET newpath\n\
 @xb @yb moveto @xb @yt lineto @xt @yt lineto @xt @yb lineto closepath } def\n\
/SC { /@s SET @s stringwidth pop 2 div sub exch moveto @s show } def\n\
/SR { /@s SET @s stringwidth pop sub exch moveto @s show } def\n\
/SL { moveto show } def\n\
/SRV { /@s SET @s stringwidth pop sub exch FSIZE 2 div sub moveto @s show } def\n\
/BV { /@x2 SET /@x1 SET FSIZE 2 div sub /@y SET /@s SET\n\
 /@l @x2 @x1 sub def /@x @x1 @x2 add 2 div def\n\
 /@w @s stringwidth pop def @l @w gt\n\
 { @y @x @s SC }\n\
 { @l FSIZE mul @w 1 add div dup MINFSIZE lt\n\
 { pop } { /svf currentfont def SF @y @x @s SC svf setfont } ifelse }ifelse\n\
} def\n\
/LE { exch GY exch GX exch SR /@x GX def\n\
 { dup stringwidth pop /@w SET /@x @x FSIZE add def @x @w add\n\
 7.5 72 mul ge { /@x GX FSIZE add def /GY GY FSIZE 2 add sub def } if\n\
 @x GY SL /@x @x @w add def } repeat /GY GY FSIZE 4 add sub def } def\n\
";
	/* print data and procedure definitions */
    fprintf( psout, "%s", defs );
    fprintf( psout, "/FSIZE %d def /MINFSIZE %d def\n", FSIZE, MINFSIZE );
	/* Switch to landscape mode */
    fprintf( psout, "%d 0 translate\n", 17 * PS_UNITS / 2 );
    fprintf( psout, "90 rotate\n" );
    fprintf( psout, "%d %d", XMARGIN, YMARGIN + YTIMES - PSY( trace.bot ) );
    fprintf( psout, " translate\n" );
    fprintf( psout, "1 setlinecap\n" );
	/* Set up text font */
    fprintf( psout,"/theFont /Helvetica findfont def FSIZE SF\n");
  }



ps_FilledRectangle( x1, y1, x2, y2 )
  {
    fprintf( psout, "%d %d %d %d BOX fill\n", PS(x1, y1), PS( x2, y2 ) );
  }


ps_Rectangle( x1, y1, x2, y2 )
  {
    fprintf( psout, "%d %d %d %d BOX stroke\n", PS( x1, y1 ), PS( x2, y2 ) );
  }

ps_Line( x1, y1, x2, y2 )
  {
    fprintf( psout,"%d %d %d %d L\n", PS( x1, y1 ), PS( x2, y2 ) );
  }



PrintStr( s )
  char *s;
  {
    putc( '(', psout );
    while( *s != '\0' )
      {
	if( (*s == '(' ) || (*s == ')') )
	    putc( '\\', psout );
        putc( *s, psout );
	s++;
      }
    putc( ')', psout );
  }

ps_StrCenter( s, x, y )
  char *s;
  {
    fprintf( psout, "%d %d ", PSY( y ), PSX( x ) );
    PrintStr( s );
    fprintf( psout, " SC\n" );
  }

ps_StrRight( s, x, y )
  char *s;
  {
    fprintf( psout, "%d %d ", PSY( y ), PSX( x ) );
    PrintStr( s );
    fprintf( psout, " SR\n" );
  }

ps_StrRightV( s, x, y )
  char *s;
  {
    fprintf( psout, "%d %d ", PSY( y ), PSX( x ) );
    PrintStr( s );
    fprintf( psout, " SRV\n" );
  }

ps_StrLeft( s, x, y )
  char *s;
  {
    PrintStr( s );
    fprintf( psout, " %d %d SL\n", PS( x, y ) );
  }

BusValue( s, x1, x2, y )
  char *s;
  {
    PrintStr( s );
    fprintf( psout, " %d %d %d BV\n", PSY( y ), PSX( x1 ), PSX( x2 ) );
  }

ps_SetPattern( p )
  float p;
  {
    fprintf( psout, "%g setgray\n", p );
  }
