defstar {
	name {Burg}
	domain {SDF}
	version {@(#)SDFBurg.pl	1.12 12/8/92}
	desc {
Burg's algorithm.  The lp output receives the linear prediction coefficients
and the refl output receives reflection coefficients.  The errPower output
gets the power of the prediction error at each stage.
	}
	author { E. A. Lee and J. T. Buck }
	copyright {
Copyright (c) 1990, 1991, 1992 The Regents of the University of California.
All rights reserved.
See the file ~ptolemy/copyright for copyright notice,
limitation of liability, and disclaimer of warranty provisions.
	}
	location { SDF dsp library }
	explanation {
.pp
This star uses Burg's algorithm to estimate the reflection coefficients
and AR parameters of an input random process.
.Id "Burg's algorithm"
.Ir "linear prediction"
.Ir "spectral estimation, maximum entropy"
.Ir "maximum entropy spectral estimation"
The number of inputs looked at is given by the \fInumInputs\fR parameter
and the order of the AR model is given by the \fIorder\fR parameter.
.Ir "autoregressive model"
.Ir "AR model"
The order specifies how many outputs appear on the \fIlp\fR and
\fIrefl\fR output portholes.
These outputs are, respectively, the autoregressive (AR) parameters
(also called the linear predictor parameters),
and the reflection coefficients.
.Ir "reflection coefficients"
.Ir "coefficients, reflection"
.pp
Note that the definition of reflection coefficients is not quite
universal in the literature.
The reflection coefficients in references [2] and [3]
are the negative of the ones generated by this star,
which correspond to the definition in most other texts,
and to the definition of partial-correlation (PARCOR)
coefficients in the statistics literature.
.Ir "partial correlation coefficients"
.Ir "coefficients, partial correlation"
.Ir "PARCOR"
.pp
The \fIerrPower\fR output is the power of the prediction error
as a function of the model order.  There are $order+1$ output samples,
where the first corresponds to the prediction error of a zero-th
order predictor.  This is simply an estimate of the input signal power.
.UH REFERENCES
.ip [1]
J. Makhoul, "Linear Prediction: A Tutorial Review",
\fIProc. IEEE\fR, Vol. 63, pp. 561-580, Apr. 1975.
.ip [2]
S. M. Kay, \fIModern Spectral Estimation: Theory & Application\fR,
Prentice-Hall, Englewood Cliffs, NJ, 1988.
.ip [3]
S. Haykin, \fIModern Filters\fR, MacMillan Publishing Company,
New York, 1989.
	}
	seealso { LevDur, linearPrediction, powerSpectrum }
	input {
		name {input}
		type {float}
		desc { Input random process. }
	}
	output {
		name {lp}
		type {float}
		desc { AR coefficients output. }
	}
	output {
		name {refl}
		type {float}
		desc { Lattice predictor coefficients output. }
	}
	output {
		name {errPower}
		type {float}
		desc { Prediction error power. }
	}
	defstate {
		name {order}
		type {int}
		default {8}
		desc {The number of reflection coefficients to generate.}
	}
	defstate {
		name {numInputs}
		type {int}
		default {64}
		desc { The number of inputs used to estimate the model.}
	}
	protected {
		double *f, *b, *aOrig, *aPrime;
		int N, M;
	}
	constructor {
		f = b = aOrig = aPrime = 0;
		N = M = 0;
	}
	destructor {
		LOG_DEL; delete f;
		LOG_DEL; delete b;
		LOG_DEL; delete aOrig;
		LOG_DEL; delete aPrime;
	}
	setup {
		if (N != int(numInputs)) {
			LOG_DEL; delete f; LOG_DEL; delete b;
			N = int(numInputs);
			LOG_NEW; f = new double[N];
			LOG_NEW; b = new double[N];
		}
		if (M != int(order)) {
			LOG_DEL; delete aOrig; LOG_DEL; delete aPrime;
			M = int(order);
			LOG_NEW; aOrig = new double[M+1];
			LOG_NEW; aPrime = new double[M+1];
		}
		refl.setSDFParams (int(order), int(order)-1);
		lp.setSDFParams (int(order), int(order)-1);
		errPower.setSDFParams (int(order)+1, int(order));
		input.setSDFParams (int(numInputs), int(numInputs)-1);
	}
	go {
	    // Define pointers so that the arrays can be swapped
	    double* a = aOrig;
	    double* aP = aPrime;
	    double gamma, dsum, nsum;

	    // initialize the forward and backward predictor errors
	    // and the prediction error power estimate
	    int count = 0;
	    double ep = 0.0;	// error power estimate
	    for (int i = N-1; i >= 0; i--) {
		double x = input%i;
		f[count] = x;
		b[count++] = x;
		ep += x * x;
	    }
	    ep = ep/N;
	    // output the zeroth order prediction error, which is simply
	    // the power estimate of the input
	    errPower%M << ep;

	    // Iterate on the predictor order
	    for (int m = 1; m <= M; m++ ) {
	        // Compute the reflection coefficients, and output them
		nsum = 0.0;
		dsum = 0.0;
		for (i = m; i < N; i++) {
		    nsum += f[i]*b[i-1];
		    dsum += f[i]*f[i] + b[i-1]*b[i-1];
		}
		gamma = -2*nsum/dsum;
		refl%(M-m) << - gamma;

		// update the forward and backward predictor errors
		for (i = N-1; i >= m; i--) {
		    double tempf = f[i];
		    f[i] = tempf + gamma*b[i-1];
		    b[i] = b[i-1] + gamma*tempf;
		}

		// update the prediction error power estimate
		ep = (1 - gamma*gamma) * ep;
		errPower%(M-m) << ep;

		// Update the FIR predictor coefficient estimates
		for (i = 1; i < m; i++) {
		    aP[i] = a[i] + gamma * a[m-i];
		}
		aP[m] = gamma;

		// Swap a and aP for next order recurrence
		double* temp = a;
		a = aP;
		aP = temp;
	    }
	    // generate the lp outputs
	    for (m = 1; m <= M; m++ ) {
		lp%(M-m) << -a[m];
	    }
	}
}
