static const char file_id[] = "DDFAutoWorm.cc";
/**************************************************************************
Version identification:
@(#)DDFAutoWorm.cc	1.20	12/5/92

Copyright (c) 1990, 1991, 1992 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY 
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES 
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF 
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF 
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.
							COPYRIGHTENDKEY

 Programmer:  Soonhoi Ha
 Date of creation: 4/24/91
    Separate the auto-wormholizing routine from the main DDF Scheduler
    for ease of maintenance.
    Notes: This process does not check whether the clustering results in
	artificial dead-lock condition or not.  If an artificial dead-lock
	is formed after clustering, the programmer is recommended to
	disable this clustering process by defining a parameter in the
	DDF domain, called "restructure", and setting it FALSE.

**************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include "type.h"
#include "SDFStar.h"
#include "SDFWormhole.h"
#include "DDFScheduler.h"
#include "DDFWormhole.h"
#include "DynamicGalaxy.h"
#include "Geodesic.h"
#include "GalIter.h"
#include <std.h>

extern const char DDFdomainName[];

// melt the SDFinDDFWormholes in a larger one.
Galaxy* deCrust(Star* dw);

// make a unique portname of an auto-galaxy
const char* galPortName(const char* starName, const char* portName); 

// correct numTokens value for each EventHorizon.
void renewNumTokens(Block* b, int flag);

// expand an auto-galaxy from the "org" block.
static void expandGal (Galaxy& galaxy, Galaxy* g, Block* org); 

// make a connection to the porthole(galP)  of an auto-wormhole
static void reConnect(PortHole& eveP, PortHole& galP); 

// make connection of auto-wormholes
static void connectWormholes (Galaxy&);

// special treat of arcs with delays
static void handleDelays();

// change DDFWormhole to SDFWormhole
Star* changeDDFtoSDF (Star* s) ;
			
/********************************************************************

	MAIN ROUTINE for AUTO-WORMHOLIZATION

 *******************************************************************/

// variable used for generating names
int DDFScheduler::nW = 0;

// generate a new name for Wormholes of SDF domain
const char* DDFScheduler::wormName() {
	char buf[16];
	sprintf (buf, "!worm%d", ++nW);
	return hashstring (buf);
}

// name a porthole of an auto-galaxy with a combination of the star name
// and a portname.
const char* galPortName(const char* starName, const char* portName) {
	char buf[32];
	int len = strlen(starName);
	strcpy(buf, starName);
	buf[len++] = '(';
	strcpy(buf+len, portName); 
	len += strlen(portName);
	strcpy(buf+len, ")"); 
	return hashstring (buf);
}

// SequentialList for the Geodesic with delays
SequentialList delayGeos;

	////////////////////////////
	// makeSDFWormholes
	////////////////////////////

// In the DDF domain, we group SDF stars and wormholes of SDF domain
// as a as large a DDFWormhole of SDF domain as possible.
// This grouping will improve the runtime speed by utilizing the
// compile-time SDF scheduler.
// Also, with modified topology, check whether it is a special domain
// or not : Case, For, DoWhile, Recur.

void DDFScheduler :: makeSDFWormholes (Galaxy& galaxy) {

	GalStarIter nextStar(galaxy);

	sdfWorms.initialize();
	delayGeos.initialize();
	canDom = Unknown;
	int flagDDF = 0;

	Star* current = nextStar++;
	while (current != NULL) {

	    Star* s = current;
	    Star* prev = current;
 
	    // detect a SDF star which is not grouped into a newly
	    // created SDF galaxy.  If found, create a new SDF galaxy (newG)
	    // and add the SDF star into the galaxy.
	    // Then, expand the galaxy as much as possible.
	
	    int branch = 0;
	    if (s->isItWormhole()) {
		const char* dom = s->scheduler()->domain();
		if (!strcmp(dom, "SDF")) branch = 2;
		else if (!strcmp(dom, "DDF")) {
			DDFScheduler* sr = (DDFScheduler*) s->scheduler();
			int temp = sr->isSDFType();
			if (temp == 1) {
				branch = 1;
				s = changeDDFtoSDF(current);
			} else if (temp == 2) branch = 2;
		}
	    } else {
		if (strcmp(s->domain(), DDFdomainName)) branch = 1;
	    }

	    if (branch) {

		LOG_NEW; DynamicGalaxy* newG = new DynamicGalaxy;
		sdfWorms.put(newG);
		newG->setBlock(wormName(), 0);
		newG->setDomain("SDF");

		if (branch == 2) {
			// take off wormhole crust
		    Galaxy* inG = deCrust(s);
		    newG->addBlock(*inG, inG->name());
		    expandGal(galaxy, newG, inG);
		} else {
			newG->addBlock(*s, s->name());
			expandGal(galaxy, newG, s);
		}
		current = nextStar++;
		if (current == prev) current = NULL;
		galaxy.removeBlock(*prev);

	    } else {

		current = nextStar++;
		flagDDF++;
	    // if it is a DDF Star or a Wormhole of non-SDF domain, 
	    // do nothing.  And check the candidate domain.

		if (s->isItWormhole())  canDom = DDF;

	    }	// end of outermost if
	}	// end of while

	if (!flagDDF) canDom = SDF;

	// Handle  the arcs of delays.
	// If the delay is between two different wormholes, create
	// wormhole portholes for the arc. Otherwise ignore it.
	handleDelays();

	ListIter nextb(sdfWorms);

	// if canDom = SDF, do not make wormholes.
	if (canDom == SDF) {
		for (int i = sdfWorms.size(); i > 0; i--) {
			DynamicGalaxy* inG = (DynamicGalaxy*) nextb++;
			galaxy.addBlock(*inG, inG->name());
		}
		galaxy.setDomain("SDF");
		return;
	}

	// Now we have SDF galaxies. Make them as DDF Wormholes.
	for (int i = sdfWorms.size(); i > 0; i--) {
		DynamicGalaxy* inG = (DynamicGalaxy*) nextb++;
		LOG_NEW; DDFWormhole* dW = new DDFWormhole(*inG);
		galaxy.addBlock(*dW, inG->name());
		dW->setup();
		renewNumTokens(dW, TRUE);
	} 

	// make connections for broken arcs.
	connectWormholes(galaxy);
}

	////////////////////////////
	// connectWormholes
	////////////////////////////

// During the procedure of making DDF Wormholes of SDF domain, we
// broke the connections at the boundary of the Wormholes.
// This routine remakes the connections.

void connectWormholes (Galaxy& g) {

	if (g.numberPorts()) {

	    BlockPortIter galps(g);
	    // verify the wormhole boundary connection.
	    Block* parW = g.parent();

	    // exclude DDF Self in the recursion construct
	    if (parW->isItWormhole()) {
		BlockPortIter nextp(*parW);
		PortHole* p;

		while((p = nextp++) != 0) {
			PortHole* eveP = p->asEH()->ghostAsPort();
			PortHole* galP = eveP->far();
			// check the connection
			if (eveP != galP->far()) {
				reConnect(*eveP, *galP);	
				// change alias of the galaxy
				PortHole* gp;
				galps.reset();
				while ((gp = galps++) != 0) {
					if (galP == &(gp->realPort())) {
						gp->setAlias(*(eveP->far()));
						break;	
					}
				}
			}
		}
	    }
	}

	// verify the connections of the DDF Stars and Wormholes.
	// Call reConnect if necessary.

	GalStarIter nextStar(g);
	Star* s;

	while ((s = nextStar++) != 0) {

	   if ((!s->isItWormhole()) ||
			strcmp(s->scheduler()->domain(), "SDF")) {

		BlockPortIter nextp(*s);
		PortHole* p;

		while((p = nextp++) != 0) {
			PortHole* galP = p->far();
			// check the connection
			if ( galP && (p != galP->far())) 
				reConnect(*p, *galP);	
		}
	    }
	}

	// Now, make connections around each arc with delays if
	// the arc connects two different auto-wormholes.
	ListIter nextGeo(delayGeos);
	Geodesic* geo;

	while ((geo = (Geodesic*) nextGeo++) != 0) {
		PortHole* src = geo->sourcePort();
		PortHole* dest = geo->destPort();
		// if delay between SDF stars or DDFWormhole of SDF domain
		if (src->atBoundary() && dest->atBoundary()) {
			PortHole* eS = src->far()->asEH()->ghostAsPort();
			PortHole* dS = dest->far()->asEH()->ghostAsPort();
			// reuse the old geodesic
			geo->disconnect(*src);
			geo->disconnect(*dest);
			geo->setSourcePort(*eS, geo->numInit());
			geo->setDestPort(*dS);
			eS->initialize();
			dS->initialize();
		}
	}
}

	////////////////////////////
	// reConnect
	////////////////////////////

// When create a new DDFWormhole of SDF domain, the connections around
// the Wormholes are all broken.  galP points to the PortHole inside the
// Wormhole and eveP points to the PortHole of the other block.
// Now, connect eveP to the proper EventHorizon of the Wormhole, which by
// turns connected to galP.

void reConnect(PortHole& eveP, PortHole& galP) {

	PortHole* realP = galP.far()->asEH()->ghostAsPort();
	int delay = eveP.numInitDelays();
	eveP.geo()->disconnect(galP);
	eveP.disconnect(1);
	if (galP.isItInput())
		eveP.connect(*realP, delay);
	else
		realP->connect(eveP, delay);
	realP->initialize();
}
		
	////////////////////////////
	// expandGal
	////////////////////////////

// Once create a SDF galaxy, expand it as much as possible.
// First, include any adjacent SDF stars into the galaxy.
// Second, if there is any Wormholes of SDF domain, remove the
// crust of Wormhole and make the inside galaxy as a normal
// galaxy.

void expandGal (Galaxy& galaxy, Galaxy* g, Block* org) {

	BlockPortIter nextp(*org);
	PortHole* p;

	while((p = nextp++) != 0) {
		PortHole& realP = p->newConnection();
		Star* s = (Star*) realP.far()->parent();;

		// do not uncover a galaxy if the galaxy is included
		// in a auto-galaxy already.
		int touched = 0;
		Block* sB = s->parent();
		while (sB && !sB->isItWormhole()) {
			if (sB == (Block*) g) { touched = 1;
					        break; }
			else sB = sB->parent();
		}

	   if (!touched) {

		// check which branch the star is in.
	    	int branch = 0;
	    	if (s->isItWormhole()) {
			const char* dom = s->scheduler()->domain();
			if (realP.atBoundary() == FALSE) {
				if (!strcmp(dom, "SDF")) branch = 2;
				else if (!strcmp(dom, "DDF")) {
					DDFScheduler* sr = 
					(DDFScheduler*) s->scheduler();
					int temp = sr->isSDFType();
					if (temp == 1) {
						branch = 1;
						s = changeDDFtoSDF(s);
					} else if (temp == 2) branch = 2;
				}
			}
	    	} else {
			if (strcmp(s->domain(), DDFdomainName)) branch = 1;
	    	}

		int initNum = realP.numInitDelays();
		if (branch == 1) {
			// if SDF star, put it into the galaxy and
			// call expandGal for this star.
		   if (!initNum) {
			g->addBlock(*s, s->name());
			galaxy.removeBlock(*s);
			expandGal(galaxy, g, s);
		   } else {
			// put the geodesic only once in case there is a delay
			// on the arc
			if (realP.isItInput())
				delayGeos.put(realP.geo());
		   }
		} else if (!branch) {
			// if DDF star, DDFWormhole of non-SDF domain,
			// or wormhole boundary, 
			// create a galaxy porthole and alias it.
			LOG_NEW; GalPort* gp = new GalPort(*p);
			g->addPort(gp->setPort(
			   galPortName(org->name(), realP.name()), g));
		} else {
		   if (!initNum) {
			// DDF wormhole of SDF domain
			// First, take off wormhole crust
			Galaxy* inG = deCrust(s);
			g->addBlock(*inG, inG->name());
			galaxy.removeBlock(*s);
			expandGal(galaxy, g, inG);
		   } else {
			// put the geodesic only once in case there is a delay
			// on the arc
			if (realP.isItInput())
				delayGeos.put(realP.geo());
		   }
		}
	    }
	}
}

// method for tracing back porthole aliases.
static PortHole& traceBack(PortHole* ptr) {
	GenericPort* g;
	while ((g = ptr->aliasFrom()) != 0)
		// cast is safe: alias for PortHole is always PortHole
		ptr = (PortHole*)g;
	return *ptr;
}

	////////////////////////////
	// handleDelays
	////////////////////////////

void handleDelays() {
	
	ListIter nextGeo(delayGeos);
	Geodesic* geo;

	while ((geo = (Geodesic *) nextGeo++) != 0) {
		PortHole* src = geo->sourcePort();
		PortHole* dest = geo->destPort();

		Block* sB = src->parent()->parent();
		while(sB && strncmp(sB->name(), "!worm", 5))
			sB = sB->parent();

		Block* dB = dest->parent()->parent();
		while(dB && strncmp(dB->name(), "!worm", 5))
			dB = dB->parent();

		// compare galaxies
		// if the geodesic will connect two different wormholes,
		// create  galaxy portholes.
		if (sB != dB) {
			// if different create galaxy ports.
			LOG_NEW; GalPort* ip = new GalPort(traceBack(dest));
			dB->addPort(ip->setPort( 
				galPortName(dest->parent()->name(), 
				dest->name()), dB));
	
			LOG_NEW; GalPort* op = new GalPort(traceBack(src));
			sB->addPort(op->setPort(
			   	galPortName(src->parent()->name(), 
				src->name()), sB));
		}
	}
}

	////////////////////////////
	// deCrust
	////////////////////////////

Galaxy* deCrust(Star* dw) {

	BlockPortIter nextp(*dw);
	PortHole* p;
	Galaxy* myGal = 0;

	while((p = nextp++) != 0) {
		int delay = p->numInitDelays();
		PortHole* tempP = p->far();
		EventHorizon* eveP = p->asEH();
		PortHole* galP = eveP->ghostAsPort()->far();
		GenericPort* alp = p->aliasFrom();
		Block* galS = galP->parent();
		while (!galS->parent()->isItWormhole())
			galS = galS->parent();
		myGal = (Galaxy*) galS;
		tempP->disconnect(1);
		galP->disconnect(1);
		// remove parent pointers, so portholes won't try to
		// remove themselves from their parent.
		eveP->ghostAsPort()->setPort("",0);
		eveP->asPort()->setPort("",0);
		// now remove the event horizon pair.
                LOG_DEL; delete eveP;
		if (tempP->isItInput()) 
			galP->connect(*tempP, delay);
		else
			tempP->connect(*galP, delay);

		// update aliase pointer if necessary.
		// since deleting EventHorizons may corrupt aliase pointers.
		if (alp != 0)
			alp->setAlias(*(galP->aliasFrom()));
		
	}

	// reclaim the memory taken by "dw"
	DDFStar* ds = (DDFStar*) dw;
	DDFWormhole* w = ds->myWorm();
	w->stickGal();		// not to destroy myGal.
	LOG_DEL; delete w;

	return myGal;
}

	////////////////////////////
	// renewNumTokens
	////////////////////////////

// after compile-time scheduling, change the parameter
// of the EventHorizons.

void renewNumTokens (Block* b, int flag) {

	BlockPortIter nextp(*b);
	PortHole* p;

	while ((p = nextp++) != 0) {
		EventHorizon* eveP = p->asEH();
		PortHole* ghostP = eveP->ghostAsPort();
		PortHole* bP = ghostP->far();
		int num = bP->numXfer();
		Star* s = (Star*) bP->parent();
		// if b is a SDF wormhole
		if (flag == TRUE) {
			SDFStar* sS = (SDFStar*) s;
			num *= sS->repetitions.num();
		} else {	// DDF wormhole
			if (num == 0) num = 1;
		}
		ghostP->asEH()->setParams(num);
	}
}

	////////////////////////////
	// changeDDFtoSDF
	////////////////////////////

// if the DDF domain is a predefined construct, change the crust
// from DDFWormhole to SDFWormhole. ->necessary for restructuring.

Star* changeDDFtoSDF (Star* s) {

	SequentialList ports;

	ports.initialize();
	
	// save far() ports of the EventHorizons.
	BlockPortIter nextp(*s);
	PortHole* p;

	while ((p = nextp++) != 0) {
		ports.put(p->far());
	}
	
	// deCrust the DDFWormhole
	Galaxy* inGal = deCrust(s);

	// make SDFWormhole
	LOG_NEW; SDFWormhole* sW = new SDFWormhole(*inGal);

	// remake connections.
	ListIter farp(ports);

	while ((p = (PortHole*) farp++) != NULL) 
		reConnect(*p, *(p->far()));
		
	renewNumTokens(sW, FALSE);
	return sW;
}
