defstar {
	name {Window}
	domain {CG96}
	desc {
Apply a window function of length N and of the specified
type to the input block.
        }
	version {@(#)CG96Window.pl	1.5 12/8/92}
	author { Chih-Tsung Huang (ported from gabriel) }
        acknowledge { Gabriel version by Anthony Wong }
	copyright {
Copyright (c) 1990, 1991, 1992 The Regents of the University of California.
All rights reserved.
See the file ~ptolemy/copyright for copyright notice,
limitation of liability, and disclaimer of warranty provisions.
	}
	location { CG96 demo library }
        explanation {
.PP
The star windows a block of input of length \fIlength\fR.
If \fIname\fR is "special",
the block of data is scaled by the coefficients given by
\fIwindowSequence\fR.
The sequence is zero-padded or truncated to \fIlength\fR as necessary.
Otherwise, the block of data is scaled by the coefficients of the
type of window specified by \fIname\fR.
.PP
\fIname\fR can be one of the following 4 types:
.PP
"triangular" or "bartlett" - given by
.IP
W(n) = length/2 + sgn(n - length/2) * (n - length/2), where sgn is
the signum function
.PP
"hamming" - given by
.IP
W(n) = 0.54 - 0.46*cos(2*pi*n/(length-1))
.PP
"hanning" - given by
.IP
W(n) = sqr(sin(pi*n/(length-1))), where sqr(x) = x^2
.PP
"blackman"
.IP
W(n) = 0.42 - 0.5*cos(2*pi*n/(length-1)) + 0.08*cos(4*pi*n/(length-1))
	}
	input {
		name{input}
		type{float}
	}
	output {
		name{output}
		type{float}
	}
	state {
		name {name}
		type {string}
		default {"hamming"}
                desc {Name of the window function to generate.}
	}
        state {
                name {length}
                type {int}
                default {16}
                desc { Length of the window function to produce.}
        }
	state {
		name {windowSequence}
		type {floatarray}
		default {"0.1 0.2 0.3 0.4"}
		desc {internal}
        }
	state {
		name {coef}
		type {floatarray}
		default {"0"}
		desc {internal}
	        attributes {A_NONCONSTANT|A_NONSETTABLE}
        }
	state {
		name {dcVal}
		type {float}
		default {0}
		desc {internal}
	        attributes {A_NONCONSTANT|A_NONSETTABLE}
        }
	state {
		name {X}
		type {int}
		default {"0"}
		desc {internal}
	        attributes {A_NONCONSTANT|A_NONSETTABLE}
        }
        protected {
        int winType;
        double half;
        }

        code {
        extern "C" {
            extern int strcasecmp(const char*,const char*);
        }
#define CG96WinType_Null         (0)
#define CG96WinType_Triangular   (1)
#define CG96WinType_Bartlett     (2)
#define CG96WinType_Hanning      (3)
#define CG96WinType_Hamming      (4)
#define CG96WinType_Blackman     (5)
#define CG96WinType_Special      (6)
        } 

	setup {
		const char     *wn = name;
		 /* IF */ if (strcasecmp(wn, "Triangular") == 0) {
			winType = CG96WinType_Triangular;
		} else if (strcasecmp(wn, "Bartlett") == 0) {
			winType = CG96WinType_Bartlett;
		} else if (strcasecmp(wn, "Hanning") == 0) {
			winType = CG96WinType_Hanning;
		} else if (strcasecmp(wn, "Hamming") == 0) {
			winType = CG96WinType_Hamming;
		} else if (strcasecmp(wn, "Blackman") == 0) {
			winType = CG96WinType_Blackman;
		} else if (strcasecmp(wn, "Special") == 0) {
			winType = CG96WinType_Special;
		} else {
			Error::abortRun(*this, ": Unknown window name");
			return;
		}

		switch (winType) {
		case CG96WinType_Special:
			windowSequence.setAttributes(A_YMEM);
			windowSequence.resize(int (length));
			break;
		default:
			coef.setAttributes(A_YMEM|A_NOINIT);
			coef.resize(int (ceil (double (length) / 2.0)));
		}
		input.setSDFParams(int (length), int (length) - 1);
		output.setSDFParams(int (length), int (length) - 1);
	}


	initCode {
           half=double(length)/2.0;
           double twoPi = 2.0 * M_PI;
	       
           if(length>1 && winType!=CG96WinType_Special) {
		addCode(orgdc);
             for(int i=0; i< half; i++) {
               switch ( winType ) {
               case CG96WinType_Triangular:
                   dcVal= double(i)/(double(half)-0.5);
                   break;
               case CG96WinType_Bartlett:
          	   dcVal= double(i)/(double(half)-0.5);
                   break;
               case CG96WinType_Hanning:
                   dcVal= 0.5 * (1-cos(twoPi* double(i)/(double(length)-1)));
                   break;
               case CG96WinType_Hamming:
                   dcVal= 0.54 - 0.46 * (cos(twoPi* double(i)/(double(length)-1)));
                   break;
               case CG96WinType_Blackman:
                   dcVal= (0.42 - 0.5 * (cos(twoPi* double(i)/(double(length)-1)))
		           + 0.08 * (cos(2 * twoPi* double(i)/(double(length)-1))));
                   break;
               case CG96WinType_Special:
                   return;
               default:
                   Error::abortRun(*this, ": Invalid window type");
                   return;
               }
               addCode(dc);
             }
          }
	}

 	go {
             if(winType==CG96WinType_Special)
		 addCode(special);
	     else if(length>1) {
                 X=int((floor(length/2.0)));
                 int modtemp=length%2;

	         addCode(others);

                 if(modtemp!=0)
			addCode(odd);
		 addCode(cont);
             }
	     else addCode(none);
	}

        execTime {
		return 7+4*int(length);
		if(winType==CG96WinType_Special) return 5+4*int(length);
		else if(length>1) return 7+4*int(length);
		if(length==1) return 19+ (4*int(length));
		else return 5;
	}

	codeblock(orgdc) {
; Compute cofficients -- $fullname()
	org	y:$addr(coef)
	}
        codeblock(dc) {
        dc      $val(dcVal)
        }

        codeblock(special) {
        move    #<$addr(input),r0
        move    #<$addr(windowSequence),r1
        move    #<$addr(output),r2
        do      #$val(length),$label(loop1)
        move    x:(r0)+,d0.s
        move    y:(r1)+,d1.s
        fmpy.s  d0,d1,d0
        move    d0.s,x:(r2)+
$label(loop1)
        }
        codeblock(others) {
        move    #<$addr(input),r0
        move    #<$addr(coef),r1
        move    #<$addr(output),r2
        do      #$val(X),$label(loop2)
        move    x:(r0)+,d0.s
        move    y:(r1)+,d1.s
        fmpy.s  d0,d1,d0
        move    d0.s,x:(r2)+
$label(loop2)
        }
        codeblock(odd) {
        move    x:(r0)+,d0.s
        move    y:(r1),d1.s
        fmpy.s  d0,d1,d0
        move    d0.s,x:(r2)+
        }
        codeblock(cont) {
        do      #$val(X),$label(loop3)
        move    x:(r0)+,d0.s
        move    y:-(r1),d1.s
        fmpy.s  d0,d1,d0
        move    d0.s,x:(r2)+
$label(loop3)
        }
    
        codeblock(none) {
        move    x:(r0),d0.s
        move    d0.s,x:(r2)
        }    
}
