static const char file_id[] = "CGForScheduler.cc";

#ifdef __GNUG__
#pragma implementation
#endif

#include "CGForScheduler.h"
#include "Galaxy.h"
#include "DataStruct.h"
#include "CGDDFWormhole.h"
#include "distributions.h"
#include "CodeBlock.h"
#include <math.h>

/**************************************************************************
Version identification:
@(#)CGForScheduler.cc	1.6	11/25/92

Copyright (c) 1990, 1991, 1992 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY 
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES 
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF 
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF 
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.
							COPYRIGHTENDKEY

 Programmer:  Soonhoi Ha

 Methods for CGForScheduler class.
**************************************************************************/

int CGForScheduler :: getStatistics(Target* tg) {

	// read the type of the distribution
	if (!dist.setType(tg)) return FALSE;

	// set the paramter information
	return (dist.myParam()->setParams(tg)); 
}

int CGForScheduler :: examine(Galaxy& galaxy) {
        For::clearSchedules();
        if (!For::checkTopology(galaxy)) {
           Error::abortRun(galaxy, " has wrong topology for For construct ");
           return FALSE;
        }
	return TRUE;
}

int CGForScheduler :: closerExamine() {

	// check whether there is only one wormhole or more.
	if (schedules[1].size() != 1) {
		Error::abortRun("For construct wants a single wormhole!");
		return FALSE;
	}
	return TRUE;
}

static int downLoaded;

	///////////////////////
	//  initMembers
	///////////////////////

void CGForScheduler :: initMembers() {
	// Obtain the profile of the iteration body.
	ListIter bodyWorm(schedules[1]);
	CGDDFStar* wormStar = (CGDDFStar*) bodyWorm++;
	worm = wormStar->myWormhole();
	body = worm->myProfile();
	downLoaded = FALSE;
}

	///////////////////////
	//  assumeExecTime
	///////////////////////

int CGForScheduler :: assumeExecTime() {

	// take the total work of the wormhole
	int work = worm->getTotalWork();
	double avg = dist.myParam()->assumedValue();
	avg = avg * double(work);

	return int(avg);
}

// local to this file
static int tau;
static int bestK;
static int bestX;

	///////////////////////
	//  setProfile
	///////////////////////

int CGForScheduler :: setProfile(int num, int resWork,
			Profile* prof) {

	// compute the optimal number of assigned processors.
	Profile& p = body[num-1];
	
	// check the effective number of processors
	if (p.getEffP() < num) return FALSE;

	// calculate best K value.
	bestK = numProcs / num;
	int tk = p.ToverK();
	if (tk && bestK > tk) bestK = tk;

	// compute the profile
	tau = p.getMakespan();
	double totalCost = calcCost(num, resWork);

	// record the profile
	int offset = p.getMinDisplacement();
	int m1, m2, span;
	m1 = bestX / bestK;
	m2 = bestX % bestK;
	for (int i = 0; i < bestK; i++) {
		int move = offset * i;
		if (i < m2) span = (m1 + 1) * tau; 
		else	    span = m1 * tau;
		for (int j = 0; j < num; j++) {
			int index = j + i * num;
			prof->setStartTime(index, move);
			prof->setFinishTime(index, move + span);
		}
	}
	
	// add control codes to the profile.
	totalCost += addControlCodes(num, prof);

	prof->setEffP(bestK*num);
	prof->setTotalCost(totalCost);
	prof->summary();
	return TRUE;
}
	
        ///////////////////////
        //  addControlCodes
        ///////////////////////

int CGForScheduler :: addControlCodes(int num, Profile* prof) {

	ListIter up(schedules[0]);
	ListIter down(schedules[2]);
	CGDDFStar* s;

	// add upSample codes
        int sumUp = 0;
	
	while ((s = (CGDDFStar*) up++) != 0) {
		sumUp += s->myExecTime();
	}
        prof->setFinishTime(0, prof->getFinishTime(0) + sumUp);
	int temp = num * bestK;
        for (int i = 1; i < temp; i++) {
                prof->setStartTime(i, prof->getStartTime(i) + sumUp);
                prof->setFinishTime(i, prof->getFinishTime(i) + sumUp);
        }

	// add downSample codes
        int sumDown = 0;
	while ((s = (CGDDFStar*) down++) != 0) {
		sumDown += s->myExecTime();
	}

	int ix = num * (bestX % bestK - 1);
	if (ix < 0) ix = num * (bestK - 1);
        prof->setFinishTime(ix, prof->getFinishTime(ix) + sumDown);
        prof->setSyncPoint(ix);

        return (sumUp + sumDown);
}

	///////////////////////
	//  calcCost
	///////////////////////

double CGForScheduler :: calcCost(int num, int resWork) {

	// calculate bestX based on the distribution of the
	// the iteration steps.
	switch (dist.readType()) {
		case GEOMETRIC:	return costInGeometric(num,resWork);
		case UNIFORM:	return costInUniform(num,resWork);
		case GENERAL:	return costInGeneral(num,resWork);
		default:   return FALSE;
	}
}
		
// Geomoteric distribution
double CGForScheduler :: costInGeometric(int num, int resWork) {

	// calculate x
	double procRatio = double(num) / double(numProcs);
	DistGeometric* geo = (DistGeometric*) dist.myParam();
	double p = geo->getP();
	double z = 1 - pow(p,bestK);
	double temp = procRatio * z / (1 - p);
	double ratio = log(temp) / log(p);
	bestX = adjustX(geo->getMin(), int(ratio), resWork, num);
	int dyX = bestX - geo->getMin();
	
	// calculate total cost due to the non-deterministic part.
	double total = tau * 
		(num*bestX + numProcs*pow(p, dyX + 1)/z);

	return total;
}

// Uniform distribution
double CGForScheduler :: costInUniform(int num, int resWork) {

	// calculate x
	double procRatio = double(num) / double(numProcs);
	DistUniform* uni = (DistUniform*) dist.myParam();
	int tmax = uni->getMax();
	int tmin = uni->getMin();
	double p = tmax - tmin + 1;
	double tsum = procRatio*p;
	double temp = ceil(tsum - 1) * bestK;
	int tx = tmax - int(temp) - tmin;
	bestX = adjustX(tmin, tx, resWork, num);
	
	// calculate total cost
	double l = floor( double(tmax - bestX)/double(bestK) );
	// keep the second term since bestX can be modified by adjustX().
	tsum = bestK * l * (l+1) / 2 + (l+1) * (tmax - bestX - l * bestK);
	double total = tau * (num*bestX + numProcs*tsum/p);

	return total;
}

// adjust "x" - if too much idle time, reduce "x".
// But retain the skewed structure.
int CGForScheduler :: adjustX(int min, int x, int resWork, int num) {

	int newX = min + x;
	if (numProcs == num * bestK) return newX;

	double make = double (resWork) / double (numProcs - num * bestK);
	int temp = newX / bestK;
	int skew = newX % bestK;
	// compare with remaining work, the processor-time space in the
	// non-iteration processors.
	if ( (tau*temp) > int(make)) {
		// adjust "x" - reduce it.
		temp = int(make / tau);
		if (temp*tau == make && temp > 0) temp--;
		if (skew == 0) skew = bestK;
		newX = temp*bestK + skew;
	}
	if (newX < min) newX = (min/bestK)*bestK + skew;
	return newX;
}
		

// General distribution
double CGForScheduler :: costInGeneral(int num, int resWork) {

	// calculate x
	double procRatio = double(num) / double(numProcs);
	DistGeneral* geo = (DistGeneral*) dist.myParam();
	double psum = 0;
	int size = geo->tableSize();

	// Among "bestK" values of x, obtain the largest
	// Then, the "bestX" is among [x, x+bestK].
	double limit = procRatio*double(bestK);
	int i = 0;
	while (psum < limit && i < size) {
		psum += geo->getTable(i).value;
		i++;
	}

	int largestX = geo->getTable(i-1).index - 1 + bestK;

	// adjust "largestX" if too much idle time.
	if (numProcs > num * bestK) {
		double make = double(resWork) / double(numProcs - num * bestK);
		int temp = largestX / bestK;
		// compare with remaining work, the processor-time space in 
		// the non-iteration processors.
		if ( (tau*temp) > int(make)) {
			// adjust "largestX" - reduce it.
			largestX = (int(make / tau) + 1) * bestK;
		}
	}

	// Compare total costs with x with [largestX-bestK+1, largestX].
	// Select x with smallest total cost.
	double minSum = numProcs * tau * geo->getTable(0).index;
	bestX = largestX;		// initialize.

	for (int k = 0; k > bestK; k++) {
		int tempx = largestX - k;

		// calculate total cost due to the non-deterministic part.
		double tot = 0;
		i = 0;
		int ix = geo->getTable(i).index;
		while (ix > tempx && i < size) {
			psum = double(ix - tempx)/double(bestK);
			tot += geo->getTable(i).value * ceil(psum);
			i++;
			if (i < size) ix = geo->getTable(i).index;
		}
		tot = (tot*numProcs + num * tempx) *tau;

		// update minSum and optimal X.
		if (tot < minSum) {
			minSum = tot;
			bestX = tempx;
		}
	}

	return minSum;
}

static int indexMax;
int indexOfLongest(Profile*);

        ///////////////////////
        //  download the code
        ///////////////////////

static CodeBlock Start("\n .... start of (For) ....\n");
static CodeBlock End("\n .... end of (For) ....\n");

void CGForScheduler :: downLoadCode(int ix, Target* tg, Profile* prof) {

	if (!downLoaded) {
		indexMax = indexOfLongest(prof);
		downLoaded = TRUE;
	}

        addCode(tg, Start.getText());

        if (ix == 0) {
		// UpSample actors
		ListIter up(schedules[0]);
		Star* s;
		while ((s = (Star*)up++) != 0) s->run();

		// add Control.
                addCode(tg, controlCode());
        }

        // inside wormholes.
        worm->downLoadCode(ix);
        
	if (ix == indexMax) {
		// Add syncCode
		addCode(tg, syncCode());

		// DownSample actors
		ListIter down(schedules[2]);
		Star* s;
		while ((s = (Star*)down++) != 0) s->run();
	}

        addCode(tg, End.getText());
}

int indexOfLongest(Profile* prof) {
	int ix = prof->getEffP() - 1;
	int maxVal = 0;
	
	while (ix >= 0 && maxVal <= prof->getFinishTime(ix)) {
		maxVal = prof->getFinishTime(ix);
		ix--;
	}
	ix++;
	return ix;
}
		

// virtual methods --- default

static CodeBlock Control("-- check the control value.\n");
static CodeBlock Sync("-- synchronization code.\n");

const char* CGForScheduler :: controlCode()  const {
        return Control.getText();
}

const char* CGForScheduler :: syncCode()  const {
        return Sync.getText();
}

