/* $Id: draw.c,v 1.32 90/05/09 22:11:46 pturner Exp Locker: pturner $
 *
 * interface to device drivers
 *
 *
 * 	TODO: finish implementing log code (log plots will be
 * 	done by the graphics routines rather than transforming the
 * 	data).
 *
 * 	TODO: finish implementing code for filled barcharts
 */

#include <stdio.h>
#include <math.h>

double charsize = 1.0;

static double xg1, xg2, yg1, yg2;	/* world coordinates */
static double dxg, dyg;		/* delta x, y of world co-ordinates */
static double xv1, xv2, yv1, yv2;	/* viewpoint coordinates */
static double dxv, dyv;		/* distance covered by viewport */
static double rx, ry;		/* dxv/dxg & dyv/dyg */
static double crx, cry;		/* constants for world -> normal */
static double curx, cury;	/* current location of pen */
static int color;		/* current color */
static int lines;		/* current linestyle */
static int ylabpos;		/* info for positioning y-axis label */
static int devtype;		/* current device */
static int clipflag = 1;	/* clip in draw2 */
static int logx = 0, logy = 0;	/* log scaling flags */

double devcharsize;		/* device charsize */
int (*devsetcolor) ();		/* device set color */
int (*devsetfont) ();		/* device set font */
int (*devsetline) ();		/* device set line style */
int (*devsetpat) ();		/* device set fill pattern */
int (*devfill) ();		/* device fill polygon */
int (*devconvx) ();		/* viewport to device mapping for x */
int (*devconvy) ();		/* viewport to device mapping for y */
int (*vector) ();		/* device draw line */
int (*devwritestr) ();		/* device write string */
int (*devdrawtic) ();		/* device draw tic */
int (*devleavegraphics) ();	/* device exit */
int devxticl;			/* device tic length for x-axis */
int devyticl;			/* device tic length for y-axis */
int devarrowlength;		/* device tic length for y-axis */
int devfontsrc;			/* device or hershey font selector */
static int savexticl;		/* save device tic length for x-axis */
static int saveyticl;		/* save device tic length for y-axis */

int curfontd = 0;		/* current font */
int current_pattern;
static double hdelta;

double xconv(), yconv();
void move2();

extern int stringextentx();	/* in chersh.c */
extern int stringextenty();

/*
 * get_world - given (x,y) in screen coordinates, return the world coordinates
 *             in (wx,wy) used for the display only - in this case the canvas
 */
void get_world(x, y, wx, wy)
    int x, y;
    double *wx, *wy;
{
    extern int win_h, win_w;

    *wx = ((double) x / win_w - crx) / rx;
    *wy = ((double) (y - win_h) / (-win_h) - cry) / ry;
}

/*
 * get_view - given (x,y) in world coordinates, return viewport coordinates
 *            in (vx,vy)
 */
void get_view(x, y, vx, vy)
    double x, y;
    double *vx, *vy;
{
    *vx = xconv(x);
    *vy = yconv(y);
}

/*
 * get_view2world - given (x,y) in viewport coordinates, return world coordinates
 *            in (vx,vy)
 */
void get_view2world(vx, vy, x, y)
    double vx, vy;
    double *x, *y;
{
    *x = (vx - crx) / rx;
    *y = (vy - cry) / ry;
}

/*
 * get_device - given world coordinates (wx,wy) return the device coordinates,
 *              for the display only
 */
void get_device(wx, wy, x, y)
    double wx, wy;
    int *x, *y;
{
    extern int win_h, win_w;

    *x = (*devconvx) (wx);
    *y = win_h - (*devconvy) (wy);
}

/*
 * map world co-ordinates to viewport
 */
double xconv(x)
    double x;
{
    return (rx * x + crx);
}

double yconv(y)
    double y;
{
    return (ry * y + cry);
}

/*
 * defineworld - really should be called definewindow, defines the scaling
 *               of the plotting rectangle to be used for clipping
 */
void defineworld(x1, y1, x2, y2)
    double x1, y1, x2, y2;
{
    xg1 = x1;
    xg2 = x2;
    yg1 = y1;
    yg2 = y2;
    dxg = xg2 - xg1;
    dyg = yg2 - yg1;
}

/*
 * viewport - define the location of the clipping rectangle defined by
 *            defineworld on the display device
 */
void viewport(x1, y1, x2, y2)
    double x1, x2, y1, y2;
{
    xv1 = x1;
    xv2 = x2;
    yv1 = y1;
    yv2 = y2;
    dxv = x2 - x1;
    dyv = y2 - y1;
    rx = dxv / dxg;
    ry = dyv / dyg;
    crx = -rx * xg1 + xv1;
    cry = -ry * yg1 + yv1;
}

#ifdef D3D

#define MAXSCAN 17000
#define MAYSCAN 17000

static short amax[MAXSCAN], amin[MAXSCAN];
int hidflag = 0;

/*
    initialize the upper and lower horizon arrays
*/
void inithorizon()
{
    int i;

    for (i = 0; i < MAXSCAN; i++) {
	amax[i] = -20000;
	amin[i] = 20000;
    }
}

/*
    to avoid indexing problems
*/
int aok(x, y)
    int x, y;
{
    return ((x < MAXSCAN) && (x >= 0) && (y < MAYSCAN) && (y >= 0));
}

/*
   check for visibility on top
*/
int vistop(x, y)
    int x, y;
{
    int ok = 0;

    if (aok(x, y)) {
	if (y >= amax[x]) {
	    amax[x] = y;
	    return 1;
	} else {
	    return 0;
	}
    }
    return 0;
}

/*
   check for visibility on bottom
*/
int visbot(x, y)
    int x, y;
{
    int ok = 0;

    if (aok(x, y)) {
	if (y <= amin[x]) {
	    amin[x] = y;
	    return 1;
	} else {
	    return 0;
	}
    }
    return 0;
}

/*
   a Bresenham with a check for visibility
   2 pass method:
      1st is to check for visibility above the upper horizon
      2nd to check for vis below the lower horizon

   lines must be draw starting from the min(y1,y2) for the 1st pass
   and max(y1,y2) for the 2nd pass.
*/
void bres(x1, y1, x2, y2)
    int x1, y1, x2, y2;
{
    int first, tmp, startx, starty, plotox, plotoy, x, y, dx, dy, xs, ys, dir;
    int linetodraw = 0;

    /* swap points if (x1,y1) lies above (x2,y2) */
    if (y1 >= y2) {
	tmp = x1;
	x1 = x2;
	x2 = tmp;
	tmp = y1;
	y1 = y2;
	y2 = tmp;
    }
    x = x1;
    y = y1;
    xs = 1;
    ys = 1;
    if (x1 > x2)
	xs = -1;
    if (y1 > y2)
	ys = -1;
    dx = abs(x2 - x1);
    dy = abs(y2 - y1);
    if (dx == 0) {
	dir = -1;
    } else {
	dir = 0;
    }
    first = 1;
    /*
     * to avoid problems at the end of the line if (y2 == amax[x2]) {
     * amax[x2] = -20000; }
     */
    while (!((x == x2) && (y == y2))) {

	if (vistop(x, y) || (y2 >= amax[x2] && x == x2)) {
	    if (first) {
		startx = x;
		starty = y;
		plotox = x;
		plotoy = y;
		linetodraw = 1;
		first = 0;
	    } else {
		plotox = x;
		plotoy = y;
	    }
	} else {
	    if (linetodraw) {
		(*vector) (startx, starty, 0);
		(*vector) (plotox, plotoy, color);
		linetodraw = 0;
		first = 1;
	    }
	}
	if (dir < 0) {
	    y = y + ys;
	    dir = dir + dx;
	} else {
	    x = x + xs;
	    dir = dir - dy;
	}
    }
    if (linetodraw) {
	(*vector) (startx, starty, 0);
	(*vector) (plotox, plotoy, color);
    }
    if (y1 <= y2) {
	tmp = x1;
	x1 = x2;
	x2 = tmp;
	tmp = y1;
	y1 = y2;
	y2 = tmp;
    }
    x = x1;
    y = y1;
    xs = 1;
    ys = 1;
    if (x1 > x2)
	xs = -1;
    if (y1 > y2)
	ys = -1;
    dx = abs(x2 - x1);
    dy = abs(y2 - y1);
    if (dx == 0) {
	dir = -1;
    } else {
	dir = 0;
    }
    first = 1;
    while (!((x == x2) && (y == y2))) {

	if (visbot(x, y) || (y2 <= amin[x2] && x == x2)) {
	    /* cgaplot(x, y, 1, 1); */
	    if (first) {
		startx = x;
		starty = y;
		plotox = x;
		plotoy = y;
		linetodraw = 1;
		first = 0;
	    } else {
		plotox = x;
		plotoy = y;
	    }
	} else {
	    if (linetodraw) {
		(*vector) (startx, starty, 0);
		(*vector) (plotox, plotoy, color);
		linetodraw = 0;
		first = 1;
	    }
	}
	if (dir < 0) {
	    y = y + ys;
	    dir = dir + dx;
	} else {
	    x = x + xs;
	    dir = dir - dy;
	}
    }
    if (linetodraw) {
	(*vector) (startx, starty, 0);
	(*vector) (plotox, plotoy, color);
    }
}

#endif				/* D3D */

/*
 * clip if clipflag = TRUE
 */
void setclipping(fl)
{
    clipflag = fl;
}

/*
 * clip lines within rectangle defined in world coordinates
 */
static int region(xend, yend)
    double xend, yend;
{
    int endpoint;

    endpoint = 0;
    if (xend < xg1)
	endpoint = 1;		/* left */
    else {
	if (xend > xg2)
	    endpoint = 2;	/* right */
    }
    if (yend < yg1)
	endpoint |= 4;		/* bottom */
    else {
	if (yend > yg2)
	    endpoint |= 8;	/* top */
    }
    return (endpoint);
}

/*
 * clip the line (x1,y1)-(x2,y2)
 */
int clipline(xg1, yg1, xg2, yg2, x1, y1, x2, y2)
    double xg1, yg1, xg2, yg2;
    double *x1, *y1, *x2, *y2;
{
    double m, minverse, x, y;
    int dir, dir1, dir2, region();

    dir1 = region(*x1, *y1);
    dir2 = region(*x2, *y2);
    if (*x1 != *x2)
	m = (*y2 - *y1) / (*x2 - *x1);
    if (*y2 != *y1)
	minverse = (*x2 - *x1) / (*y2 - *y1);
    while ((dir1 != 0) || (dir2 != 0)) {
	if (dir1 & dir2) {
	    return 0;
	}
	if (dir1 == 0) {
	    dir = dir2;
	    x = *x2;
	    y = *y2;
	} else {
	    dir = dir1;
	    x = *x1;
	    y = *y1;
	}
	if (1 & dir) {
	    y = m * (xg1 - *x1) + *y1;
	    x = xg1;
	} else {
	    if (2 & dir) {
		y = m * (xg2 - *x1) + *y1;
		x = xg2;
	    } else {
		if (4 & dir) {
		    x = minverse * (yg1 - *y1) + *x1;
		    y = yg1;
		} else {
		    if (8 & dir) {
			x = minverse * (yg2 - *y1) + *x1;
			y = yg2;
		    }
		}
	    }
	}
	if (dir == dir1) {
	    *x1 = x;
	    *y1 = y;
	    dir1 = region(*x1, *y1);
	} else {
	    *x2 = x;
	    *y2 = y;
	    dir2 = region(*x2, *y2);
	}
    }
    return 1;
}

/*
 * draw2 - draw a line from the current point (curx,cury) to (x2,y2) clipping the
 *         line to the viewport providing clipflag is TRUE
 */
void draw2(x2, y2)
    double x2, y2;
{
    double m, minverse, x, y, x1, y1, xtmp, ytmp;
    int dir, dir1, dir2, region();

    x1 = curx;
    y1 = cury;
    xtmp = x2;
    ytmp = y2;
    if (!clipflag) {
	move2(x1, y1);
	(*vector) ((*devconvx) (x2), (*devconvy) (y2), color);
	curx = xtmp;
	cury = ytmp;
	return;
    }
    dir1 = region(x1, y1);
    dir2 = region(x2, y2);
    if (x1 != x2) {
	m = (y2 - y1) / (x2 - x1);
    }
    if (y2 != y1) {
	minverse = (x2 - x1) / (y2 - y1);
    }
    while ((dir1 != 0) || (dir2 != 0)) {
	if (dir1 & dir2) {
	    curx = xtmp;
	    cury = ytmp;
	    return;
	}
	if (dir1 == 0) {
	    dir = dir2;
	    x = x2;
	    y = y2;
	} else {
	    dir = dir1;
	    x = x1;
	    y = y1;
	}
	if (1 & dir) {
	    y = m * (xg1 - x1) + y1;
	    x = xg1;
	} else {
	    if (2 & dir) {
		y = m * (xg2 - x1) + y1;
		x = xg2;
	    } else {
		if (4 & dir) {
		    x = minverse * (yg1 - y1) + x1;
		    y = yg1;
		} else {
		    if (8 & dir) {
			x = minverse * (yg2 - y1) + x1;
			y = yg2;
		    }
		}
	    }
	}
	if (dir == dir1) {
	    x1 = x;
	    y1 = y;
	    dir1 = region(x1, y1);
	} else {
	    x2 = x;
	    y2 = y;
	    dir2 = region(x2, y2);
	}
    }
    move2(x1, y1);
#ifdef D3D
    if (hidflag) {
	bres((*devconvx) (x1), (*devconvy) (y1), (*devconvx) (x2), (*devconvy) (y2));
    } else {
	(*vector) ((*devconvx) (x2), (*devconvy) (y2), color);
    }
#else				/* 2D version */
    (*vector) ((*devconvx) (x2), (*devconvy) (y2), color);
#endif				/* D3D */
    curx = xtmp;
    cury = ytmp;
}

/*
 * move2 - make (x,y) the current point (curx,cury)
 */
void move2(x, y)
    double x, y;
{
    (*vector) ((*devconvx) (x), (*devconvy) (y), 0);
    curx = x;
    cury = y;
}

/*
 * setfont - make f the current font to use for writing strings
 */
void setfont(f)
    int f;
{
    (*devsetfont) (f);
    curfontd = f;
}

/*
 * rect - draw a rectangle using the current color and linestyle
 */
void rect(x1, y1, x2, y2)
    double x1, x2, y1, y2;
{
    move2(x1, y1);
    draw2(x1, y2);
    draw2(x2, y2);
    draw2(x2, y1);
    draw2(x1, y1);
}

/*
 * symok - return TRUE if (x,y) lies within the currently defined window in
 *         world coordinates. used to clip symbols that bypass the clipping
 *         done in draw2.
 */
int symok(x, y)
    double x, y;
{
    return ((x >= xg1) && (x <= xg2) && (y >= yg1) && (y <= yg2));
}

/*
 * histbox - draw a box of width 2*hdelta centered at x with height y starting
 *           from y = 0.0
 */
static void histbox(x, y)
    double x, y;
{
/*    double tmpx[4];
    double tmpy[4];
    int i;
    tmpx[0] = x - hdelta;
    tmpy[0] = 0.0;
    tmpx[1] = x - hdelta;
    tmpy[1] = y;
    tmpx[2] = x + hdelta;
    tmpy[2] = y;
    tmpx[3] = x + hdelta;
    tmpy[3] = 0.0;
    for (i=0;i<4;i++) {
       if (tmpx[i] < xg1)
	   tmpx[i] = xg1;
       else if (tmpx[i] > xg2)
	   tmpx[i] = xg2;
       if (tmpy[i] < yg1)
	   tmpy[i] = yg1;
       else if (tmpy[i] > yg2)
	   tmpy[i] = yg2;
    }
    setpattern(current_pattern);
    fillpattern(4,tmpx,tmpy);
*/
    move2(x - hdelta, 0.0);
    draw2(x - hdelta, y);
    draw2(x + hdelta, y);
    draw2(x + hdelta, 0.0);
    draw2(x - hdelta, 0.0);
}

/*
 * histbox2 - draw a stairstep, each step is width 2*hdelta
 */
static void histbox2(x, y, y2, hflag)	/* h is flag set to true if this is
					 * the last point */
    double x, y, y2;
    int hflag;
{
    move2(x - hdelta, y);
    draw2(x + hdelta, y);
    if (!hflag)
	draw2(x + hdelta, y2);
}

/*
 * histbox3 - draw a box of width 2*hdelta centered at y with length x starting
 *           from x = 0.0
 */
static void histbox3(x, y)
    double x, y;
{
    int i;

/*    double tmpx[4];
    double tmpy[4];
    tmpy[0] = y - hdelta;
    tmpx[0] = 0.0;
    tmpy[1] = y - hdelta;
    tmpx[1] = x;
    tmpy[2] = y + hdelta;
    tmpx[2] = x;
    tmpy[3] = y + hdelta;
    tmpx[3] = 0.0;
    for (i=0;i<4;i++) {
       if (tmpx[i] < xg1)
	   tmpx[i] = xg1;
       else if (tmpx[i] > xg2)
	   tmpx[i] = xg2;
       if (tmpy[i] < yg1)
	   tmpy[i] = yg1;
       else if (tmpy[i] > yg2)
	   tmpy[i] = yg2;
    }
    setpattern(current_pattern);
    fillpattern(4,tmpx,tmpy);
*/
    move2(0.0, y - hdelta);
    draw2(x, y - hdelta);
    draw2(x, y + hdelta);
    draw2(0.0, y + hdelta);
    draw2(0.0, y - hdelta);
}

/*
 * histbox4 - draw a stairstep, each step is length 2*hdelta
 */
void histbox4(x, y, x2, hflag)	/* h is flag set to true if this is the last
				 * point */
    double x, y, x2;
    int hflag;
{
    move2(x, y - hdelta);
    draw2(x, y + hdelta);
    if (!hflag)
	draw2(x2, y + hdelta);
}

/*
 * draw symbol sym at (x,y) in world coordinates - symbols 1 through
 * 15 are from the Hershey fonts, the rest are generated here in draw.c and plotone.c
 */
void drawsym(x, y, sym)
    double x, y;
    int sym;
{
    double ytmp;

    if (sym == 0)
	return;
    if (symok(x, y)) {
	if ((sym > 0) && (sym < 16))
	    hwritesym(sym, (*devconvx) (x), (*devconvy) (y), charsize * devcharsize, color, vector);
	else {
	    switch (sym) {
	    case 16:		/* impulse from yg1 or y = 0.0 */
		if (yg1 < 0.0 && yg2 > 0.0)
		    ytmp = 0.0;
		else
		    ytmp = yg1;
		move2(x, ytmp);
		draw2(x, y);
		break;
	    case 17:		/* vertical line at x */
		move2(x, yg1);
		draw2(x, yg2);
		break;
	    case 18:		/* horizontal line at y */
		move2(xg1, y);
		draw2(xg2, y);
		break;
	    }
	}
    }
}

/*
 * drawpoly - draw a connected line in the current color and linestyle
 *            with nodes given by (x[],y[])
 */
void drawpoly(x, y, n)
    double x[], y[];
int n;

{
    int i;

    move2(x[0], y[0]);
    for (i = 1; i < n; i++) {
	draw2(x[i], y[i]);
    }
}

/*
 * drawpolyseg - draw segments, treating each successive pairs of points
 *               as a line segment
 */
void drawpolyseg(x, y, n)
    double x[], y[];
int n;

{
    int i, itmp;

    for (i = 0; i < (n / 2); i++) {
	itmp = i * 2;
	move2(x[itmp], y[itmp]);
	draw2(x[itmp + 1], y[itmp + 1]);
    }
}

/*
 * drawerrbar - draw (x[],y[]) with errorbars (el[],et[])
 *              etype % 2 = 0 -> errorbars on y
 *                    0 : errorbars on top and bottom
 *                    2 : errorbars on top only
 *                    4 : errorbars on bottom only
 *              etype % 2 != 0 -> errorbars on x
 *                    1 : errorbars on left and right
 *                    3 : errorbars on left only
 *                    5 : errorbars on right only
 *              width is ebarlen
 */
void drawerrbar(x, y, n1, el, et, n2, etype, ebarlen)
    double x[], y[], el[], et[], ebarlen;
    int n1, n2, etype;

{
    int i, n, ebarleny, ebarlenx;

    n = n2;
    if (n1 < n2) {
	n = n1;
    }
    ebarleny = (int) (ebarlen * devyticl);
    ebarlenx = (int) (ebarlen * devxticl);
    for (i = 0; i < n; i++) {
	switch (etype) {
	case 0:
	    move2(x[i], y[i] - el[i]);
	    draw2(x[i], y[i] + et[i]);
	    if (symok(x[i], y[i] - el[i])) {
		(*vector) ((*devconvx) (x[i]) - ebarleny, (*devconvy) (y[i] - el[i]), 0);
		(*vector) ((*devconvx) (x[i]) + ebarleny, (*devconvy) (y[i] - el[i]), color);
	    }
	    if (symok(x[i], y[i] + et[i])) {
		(*vector) ((*devconvx) (x[i]) - ebarleny, (*devconvy) (y[i] + et[i]), 0);
		(*vector) ((*devconvx) (x[i]) + ebarleny, (*devconvy) (y[i] + et[i]), color);
	    }
	    break;
	case 2:
	    move2(x[i], y[i]);
	    draw2(x[i], y[i] + et[i]);
	    if (symok(x[i], y[i] + et[i])) {
		(*vector) ((*devconvx) (x[i]) - ebarleny, (*devconvy) (y[i] + et[i]), 0);
		(*vector) ((*devconvx) (x[i]) + ebarleny, (*devconvy) (y[i] + et[i]), color);
	    }
	    break;
	case 4:
	    move2(x[i], y[i] - el[i]);
	    draw2(x[i], y[i]);
	    if (symok(x[i], y[i] - el[i])) {
		(*vector) ((*devconvx) (x[i]) - ebarleny, (*devconvy) (y[i] - el[i]), 0);
		(*vector) ((*devconvx) (x[i]) + ebarleny, (*devconvy) (y[i] - el[i]), color);
	    }
	    break;
	case 1:
	    move2(x[i] - el[i], y[i]);
	    draw2(x[i] + et[i], y[i]);
	    if (symok(x[i] - el[i], y[i])) {
		(*vector) ((*devconvx) (x[i] - el[i]), (*devconvy) (y[i]) - ebarlenx, 0);
		(*vector) ((*devconvx) (x[i] - el[i]), (*devconvy) (y[i]) + ebarlenx, color);
	    }
	    if (symok(x[i] + et[i], y[i])) {
		(*vector) ((*devconvx) (x[i] + et[i]), (*devconvy) (y[i]) - ebarlenx, 0);
		(*vector) ((*devconvx) (x[i] + et[i]), (*devconvy) (y[i]) + ebarlenx, color);
	    }
	    break;
	case 3:
	    move2(x[i] - el[i], y[i]);
	    draw2(x[i], y[i]);
	    if (symok(x[i] - el[i], y[i])) {
		(*vector) ((*devconvx) (x[i] - el[i]), (*devconvy) (y[i]) - ebarlenx, 0);
		(*vector) ((*devconvx) (x[i] - el[i]), (*devconvy) (y[i]) + ebarlenx, color);
	    }
	    break;
	case 5:
	    move2(x[i], y[i]);
	    draw2(x[i] + et[i], y[i]);
	    if (symok(x[i] + et[i], y[i])) {
		(*vector) ((*devconvx) (x[i] + et[i]), (*devconvy) (y[i]) - ebarlenx, 0);
		(*vector) ((*devconvx) (x[i] + et[i]), (*devconvy) (y[i]) + ebarlenx, color);
	    }
	    break;
	}
    }
}

/*
 * make the current color col
 */
void setcolor(col)
    int col;
{
    color = (*devsetcolor) (col);
}

/*
 * make the current linestyle style
 */
void setlinestyle(style)
    int style;
{
    lines = (*devsetline) (style);
}

/*
 * drawtic - interface to device driver routines, done this way
 *           as there were problems with low resolution devices
 *           (this is probably not necessary now)
 *      dir = 0 = draw in up (for x-axis) or right (for y-axis)
 *      axis = 0 = draw x-axis tic,   1 = y-axis
 */
static void drawtic(x, y, dir, axis)
    double x, y;
    int dir, axis;
{
    (*devdrawtic) ((*devconvx) (x), (*devconvy) (y), dir, axis);
}

/*
 * set the current character size to size
 */
void setcharsize(size)
    double size;
{
    charsize = size;
}

/*
 * set the current length of the tick marks
 */

void setticksize(sizex, sizey)
    double sizex, sizey;
{
    devxticl = (int) savexticl *sizex;
    devyticl = (int) saveyticl *sizey;
}

/*
 * writestr - user interface to the current device text drawing routine
 */
void writestr(x, y, dir, s)
    double x, y;
    int dir;
    char *s;
{
    (*devwritestr) ((*devconvx) (x), (*devconvy) (y), dir, s);
}

/*
 * draw the title, subtitle, x-axis and y-axis labels
 */
void drawlabels(xl, yl, title, stitle)
    char xl[], yl[], title[], stitle[];

{
    int fudge, ix, iy;
    double tmp;

    fudge = 4;
    tmp = (xg2 + xg1) / 2.0;	/* center x-axis label and title */
    if (xl[0]) {
	ix = stringextentx(charsize * devcharsize, xl);
	iy = stringextenty(charsize * devcharsize, "Xy");
	(*devwritestr) ((*devconvx) (tmp) - ix / 2, (*devconvy) (yg1) - fudge * iy, 0, xl);
    }
    if (title[0]) {
	charsize *= 1.5;
	iy = (int) (3.0 * stringextenty(charsize * devcharsize, "X"));
	ix = stringextentx(charsize * devcharsize, title);
	(*devwritestr) ((*devconvx) (tmp) - ix / 2, (*devconvy) (yg2) + iy, 0, title);
	charsize /= 1.5;
    }
    if (stitle[0]) {
	ix = stringextentx(charsize * devcharsize, stitle);
	iy = stringextenty(charsize * devcharsize, "Xy");
	(*devwritestr) ((*devconvx) (tmp) - ix / 2, (*devconvy) (yg2) + iy, 0, stitle);
    }
    if (yl[0]) {
	tmp = (yg2 + yg1) / 2.0;
	ix = stringextentx(charsize * devcharsize, yl);
	iy = stringextenty(charsize * devcharsize, yl);
	(*devwritestr) ((*devconvx) (xg1) - (int) (1.4 * ylabpos), (*devconvy) (tmp) - ix / 2, 90, yl);
    }
}

/*
 *      write axis labels from start to end starting at y with format
 *      form and decimal or exponential labels selected by style
 *
 *	start - start of labeling, always xg1 in grtool
 *	end - end of labeling, always xg2 in grtool
 *	y - either yg1 or yg2 depending on external flag
 *	step - spacing of labels
 *	form - precision of label
 *	style - decimal or exponential
 *	logflag - label logarithmic
 *	absflag - use absolute value
 *	angle - draw the label at angle in integer degrees
 *	topbot - draw on top or bottom
 *	skip - number of steps to skip for labels
 */
void xlabels(start, end, y, step, form, style, logflag, absflag, angle, topbot, skip)
    double start, end, y, step;
    int form, style, logflag, absflag, angle, topbot;
{
    char s[40];
    char format[20];
    double loc = start;
    int i = 0, ix, iy;

    skip++;
    if (topbot)
	topbot = -1;
    else
	topbot = 1;
    while (loc <= end) {
	if (style)
	    strcpy(format, "%.*lf");
	else
	    strcpy(format, "%.*le");
	if (logflag) {
	    strcpy(format, "10\\S%.*lf\\N");
	    sprintf(s, format, form, loc);
	} else {
	    if (absflag) {
		sprintf(s, format, form, fabs(loc));
	    } else {
		sprintf(s, format, form, loc);
	    }
	}
	ix = stringextentx(charsize * devcharsize, s);
	iy = stringextenty(charsize * devcharsize, s);
	if (angle) {
	    ix = 0;
	} else {
	    ix = stringextentx(charsize * devcharsize, s);
	}
	(*devwritestr) ((*devconvx) (loc) - ix / 2, (*devconvy) (y) - 2 * iy * topbot, angle, s);
	i += skip;
	loc = start + i * step;
    }
}

/*
 *      tickmarks from start to end starting at y1 and y2
 *      depending on opflag
 */
void xmajor(start, end, y1, y2, step, opflag, outflag)
    double start, end, y1, y2, step;
    int opflag, outflag;
{
    double s = start;

    while (start <= end) {
	if (!outflag)
	    drawtic(start, y1, 0, 0);	/* draw up from y1 */
	else
	    drawtic(start, y1, 0, 1);	/* draw down from y2 */
	start += step;
    }
    if (opflag) {
	start = s;
	while (start <= end) {
	    if (!outflag)
		drawtic(start, y2, 0, 1);	/* draw down from y2 */
	    else
		drawtic(start, y2, 0, 0);	/* draw up from y2 */
	    start += step;
	}
    }
}

/*
 * minor tick marks
 */
void xminor(start, end, y1, y2, step, opflag, outflag)
    double start, end, y1, y2, step;
    int opflag, outflag;
{
    double s = start;

    devxticl /= 2;
    while (start <= end) {
	if (!outflag)
	    drawtic(start, y1, 0, 0);	/* draw up from y1 */
	else
	    drawtic(start, y1, 0, 1);	/* draw down from y2 */
	start += step;
    }
    if (opflag) {
	start = s;
	while (start <= end) {
	    if (!outflag)
		drawtic(start, y2, 0, 1);	/* draw down from y2 */
	    else
		drawtic(start, y2, 0, 0);	/* draw up from y2 */
	    start += step;
	}
    }
    devxticl = savexticl;
}

/*
 * draw the y-axis tick labels - see xlabels for a description of parameters
 */
void ylabels(start, end, x, step, form, style, logflag, absflag, angle, leftright, skip)
    double start, end, x, step;
    int form, style, logflag, absflag, leftright, angle, skip;
{
    char s[20];
    char format[20];
    int i = 0, ix, ifudge, is = 0;
    double loc = start;

    ylabpos = 0;
    skip++;
    ifudge = stringextentx(charsize * devcharsize, "N");
    while (loc <= end) {
	if (style)
	    strcpy(format, "%.*lf");
	else
	    strcpy(format, "%.*le");
	if (logflag) {
	    strcpy(format, "10\\S%.*lf\\N");
	    sprintf(s, format, form, loc);
	} else {
	    if (absflag) {
		sprintf(s, format, form, fabs(loc));
	    } else {
		sprintf(s, format, form, loc);
	    }
	}

	ix = stringextentx(charsize * devcharsize, s);
	if (ix > ylabpos)
	    ylabpos = ix;
	if (leftright)
	    (*devwritestr) ((*devconvx) (x) + ifudge, (*devconvy) (loc), angle, s);
	else
	    (*devwritestr) ((*devconvx) (x) - ix - ifudge, (*devconvy) (loc), angle, s);
	i += skip;
	loc = start + i * step;
    }
    ylabpos = leftright ? ifudge : ylabpos + ifudge;
}

/*
 *      ticmarks from start to end starting at x1 and
 *      x2 depending on opflag
 */
void ymajor(start, end, x1, x2, step, opflag, outflag)
    double start, end, x1, x2, step;
    int opflag, outflag;
{
    double s = start;

    while (start <= end) {
	if (!outflag)
	    drawtic(x1, start, 1, 0);
	else
	    drawtic(x1, start, 1, 1);
	start += step;
    }
    if (opflag) {
	start = s;
	while (start <= end) {
	    if (!outflag)
		drawtic(x2, start, 1, 1);
	    else
		drawtic(x2, start, 1, 0);
	    start += step;
	}
    }
}

/*
 * y-axis minor tick marks
 */
void yminor(start, end, x1, x2, step, opflag, outflag)
    double start, end, x1, x2, step;
    int opflag, outflag;
{
    double s = start;

    devyticl /= 2;
    while (start <= end) {
	if (!outflag)
	    drawtic(x1, start, 1, 0);
	else
	    drawtic(x1, start, 1, 1);
	start += step;
    }
    if (opflag) {
	start = s;
	while (start <= end) {
	    if (!outflag)
		drawtic(x2, start, 1, 1);
	    else
		drawtic(x2, start, 1, 0);
	    start += step;
	}
    }
    devyticl = saveyticl;
}

/*
 * special routine for xminor tick in log display
 */
void xminorlog(start, end, y1, y2, step, opflag, outflag)
    double start, end, y1, y2, step;
    int opflag;
{
    double begint, endt, i, j;

    devxticl /= 2;
    begint = start;
    endt = end;
    for (i = begint; i < endt; i += 1.0) {
	for (j = 1.0; j <= step; j += 1.0) {
	    if (log10(pow(10.0, i) * j) > endt)
		goto skip;
	    if (!outflag)
		drawtic(log10(pow(10.0, i) * j), y1, 0, 0);
	    else
		drawtic(log10(pow(10.0, i) * j), y1, 0, 1);
	}
    }
skip:
    if (opflag) {
	for (i = begint; i < endt; i += 1.0) {
	    for (j = 1.0; j <= step; j += 1.0) {
		if (log10(pow(10.0, i) * j) > endt)
		    return;
		if (!outflag)
		    drawtic(log10(pow(10.0, i) * j), y2, 0, 1);
		else
		    drawtic(log10(pow(10.0, i) * j), y2, 0, 0);
	    }
	}
    }
    devxticl = savexticl;
}

/*
 * special routine for yminor tick in log display
 */
void yminorlog(start, end, x1, x2, step, opflag, outflag)
    double start, end, x1, x2, step;
    int opflag;
{
    double begint, endt, i, j;

    devyticl /= 2;
    begint = start;
    endt = end;
    for (i = begint; i < endt; i += 1.0) {
	for (j = 1.0; j <= step; j += 1.0) {
	    if (log10(pow(10.0, i) * j) > endt)
		goto skip;
	    if (!outflag)
		drawtic(x1, log10(pow(10.0, i) * j), 1, 0);
	    else
		drawtic(x1, log10(pow(10.0, i) * j), 1, 1);
	}
    }
skip:
    if (opflag) {
	for (i = begint; i < endt; i += 1.0) {
	    for (j = 1.0; j <= step; j += 1.0) {
		if (log10(pow(10.0, i) * j) > endt)
		    return;
		if (!outflag)
		    drawtic(x2, log10(pow(10.0, i) * j), 1, 1);
		else
		    drawtic(x2, log10(pow(10.0, i) * j), 1, 0);
	    }
	}
    }
    devyticl = saveyticl;
}

/*
 * routines for the lines y=0 and x=0
 */
void drawzerox()
{
    move2(xg1, 0.0);
    draw2(xg2, 0.0);
}

void drawxzerotics(xt1, xt2)
    double xt1, xt2;
{
    double loc;
    int saveticl = devxticl;

    loc = xg1;
    while (loc <= xg2) {
	drawtic(loc, 0.0, 0, 0);
	drawtic(loc, 0.0, 0, 1);
	loc = loc + xt1;
    }
    loc = xg1;
    devxticl /= 2;
    while (loc <= xg2) {
	drawtic(loc, 0.0, 0, 0);
	drawtic(loc, 0.0, 0, 1);
	loc = loc + xt2;
    }
    devxticl = saveticl;
}

void drawzeroy()
{
    move2(0.0, yg1);
    draw2(0.0, yg2);
}

void drawyzerotics(yt1, yt2)
    double yt1, yt2;
{
    double loc;

    loc = yg1;
    while (loc <= yg2) {
	drawtic(0.0, loc, 1, 0);
	drawtic(0.0, loc, 1, 1);
	loc += yt1;
    }
    loc = yg1;
    devyticl /= 2;
    while (loc <= yg2) {
	drawtic(0.0, loc, 1, 0);
	drawtic(0.0, loc, 1, 1);
	loc += yt2;
    }
    devyticl = saveyticl;
}

/*
        draw grid lines rather than ticmarks
*/
void drawxgrid(start, end, y1, y2, step, cy, ly)
    double start, end, y1, y2, step;
    int ly, cy;
{
    int scol, slin;

    scol = color;
    slin = lines;
    setcolor(cy);
    setlinestyle(ly);
    while (start <= end) {
	move2(start, y1);
	draw2(start, y2);
	start += step;
    }
    setcolor(scol);
    setlinestyle(slin);
}

void drawygrid(start, end, x1, x2, step, cy, ly)
    double start, end, x1, x2, step;
    int ly, cy;
{
    int scol, slin;

    scol = color;
    slin = lines;
    setcolor(cy);
    setlinestyle(ly);
    while (start <= end) {
	move2(x1, start);
	draw2(x2, start);
	start += step;
    }
    setcolor(scol);
    setlinestyle(slin);
}

/*
        place symbols at the vertices of a polygon specified by x & y.
*/

double barwid = 0.01;

void drawpolysym(x, y, len, sym)
    double x[], y[];
int len, sym;

{
    int i;
    char s[10];

    if (sym >= 19) {
	if (sym == 23)
	    hdelta = barwid * (xg2 - xg1);
	else if (sym == 24)
	    hdelta = barwid * (yg2 - yg1);
	else if (sym == 19 || sym == 21)
	    hdelta = fabs(x[1] - x[0]) / 2.0;
	else if (sym == 20 || sym == 22)
	    hdelta = fabs(y[1] - y[0]) / 2.0;
	for (i = 0; i < len; i++) {
	    switch (sym) {
	    case 19:
	    case 23:
		histbox(x[i], y[i]);
		break;
	    case 20:
	    case 24:
		histbox3(x[i], y[i]);
		break;
	    case 21:
		if (i != len - 1)
		    histbox2(x[i], y[i], y[i + 1], 0);
		else
		    histbox2(x[i], y[i], y[i + 1], 1);
		break;
	    case 22:
		if (i != len - 1)
		    histbox4(x[i], y[i], x[i + 1], 0);
		else
		    histbox4(x[i], y[i], x[i + 1], 1);
		break;
	    case 25:
		if (symok(x[i], y[i])) {
		    sprintf(s, "%d", i + 1);
		    writestr(x[i], y[i], 0, s);
		}
		break;
	    }
	}
    } else {
	for (i = 0; i < len; i++) {
	    drawsym(x[i], y[i], sym);
	}
    }
}

/*
 * draw the head of an arrow
 */
void draw_head(ix1, iy1, ix2, iy2, sa)
    int ix1, iy1, ix2, iy2, sa;
{
    double dist, ang, dx, dy;
    double adj = 8.0 * M_PI / 180.0;
    int xr, yr, xl, yl, s1, s2;

    dx = ix2 - ix1;
    dy = iy2 - iy1;
    if (dx == 0.0 && dy == 0.0) {
	errwin("Can't draw arrow, dx = dy = 0.0");
	return;
    }
    dist = hypot(dx, dy);
    ang = atan2(dy, dx);
    s1 = (int) ((dist - sa) * cos(ang));
    s2 = (int) ((dist - sa) * sin(ang));
    (*vector) (ix1, iy1, 0);
    (*vector) (ix2, iy2, color);
    xr = (int) (sa * cos(ang + M_PI / 2.0) / 2.0);
    yr = (int) (sa * sin(ang + M_PI / 2.0) / 2.0);
    (*vector) (s1 + ix1 - xr, s2 + iy1 - yr, color);
    xl = (int) (sa * cos(ang - M_PI / 2.0) / 2.0);
    yl = (int) (sa * sin(ang - M_PI / 2.0) / 2.0);
    (*vector) (ix2, iy2, 0);
    (*vector) (s1 + ix1 - xl, s2 + iy1 - yl, color);
}

/*
 * draw an arrow
 */
void draw_arrow(x1, y1, x2, y2, end, asize)
    double x1, y1, x2, y2, asize;
    int end;			/* 0 = none 1 = arrow at x1, y1  2 = arrow at
				 * x2, y2 3 arrow at both ends */
{
    int ix1, iy1, ix2, iy2;

    int sa = (int) (asize * devarrowlength);

    ix1 = (*devconvx) (x1);
    ix2 = (*devconvx) (x2);
    iy1 = (*devconvy) (y1);
    iy2 = (*devconvy) (y2);
    switch (end) {
    case 0:
	(*vector) (ix1, iy1, 0);
	(*vector) (ix2, iy2, color);
	break;
    case 1:
	draw_head(ix2, iy2, ix1, iy1, sa);
	break;
    case 2:
	draw_head(ix1, iy1, ix2, iy2, sa);
	break;
    case 3:
	draw_head(ix2, iy2, ix1, iy1, sa);
	draw_head(ix1, iy1, ix2, iy2, sa);
	break;
    }
}

/*
 * draw the legend at world coordinates (x,y)
 */
void putlegend(i, xlen, ylen, size, x, y, sy, ly, cy, ib, s)
    int i, xlen, ylen, sy, ly, cy, ib;
    double size, x, y;
    char *s;
{
    int ifudgex, ifudgey, scol, slin, xtmp, ytmp;
    static int maxx = 0;

    size *= devcharsize;
    ifudgey = stringextenty(size, "N");
    ifudgex = stringextentx(size, "N");
    xtmp = (*devconvx) (x);
    ytmp = (*devconvy) (y) - i * ylen * ifudgey;
    /*
     * draw a box
     */
    if (ib == -1) {
	int yt = (*devconvy) (y);

	if (i > 0) {
	    (*vector) (xtmp - ifudgex, yt + ifudgey, 0);
	    (*vector) (maxx + 2 * ifudgex, yt + ifudgey, color);
	    (*vector) (maxx + 2 * ifudgex, ytmp - ifudgey, color);
	    (*vector) (xtmp - ifudgex, ytmp - ifudgey, color);
	    (*vector) (xtmp - ifudgex, yt + ifudgey, color);
	    maxx = 0;
	}
	return;
    }
    scol = color;
    slin = lines;
    setcolor(cy);
    setlinestyle(ly);
    (*vector) (xtmp, ytmp, 0);
    if (ly) {
	(*vector) (xtmp + xlen * ifudgex, ytmp, color);
    }
    if ((sy > 0) && (sy <= 16)) {
	setlinestyle(slin);
	if (ly)
	    hwritesym(sy, xtmp, ytmp, size, color, vector);
	hwritesym(sy, xtmp + xlen * ifudgex, ytmp, size, color, vector);
    }
    setcolor(scol);
    setlinestyle(slin);
    (*devwritestr) (xtmp + (xlen + 1) * ifudgex, ytmp, 0, s);
    if (ib == 1) {
	int itmp;

	itmp = stringextentx(size, s) + xtmp + xlen * ifudgex;
	if (itmp > maxx) {
	    maxx = itmp;
	}
    }
}

/*
 * initialize the graphics device device, retval is currently
 * unused but will eventually return a code indicating success
 * or failure opening the device
 */
int initgraphics(device)
    int device;
{
    int retval;

    switch (device) {
    case 0:
	retval = sviewinitgraphics(1);
	break;
    case 1:
	retval = hpinitgraphics(1);
	break;
    case 2:
	retval = hpinitgraphics(3);
	break;
    case 3:
	retval = hpinitgraphics(9);
	break;
    case 4:
	retval = hpinitgraphics(11);
	break;
    case 5:
	retval = psinitgraphics(1);
	break;
    case 6:
	retval = psinitgraphics(3);
	break;
    case 7:
	retval = hpinitgraphics(5);
	break;
    case 8:
	retval = hpinitgraphics(7);
	break;
    case 9:
	retval = rasterinitgraphics(1);
	break;
    case 10:
	retval = rasterinitgraphics(2);
	break;
    case 11:
    case 12:
	retval = geninitgraphics(1);
	break;
    case 13:
	retval = hpljinitgraphics(1);
	break;
    case 14:
	retval = hpljinitgraphics(3);
	break;
    case 15:
	retval = hpljinitgraphics(5);
	break;
    case 16:
	retval = hpljinitgraphics(7);
	break;
    case 17:
	retval = hpljinitgraphics(9);
	break;
    case 18:
	retval = hpljinitgraphics(11);
	break;
    default:
	fprintf(stderr, "No such device %d\n", device);
	exit(1);
    }
    devtype = device;
    savexticl = devxticl;
    saveyticl = devyticl;
    return retval;
}

void leavegraphics()
{
    (*devleavegraphics) ();
}
