/*
 *           PVM 3.0:  Parallel Virtual Machine System 3.0
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *          R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM 3.0 was funded in part by the U.S. Department of Energy, the
 * National Science Foundation and the State of Tennessee.
 */

/*
 * PVM3.0-I860
 *
 *  cube.c:  routines for managing the list of active cubes
 *
 *		send bug reports to:	wjiang@cs.utk.edu
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#ifdef  SYSVSTR
#include <string.h>
#else
#include <strings.h>
#endif
#include <cube.h>
#include "protoglarp.h"
#include "myalloc.h"
#include "listmac.h"
#include "mycube.h"
#include "ddpro.h"
#include "task.h"

/* global */
extern void     bailout();
extern short    qm_cubeid;	/* from libhost.a */
extern int      debugmask;	/* from pvmd.c */

struct cube    *cubes = 0;	/* allocated cubes */

/* private */
static char     etext[512];	/* scratch for error log */

/* make list head */
void
cube_init()
{
	if (!cubes) {
		cubes = TALLOC(1, struct cube, "cube");
		bzero((char *) cubes, sizeof(struct cube));
		cubes->c_link = cubes;
		cubes->c_rlink = cubes;
		cubes->c_size = -1;
	}
}

/* make a new cube struct; put it on cubes list; return 0 if failed */
struct cube    *
cube_new()
{
	struct cube    *cp;
	int             i;

	if (!(cp = TALLOC(1, struct cube, "cube"))) {
		log_error("cube_new() can't get memory\n");
		bailout(0);
	}
	bzero((char *) cp, sizeof(struct cube));
	cp->c_size = -1;
	LISTPUTBEFORE(cubes, cp, c_link, c_rlink);
	return cp;
}

/* locate a cube on active list by its id */
struct cube    *
cube_find(cid)
	int             cid;
{
	struct cube    *cp;

	for (cp = cubes->c_link; cp != cubes; cp = cp->c_link)
		if (cp->c_id == cid)
			return cp;
	return (struct cube *) 0;
}

/* locate a cube on acitve list by its name */
struct cube    *
cube_byname(name)
	char           *name;
{
	struct cube    *cp;

	for (cp = cubes->c_link; cp != cubes; cp = cp->c_link)
		if (strcmp(cp->c_name, name) == 0)
			return cp;
	return (struct cube *) 0;
}

/* release the cube and clean up */
void
cube_cleanup(cp)
	struct cube    *cp;
{
	struct task *tp;

	_attachcube(cp->c_name);
	_relcube(cp->c_name);	/* this sends a SIGKILL to cubed */
	if (tp = task_find(cp->c_cubed)) {
		task_cleanup(tp);
		task_free(tp);
	}
	cube_free(cp);
	wait((int *)0);			/* wait for cubed to die */
}

/* remove a cube struct from list and free all associated storage */
void
cube_free(cp)
	struct cube    *cp;
{
	if (cp->c_link && cp->c_rlink)
		LISTDELETE(cp, c_link, c_rlink);
	if (cp->c_file)
		PVM_FREE(cp->c_file);
	if (cp->c_name)
		PVM_FREE(cp->c_name);
	PVM_FREE(cp);
}
