/******************************************************************************/
/**									     **/
/**		      Copyright 1990 by Computer Science Dept.  	     **/
/**			University College London, England		     **/
/**									     **/
/**									     **/
/**									     **/
/** Permission to use, copy and modify (but NOT distribute) this software    **/
/** and its documentation for any purpose and without fee is hereby granted, **/
/** provided the above copyright notice appears in all copies, and that both **/
/** that copyright notice and this permission notice appear in supporting    **/
/** documentation, and that the name Pygmalion not be used in advertising or **/
/** publicity of the software without specific, written prior permission of  **/
/** Thomson-CSF.							     **/
/**									     **/
/** THE DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON DISCLAIMS  **/
/** ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED       **/
/** WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE 	     **/
/** DEPARTMENT OF COMPUTER SCIENCE, UNIVERSITY COLLEGE LONDON BE LIABLE FOR  **/
/** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER **/
/** RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF     **/
/** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN      **/
/** CONJUNCTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.		     **/
/**									     **/
/******************************************************************************/

/******************************************************************************
 * Pygmalion Programming Environment v 1.02 3/3/90
 *
 * pgm 
 *
 * hf.c
 ******************************************************************************/



#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>

#include "pygmalion.h"
#include "sysdef.h"
#include "hfconfig.h"
#include "pgmrc.h"

#include "hfdef.h"
#include "built_in_fn.h"
#include "util.h"
#include "fontdefs.h"

/* --------------- functions defined for this algorithm ------------------- */
int	connect ();
int	build_rules ();

int	clear_counter ();

float	learn ();
float	recall_async ();
float	step_recall_async ();
float	recall_sync ();
float	step_recall_sync ();

/* --------------- globals ------------------------------------------------ */
int		I_LAYER, O_LAYER;			/* input, output layers */
int		c_net = 0;				/* current net */
int		neuron_count;
char		system_filename[ FILESIZE ];

float		*input_vector;				/* to receive input pattern */
float		*target_vector;				/* to receive target pattern */

int		input_pattern_control = PLUSMINUS;		/* controls representation of pattern */
int		target_pattern_control = PLUSMINUS;		/* controls representation of pattern */

/* ------------------------------------------------------------------------ */

/*  low level functions for the Hopfield model  */

int	general_update();
int	state_shift();
int	weight_update();
int	tolerance();
int	clear_counter();

/* class_type structures		    function		fn_name			gen.	ext. */

class_type learn_meta_class		= { sexec,		"sexec",		0,	1 };

class_type step_recall_async_meta_class	= { sexec,		"sexec",		0,	1 };
class_type recall_async_meta_class	= { sexec,		"sexec",		0,	1 };
class_type loop_async_meta_class	= { sexec_r,		"sexec_r",		0,	1 };

class_type step_recall_sync_meta_class	= { sexec,		"sexec",		0,	1 };
class_type recall_sync_meta_class	= { sexec,		"sexec",		0,	1 };
class_type loop_sync_meta_class		= { sexec_r,		"sexec_r",		0,	1 };

class_type weight_update_meta_class	= { pexec,		"pexec",		0,	1 };
class_type state_update_meta_class	= { pexec,		"pexec",		0,	1 };
class_type acc_update_meta_class	= { pexec,		"pexec",		0,	1 };
class_type state_shift_meta_class	= { pexec,		"pexec",		0,	1 };

class_type tolerance_test_class		= { tolerance,		"tolerance",		4,	2 };
class_type clear_counter_class		= { clear_counter,	"clear_counter",	1, 	2 };

class_type state_update_class		= { general_update,	"general_update",	4,	2 };
class_type acc_update_class		= { general_update,	"general_update",	4,	2 };
class_type state_shift_class		= { state_shift,	"state_shift",		2,	2 };
class_type weight_update_class		= { weight_update,	"weight_update",	1,	2 };

/* ------ specify maximum counts of ground, meta and parameters at all levels ---- */

int	xc	[ 6 ][ 2 ] = {	/* extra counts : rules, parameters */
		0,	0,	/* system	*/
		13,	6,	/* net		*/
		0,	0,	/* layer	*/
		0,	0,	/* cluster	*/
		4,	0,	/* neuron	*/
		0,	0	/* synapse	*/
};

/* ------ names for parameters -------------------------------------------- */

char	*pname [6] [6] = {
		{ "","","","","",""},
		{ "score", "loop_counter", "loop_limit", "time_out_limit", "activation", "lambda" },
		{ "","","","","",""},
		{ "","","","","",""},
		{ "","","","","",""},
		{ "","","","","",""}
};
/* ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------ */
/*	external functions and variables                                    */
/* ------------------------------------------------------------------------ */

/* -------- declarations from pgmrc.c ------------------------------------- */

extern	char		*rc [];

/* -------- declarations from pattern.c ----------------------------------- */

extern	int		init_patterns();
extern	int		count_patterns();
extern	int		read_pattern();
extern	pat_elem	*load_patterns();
extern	pat_elem	*indextop();
extern	void		add_to_pattern_list();
extern	int		read_input();
extern	int		read_target();
extern	int		read_xfile();

extern	pat_elem	*pattern_list;		/* patterns loaded for net */
extern	pat_elem	*current_pattern;	/* to pass widths to show_net */
extern	int		pattern_count;		/* how many */
extern	char		pattern_filename[];

/* -------- declarations from jload.c ------------------------------------- */

extern	int		sys_load();

/* -------- declarations from jsave.c ------------------------------------- */

extern	int		sys_save();

/* -------- declarations from shownet.c ----------------------------------- */

void			shownet();

/* -------- declarations from util.c -------------------------------------- */

extern system_type	*sys;
extern int		int_mode;
extern int		cycles;
extern float		rand_scale;
extern	char		*strstr();

/* -------- declarations from built_in_fn.c ------------------------------- */

extern	char		*jalloc();
extern	int		sexec();
extern	int		sexec_r();
extern	int		pexec();
extern	int		pexec_r();

extern int		err_cnt;

/* -------- declarations from alloc.c ------------------------------------- */

extern	system_type	*sys_alloc();

/* ------------------------------------------------------------------------ */
/*  Function bodies for the hopfield rule classes                   */
/* ------------------------------------------------------------------------ */

float learn()
{
	EXEC(&sys->net[c_net]->rules[ NET_R_learn ]);

	return(0.0);
}

/* ------------------------------------------------------------------------ */
float recall_async()
{
	EXEC(&sys->net[c_net]->rules[ NET_R_recall_async ]);

	return(sys->net[c_net]->parameters[ NET_P_score ].parameter.value.f);
}

/* ------------------------------------------------------------------------ */
float step_recall_async()
{
	EXEC(&sys->net[c_net]->rules[ NET_R_step_recall_async ]);

	return(sys->net[c_net]->parameters[ NET_P_score ].parameter.value.f);
}

/* ------------------------------------------------------------------------ */
float recall_sync()
{
	EXEC(&sys->net[c_net]->rules[ NET_R_recall_sync ]);

	return(sys->net[c_net]->parameters[ NET_P_score ].parameter.value.f);
}

/* ------------------------------------------------------------------------ */
float step_recall_sync()
{
	EXEC(&sys->net[c_net]->rules[ NET_R_step_recall_sync ]);

	return(sys->net[c_net]->parameters[ NET_P_score ].parameter.value.f);
}

/* ------------------------------------------------------------------------ */

/*  state_update_class and acc_update_class */

int general_update (p)
TAGVAL **p;        /* [*STATE or *ACC, *ERR, *activation, *lambda, *size, *state1, *weight1, ... ] */
{
	float	f = 0.0, prev;
	int	i, activation;

	activation = (int) p[ 2 ]->value.f;

 	f = dp( p + 4 );

	prev = p[ 0 ]->value.f;		/* old value */

	switch (activation) {

	case SIGMOID:	/*	sigmoid : -1.0 to +1.0	*/

	 	p[ 0 ]->value.f = -1.0 + ( 2.0/(1.0+ exp(- (p[ 3 ]->value.f * f ) ))) ;
		break;

	case STEP:	/*	zero threshold :  f >= 0.0 => +1.0,  f < 0.0 => -1.0	*/

		if ( f >= 0.0 ) {
			p[ 0 ]->value.f = 1.0;
		}
		else {
			p[ 0 ]->value.f = -1.0;
		}
		break;

	default:
		printf("invalid activation function switch\n");
		exit( 1 );
	}
	if ( 	( (prev > 0.0) && (p[ 0 ]->value.f > 0.0) ) || 
		( (prev < 0.0) && (p[ 0 ]->value.f < 0.0) ) )
		p[ 1 ]->value.f += 1.0;
	else
		p[ 1 ]->value.f  = 0.0;

	return( 0 );
}

/* ------------------------------------------------------------------------ */
/*  state_shift_class  */

int state_shift (p)
TAGVAL **p;        /*  [*state, *acc] */
{	/* transfer ACC to STATE */

	p[ 0 ]->value.f = p[ 1 ]->value.f;

	return( 0 );
}

/* ------------------------------------------------------------------------ */
/*  clear_counter_class  */

int clear_counter (p)
TAGVAL **p;        /*  [*parameter, size, ERR[0], STATE[0], ERR[1] ...] */
{	/* clear counters */

	int	i, size = *(int *)p[ 1 ];

	p[ 0 ]->value.f = 0.0;	/* NET_P_loop_counter */

	for (i=2; i<2*size+2; i+=2) {
 		p[ i ]->value.f = 0.0;	/* all the ERR registers */
	}
	return( 0 );
}

/* ------------------------------------------------------------------------ */
/*  weight_update_class  */

int weight_update (p)
TAGVAL **p;        /*  state[current], size, *weight[0], *state[0], .... ( not incl. current ) */
{
	/* 
		This is called when an input pattern has been clamped to the states 
	*/
	int	i, size = *(int *)p[ 1 ];

	for (i=2; i<2*size+2; i+=2) {
 		p[ i ]->value.f += p[ 0 ]->value.f * p[ i + 1 ]->value.f;
	}
	return( 0 );
}

/* ------------------------------------------------------------------------ */
/* tolerance_class */

int tolerance (p)
TAGVAL **p;	/*	*net.score, *net.loop_counter, 
				*net.loop_limit, *net.time_out_limit,
				*size,
				ERR[0], STATE[0], 
				ERR[1], STATE[1] ... */
{
	int 	count, min_count, size, loop_counter, loop_limit, time_out_limit, result, i;

	loop_counter	= (int) p[ 1 ]->value.f;
	loop_limit	= (int) p[ 2 ]->value.f;
	time_out_limit	= (int) p[ 3 ]->value.f;
	size 		= *(int *)p[ 4 ];

	result = OK;

	for (i=5; i<2*size+5; i+=2) {
		count = (int) p[ i ]->value.f;
		
		if ( count < loop_limit ) {
			result = NOTOK;
			break;
		}
	}

	if ( result == OK ) {
		p[ 0 ]->value.f = (float) count;
		return ( TERM );
	}

	p[ 1 ]->value.f += 1.0;

	if ( p[ 1 ]->value.f > time_out_limit ) {

		min_count=time_out_limit;

		for (i=5; i<2*size+5; i+=2) {
			count = (int) p[ i ]->value.f;
			if ( count < min_count ) {
				min_count = count;
			}
		}
		p[ 0 ]->value.f = (float) min_count;
		return ( TERM );
	}
	return( OK );
}

/* ------------------------------------------------------------------------ *\

		Hopfield connect

\* ------------------------------------------------------------------------ */

int connect (conf)
int	conf[];
{
	int		cn, i, j, k, l, m, p, s, x, y, index;
	register	int io_size;
	caddr_t		*free_pointer1, *free_pointer2, *free_pointer3;
	rule_type	*gp;
	para_type	*pp;
	synapse_type	*sp;

	sys = sys_alloc(&conf);		/** allocates memory for networks and neurons **/

	/* allocate memory for the rules and parameters */

	for ( x = SYS; x < SYN; x++ ) {

		y = xc [x] [0];
		if ( y ) {	/* rules */

			s = sizeof (rule_type) ;	/* basic size to allocate */
			switch (x) {
				case SYS:
					sys->n_rules = y;
					sys->rules = (rule_type *) jalloc (s, y);
					break;

				case NET:
					for ( cn = 0; cn < sys->nets; cn++ ) {
						sys->net[cn]->n_rules = y;
						sys->net[cn]->rules = (rule_type *) jalloc (s, y);
					}
					break;

				case LAY:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					gp = (rule_type *) jalloc ( s, y * sys->net[cn]->layers);
					for (i=0; i < sys->net[cn]->layers; i++, gp += y) {
						sys->net[cn]->layer[i]->n_rules = y;
						sys->net[cn]->layer[i]->rules = gp;
					}
					}
					break;

				case CLU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=0; i < sys->net[cn]->layers; i++) {
					gp = (rule_type *) jalloc ( s, y * sys->net[cn]->layer[i]->clusters);
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++, gp += y) {
						sys->net[cn]->layer[i]->cluster[j]->n_rules = y;
						sys->net[cn]->layer[i]->cluster[j]->rules = gp;
					}
					}
					}
					break;

				case NEU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=0; i < sys->net[cn]->layers; i++) {
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++) {
					gp = (rule_type *) jalloc ( s, y * sys->net[cn]->layer[i]->cluster[j]->neurons);
					for (k=0; k < sys->net[cn]->layer[i]->cluster[j]->neurons; k++, gp += y) {
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->n_rules = y;
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->rules = gp;
					}
					}
					}
					}
					break;
			
				case SYN:
					/* dunno ? */
					break;

				default:
					printf ("invalid control\n");
					exit(1);
					break;
			}
		}
	
		y = xc [x] [1];
		if ( y ) {	/* Parameters */

			s = sizeof (para_type) ;	/* basic size to allocate */
			switch (x) {
				case SYS:
					sys->n_parameters = y;
					sys->parameters = (para_type *) jalloc (s, y);
					for ( p = 0; p < y; p++ ) {
						sys->parameters[p].name = pname [x] [p];
 					}
					break;

				case NET:
					for ( cn = 0; cn < sys->nets; cn++ ) {
						sys->net[cn]->n_parameters = y;
						sys->net[cn]->parameters = (para_type *) jalloc (s, y);
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->parameters[p].name = pname [x] [p];
	 					}
					}
					break;

				case LAY:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					pp = (para_type *) jalloc ( s, y * sys->net[cn]->layers);
					for (i=0; i < sys->net[cn]->layers; i++, pp += y) {
						sys->net[cn]->layer[i]->n_parameters = y;
						sys->net[cn]->layer[i]->parameters = pp;
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->layer[i]->parameters[p].name = pname [x] [p];
	 					}
					}
					}
					break;

				case CLU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=0; i < sys->net[cn]->layers; i++) {
					pp = (para_type *) jalloc ( s, y * sys->net[cn]->layer[i]->clusters);
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++, pp += y) {
						sys->net[cn]->layer[i]->cluster[j]->n_parameters = y;
						sys->net[cn]->layer[i]->cluster[j]->parameters = pp;
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->layer[i]->cluster[j]->parameters[p].name = pname [x] [p];
	 					}
					}
					}
					}
					break;

				case NEU:
					for ( cn = 0; cn < sys->nets; cn++ ) {
					for (i=0; i < sys->net[cn]->layers; i++) {
					for (j=0; j < sys->net[cn]->layer[i]->clusters; j++) {
					pp = (para_type *) jalloc ( s, y * sys->net[cn]->layer[i]->cluster[j]->neurons);
					for (k=0; k < sys->net[cn]->layer[i]->cluster[j]->neurons; k++, pp += y) {
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->n_parameters = y;
						sys->net[cn]->layer[i]->cluster[j]->neuron[k]->parameters = pp;
						for ( p = 0; p < y; p++ ) {
							sys->net[cn]->layer[i]->cluster[j]->neuron[k]->parameters[p].name = pname [x] [p];
	 					}
					}
					}
					}
					}
					break;
			
				case SYN:
					/* dunno ? */
					break;

				default:
					printf ("invalid control\n");
					exit(1);
					break;
			}
		}
	}

	for ( cn = 0; cn < sys->nets; cn++ ) {

	sys->net[cn]->parameters[ NET_P_loop_limit ].parameter.value.f		= (float) LOOP_LIMIT;
	sys->net[cn]->parameters[ NET_P_time_out_limit ].parameter.value.f	= (float) TIME_OUT_LIMIT;
	sys->net[cn]->parameters[ NET_P_activation ].parameter.value.f		= (float) ACTIVATION;
	sys->net[cn]->parameters[ NET_P_lambda ].parameter.value.f		= (float) LAMBDA;

	/********* allocating the output, teaching and input pattern pipe pointers ********/

	neuron_count = sys->net[cn]->layer[0]->cluster[0]->neurons;

	sys->net[cn]->fanin	= neuron_count;
	input_vector		= (float *) jalloc( sizeof ( float ), neuron_count );
	target_vector		= (float *) jalloc( sizeof ( float ), neuron_count );

	sys->net[cn]->input_port  = (caddr_t *)jalloc (sizeof(caddr_t), neuron_count);
	sys->net[cn]->output_port = (caddr_t *)jalloc (sizeof(caddr_t), neuron_count);
	sys->net[cn]->target	  = (caddr_t *)jalloc (sizeof(caddr_t), neuron_count);


	/**** build up the I/O pipe pointers ****/

	free_pointer1 = sys->net[cn]->output_port;
	free_pointer2 = sys->net[cn]->target;
	free_pointer3 = sys->net[cn]->input_port;
	for (i=0; i<neuron_count; i++) {
		*free_pointer1++ = (caddr_t) &sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_STATE ].value.f;
		*free_pointer2++ = (caddr_t) &sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_TARGET ].value.f;
	        *free_pointer3++ = (caddr_t) &sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_STATE ].value.f;
	}

	/********* allocating synapses for the neurons *********/

	s = neuron_count - 1;
        for (i = 0; i < neuron_count; i++) {
		sys->net[cn]->layer[0]->cluster[0]->neuron[i]->fanin = s;
		sys->net[cn]->layer[0]->cluster[0]->neuron[i]->synapses = s;
		sys->net[cn]->layer[0]->cluster[0]->neuron[i]->synapse = (synapse_type **) jalloc (sizeof(synapse_type *), s);
		sys->net[cn]->layer[0]->cluster[0]->neuron[i]->input_neuron = (neuron_type **) jalloc (sizeof(neuron_type *), s );
	}

	/* shared synapses : Wij = Wji */
	s  = (neuron_count * (neuron_count - 1)) / 2;
	sp = (synapse_type *) jalloc (sizeof(synapse_type), s);

	for (i=0; i<neuron_count; i++) {
		for (j = i + 1; j<neuron_count; j++) {
			sys->net[cn]->layer[0]->cluster[0]->neuron[ i ]->synapse[ j - 1 ] = sp;
			sys->net[cn]->layer[0]->cluster[0]->neuron[ j ]->synapse[ i ] = sp++;
		}
	}

	/****  connecting the neurons => putting the address in the ****/
	/****       neuron.input_neuron  for the forward path       ****/

	for (i=0; i<neuron_count; i++) {
		index = 0;
		for (j=0; j<neuron_count; j++) {
			if ( i != j ) {
				sys->net[cn]->layer[0]->cluster[0]->neuron[i]->input_neuron[index++] =
					sys->net[cn]->layer[0]->cluster[0]->neuron[j];
			}
		}
	}

	/* write route[] data */

	for ( i=0; i < neuron_count; i++ ) {
		sys->net[cn]->layer[0]->cluster[0]->neuron[i]->route[0] = cn;
		sys->net[cn]->layer[0]->cluster[0]->neuron[i]->route[1] = 0;
		sys->net[cn]->layer[0]->cluster[0]->neuron[i]->route[2] = 0;
		sys->net[cn]->layer[0]->cluster[0]->neuron[i]->route[3] = i;
	}
	}	/* end for cn ... */

	return( OK );
}

/* -------------------------------------------------------------- */
/*	build_rule.c                                              */
/* -------------------------------------------------------------- */


/*****	Function Declarations *****/

int build_rules()
{
	int cn, i, j, k;

	if (!sys || !sys->net || ((0 < 0)))
		return ( FAIL );

	for ( cn = 0; cn < sys->nets; cn++ ) {

	for (i=0; i<neuron_count; i++) {

	rule_init (
		"neuron.state_update",
		&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->rules[ NEU_R_state_update ],
		&state_update_class,
		neuron_count - 1,
		&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_STATE ],
		&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_ERR ],
		&sys->net[cn]->parameters [ NET_P_activation ].parameter,
		&sys->net[cn]->parameters [ NET_P_lambda ].parameter,
		EOP );

	rule_init (
		"neuron.acc_update",
		&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->rules[ NEU_R_acc_update ],
		&acc_update_class,
		neuron_count - 1,
		&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_ACC ],
		&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_ERR ],
		&sys->net[cn]->parameters [ NET_P_activation ].parameter,
		&sys->net[cn]->parameters [ NET_P_lambda ].parameter,
		EOP );

	rule_init (
		"neuron.state_shift",
		&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->rules[ NEU_R_state_shift ],
		&state_shift_class,
		0,
		&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_STATE ],
		&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_ACC ],
		EOP );

	rule_init (
		"neuron.weight_update",
		&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->rules [ NEU_R_weight_update ],
		&weight_update_class,
		neuron_count - i - 1,	/* decreases to compensate for shared synapses */
		&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_STATE ],
		EOP );

	}		/* end for i ... */

	/*** extending rules -- to fill in the parameter pointers ***/


	for (i=0; i<neuron_count; i++) {
	for (j=0; j<neuron_count-1; j++) {
		rule_extend (
			&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->rules[ NEU_R_state_update ],
			j,
			&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->synapse[j]->weight,
			&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->input_neuron[j]->state[ N_STATE ],
			EOP );

		rule_extend (
			&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->rules[ NEU_R_acc_update ],
			j,
			&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->synapse[j]->weight,
			&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->input_neuron[j]->state[ N_STATE ],
			EOP );

	}	/* end for j ... */
	}	/* end for i ... */

	for (i = 0; i<neuron_count; i++) {
	for (j = i; j<neuron_count-1; j++) {	/* to compensate for shared synapses */
		rule_extend (
			&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->rules[ NEU_R_weight_update ],
			j - i,
			&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->synapse[j]->weight,
			&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->input_neuron[j]->state[ N_STATE ],
			EOP );

	}	/* end for j ... */
	}	/* end for i ... */

/* -------------------------------------------------------------- */
/*** Initialize the rules at the net level ***/

/*
	{ NET_R_tolerance_test,
	  NET_R_clear_counter };

	{ NET_R_learn,

	  NET_R_step_recall_async,
	  NET_R_recall_async,
	  NET_R_loop_async,

	  NET_R_step_recall_sync,
	  NET_R_recall_sync,
	  NET_R_loop_sync,

	  NET_R_state_update,
	  NET_R_acc_update,
	  NET_R_state_shift,
	  NET_R_weight_update };
*/
		rule_init (
			"net.tolerance_test",
			&sys->net[cn]->rules[ NET_R_tolerance_test ],
			&tolerance_test_class,
			neuron_count,		/* ground rule tests all ERR values */
			&sys->net[cn]->parameters[ NET_P_score ].parameter,
			&sys->net[cn]->parameters[ NET_P_loop_counter ].parameter,
			&sys->net[cn]->parameters[ NET_P_loop_limit ].parameter,
			&sys->net[cn]->parameters[ NET_P_time_out_limit ].parameter,
			EOP );

		rule_init (
			"net.clear_counter",
			&sys->net[cn]->rules[ NET_R_clear_counter ],
			&clear_counter_class,
			neuron_count,
			&sys->net[cn]->parameters[ NET_P_loop_counter ].parameter,
			EOP );

		rule_init (
			"net.learn",
			&sys->net[cn]->rules[ NET_R_learn ],
			&learn_meta_class,
			neuron_count,		/* need to call NEU_R_weight_update for each */
			EOP );

		/* asynchronous recall */

		rule_init (
			"net.step_recall_async",
			&sys->net[cn]->rules[ NET_R_step_recall_async ],
			&step_recall_async_meta_class,
			2,		/* state_update then tol_test */
			EOP );

		rule_init (
			"net.recall_async",
			&sys->net[cn]->rules[ NET_R_recall_async ],
			&recall_async_meta_class,
			2,		/* clear_counter then loop_async */
			EOP );

		rule_init (
			"net.loop_async",
			&sys->net[cn]->rules[ NET_R_loop_async ],
			&loop_async_meta_class,
			2,		/* first state_update for all then tol_test */
			EOP );

		/* synchronous recall */

		rule_init (
			"net.step_recall_sync",
			&sys->net[cn]->rules[ NET_R_step_recall_sync ],
			&step_recall_sync_meta_class,
			3,		/* first acc_update for all - then state_shift for all, then tol_test */
			EOP );

		rule_init (
			"net.recall_sync",
			&sys->net[cn]->rules[ NET_R_recall_sync ],
			&recall_sync_meta_class,
			2,		/* clear_counter then loop_sync */
			EOP );

		rule_init (
			"net.loop_sync",
			&sys->net[cn]->rules[ NET_R_loop_sync ],
			&loop_sync_meta_class,
			3,		/* first acc_update for all - then state_shift for all, then tol_test */
			EOP );

 		rule_init (
			"net.state_update",
			&sys->net[cn]->rules[ NET_R_state_update ],
			&state_update_meta_class,
			neuron_count,
			EOP );

 		rule_init (
			"net.acc_update",
			&sys->net[cn]->rules[ NET_R_acc_update ],
			&acc_update_meta_class,
			neuron_count,
			EOP );

 		rule_init (
			"net.state_shift",
			&sys->net[cn]->rules[ NET_R_state_shift ],
			&state_shift_meta_class,
			neuron_count,
			EOP );

 		rule_init (
			"net.weight_update",
			&sys->net[cn]->rules[ NET_R_weight_update ],
			&weight_update_meta_class,
			neuron_count,
			EOP );

/* ---------------- rule extend at NET level -------------------------- */

		/* asynchronous recall */

		rule_extend (
			&sys->net[cn]->rules[ NET_R_step_recall_async ],
			0,
			&sys->net[cn]->rules[ NET_R_state_update ],
			EOP );

		rule_extend (
			&sys->net[cn]->rules[ NET_R_step_recall_async ],
			1,
			&sys->net[cn]->rules[ NET_R_tolerance_test ],
			EOP );


		rule_extend (
			&sys->net[cn]->rules[ NET_R_recall_async ],
			0,
			&sys->net[cn]->rules[ NET_R_clear_counter ],
			EOP );

		rule_extend (
			&sys->net[cn]->rules[ NET_R_recall_async ],
			1,
			&sys->net[cn]->rules[ NET_R_loop_async ],
			EOP );


		rule_extend (
			&sys->net[cn]->rules[ NET_R_loop_async ],
			0,
			&sys->net[cn]->rules[ NET_R_state_update ],
			EOP );

		rule_extend (
			&sys->net[cn]->rules[ NET_R_loop_async ],
			1,
			&sys->net[cn]->rules[ NET_R_tolerance_test ],
			EOP );


		/* synchronous recall */

		rule_extend (
			&sys->net[cn]->rules[ NET_R_step_recall_sync ],
			0,
			&sys->net[cn]->rules[ NET_R_acc_update ],
			EOP );

		rule_extend (
			&sys->net[cn]->rules[ NET_R_step_recall_sync ],
			1,
			&sys->net[cn]->rules[ NET_R_state_shift ],
			EOP );

		rule_extend (
			&sys->net[cn]->rules[ NET_R_step_recall_sync ],
			2,
			&sys->net[cn]->rules[ NET_R_tolerance_test ],
			EOP );


		rule_extend (
			&sys->net[cn]->rules[ NET_R_recall_sync ],
			0,
			&sys->net[cn]->rules[ NET_R_clear_counter ],
			EOP );

		rule_extend (
			&sys->net[cn]->rules[ NET_R_recall_sync ],
			1,
			&sys->net[cn]->rules[ NET_R_loop_sync ],
			EOP );


		rule_extend (
			&sys->net[cn]->rules[ NET_R_loop_sync ],
			0,
			&sys->net[cn]->rules[ NET_R_acc_update ],
			EOP );

		rule_extend (
			&sys->net[cn]->rules[ NET_R_loop_sync ],
			1,
			&sys->net[cn]->rules[ NET_R_state_shift ],
			EOP );

		rule_extend (
			&sys->net[cn]->rules[ NET_R_loop_sync ],
			2,
			&sys->net[cn]->rules[ NET_R_tolerance_test ],
			EOP );


/* ---------------- others require NEURONS extentions -------------------------- */

		for (i=0, j=0; i < neuron_count; i++) {

			rule_extend (
				&sys->net[cn]->rules[ NET_R_clear_counter ],
				i,
				&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_ERR ],
				&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_STATE ],
				EOP );

			rule_extend (
				&sys->net[cn]->rules[ NET_R_tolerance_test ],
				i,
				&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_ERR ],
				&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->state[ N_STATE ],
				EOP );

			rule_extend (
				&sys->net[cn]->rules[ NET_R_learn ],
				i,
				&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->rules[ NEU_R_weight_update ],
				EOP );

			rule_extend (
				&sys->net[cn]->rules[ NET_R_state_update ],
				i,
				&sys->net[cn]->layer[0]->cluster[0]->neuron[j]->rules[ NEU_R_state_update ],
				EOP );
/*									    ^ change to something PRIME */
				j = ( j + 31 ) % 96;
/*				printf("%d\t",j); */

			rule_extend (
				&sys->net[cn]->rules[ NET_R_acc_update ],
				i,
				&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->rules[ NEU_R_acc_update ],
				EOP );

			rule_extend (
				&sys->net[cn]->rules[ NET_R_state_shift ],
				i,
				&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->rules[ NEU_R_state_shift ],
				EOP );

			rule_extend (
				&sys->net[cn]->rules[ NET_R_weight_update ],
				i,
				&sys->net[cn]->layer[0]->cluster[0]->neuron[i]->rules[ NEU_R_weight_update ],
				EOP );
    		}


	}	/* end for cn ... */

	return ( OK );
}

/* --------------------------------------------------------------------------- */
#ifndef	OMIT_MAIN

/*  Main program */

main (argc, argv)
int	argc;
char	*argv[];
{
	int		i, count, somefail, cycle;
	int		wh[4];
	float		result;

	rc_read();
	printf ( "local_user %s\n", rc[ RC_local_user ] );

	srandom(0xd5a4793c);	/*  Initialize drand48  */
	srand48(random());

	/*  Construct the network */

	if (argc == 1) {
		if (connect(&config))
			exit(3);
		if (build_rules())
			exit(3);
	}
	else {
		strcpy(system_filename, argv[1]);
		sys_load(system_filename);
	}

	/*  Initialise patterns specified on command line */

	if ( argc > 2 ) {
		i = init_patterns ( argv [ 2 ] );
		printf ( "init_patterns returns %d for %s\n", i, argv[ 2 ] );
	}

	/* learn patterns  - one pass */

	for ( i  = 0; i < pattern_count; i++ ) {
		current_pattern = indextop( i );
		read_input ( i );
		result = learn ();
		printf ( "pattern [ %s \'%c\' ]\tlearn [ %f ]\n", current_pattern->input_file, current_pattern->input_character, result );
	}


	if ( argc > 3 ) {
		i = init_patterns ( argv [ 3 ] );
		printf ( "init_patterns returns %d for %s\n", i, argv[ 3 ] );
	}

	for ( i  = 0; i < pattern_count; i++ ) {
		current_pattern = indextop( i );
		read_input ( i );
		shownet();
		cycle = 0;
		while ( TRUE ) {
			printf ( " recall_async ?\n" );
			if ( YES() ) {
				result = recall_async ();
				printf ( "cycle [%d] pattern [ %s \'%c\' ]\trecall_async [ %f ]\n", cycle, current_pattern->input_file, current_pattern->input_character, result );
				shownet();
			}
			else {
				printf ( " step_recall_sync ?\n" );
				if ( YES() ) {
					result = step_recall_sync ();
					printf ( "cycle [%d] pattern [ %s \'%c\' ]\tstep_recall_sync [ %f ]\n", cycle, current_pattern->input_file, current_pattern->input_character, result );
					shownet();
				}
				else {
					break;
				}
			}
			cycle++;
		}
	}

	if (system_filename[0] == '\0') {
		sys_save ( "test" );
	}
	else {
		sys_save( system_filename );
	}
}
#endif
/* --------------------------------------------------------------------------- */
