/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/********************************************************************** 
 * Graphics Interface
 * ------------------
 * This routine contain routines that interface between the
 * Graphics Interface package and the Rochester Connectionist
 * Simulator.
 *
 * Note: the intention is that ANY and ALL routines that
 *       interface to the simulator should reside here so 
 *       that changes to the simulator or attachment to
 *       new simulators should only necessitate changes
 *       to routines in this file.  All such "interface" 
 *       routines should by convention have the prefix
 *       "gi_sim_" as part of their name.
 * 
 *
 * Kenton Lynne
 *********************************************************************/

#include "macros.h"
#include "externs.h"
#include "uniproc.h"

/* entry point to the simulator command reader */
extern char *extern_command_reader();

/********************************************************************/
gi_sim_get_clock()
{
/* get the current simulator clock value
 */
   return(Clock);
}

/********************************************************************/
gi_sim_read_cmd(cmdstring)
  char *cmdstring;
{
/* send a command to the simulator which will return an error
 * message if something went wrong,     
 * otherwise returns NULL
 */
  char *ret_string;

  /* is this really necessary ? */
  if (cmdstring[strlen(cmdstring)-1] == '\n')
    cmdstring[strlen(cmdstring)-1] = '\0'; 
  if ((ret_string=extern_command_reader(cmdstring))!=NULL)
  {
    gi_put_error(ret_string);
    return(ERROR);
  }
  return(OK);         
}

/********************************************************************/
char *gi_sim_get_name(ui)
  int ui;
{
/* given a unit index, returns a pointer
 * the unit name 
 */
  return(IndToName(ui));
}

/********************************************************************/
char *gi_sim_get_type(ui)
   int ui;
{
/* given a unit index, returns a
 * pointer to the unit name
 */
  return((UnitList+ui)->type);
}

/********************************************************************/
gi_get_info(col,info_flag)
  int col, info_flag;
{
/* Fills in all the correct values for a particular gi_info_item.     
 * info_flag: TRUE requests that all information is needed 
 *            FALSE requests that only the dynamic info is needed
 * Assumes that ui, what, target and site have been filled in by caller.
 */
 
  int ui;
  char *sptr, *dptr;

  /* make sure that this is a valid unit index */
  if (gi_info_item[col].ui >= 0)
  {
    /* get the unit potential, output, state and data fields */
    ui = gi_info_item[col].ui;
    gi_info_item[col].pot = gi_sim_get_pot(ui);
    gi_info_item[col].out = gi_sim_get_output(ui);
    gi_info_item[col].state = gi_sim_get_state(ui);
    gi_info_item[col].data = gi_sim_get_data(ui);
 
    /* if in link mode, then get link info based on current target */
    if (gi_mode==MODE_LINK && gi_cur_link_target >= 0)
    {
      if (gi_link_direction==TO_TARGET)
        gi_info_item[col].link 
             = gi_sim_get_link(ui,gi_cur_link_target,gi_cur_link_site);
      else
        gi_info_item[col].link 
             = gi_sim_get_link(gi_cur_link_target,ui,gi_cur_link_site);
    }
    else

    /* only get link info if this unit is displaying a link value */
    {
      if (gi_info_item[col].what==LINKIN)
        gi_info_item[col].link 
            = gi_sim_get_link(ui,gi_info_item[col].target,
                              gi_info_item[col].site); 
      if (gi_info_item[col].what==LINKOUT)
        gi_info_item[col].link 
            = gi_sim_get_link(gi_info_item[col].target,
                              ui,gi_info_item[col].site); 
    }

    /* if flag set, then get static info as well */
    if (info_flag)
    {
      /* get name (and truncate length if necessary) */
      for (dptr=gi_info_item[col].name, sptr=gi_sim_get_name(ui);
           dptr < gi_info_item[col].name+MAX_NAME_LEN && *sptr!='\0'; )
        *dptr++ = *sptr++;
      *dptr = '\0';

      /* get type (and truncate length if necessary) */
      for (dptr=gi_info_item[col].type, sptr=gi_sim_get_type(ui);
           dptr < gi_info_item[col].type+MAX_NAME_LEN && *sptr!='\0'; )
        *dptr++ = *sptr++;
      *dptr = '\0';
    }
  }
}

/********************************************************************/
gi_get_values()
{
/* fills in all the correct values for grobjs on the 
 * display chain paying attention to the value of the
 * reshow_flag 
 */
  struct grobj *ptr;
  int i;
  FLINT new_val;

  /* look at each grobj on the display chain */
  for (ptr=gi_marker.next; ptr!=&gi_marker; ptr=ptr->next)
  {
    /* if all values need to be gotten or specifically
       this unit's value, then proceed to get the
       value that this grobj is showing
    */
    if (gi_reshow_flag & ALL_VALS_NEEDED || !(ptr->flag & VALUE_OK))
    {
      switch(ptr->u_what)
      {
        case POT:     
               new_val = gi_sim_get_pot(ptr->u_index);
               break;
        case OUTPUT:  
               new_val = gi_sim_get_output(ptr->u_index);
               break;
        case STATE:   
               new_val = gi_sim_get_state(ptr->u_index);
               break;
        case LINKIN:  
               new_val = gi_sim_get_link(ptr->u_index, ptr->target, ptr->site);
               break;
        case LINKOUT: 
               new_val = gi_sim_get_link(ptr->target, ptr->u_index, ptr->site);
               break;
        case DATA:    
               new_val = gi_sim_get_data(ptr->u_index);
               break;
      } 
      
      /* if the value has changed, set the VALUE_CHG flag on 
         (checked by display_grobj_chain)
      */
      
      if (new_val!=ptr->val) 
      {  
        ptr->val = new_val;
        ptr->flag |= VALUE_CHG;
      }

      /* indicate that the value is now up-to-date */
      ptr->flag |= VALUE_OK;
    }
  }
     
  /* now update all the values for the info panel */
  for (i=0; i<MAX_INFO_COLS; i++)
    gi_get_info(i,FALSE);
}

/********************************************************************/
gi_get_links()
{
/* fills in all the correct link values for
 * grobjs on the display grobj chain paying attention to the
 * reshow_flag (ALL_LINKS_NEEDED)
 */
  struct grobj *ptr;
  int i;
  FLINT new_val;

  for (ptr=gi_marker.next; ptr!=&gi_marker; ptr=ptr->next)
  {
    if (gi_reshow_flag & ALL_LINKS_NEEDED || !(ptr->flag & LINK_OK))
    {
      /* if a target has been picked out, get link value from simulator */
      if (gi_cur_link_target >= 0)
      {
        if (gi_link_direction==TO_TARGET)
          new_val = gi_sim_get_link(ptr->u_index,gi_cur_link_target,
                                    gi_cur_link_site);
        else
          new_val = gi_sim_get_link(gi_cur_link_target,ptr->u_index,
                                    gi_cur_link_site);
      }
      else  /* return 0 */
        new_val = 0;

      /* if the value has changed, set the LINK_CHG flag
         (checked by display_grobj_chain)
      */
      if (new_val!=ptr->link_val) 
      {  
        ptr->link_val = new_val;
        ptr->flag |= LINK_CHG;
      }

      /* indicate that the link value now up-to-date */
      ptr->flag |= LINK_OK;
    }
  }

  /* now update all the values for the info panel */
  for (i=0; i<MAX_INFO_COLS; i++)
    gi_get_info(i,FALSE);
}

/********************************************************************/
FLINT gi_sim_get_link(from_unit, to_unit, sitename)
  int from_unit, to_unit;
  char *sitename;
{
/* returns the value of a link from one
 * unit to another
 * 0 is returned if no link exists
 * only looks at links within the given sitename
 * unless sitename is NULL in which case all sites
 * are looked at with the search terminating with
 * the first matching one found
 */
  FLINT value = 0; 
  int done=FALSE;
  Site *sp; 
  Link *lp;

  /* get first site pointer for to-unit */
  sp = (UnitList+to_unit)->sites;

  /* get matching site name, if any */
  if (sitename)
  {
    for (; sp!=NULL; sp=sp->next)
    {
      if (strcmp(sp->name,sitename)==0)
        break;
    }
  } 
    
  /* look through the sites for the particular frrom_unit */
  while (!done && sp!=NULL)
  {
     /* get first link pointer at this site */
     lp = sp->inputs;
     while (!done && lp!=NULL)
     {
       if (lp->from_unit==from_unit)
       {
          value = lp->weight;
          done = TRUE;
       }
       lp = lp->next;
     }
     /* if sitename given execute inner loop only once */
     if (sitename)
       done = TRUE;
     else
       sp = sp->next;
  }

  return(value);
}


/********************************************************************/
gi_sim_go(num)
  int num;
{
/* runs num simulator steps   
 */
   Step(num);
}

/********************************************************************/
gi_sim_get_unit(string, flag) 
   char *string;
   int flag;
{
/* this routine is passed a character string    
 * and returns a unit index or ERROR if not found
 * the string can be either an (ASCII) unit number
 * a unit name, a unit set name or a unit type
 * flag is either GET_FIRST : return 1st unit number that
 *                        meets description
 *             or GET_NEXT  : return the next unit (must be
 *                        used after a previous call with
 *                        flag set to FIRST or NEXT)
 */

  static int save_ui;
  int ui, i; 
  NameDesc *nt,nte;

  /* see if a unit index was passed */
  if (gi_check_num(string)==OK)
  {
    /* make sure unit index is not negative */
    if((ui=atoi(string))<0)
    {
      ui = ERROR;
    }
    else
    {
      /* if GET_FIRST call, then save unit index for next time */
      if (flag==GET_FIRST)
        save_ui = ui;

      /* if GET_NEXT call, then bump unit index by one */
      else
        ui = ++save_ui;

      /* check that unit index asked for is within range */
      if (ui>=NoUnits) 
      {
        ui = ERROR;
      }
    } 
  }

  /* check if it is a unit name */
  else if ((nt=FindName(string,&nte))) 
  {
    /* is either a setname or a unit name */
    if (nt->type<SET_SYM) 
    {
       /* is a unit name */
       if (flag==GET_FIRST)
       {
         /* if GET_FIRST and unit name, then return first index */
         ui = nt->index;
         save_ui = ui;
       }
       else  /* flag is GET_NEXT */
       {
         switch (nt->type)
         {
           case SCALAR: 
             /* only one unit, therfore GET_NEXT must fail */
             ui = ERROR;
             break;
           case VECTOR:
             /* vector name, make sure "next" unit is within range */
             if ((save_ui - nt->index + 1) < nt->size)
               ui = ++save_ui;
             else
               ui = ERROR;
             break;
           case ARRAY:
             /* array name, make sure "next" unit is still in array */
             if ((save_ui - nt->index +1) < (nt->size * nt->length))
               ui = ++save_ui;
             else
               ui = ERROR;
             break;
           default:
             fprintf(stderr,"unexpected name type for unit in grsim.c\n");
         }
       }
    }

    /* check if it is a set name */
    else if (nt->type==SET_SYM)
    {
       /* is a set name */
       ui = ERROR;  /* initialize */
       if (flag==GET_FIRST) 
         i = 0;
       else 
         i = ++save_ui;
       for (; i<NoUnits; i++)
       {
           if (MemberSet(string,i))
           {
             ui = save_ui = i;
             break;
           }
       }
     }


     /* check if it is a type name */
     else if (nt->type==TYPE_SYM)
     {
       /* search the whole UnitList array */
       ui = ERROR;
       if (flag==GET_FIRST) 
         i = 0;
       else 
         i = ++save_ui;
       for (; i<NoUnits; i++)
       { 
         /* check if this unit si of correct type */
         if (strcmp(string,(UnitList+i)->type)==0)
         {
            ui = save_ui = i;
            break;
         }
       }
     }
     else /* string is not a name, setname or type name -- user error */
     {
       ui = ERROR;
     }
   }
   else  /* string not found - must be user error */
   {
     ui = ERROR;
   }
   return (ui);
 }

/********************************************************************/
FLINT gi_sim_get_output(ui)
  int ui;
{
/* returns the current output value of the
 * unit indicated by ui  
 */

#ifdef DEBUG
   if (ui>=NoUnits)
   {
      gi_put_error("Bad unit index sent to sim_get_output");
      return(ERROR);
   }
#endif
   return ((UnitList+ui)->output);
}

/********************************************************************/
FLINT gi_sim_get_pot(ui)
  int ui;
{
/* returns the current potential value of the
 * unit indicated by ui
 */

#ifdef DEBUG
   if (ui>=NoUnits)
   {
      gi_put_error("Bad unit index sent to sim_get_pot");
      return(ERROR);
   }
#endif

   return ((UnitList+ui)->potential);
}
 
/********************************************************************/
FLINT gi_sim_get_state(ui)
  int ui;
{
/* returns the current potential value of the
 * unit indicated by ui
 */

#ifdef DEBUG
   if (ui>=NoUnits)
   {
      gi_put_error("Bad unit index sent to sim_get_state");
      return(ERROR);
   }
#endif
   return ((FLINT) (UnitList+ui)->state);
}  

/********************************************************************/
FLINT gi_sim_get_data(ui)
  int ui;
{
/* given a unit index, returns the
 * value of the data field of the unit
 */

#ifdef DEBUG
   if (ui>=NoUnits)
   {
      gi_put_error("Bad unit index sent to sim_get_data");
      return(ERROR);
   }
#endif
  return ((UnitList+ui)->data);
}
