/******************************************************************************
**  The Rochester Connectionist Simulator - a neural network simulator.      **
**  COPYRIGHT (C) 1989  UNIVERSITY OF ROCHESTER.                             **
**                                                                           **
**  This program is free software; you can redistribute it and/or modify it  **
**  under the terms of the GNU General Public License as published by the    **
**  Free Software Foundation; either version 1, or (at your option) any      **
**  later version.                                                           ** 
**                                                                           **
**  This program is distributed in the hope that it will be useful, but      **
**  WITHOUT ANY WARRANTY; without even the implied warranty of               **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     **
**  See the GNU General Public License for more details.                     **
*******************************************************************************/

/**********************************************************************
 * Graphics Interface
 * ------------------
 * This file contains the routines used to set up, access and change
 * values and structures associated with the display (or graphics)
 * panel, including processing all input events (mouse buttons, moves,
 * and keystrokes) that occur in the display panel..
 *
 * Kenton Lynne
 *********************************************************************/

#include "macros.h"
#include "externs.h"
#include "panel.h"
#include "cmd_panel.h"
#include "mode_panel.h"
#include "control_panel.h"

/* place for saving the old cursor image for later restoration */
DEFINE_CURSOR(oldcursor, 0, 0, 0,
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

/* get and format cursor for moving the display */
static short gall_image[16] = {
#include "icon/graball"
};
DEFINE_CURSOR_FROM_IMAGE(gball_cursor,HOTX,HOTY,PIX_SRC ^ PIX_DST,gall_image);
static struct cursor *graball = &gball_cursor;

/* get and format cursor for moving graphic objects */
static short grab_image[16] = {
#include "icon/grab"
};
DEFINE_CURSOR_FROM_IMAGE(gb_cursor,HOTX,HOTY,PIX_SRC ^ PIX_DST,grab_image);
static struct cursor *grab = &gb_cursor;

/* the following static variables are accessed "globally" by
 * routines in this file
 */
static struct inputevent ie;        /* the current input event */
static short blob_set;              /* indicates whether blob is shown */
static int old_blob_x, old_blob_y;  /* current location of blob */
static int old_font_x, old_font_y;  /* x/y spacing for current font */
static int reshow;                  /* reshow should be done */
static int first_move;              /* indicates first move of drawn object */
static int mouse_mode=NORMAL;       /* current state of the mouse */

static short vertex_set;            /* a vertex has been set */
static struct drobj 
  *deldraw_ptr=NULL,                /* to-be-deleted drawn object */
  *drobj_ptr,                       /* current drawn object */ 
  *cur_draw;                        /* current drawn object */

static struct txobj 
    *deltext_ptr=NULL,              /* to-be-deleted text object */
    *txobj_ptr;                     /* current text object */
static struct grobj *grobj_ptr;

static int last_disp_move_x=0, 
           last_disp_move_y=0;
static int old_vertex_x, old_vertex_y;
static int old_move_x, old_move_y;
static int cur_x, cur_y;
static int last_x, last_y;
static int base_x, base_y;  
static int cur_text_x, cur_text_y, cur_text_len=0;

/* static procedures in this file that are forward referenced */
static display_select(), display_redraw();

/*************************************************************************/
gi_make_display_panel()
{
/* initial set up and display of the display (graphics) panel
 */

  struct toolsw *display_sw;

  /* make the display panel */
  if ((display_sw = gfxsw_createtoolsubwindow(gi_tool,"Display",
            TOOL_SWEXTENDTOEDGE, 
            CNTL_HEIGHT,
            0)) == NULL) 
  {
    fputs("Can't create display graphics subwindow\n", stderr);
    exit(1);
  }

  /* get the pointer to the graphics subwindow */
  gi_gfx = (struct gfxsubwindow *)display_sw->ts_data;

  /* make the graphics window retained */
  gfxsw_getretained(gi_gfx); 

  /* set the routine to be called for input events */
  display_sw->ts_io.tio_selected = display_select;

  /* set the routine to be called for redrawing the screen */
  display_sw->ts_io.tio_handlesigwinch = display_redraw;
}

/*************************************************************************/
/*ARGSUSED*/
static display_redraw(data)
   caddr_t data;
{
/* the graphics (and info) screen have been resized
 * thus change the rectlist accordingly and
 * redraw all the objects in it
 */

   int off_flag, on_flag;

   /* do the standard thing for graphics windows */
   gfxsw_interpretesigwinch(gi_gfx);

   /* first check if the panel has been damaged */
   if (gi_gfx->gfx_flags & GFX_DAMAGED)
   {
     /* call the default damage handler */
     gfxsw_handlesigwinch(gi_gfx);
   }

   /* now check if the window has been resized */
   if (gi_gfx->gfx_flags & GFX_RESTART)
   {
     /* screen has been resized */
     gi_gfx->gfx_flags &= ~GFX_RESTART;

     /* if the new size is larger in either dimension, then
        we need to determine what new objects have to 
        be displayed 
     */

     if (gi_rects[GRAPH].r_width > gi_old_width
     || gi_rects[GRAPH].r_height > gi_old_height)
       off_flag = TRUE;
 
     if (gi_rects[GRAPH].r_width < gi_old_width
     || gi_rects[GRAPH].r_height < gi_old_height)
       on_flag = TRUE;
 
     if (on_flag || off_flag)
     {
       gi_update_grobj_chain(off_flag, on_flag);
       gi_reshow_flag |= RESHOW_ALL;
       gi_reshow();
     }
     
     /* keep the log items on the extreme right side of the panel */
     panel_set(gi_log_item,PANEL_LABEL_X,gi_log_start_x,0);
     panel_set(gi_logfile_item,PANEL_LABEL_X,PANEL_CU(9)+gi_log_start_x,0);
   }
}
    
/*************************************************************************/
/*ARGSUSED*/
static display_select(sw, ibits, obits, ebits, timer)
     caddr_t sw;
     int *ibits, *obits, *ebits;
     struct timeval **timer;
{
/* handles all the input events for the display panel 
 */

   /* get the next input event on the queue for display window */
   input_readevent(gi_gfx->gfx_windowfd, &ie);

   /* set up the current display coordinates */
   cur_x = ie.ie_locx + gi_origin_x;
   cur_y = ie.ie_locy + gi_origin_y;
   reshow = FALSE;

   if ((deldraw_ptr || deltext_ptr)
      && ie.ie_code != MS_RIGHT)
   {
     gi_put_message("Delete cancelled");
     deldraw_ptr = (struct drobj *) deltext_ptr = NULL;
   }

   /* based on current mode, call appropriate handler */
   switch (gi_mode)
   {
     case MODE_MAIN:
            do_main_mode();
            break;
     case MODE_LINK:
            do_link_mode();
            break;
     case MODE_TEXT:
            do_text_mode();
            break;
     case MODE_DRAW:
            do_draw_mode();
            break;
     case MODE_CUSTOM:
            do_custom_mode();
            break;
   } 

   /* update the screen if necessary */
   if (reshow)
   {
     if ((gi_reshow_flag & (RESHOW_NEEDED+RESHOW_ALL+CLEAR_NEEDED))
     || (gi_mode==MODE_LINK 
         && gi_reshow_flag & (SOME_LINKS_NEEDED+ALL_LINKS_NEEDED))
     || (gi_mode!=MODE_LINK 
         && gi_reshow_flag & (SOME_VALS_NEEDED+ALL_VALS_NEEDED)))
       gi_reshow();
   } 

   *ibits = *obits = *ebits = 0;
}

/*************************************************************************/
do_main_mode()
{
/* process a main mode input event on
 * the display screen
 */
   static struct grobj *obj_ptr;
   char buf[MAX_ITEM_LENGTH+1];
   char *sptr;

   /*************************************/
   /* mouse has left the display window */ 
   /*************************************/
   if (ie.ie_code==LOC_WINEXIT && mouse_mode!=NORMAL)
   {
     drop_move_obj(cur_x, cur_y);
   }

   /* mouse being moved while button pushed DOWN */
   if (ie.ie_code==LOC_MOVEWHILEBUTDOWN) 
   {
     cont_move();
   }

   /* left or right mouse button down while in MOVE_DISPLAY mode */
   else if (mouse_mode==MOVE_DISPLAY && !win_inputnegevent(&ie) &&
            (ie.ie_code==MS_LEFT || ie.ie_code==MS_RIGHT))
   {
     /* jump move the display */
     jump_move();
   }

   /**************************/
   /* left mouse button DOWN */
   /**************************/
   else if (ie.ie_code==MS_LEFT 
        && mouse_mode==NORMAL 
        && !win_inputnegevent(&ie))
   {
       /* update info panel for this unit if it exists */
       if (gi_do_info(cur_x, cur_y, TRUE)==ERROR) 
       {
         /* no unit is here : so move marker here instead */
         gi_move_marker(cur_x-HOTX,cur_y-HOTY);

         /* if marker currently not shown, show it */
         if (gi_marker.flag & NOSHOW)
         {
           gi_marker.flag &= ~NOSHOW;
           gi_display_grobj(&gi_marker, PIX_SRC ^ PIX_DST);
         }
       }
   }

   /**********************************/ 
   /* middle mouse button UP or DOWN */
   /**********************************/ 
   else if (ie.ie_code==MS_MIDDLE)
   {
     do_middle();
   }

   /***************************/
   /* right mouse button DOWN */
   /***************************/
   else if (ie.ie_code==MS_RIGHT 
        && mouse_mode==NORMAL
        && !win_inputnegevent(&ie))
   {
     /* check if the marker is here and if so, blank it out */
     if (gi_marker_here(cur_x,cur_y) && !(gi_marker.flag & NOSHOW))
     {
       /* marker is underneath, set flag to NOSHOW */
       gi_marker.flag |= NOSHOW;
       gi_display_grobj(&gi_marker,PIX_SRC ^ PIX_DST);
     }

     /* check if a grobj is here and tag or untag it as appropriate */
     else if ((obj_ptr=gi_find_xy(cur_x,cur_y))!=NULL)
     {
       if ((obj_ptr->flag & TAGGED)==TAGGED)
       {
          /* unmark this item as current LINK target */ 
          gi_last_tagged = NULL;
          obj_ptr->flag = obj_ptr->flag & ~TAGGED;
          gi_display_grobj(obj_ptr, PIX_SRC);
       }
       else 
       {
          /* mark this node as the current LINK target */
          obj_ptr->flag = obj_ptr->flag | TAGGED;
          gi_display_grobj(obj_ptr,PIX_NOT(PIX_SRC));

          /* update last_tagged to point to this object */
          if (gi_last_tagged!=NULL)
          {
            gi_last_tagged->flag &= ~TAGGED;
            gi_display_grobj(gi_last_tagged, PIX_SRC);
          }
          gi_last_tagged = obj_ptr;

          /* update the target prompt on the control panel */
          (void) sprintf(buf,"%1d",obj_ptr->u_index);
          
          /* make sure to keep the site prompt, if any */
          for (sptr=(char *) panel_get_value(gi_mitem[TARGET_ITEM]); 
               *sptr!='\0'; sptr++)
          {
            if (*sptr==SITE_SEP_CHAR)
            {
              strcat(buf,sptr);
              break;
            }
          }
    
          /* copy new target/site arg to panel prompt */
          panel_set_value(gi_mitem[TARGET_ITEM],buf);
          panel_set(gi_control_panel,PANEL_CARET_ITEM,gi_mitem[TARGET_ITEM],0);
       }
     }
   }       
}

/**********************************************************************/
gi_set_new_target(ptr,ui)
  struct grobj *ptr;
  int ui;
{
/* Sets up the unit indicated by 
 * by the passed unit index
 * or the ptr (if not NULL)
 * as the new target in link mode.
 * Note: if passed ptr is NULL, then the grobj chains will
 *       be searched for non-auxiliary unit with ui.
 *       In this case, we assume that the caller is "target_proc"
 *       and thus there is no need to update the control panel
 *       prompt with the new ui/site string 
 *       
 *       
 */
  char buf[MAX_ITEM_LENGTH+1];
  char *site_string;
  int update_prompt;
 
  /* if ptr is null, then get the pointer, if any,
     to the grobj for that unit
  */
  if (ptr)
  {
    ui = ptr->u_index;
    update_prompt = TRUE;
  }
  else
  {
    ptr = gi_find_unit(ui,0);
    update_prompt = FALSE;
  }

  /* make sure the old link target (if any) gets redisplayed */
  if (gi_cur_target_ptr)
    gi_cur_target_ptr->flag &= ~DISPLAYED;

  /* make sure the new link target (if any) gets redisplayed */
  if (ptr)
    ptr->flag &= ~DISPLAYED;

  /* get the current link site */
  gi_cur_link_site 
      = gi_get_site((char *) panel_get_value(gi_litem[LSITE_ITEM]));

  /* update globals for current target */
  gi_cur_link_target = ui;
  gi_cur_target_ptr = ptr;  /* note: could be NULL */

  /* update the prompt for the target field (if no ptr supplied) */
  if (update_prompt)
  {
    if (gi_cur_link_site)
      site_string = gi_cur_link_site;
    else
      site_string = ANY_SITE;
    (void) sprintf(buf,"%1d/%s",gi_cur_link_target,site_string);
    panel_set_value(gi_litem[LSITE_ITEM],buf);
  }

  /* set the reshow flags appropriately */
  gi_reshow_flag |= ALL_LINKS_NEEDED;
}

/*************************************************************************/
static do_link_mode()
{
/* process an input event on the display panel while
 * in "link"  mode
 */
  struct grobj *ptr;

   /**********************************************/
   /* mouse being moved while button pushed DOWN */
   /**********************************************/
   if (ie.ie_code==LOC_MOVEWHILEBUTDOWN)
   {
     cont_move();
   }

   /**********************************/
   /* middle mouse button UP or DOWN */
   /**********************************/
   else if (ie.ie_code==MS_MIDDLE)
   {
      do_middle(); 
   }

   /*************************************/
   /* mouse has left the display window */ 
   /*************************************/
   else if (ie.ie_code==LOC_WINEXIT)
   {
     if (mouse_mode!=NORMAL)
       drop_move_obj(cur_x, cur_y);
   }

   /**************************/
   /* left mouse button DOWN */
   /**************************/
   else if (ie.ie_code==MS_LEFT && !win_inputnegevent(&ie))
   {
     gi_do_info(cur_x, cur_y, TRUE);
   }

   /***********************************/    
   /* right  mouse button pushed DOWN */
   /***********************************/    
   else if (ie.ie_code==MS_RIGHT && !win_inputnegevent(&ie))
   {
     /* if there is a unit at this location make it
        the target and set up gi_reshow so that
        new values are gotten for all the objects
     */
     if ((ptr=gi_find_xy(cur_x, cur_y))!=NULL)
     {
       gi_set_new_target(ptr,0); 
       reshow = TRUE;
     }
   }
}

/*************************************************************************/
static do_text_mode()
{
/* process an input event in text mode on
 * the display screen
 */
   char buf[MAX_FONT_LENGTH+1];

   /******************/
   /* keyboard input */
   /******************/
   if (ie.ie_code > ASCII_FIRST 
       && ie.ie_code <= ASCII_LAST
       && blob_set)
   {
     process_ASCII((char) ie.ie_code);
   }
       
   /* if text is currently being entered any other significant
      mouse action will cause the current text object to be
      saved immediately
   */

   else if (blob_set && !win_inputnegevent(&ie))
   {
     /* erase text blob */
     clear_blob();

     /* if any current text, save it */
     if (cur_text_len > 0)
       save_text();
   }

   /*************************************/
   /* mouse has left the display window */ 
   /*************************************/
   if (ie.ie_code==LOC_WINEXIT)
   {
     if (mouse_mode!=NORMAL)
     {
       /* drop object currently being moved */
       drop_move_obj(cur_x, cur_y);
     }
   }

   /****************************************/
   /* mouse moved while button pushed DOWN */
   /****************************************/
   else if (ie.ie_code==LOC_MOVEWHILEBUTDOWN)
   {
     cont_move();
   }

   /**************************************************************/
   /* left or right mouse button down while in MOVE_DISPLAY mode */
   /**************************************************************/
   else if (mouse_mode==MOVE_DISPLAY && !win_inputnegevent(&ie) &&
            (ie.ie_code==MS_LEFT || ie.ie_code==MS_RIGHT))
   {
     /* jump move the display */
     jump_move();
   }

   /**************************/
   /* left mouse button DOWN */
   /**************************/
   else if (ie.ie_code==MS_LEFT && !win_inputnegevent(&ie))
   {
     /* get font name from panel prompts */
     strcpy(buf,(char *) panel_get_value(gi_titem[TFONT1_ITEM]));
     strcat(gi_strip(buf),(char *) panel_get_value(gi_titem[TFONT2_ITEM]));
     strcat(gi_strip(buf),(char *) panel_get_value(gi_titem[TFONT3_ITEM]));
     gi_cur_font = gi_get_font(gi_strip(buf));

     /* find this font and load it in if necessary */
     if (gi_cur_font==NULL) gi_cur_font = gi_dft_font;

     /* write blob out */
     write_blob(ie.ie_locx+1,ie.ie_locy+5);
   }

   /**********************************/
   /* middle mouse button UP or DOWN */
   /**********************************/
   else if (ie.ie_code==MS_MIDDLE)
   {
     do_middle();
   }

   /***************************/
   /* right mouse button DOWN */
   /***************************/
   else if (ie.ie_code==MS_RIGHT && !win_inputnegevent(&ie))
   {
      if (deltext_ptr==NULL)
      {
        if ((deltext_ptr=gi_find_txobj(cur_x,cur_y))!=NULL)
        {
          gi_put_message("Confirm delete with right button");
        }
      }
      else
      {
        gi_put_message("Delete confirmed");

	/* blank out the the object on the screen */
        gi_display_txobj(deltext_ptr,PIX_SRC ^ PIX_DST);

	/* delete text object from chain */
        gi_erase_txobj(deltext_ptr);
        deltext_ptr = NULL;

        /* if logging enabled, log appropriate delete command */

        if (gi_log_on) gi_log_delete(cur_x, cur_y);
      }
   }
}

/*************************************************************************/
static do_draw_mode()
{
/* process an input event in draw mode on
 * the display screen
 */
   int i;

   /*************************************/
   /* mouse has left the display window */
   /*************************************/
   if (ie.ie_code==LOC_WINEXIT)
   {
     if (mouse_mode!=NORMAL)
       drop_move_obj(cur_x, cur_y);
     else if (vertex_set)
     {
       if (gi_draw_type==LINES)
         draw_lines();
       else if (gi_draw_type==BOXES)
         draw_box(0);
       else if (gi_draw_type==BOUND)
         draw_box(BOUND_BOX);
     }
   }

   /**********************************************/
   /* mouse being moved while button pushed DOWN */
   /**********************************************/
   else if (ie.ie_code==LOC_MOVEWHILEBUTDOWN)
   {
     cont_move();
   }

   /* left or right mouse button down while in MOVE_DISPLAY mode */
   else if (mouse_mode==MOVE_DISPLAY && !win_inputnegevent(&ie) &&
            (ie.ie_code==MS_LEFT || ie.ie_code==MS_RIGHT))
   {
     /* jump move the display */
     jump_move();
   }

   /*********************************/
   /* left mouse button pushed DOWN */
   /*********************************/
   else if (ie.ie_code==MS_LEFT 
	&& mouse_mode==NORMAL
	&& !win_inputnegevent(&ie))
   {
     /* see if a vertex has already been set up */
     if (!vertex_set)
     {
       /* no vertex yet: save this position in relevant variables */
       old_vertex_x = last_x = cur_x;
       old_vertex_y = last_y = cur_y;
       vertex_set = TRUE;

       /* write out a single pixel at cursor "hot spot" */
       pw_put(gi_gfx->gfx_pixwin, ie.ie_locx, ie.ie_locy, 1);
     }
     else /* a vertex has already been set */
     {
       if (gi_draw_type==LINES)
       {
         /* check if this is near the last vertex set */
         if (cur_draw==NULL) /* 2nd vertex */
         {
	   /* set up a new graphic object for this drawing */
	   cur_draw = gi_remember_drobj(old_vertex_x, old_vertex_y,
			             cur_x, cur_y, 0);
	   old_vertex_x = cur_x;
	   old_vertex_y = cur_y;
         }
         else if (vertex_nearby(cur_x, cur_y))
         {
           /* finish drawing the object */
           draw_lines();
         }
         else /* add another vertex to this object */
         {
           /* limit the size of each object to 10 vertices */
           if (cur_draw->num_vertices >= MAX_VERTICES)
           {
             /* over vertex limit: finish up old object and
              * then create a new object and continue
              */
             draw_lines();
             vertex_set = TRUE;
	     cur_draw = gi_remember_drobj(old_vertex_x, old_vertex_y,
			             cur_x, cur_y, 0);
           }

           else
           {
             /* save this vertex with the object */
	     gi_save_vertex(cur_draw, cur_x, cur_y);
           }

           /* update the old vertex values to the current vertex */
	   old_vertex_x = cur_x;
	   old_vertex_y = cur_y;
         }
       }

       else if (gi_draw_type==BOXES)
       {
         draw_box(0);
       }

       else if (gi_draw_type==BOUND)
       {
         draw_box(BOUND_BOX);
       }
     }
   }


   /************************/
   /* mouse is being moved */
   /************************/
   else if (ie.ie_code==LOC_MOVE && vertex_set)
   {
     if (gi_draw_type==LINES)
     {
       for (i=0; i<2; i++)
       {
         /* on first pass, erase the previous line */
         /* on second pass, write new line */
         pw_vector(gi_gfx->gfx_pixwin, 
	           old_vertex_x-gi_origin_x, old_vertex_y-gi_origin_y,
	           last_x-gi_origin_x, last_y-gi_origin_y, 
	           PIX_SRC ^ PIX_DST, 1);
	  /* update last vertex values */
          last_x = cur_x;
          last_y = cur_y;
       }
     }
     else /* if (gi_draw_type==BOXES) */
     {
       /* erase the old box and draw a new one */
       for (i=0; i<2; i++)
       {
	 /* on first pass, erase old box */
	 /* on next pass, write out new box */
         pw_vector(gi_gfx->gfx_pixwin, 
		 old_vertex_x-gi_origin_x, old_vertex_y-gi_origin_y, 
                 old_vertex_x-gi_origin_x, last_y-gi_origin_y, 
                 PIX_SRC ^ PIX_DST, 1);
         pw_vector(gi_gfx->gfx_pixwin, 
		 old_vertex_x-gi_origin_x, last_y-gi_origin_y, 
                 last_x-gi_origin_x, last_y-gi_origin_y, 
                 PIX_SRC ^ PIX_DST, 1);
         pw_vector(gi_gfx->gfx_pixwin, 
		 last_x-gi_origin_x, last_y-gi_origin_y, 
                 last_x-gi_origin_x, old_vertex_y-gi_origin_y, 
                 PIX_SRC ^ PIX_DST, 1);
         pw_vector(gi_gfx->gfx_pixwin, 
		 last_x-gi_origin_x, old_vertex_y-gi_origin_y, 
                 old_vertex_x-gi_origin_x, old_vertex_y-gi_origin_y, 
                 PIX_SRC ^ PIX_DST, 1);

         /* set up new corner coordinates */		
         last_x = cur_x;
         last_y = cur_y;
       }
     }
   }

   /**********************************/
   /* middle mouse button UP or DOWN */
   /**********************************/
   else if (ie.ie_code==MS_MIDDLE)
   {
      do_middle();
   }

   /***************************/
   /* right mouse button DOWN */ 
   /***************************/
   else if (ie.ie_code==MS_RIGHT && !win_inputnegevent(&ie))
   {
      if (deldraw_ptr==NULL)
      {
        if ((deldraw_ptr=gi_find_drobj(cur_x,cur_y))!=NULL)
        {
          gi_put_message("Confirm delete with right button");
        }
      }
      else
      {
        gi_put_message("Delete confirmed");

	/* blank out the the object on the screen */
        gi_display_drobj(deldraw_ptr,PIX_CLR);

	/* delete text object from chain */
        gi_erase_drobj(deldraw_ptr);
        deldraw_ptr = NULL;
        
        /* if logging enabled, log appropriate delete command */

        if (gi_log_on) gi_log_delete(cur_x, cur_y);
      }
   }
}
 
/*************************************************************************/
static do_custom_mode()
{
/* process an input event in CUSTOM mode
 * by passing the index of the button panel prompt
 * to the process_button routine
 */

   int index= -1;

   /* check for positive (click down) mouse event */
   if (!win_inputnegevent(&ie))
   {
     if (ie.ie_code==MS_LEFT) 
     {
       /* set index to left button down panel prompt */
       index = CLBDN1_ITEM;
     }
     else if (ie.ie_code==MS_MIDDLE) 
     {
       /* set index to middle button down panel prompt */
       index = CMBDN1_ITEM;
     }
     else if (ie.ie_code==MS_RIGHT) 
     {
       /* set index to right button down panel prompt */
       index = CRBDN1_ITEM;
     }
   }

   else /* button was lifted */
   {
     if (ie.ie_code==MS_LEFT) 
     {
       /* set index to left button up panel prompt */
       index = CLBUP1_ITEM;
     }
     else if (ie.ie_code==MS_MIDDLE) 
     {
       /* set index to middle button up panel prompt */
       index = CMBUP1_ITEM;
     }
     else if (ie.ie_code==MS_RIGHT) 
     {
       /* set index to right button up panel prompt */
       index = CRBUP1_ITEM;
     }
   }

   /* if a button press or release, process that button */
   if (index >= CLBDN1_ITEM)
     process_button(index);

   /* else process mouse event as if in main mode */
   else
     do_main_mode();
}

/*************************************************************************/
static move_cur_object(x, y)
  int x, y;
{
/* given the current display space coordinates of the cursor,
 * moves the currently "moving" object to this position after
 * erasing the old object
 * note: assumes old_move_x, old_move_y are correct
 */
    int offset_x, offset_y, erase_op;
    char buf[2*MAX_ITEM_LENGTH+1];
    struct drobj *dptr;

    /* calculate the new offset since last position */
    offset_x = x - old_move_x;
    offset_y = y - old_move_y;

    switch(mouse_mode)
    {
      case MOVE_DISPLAY:
           (void) sprintf(buf,"%1d %1d", 
                         gi_origin_x + (base_x - x),
                         gi_origin_y + (base_y - y));
           panel_set(gi_origin_item, PANEL_LABEL_STRING, buf, 0);
           break;

      case MOVE_GROBJ:
           gi_move_grobj(grobj_ptr, offset_x, offset_y, TRUE);
           break;

      case MOVE_TXOBJ:
           gi_move_txobj(txobj_ptr,offset_x, offset_y, TRUE);
           break;

      case MOVE_DROBJ:
           /* determine what pixop to use for the erase */
           if (first_move)
           {
             /* use PIX_CLR so as not to leave garbage around */
             first_move = FALSE;
             erase_op = PIX_CLR;
           }
           else  /* use a mask so as not to disturb other objects */
             erase_op = PIX_SRC ^ PIX_DST;

           /* check if this is a bounding box */
           if (drobj_ptr->flag & BOUND_BOX)
           {
             /* go through the draw chain looking for bounding boxes */
             for (dptr=gi_draw_head.next; dptr!=&gi_draw_head; dptr=dptr->next)
             {
               if (dptr->flag & MOVING && dptr->flag & BOUND_BOX)
                 gi_move_drobj(dptr,offset_x,offset_y,
                               erase_op,PIX_SRC^PIX_DST);
             }
           }

           else  /* just a normal object to move */
             gi_move_drobj(drobj_ptr,offset_x,offset_y,
                           erase_op,PIX_SRC^PIX_DST);
           
           break;
    }
}

/*************************************************************************/
static jump_move()
{
/* executes a "jump move" of the display window
 * by the indicated offsets
 * called when the user presses either the right of
 * left mouse button while in DISPLAY_MOVE mode
 */

  /* only do something if displacement non-zero */
  if (last_disp_move_x || last_disp_move_y)
  {
    /* jump in opposite directions depending on button pushed */
    if (ie.ie_code==MS_RIGHT)
      gi_change_origin(last_disp_move_x, last_disp_move_y);
    else
      gi_change_origin(-last_disp_move_x, -last_disp_move_y);

    /* update the grobj chain */
    gi_update_grobj_chain(TRUE,TRUE);

    /* set flags to clear the display and reshow everything */

    gi_reshow_flag |= RESHOW_ALL+CLEAR_NEEDED;
    reshow = TRUE; 

    /* reinitialize base variables */
    base_x = ie.ie_locx + gi_origin_x;
    base_y = ie.ie_locy + gi_origin_y;
   }
}

/*************************************************************************/
static drop_move_obj(x, y)
  int x, y;
{
/* given display coordinates of the current
 * cursor, "drops" the object currently
 * being carried at that location
 * note: assumes base_x and base_y contain the
 *       original relative coordinates that
 *       the object was picked up from
 */

  switch(mouse_mode)
  {
    case MOVE_DISPLAY:
           /* only do something if the mouse has actually moved */
           if (base_x!=x || base_y!=y)
           {
             /* if at least 10 pixels of change
              * save the last move displacement  
              * for future "jump moves" 
              */
             if (abs(base_x-x) > 9 || abs(base_y-y) > 9)
             {
               last_disp_move_x = base_x - x;
               last_disp_move_y = base_y - y;
             }

             /* officially change the display window parameters */
             gi_change_origin(base_x-x, base_y-y);

             /* update the display/off_display chains */
             gi_update_grobj_chain(TRUE, TRUE);

             /* set flags to clear screen and redisplay all objects */
             gi_reshow_flag |= RESHOW_ALL+CLEAR_NEEDED;
             reshow = TRUE;
            }
            break;
      case MOVE_GROBJ:
             /* if not in link mode, display object normally */ 

             if (gi_mode!=MODE_LINK)
               gi_display_grobj(grobj_ptr,PIX_SRC);

             else  /* link mode */
             {
               /* if not the current target, display unit in link mode */
               if (grobj_ptr!=gi_cur_target_ptr)
                 gi_display_link(grobj_ptr,PIX_SRC);

               else  
                 /* unit is current target, reverse image it */
                 gi_display_link(grobj_ptr,PIX_NOT(PIX_SRC));
             }
             break;
      case MOVE_TXOBJ:
             gi_display_txobj(txobj_ptr,PIX_SRC | PIX_DST);
             break;
      case MOVE_DROBJ:
             /* check if this is a bounding box */
             if (drobj_ptr->flag & BOUND_BOX)
             {
               /* move all MOVING "bound" objects to new positions */
               gi_move_bound(cur_x-base_x, cur_y-base_y, TRUE);
             }
             else
             {
               /* move single object */
               gi_display_drobj(drobj_ptr,PIX_SRC);
             }

             break;
    }
    
    /* if logging is enabled and an object has just been
       moved, log the appropriate "move" command
    */

    if (mouse_mode!=MOVE_DISPLAY && gi_log_on)
      gi_log_move(base_x, base_y, cur_x, cur_y);
      
    mouse_mode = NORMAL;
    win_setcursor(gi_gfx->gfx_windowfd, &oldcursor);
}

/*************************************************************************/
static save_text()
{
/* saves the current text parameters by making
 * a permanent text object on the text chain and
 * then logging (if necessary) the text command
 * that would recreate this object
 */
  struct txobj *ptr;

  ptr = gi_remember_txobj(cur_text_x,cur_text_y,cur_text_len,gi_cur_font);
  cur_text_len = 0;
  if (gi_log_on)
  {
    gi_log_text(ptr);
  }
}

/*************************************************************************/
static clear_blob()
{
/* clears out the old blob 
 * (using the old blob and old font values)
 * and sets blob_set FALSE
 */
  
   pw_writebackground(gi_gfx->gfx_pixwin,old_blob_x,old_blob_y-old_font_y+4,
	              old_font_x, old_font_y, PIX_CLR);
   blob_set = FALSE;
}


/*************************************************************************/
static write_blob(xpos, ypos)
  int xpos, ypos;
{
/* write the text "blob" at indicated cursor "hot spot"
 * (using current font) and updates the global
 * blob and font variables
 */

   /* update the fonts and position for blob and write it out */
   old_font_x = gi_cur_font->pf_defaultsize.x;
   old_font_y = gi_cur_font->pf_defaultsize.y;
   old_blob_x = xpos;
   old_blob_y = ypos;
   pw_writebackground(gi_gfx->gfx_pixwin,old_blob_x,old_blob_y-old_font_y+4,
	              old_font_x, old_font_y, PIX_SET);
   blob_set = TRUE;
}
	
/*************************************************************************/
static find_move_obj(x,y)
  int x, y;
{
/* finds what object is at (or near) the 
 * display coordinates given and
 * sets mouse_mode and a pointer to that
 * object 
 */
   /* save current cursor and set it to a grab icon */
   win_getcursor(gi_gfx->gfx_windowfd, &oldcursor);
   win_setcursor(gi_gfx->gfx_windowfd, grab);

   /* check if cursor is over a drawn object */
   if ((drobj_ptr=gi_find_drobj(x,y))!=NULL)
   {
     mouse_mode = MOVE_DROBJ;

     /* further check if this is a bounding box */
     if (drobj_ptr->flag & BOUND_BOX)
     {
       /* blank out and mark as MOVING all objects contained within it */
       gi_mark_bound(drobj_ptr, TRUE);
     }
   }

   /* check if cursor is over a text object */
   else if ((txobj_ptr=gi_find_txobj(x,y))!=NULL)
   {
     mouse_mode = MOVE_TXOBJ;
   }
   
   /* check if a unit object is at the cursor */
   else if ((grobj_ptr=gi_find_xy(x,y))!=NULL)
   {
     mouse_mode = MOVE_GROBJ;
   }

   /* otherwise assume the intent is to move entire display */
   else
   {
     mouse_mode = MOVE_DISPLAY;
     win_setcursor(gi_gfx->gfx_windowfd, graball);
   }
}

/*************************************************************************/
static cont_move()
{
/* check if an object is currently being moved
 * and if so, continue to have it track 
 * the cursor
 */
   if (mouse_mode!=NORMAL)
   {
      move_cur_object(cur_x, cur_y);
      old_move_x = cur_x;
      old_move_y = cur_y;
   }
}
      
/*************************************************************************/
static do_middle()
{
/* processes the UP or DOWN action of the 
 * middle (move) mouse button in all modes
 * (except possibly in Custom mode)
 */
  if (mouse_mode==NORMAL && !win_inputnegevent(&ie))
  {
    /* see what object is below 
     * note: find_obj sets mouse_mode and either   
     * grobj_ptr, drobj_ptr or txobj_ptr depending
     * on what is below 
     */
    find_move_obj(cur_x,cur_y);
    base_x = old_move_x = cur_x;
    base_y = old_move_y = cur_y;

    /* if moving a drawn object, set first_move flag to TRUE
       (so that we can correctly set the first erase operation */
    if (mouse_mode==MOVE_DROBJ)
      first_move = TRUE;
  }   
  else if (win_inputnegevent(&ie) && mouse_mode!=NORMAL)
  {
    drop_move_obj(cur_x, cur_y);
  }
}


/*************************************************************************/
static process_ASCII(this_char)
  char this_char;
{
/* processes an ASCII character
 * assumes blob_set and blob and font variables are current
 */
     switch (this_char)
     {
       case BACKSPACE:
       case DELETE:
            /* if the string has a positive length
             * backup and write a blank at new position
             */

            clear_blob();
	    write_blob(old_blob_x-old_font_x, old_blob_y);

            if (cur_text_len > 0)
            {  
              cur_text_len--;
            }
            break;

        case CR:
            /* if string has a positive length, save it
             * in a text object and link it onto the chain
             * and move the text blob to the next line
             */

	    clear_blob();
            if (cur_text_len > 0)
            {  
              save_text();
	      write_blob(cur_text_x-gi_origin_x, old_blob_y+old_font_y);
            }
            else
            {
	      write_blob(old_blob_x, old_blob_y+old_font_y);
            }
            break;
             
       default: /* (printable) character */
               
           if (cur_text_len==0)
           {
             /* save starting position of the string */
             /* using "display space" coordinates    */
             cur_text_x = old_blob_x + gi_origin_x;
             cur_text_y = old_blob_y + gi_origin_y;
           }
 
           /* save character in string space and increment length */
	   gi_save_char(this_char,cur_text_len);
           cur_text_len++;

           /* if string is at maximum length, 
              make a text object for it now and
              start a new object to contain future
              characters
           */

           if (cur_text_len >= MAX_TEXT_LENGTH)
             save_text();
             
          clear_blob();
          /* output character at current blob position */
          pw_char(gi_gfx->gfx_pixwin,
                  old_blob_x,
		  old_blob_y,
                  PIX_SRC,
                  gi_cur_font,
                  this_char);
 
           /* move the blob right one character */
	   write_blob(old_blob_x+old_font_x, old_blob_y);

           break;
      } /* end of switch */
}

/*************************************************************************/
static vertex_nearby(x,y)
  int x, y;
{
/* returns true if the display coordinates are
 * "near" (within 3 pixels) the last vertex set coordinates
 */

  if (abs(x-old_vertex_x) < 4 
  && abs(y-old_vertex_y) < 4)
     return(TRUE);
  else
    return(FALSE);
}


/*************************************************************************/
static draw_box(flag)
  int flag;
{
/* saves the box parameters from the global variables
 * on the drobj chain and then logs the command
 * to recreate that object (if logging enabled)
 * and writes is out to the display
 * note: flag indicates whether the box is a bounding box
 */
  
  struct drobj *box_ptr;

  box_ptr = gi_remember_drobj(old_vertex_x, old_vertex_y,
				 old_vertex_x, cur_y, flag);
  gi_save_vertex(box_ptr, cur_x, cur_y); 
  gi_save_vertex(box_ptr, cur_x, old_vertex_y);
  gi_save_vertex(box_ptr, old_vertex_x, old_vertex_y);
  gi_display_drobj(box_ptr,PIX_SRC);
  vertex_set = FALSE;

  if (gi_log_on) gi_log_draw(box_ptr);
}

/*************************************************************************/
static draw_lines()
{
/* finish up drawing the 
 * current line object
 * and log the command that
 * would recreate it if
 * logging is enabled
 */
    /* finish partially drawn object */
  if (cur_draw!=NULL)
  {
     gi_display_drobj(cur_draw,PIX_SRC);
     if (gi_log_on) gi_log_draw(cur_draw);
     cur_draw = NULL;
  }
  vertex_set = FALSE;
}

/*************************************************************************/
static build_cmd(numargs, args, ui, new_cmd)
  int numargs, ui;
  char **args, *new_cmd;
{
/* assumes that args contains pointers to the arg strings
 * and tries to substitute for the unit index and
 * coordinates and then builds the new command string
 * in the buffer new_cmd
 * and then returns the number of arguments still 
 * left in cmd_args (in case of multiple commands)
 * the substituted command is put in the static array "new_cmd"
 */

  char arg[MAX_CMD_LENGTH+1];
  static next_index=0;
  int i;

  /* blank out the destination buffer */
  new_cmd[0] = '\0';

  /* check each argument for substitution possibilities and
     and make the appropriate substitution */
  for (i=next_index; i<numargs; i++)
  {
    if (strcmp(args[i],UI_SUB_STRING)==0)
    {
      /* if ui is not valid, indicate an error */
      if (ui < 0)
      {
        gi_put_error("No valid substitution for $u");   
	return(NULL);
      }
      (void) sprintf(arg,"%1d ",ui);
    }

    /* check for "$x" and substitute x coordinate if necessary */
    else if (strcmp(args[i],X_SUB_STRING)==0)
      (void) sprintf(arg,"%1d ",cur_x);

    /* check for "$y" and substitute y coordinate if necessary */
    else if (strcmp(args[i],Y_SUB_STRING)==0)
      (void) sprintf(arg,"%1d ",cur_y);

    /* check for "--" making sure NOT to add it to the buffer */
    else if (strcmp(args[i],CONT_STRING)==0)
      (void) sprintf(arg," ");

    /* check for ";" indicating end of this command and exit early */
    else if (strcmp(args[i],TERM_STRING)==0)
    {
      next_index = i + 1;
      /* return the number of arguments left unprocessed in args array */
      return(numargs-i-1);
    }
    else
      (void) sprintf(arg,"%s ",args[i]);

    /* append the argument just found to the new command buffer */
    strcat(new_cmd,arg);
  }

  /* normal exit, reset next_index to start fresh on next call */
  next_index = 0;
  return(0);
}

/*************************************************************************/
static process_button(index)
  int index;
{
/* given the index of the panel item containing
 * the custom string for a particular button action,
 * this does the appropriate
 * substitution and executes the command
 * (if command is not null)
 */

  static int cmd_waiting=FALSE;
  static char command[MAX_CMD_LENGTH+1];
  static char new_cmd[MAX_CMD_LENGTH+1];

  char buf[MAX_CUST_TEXT_LEN+1];
  char *args[MAX_ARGS]; 
  char arg_buf[MAX_CMD_LENGTH+1];
  int args_left, num_args, ui;
  struct grobj *ptr;

  /* concatenate the two lines of the input panel prompt */
  strcpy(buf,(char *) panel_get_value(gi_citem[index]));
  strcat(buf,(char *) panel_get_value(gi_citem[index+1]));
  
  /* parse the panel string into args array and arg_buf */
  num_args = gi_parse_cmd(buf,args,arg_buf);

  /* if there is no command specified, then do main_mode processing */
  if (num_args <= 0 || strcmp(args[0],NULL_STRING)==0)
  {
    do_main_mode();
  }

  /* otherwise execute the command specified */
  else
  {
    /* find any grobj at this cursor location */
    if ((ptr=gi_find_xy(cur_x, cur_y))!=NULL)
      ui = ptr->u_index;
    else
      ui = -1;

    /* call build_cmd to process the next whole command
       in the cmd_args array, looping until no more
       arguments are left             
    */
    do
    {
      /* build new command substituting ui, and x and y location */
      args_left = build_cmd(num_args,args,ui,new_cmd);

      /* if there is no "continued" command already in the buffer
         initialize our command buffer to NULL
      */
      if (!cmd_waiting)
        command[0] = '\0';

      /* check that the maximum command length is not exceeded */
      if (strlen(command)+strlen(new_cmd) > MAX_CMD_LENGTH)
      {
        gi_put_error("Flushing command line : too long");
        cmd_waiting = FALSE;
      }

      /* check that new command has at least one character in it */
      if (strlen(new_cmd) > 0);
      {

        /* append the new command to the command buffer */
        strcat(command,new_cmd);
    
        /* if this command is "continued" delay execution */
        if (strcmp(args[num_args-args_left-1],CONT_STRING)==0)
        {
          cmd_waiting = TRUE;
          gi_put_message("Waiting for rest of command");
        }
        else /* execute it immediately */
        {
      
          /* execute this command */
          gi_command(command);
          reshow = TRUE;
    
          /* log command if logging enabled */
          if (gi_log_on) gi_log(command);
  
          /* write this command to the command panel */
          panel_set_value(gi_cmd_prev_item, command);
     
          /* reset cmd_waiting flag */
          cmd_waiting = FALSE;
        }
      }
    } while (args_left > 0);
  }
}
