/*-------------------------------------------------------------------------------------------
	
	Linda-C code by Denis Dancanet 
	parts of this file (C) 1989 Chorus Supercomputer Inc.
	the rest (C) 1988 University of Rochester
	8/2/1989  -- version 1.0
	
	SIMAIN.C
		
  -------------------------------------------------------------------------------------------
  
	This file contains code that will (supposedly) parallelize the simulator. It takes care
	of the following things:
				1. Setting up the NameTable and other initial data structures.
				2. Building the network on every transputer after 'call build'.
				3. Executing any command that changes the network structure on the transputers.
				4. Executing the 'Step' function in parallel.
				
  -------------------------------------------------------------------------------------------*/

#include "Stupido:rochester:include:uniproc.h"
#include "Stupido:rochester:include:lex.h"
#include "simlinda.h"							/* Linda-C version constants */

short BuiltNet = 0;

lmain(argc,argv)								/* the topmost function of the simulator */
int argc;
char **argv;
{
	int i;
	
	simulator_setup(argc,argv);
	
	parallel_start();							/* start the infinite while loops */
	out("command", L_INIT);						/* initialize stuff for all processes */
	for (i = 0; i < NUM_PROC; i++)
		in("finished_command");					/* synchronize */
	in("command", ? int *);						/* clean up */
	
	simulator_run();
	simulator_quit();
}

/* ************************************************************************************ */

parallel_start()
{
	int i;
	
	for (i = 0; i < NUM_PROC; i++)
		eval(parallel_loop());
}

/* ************************************************************************************ */

parallel_loop()
{
	int					command, no_steps, seed, argc, i, clock;
	linda char			*argv[20];				/* maximum size for command line */

	for (i=0; i<20; i++)						/* allocate space for commands */
		argv[i] = (char *) malloc(30);
	
	while(1)  {									/* BIG LOOP */
		rd("command", ? &command);
		switch (command)  {
		
		case L_INIT:
				StateNames = (char **) si_calloc(NoStates,sizeof(char *));
    			si_InitFuncNames();
	  			si_init_make();
				printf("parallel_loop() - done L_INIT.\n");
				out("finished_command");
				while (rdp("command", command)) {}	  /* do nothing until all processes done */
				break;
				
		case L_BUILD:
				rd("parameters", ? argv : &argc);		/* not used at present */
				if (!BuiltNet)
					build();
				printf("parallel_loop() - done L_BUILD.\n");
				out("finished_command");
				while (rdp("command", command)) {}
				break;
				
		case L_STEP_SYNC:
				in("steps", ? &no_steps, ? &seed);
				linda_sync(no_steps, seed);
				printf("parallel_loop() - done L_STEP_SYNC.\n"); fflush(stdout);
				out("finished_command");
				while (rdp("command", command)) {}
				break;
				
		case L_STEP_ASYNC:
				in("steps", ? &no_steps, ? &seed, ? &clock);
				linda_async(no_steps, seed, clock);
				printf("parallel_loop() - done L_STEP_ASYNC.\n");
				out("finished_command");
				while (rdp("command", command)) {}
				break;
				
		case L_STEP_FSYNC:
				in("steps", ? &no_steps, ? &seed, ? &clock, ? &ExecFract, ? &ExecLimit);
				linda_fsync(no_steps, seed, clock);
				printf("parallel_loop() - done L_STEP_FSYNC.\n");
				out("finished_command");
				while (rdp("command", command)) {}
				break;
				
		case L_WEIGHT:
				rd("parameters", ? argv : &argc);
				linda_weight(argc, argv);
				printf("parallel_loop() - done L_WEIGHT.\n");
				out("finished_command");
				while (rdp("command", command)) {}
				break;
				
		case L_LFUNC:
				rd("parameters", ? argv : &argc);
				linda_lfunc(argc, argv);
				printf("parallel_loop() - done L_LFUNC.\n");
				out("finished_command");
				while (rdp("command", command)) {}
				break;
				
		case L_SFUNC:
				rd("parameters", ? argv : &argc);
				linda_sfunc(argc, argv);
				printf("parallel_loop() - done L_SFUNC.\n");
				out("finished_command");
				while (rdp("command", command)) {}
				break;

		case L_UFUNC:
				rd("parameters", ? argv : &argc);
				linda_ufunc(argc, argv);
				printf("parallel_loop() - done L_UFUNC.\n");
				out("finished_command");
				while (rdp("command", command)) {}
				break;

		case L_STATE:
				rd("parameters", ? argv : &argc);
				linda_state(argc, argv);
				printf("parallel_loop() - done L_STATE.\n");
				out("finished_command");
				while (rdp("command", command)) {}
				break;

		case L_OUT:
				rd("parameters", ? argv : &argc);
				linda_out(argc, argv);
				printf("parallel_loop() - done L_OUT.\n");
				out("finished_command");
				while (rdp("command", command)) {}
				break;

		case L_POT:
				rd("parameters", ? argv : &argc);
				linda_pot(argc, argv);
				printf("parallel_loop() - done L_POT.\n");
				out("finished_command");
				while (rdp("command", command)) {}
				break;

		case L_RESTORE:
				linda_restore();
				printf("parallel_loop() - done L_RESTORE.\n"); fflush(stdout);
				out("finished_command");
				while (rdp("command", command)) {}
				break;
		}
	}
}

/* ************************************************************************************ */

skip_blanks(no_blanks, buffer)
int no_blanks;
char buffer[];
{
	char c;
	int i=0, j;
	
	for (j = 0; j < no_blanks; j++)  {
		while ((c = buffer[i]) != ' ')
			i++;
		i++;
	}
	return(i);
}

/* ************************************************************************************ */

Step(no_steps)
int no_steps;
{
    register Link 	*ip;
    register Site 	*sp;
	register Unit	*up;
    register int 	ucount, i, j;
    int 			step_count, s, no_sites, no_inputs, no_matched, buff_index, uindex;
	unsigned int 	outflags, outsets;
	pot_type		outpotential;
	short			outstate;
	data_type		outdata;
    static linda char	buff[ST_SIZE];						/* to hold each SuperTuple */
	
    switch (SyncFlag)
      {
      case SYNC:							/* synchronous simulation */
	  	out("command", L_STEP_SYNC);
		for (i = 0; i < NUM_PROC; i++)					/* feed the tasks */
			out("steps", no_steps, i);
			
        for (step_count = 0; step_count < no_steps; step_count++)  {
            Clock += 1;
            
			out("outputs_array", Outputs : NoUnits);
			
			for (ucount = 0; ucount < NoUnits; ucount++)
				in("result", ucount, ? &Outputs[ucount]);

			if (step_count == no_steps-1)	{			/* last step: update Mac copy of net */
				out("transfer_token");					/* enable processes to start sending data */
				
				for (j = 0; j < NoUnits; j++)  {
					in("final", ? &uindex, ? &outflags, ? &outpotential, ? &outstate, 
										? &outdata, ? &outsets);
					up = UnitList + uindex;
					up->flags 		= outflags;
					up->potential	= outpotential;
					up->state 		= outstate;
					up->data 		= outdata;
					up->sets 		= outsets;
					in("done_sync", uindex, ? buff);

					no_matched = sscanf(buff, "%d ", &no_sites);
					buff_index = 0;
					
					for (s = 0, sp = up->sites; s < no_sites && sp != NULL; s++, sp = sp->next)  {
						buff_index += skip_blanks(no_matched, buff+buff_index);
						no_matched = sscanf(buff+buff_index, "%f %d ", &(sp->value), &no_inputs);
						
						for (i = 0, ip = sp->inputs; i < no_inputs && ip != NULL; i++, ip = ip->next)  {
							buff_index += skip_blanks(no_matched, buff+buff_index);
							no_matched = sscanf(buff+buff_index, "%f ", &(ip->weight));
						}
					}
				}
				in("transfer_token");						/* all done transmitting now */
			}

			in("outputs_array", ? float * : int *);
			
			if(Echo && ((step_count + 1) % EchoStep) == 0) 
			  printf("finished %d out of %d steps\n",step_count + 1,no_steps);

			if(Show && ((step_count + 1) % ShowStep) == 0)   {
				ShowUnits();
				printf("\n");
				if(Pause && step_count + 1 != no_steps)
				  if (UserWait("PAUSE (<any key> - continue; q - quit)") == 'q')
					break;
			}
		}
		for (i = 0; i < NUM_PROC; i++)
			in("finished_command");
		in("command", L_STEP_SYNC);

		break;

      case ASYNC:							/* asynchronous simulation */
		out("command", L_STEP_ASYNC);
		for (i = 0; i < NUM_PROC; i++)				/* seed tuple space; only 0 does anything */
			out("steps", no_steps, i, Clock);
			
		linda_spread(1);							/* send the latest copy of the net */
		linda_get();								/* get the answer */
		
		for (i = 0; i < NUM_PROC; i++)				/* synchronize processes */
			in("finished_command");
		in("command", ? int *);
		
		Clock += no_steps;
        break;

      case FAIRASYNC:						/* fair asynchronous */
		out("command", L_STEP_FSYNC);
		for (i = 0; i < NUM_PROC; i++)				/* seed tuple space; only 0 does anything */
			out("steps", no_steps, i, Clock, ExecFract, ExecLimit);
			
		linda_spread(1);							/* send the latest copy of the net */
		linda_get();								/* get the answer */

		for (i = 0; i < NUM_PROC; i++)
			in("finished_command");
		in("command", ? int *);

		Clock += no_steps;
        break;

      default:
        fprintf(stderr, "Internal error: execution mode not SYNC, ASYNC or FSYNC\n");
      }

} /* Step */

/* ************************************************************************************ */

linda_send(no_copies)								/* works with linda_restore() */
int no_copies;
{
	register int	u, s, i;
    int no_sites,no_inputs;
	register Unit *up;
    register Link *ip;
    register Site *sp;
    static char buff[ST_SIZE];						/* to hold each SuperTuple */
	char temp[30];									/* for temporary results */

	for (u = 0, up = UnitList; u < NoUnits; u++, up++)  {
		out("unit", u, up->flags, up->potential, up->output, up->state, up->data, up->sets);
						
		buff[0] = '\0';						/* 'empty' buffer */
		no_sites = up->no_site;				/* number of sites as header */
		sprintf(temp, "%d ", no_sites);
		strcat(buff, temp);
		
		for (s = 0, sp = up->sites; s < no_sites && sp != NULL; s++, sp = sp->next)  {
			no_inputs = sp->no_inputs;
			sprintf(temp, "%f %d ", sp->value, no_inputs);
			strcat(buff, temp);
	
			for (i = 0, ip = sp->inputs; i < no_inputs && ip != NULL; i++, ip = ip->next) {
				sprintf(temp, "%f ", ip->weight);
				strcat(buff, temp);
			}
		}
		out("restore", u, (linda char *) buff);		/* send out the SuperTuple */
		
		for (i = 0; i < no_copies; i++)				/* synchronize proceses */
			in("got_tuples");
		in("unit", ? int *, ? unsigned int *, ? pot_type *, ? Output *, ? short *,
					? data_type *, ? unsigned int *);
		in("restore", ? int *, ? linda char *);
	}
}

/* ************************************************************************************ */

linda_spread(no_copies)								/* works with linda_get() */
int no_copies;
{
	register int	u, s, i;
    int no_sites,no_inputs;
	register Unit *up;
    register Link *ip;
    register Site *sp;
    static char buff[ST_SIZE];						/* to hold each SuperTuple */
	char temp[30];									/* for temporary results */

	for (u = 0, up = UnitList; u < NoUnits; u++, up++)  {
		out("unit", u, up->flags, up->potential, up->output, up->state, up->data, up->sets);
						
		buff[0] = '\0';						/* 'empty' buffer */
		no_sites = up->no_site;				/* number of sites as header */
		sprintf(temp, "%d ", no_sites);
		strcat(buff, temp);
		
		for (s = 0, sp = up->sites; s < no_sites && sp != NULL; s++, sp = sp->next)  {
			no_inputs = sp->no_inputs;
			sprintf(temp, "%f %d ", sp->value, no_inputs);
			strcat(buff, temp);
	
			for (i = 0, ip = sp->inputs; i < no_inputs && ip != NULL; i++, ip = ip->next) {
				sprintf(temp, "%f ", ip->weight);
				strcat(buff, temp);
			}
		}
		out("restore", u, (linda char *) buff);		/* send out the SuperTuple */
		
		for (i = 0; i < no_copies; i++)				/* synchronize proceses */
			in("got_tuples");
	}
}

/* ************************************************************************************ */

linda_restore()
{
	register int	u, s, i;
    int no_sites,no_inputs;
	register Unit *up;
    register Link *ip;
    register Site *sp;
    static linda char buff[ST_SIZE];					/* to hold each SuperTuple */
	int no_matched, buff_index;

	for (u = 0, up = UnitList; u < NoUnits; u++, up++)  {
		rd("unit", u, ? &(up->flags), ? &(up->potential), ? &(up->output), ? &(up->state),
					? &(up->data), ? &(up->sets));
		Outputs[u] = up->output;
		
		rd("restore", u, ? &buff[0]);					/* the SuperTuple !!!!! */
		out("got_tuples");
		
		no_matched = sscanf(buff, "%d ", &no_sites);	/* read the number of sites */
		buff_index = 0;
		
		for (s = 0, sp = up->sites; s < no_sites && sp != NULL; s++, sp = sp->next)  {
			buff_index += skip_blanks(no_matched, buff+buff_index);
			no_matched = sscanf(buff+buff_index, "%f %d ", &(sp->value), &no_inputs);
			
			for (i = 0, ip = sp->inputs; i < no_inputs && ip != NULL; i++, ip = ip->next)  {
				buff_index += skip_blanks(no_matched, buff+buff_index);
				no_matched = sscanf(buff+buff_index, "%f ", &(ip->weight));
			}
		}
	}
}

/* ************************************************************************************ */

linda_get()									/* same as linda_restore with in instead of rd */
{
	register int	u, s, i;
    int no_sites,no_inputs;
	register Unit *up;
    register Link *ip;
    register Site *sp;
    static linda char buff[ST_SIZE];					/* to hold each SuperTuple */
	int no_matched, buff_index;

	for (u = 0, up = UnitList; u < NoUnits; u++, up++)  {
		in("unit", u, ? &(up->flags), ? &(up->potential), ? &(up->output), ? &(up->state),
					? &(up->data), ? &(up->sets));
		Outputs[u] = up->output;
		
		in("restore", u, ? &buff[0]);					/* the SuperTuple !!!!! */
		out("got_tuples");
		
		no_matched = sscanf(buff, "%d ", &no_sites);	/* read the number of sites */
		buff_index = 0;
		
		for (s = 0, sp = up->sites; s < no_sites && sp != NULL; s++, sp = sp->next)  {
			buff_index += skip_blanks(no_matched, buff+buff_index);
			no_matched = sscanf(buff+buff_index, "%f %d ", &(sp->value), &no_inputs);
			
			for (i = 0, ip = sp->inputs; i < no_inputs && ip != NULL; i++, ip = ip->next)  {
				buff_index += skip_blanks(no_matched, buff+buff_index);
				no_matched = sscanf(buff+buff_index, "%f ", &(ip->weight));
			}
		}
	}
}

/*----------------------------------------------------------------------------

	Pieces of "commoncm.c" and "conunicm.c" follow.
	
----------------------------------------------------------------------------*/

Cmd_pot(argc,argv)
     int argc;
     char ** argv;

{
  register int u, i;
  register pot_type val;
  register Unit * up;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 3) goto synerror;

	out("command", L_POT);								/* launch! */
	out("parameters", (linda char **) argv : argc);

    Curarg = 1;
    while (argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case INT :
            val = Yyint;
            break;
          case FLOAT :
            val = Yyfloat;
            break;
          default:
            LOGfprintf(stdout,"Illegal potential: %s\n",argv[Curarg-1]);
            LOGfprintf(stdout,"Rest of line ignored\n\n");
            EAT;
            return 0;
          }
		  
        FOR_UNITS_P(u,up)
          up->potential = val;
      }
	  
	for (i=0; i<NUM_PROC; i++)							/* synchronize processes */
		in("finished_command");
	in("command", ? int *);								/* clean up tuple space */
	in("parameters", ? linda char ** : int *);
		
    return 0;

 helpinfo:
  Format = TRUE;                                          /* print detailed help */
    LOGfprintf(Dispf,"The pot(ential) command is used to set the potential of one or more units.  It expects one or more unit-identifier/potential-value pairs.  The unit identifiers may specified in any of the usual ways (try `help UnitId' for information).\n\n");

  synerror:
    Format = FALSE;                                       /* print syntax */
    LOGfprintf(Dispf,"\nUsage: pot <UnitID> <value> [<UnitID> <value>]*\n\n");
    return 0;
BFLYTHROW("Cmd_pot",0)
} /* pot */

Cmd_out(argc,argv)
     int argc;
     char ** argv;

{
  register int u, i;
  register Output val;
  register Unit * up;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 3) goto synerror;
	
	out("command", L_OUT);				/* out stuff into the black emptiness of tuple space */
	out("parameters", (linda char **) argv : argc);

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case INT :
            val = Yyint;
            break;
          case FLOAT :
            val = Yyfloat;
            break;
          default:
            LOGfprintf(stdout,"Illegal output: %s\n",argv[Curarg-1]);
            LOGfprintf(stdout,"Rest of line ignored\n\n");
            EAT;
            return 0;
          }
		  
        FOR_UNITS_P(u,up)
            up->output = Outputs[u] = val;
      }
	  
	for (i=0; i<NUM_PROC; i++)							/* synchronize processes */
		in("finished_command");
	in("command", ? int *);								/* clean up tuple space */
	in("parameters", ? linda char ** : int *);

    return 0;

  helpinfo:
  Format = TRUE;                                          /* print detailed help */
    LOGfprintf(Dispf,"The out(put) command is used to set the output of one or more units.  It expects one or more unit-identifier/output-value pairs.  The unit identifiers may be specified in any of the usual ways (try `help UnitId' for information).\n\n");


  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: out <UnitID> <value> [<UnitID> <value>]*\n\n");
    return 0;
BFLYTHROW("Cmd_out",0)
} /* out */

/***** state *****/

Cmd_state(argc,argv)
     int argc;
     char ** argv;

{
    register int u, i;
    register int val;
    register Unit * up;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 3) goto synerror;
	
	out("command", L_STATE);					/* out stuff in tuple space */
	out("parameters", (linda char **) argv : argc);

    Curarg=1;									/* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case INT :
            val = Yyint;
            break;
          default:      
            if((val = NameToState(Yyarg)) == -1)
              {
                LOGfprintf(stdout,"State not known: %s\n",argv[Curarg-1]);
                LOGfprintf(stdout,"Rest of line ignored\n\n");
                EAT;
                return 0;
              }
          }

        FOR_UNITS_P(u,up)
          up->state = val;
      }
	  
	for (i=0; i<NUM_PROC; i++)							/* synchronize processes */
		in("finished_command");
	in("command", ? int *);								/* clean up tuple space */
	in("parameters", ? linda char ** : int *);

    return 0;

  helpinfo:
  Format = TRUE;                                          /* print detailed help */
    LOGfprintf(Dispf,"The state command is used to set the state of one or more units.  It expects one or more unit-identifier/state-value pairs.  The unit identifiers may be specified in any of the usual ways.  The state values may be a known state name or an integer.\n\n");

    
  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: state <UnitID> <value> [<UnitID> <value>]*\n\n");
    return 0;
BFLYTHROW("Cmd_state",0)
} /* state */

/***** function *****/

Cmd_ufunc(argc,argv)
     int argc;
     char ** argv;

{
    register int u;
    register func_ptr func;
    register Unit * up;
    int flag, i;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 3) goto synerror;
	
	out("command", L_UFUNC);					/* put stuff in tuple space */
	out("parameters", (linda char **) argv : argc);

    Curarg=1;                             		/* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case OFF:
            flag = -1;
            break;
          case ON:
            flag = 1;
            break;
          default:
            flag = 0;
            if((func = NameToFunc(Yyarg)) == NULL)
              {
                LOGfprintf(stdout,"function not known: %s\n",argv[Curarg-1]);
                LOGfprintf(stdout,"Rest of line ignored\n\n");
                EAT;
                return 0;
              }
          }

		if (flag == 0)
          {
            FOR_UNITS_P(u,up)
              up->unit_f = func; /* reset function */
          }
        else
          if (flag > 0)         /* turn unit on */
            FOR_UNITS_P(u,up)
              UnsetFlagP(up,NO_UNIT_FUNC_FLAG);
          else
            FOR_UNITS_P(u,up)   /* turn unit off */
              SetFlagP(up,NO_UNIT_FUNC_FLAG);
      }
	  
	for (i=0; i<NUM_PROC; i++)							/* synchronize processes */
		in("finished_command");
	in("command", ? int *);								/* clean up tuple space */
	in("parameters", ? linda char ** : int *);
		
    return 0;

  helpinfo:
  Format = TRUE;                /* print detailed help */
    LOGfprintf(Dispf,"The ufunc command is used to set the unit function of one or more units, or alternatively to turn the unit function on or off for one or more units.  It expects one or more unit-identifier/function-name pairs.  The unit identifiers may be specified in any of the usual ways.  The functions may be any global user function or a library function, or one of the keywords `on' or `off'.  If one of the keywords is given, the effect is to set the NO_UNIT_FUNC_FLAG on or off.\n\n");

    
  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: ufunc <UnitID> <function> [<UnitID> <function>]*\n\n");
    return 0;
BFLYTHROW("Cmd_ufunc",0)
} /* function */

Cmd_sfunc(argc,argv)
     int argc;
     char ** argv;

{
    register int u;
    register func_ptr func;
    register Unit * up;
    char * sname;
    Site * sp;
    int flag, i;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 4) goto synerror;
	
	out("command", L_SFUNC);					/* put stuff in tuple space */
	out("parameters", (linda char **) argv : argc);

    Curarg=1;                             		/* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        Lex(argv[Curarg]);      				/* process it */
        sname = argv[Curarg-1];
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case OFF:
            flag = -1;
            break;
          case ON:
            flag = 1;
            break;
          default:
            flag = 0;
            if ((func = NameToFunc(Yyarg)) == NULL)
               {
                 LOGfprintf(stdout,"function not known: %s\n",argv[Curarg-1]);
                 LOGfprintf(stdout,"Rest of line ignored\n\n");
                 EAT;
                 return 0;
               }
          }

        if (!(strcmp(sname,"all")))
          {
            FOR_UNITS_P(u,up)
              for (sp = up->sites;
                   sp != NULL;
                   sp = sp->next)
                if (flag == 0)
                  sp->site_f = func;
                else
                  if (flag > 0)
                    UnsetFlagP(up,NO_SITE_FUNC_FLAG);
                  else
                    SetFlagP(up,NO_SITE_FUNC_FLAG);
          }
        else
          if (flag == 0)
            {
              FOR_UNITS_P(u,up)
                {
                  for (sp = up->sites;
                       sp != NULL && strcmp(sp->name,sname);
                       sp = sp->next);
                  if (sp != NULL)
                    sp->site_f = func;
                  else
                    LOGfprintf(stderr,
                               "Can't find site %s on unit %d\n",sname,u);
                }
            }
          else
            LOGfprintf(stderr,"Cannot turn single site function on or off\n");
      }
	  
	for (i=0; i<NUM_PROC; i++)							/* synchronize processes */
		in("finished_command");
	in("command", ? int *);								/* clean up tuple space */
	in("parameters", ? linda char ** : int *);
		
    return 0;

  helpinfo:
  Format = TRUE;                                          /* print detailed help */
    LOGfprintf(Dispf,"The sfunc command is used to set the site function of one or all sites on one or more units, or alternatively to turn on or off all the site functions on one or more units  It expects one or more unit-identifier/site-name/function-name triples.  The unit identifiers may be specified in any of the usual ways.\n\n");
    LOGfprintf(Dispf,"The site name may be the name of any site on the unit, or `all' meaning all sites.  The functions may be any global user function or a library function, or the keywords `on' or `off'.  If one of the keywords is given, the sitename must be `all'.  The effect then is to set the NO_SITE_FUNC_FLAG on or off.\n\n");

    
  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: sfunc <UnitID> <site> <func|on|off> [<UnitID> <site> <func|on|off>]*\n\n");
    return 0;
BFLYTHROW("Cmd_sfunc",0)
} /* function */

Cmd_lfunc(argc,argv)
     int argc;
     char ** argv;

{
    register int u;
    register func_ptr func;
    register Unit * up;
    char * sname;
    Site * sp;
    register Link * lp;
    int flag,ul,uh,us,usind,i;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
    if (argc < 4) goto synerror;
	
	out("command", L_LFUNC);				/* put command and parameters in tuple space */
	out("parameters", (linda char **) argv : argc);

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        ul = Ulow; uh = Uhigh; us = Uset; usind = Usetind;
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        Lex(argv[Curarg]);      /* process it */
        sname = argv[Curarg-1];
        if (argc <= Curarg) goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case OFF:
            flag = -1;
            break;
          case ON:
            flag = 1;
            break;
          default:
            flag = 0;
            if ((func = NameToFunc(Yyarg)) == NULL)
               {
                 LOGfprintf(stdout,"function not known: %s\n",argv[Curarg-1]);
                 LOGfprintf(stdout,"Rest of line ignored\n\n");
                 EAT;
                 return 0;
               }
          }

        if (!(strcmp(sname,"all")))
          {
            if (flag == 0)      /* reset to new link function */
              FOR_UNITS_P(u,up) /* for all destination units */
                for (sp = up->sites; /* for all sites */
                     sp != NULL;
                     sp = sp->next)
                  for (lp = sp->inputs; lp != NULL; lp = lp->next)
                    {
                      if (lp->from_unit >= ul && lp->from_unit <= uh &&
                          (!us || (UnitList[lp->from_unit].sets >> usind) & 1))
                        lp->link_f = func;
                    }
            else
              if (ul == 0 && uh == NoUnits-1 && !us) /* source all units */
                if (flag > 0)
                  FOR_UNITS_P(u,up)     /* for all destination units */
                    UnsetFlagP(up,NO_LINK_FUNC_FLAG);
                else
                  FOR_UNITS_P(u,up)     /* for all destination units */
                    SetFlagP(up,NO_LINK_FUNC_FLAG);
              else
                LOGfprintf(stderr,"Can only turn linkfunction on or off for all links to a unit\n");
          }
        else                    /* a particular site */
          if (flag == 0)        /* so should be function change */
            {
              FOR_UNITS_P(u,up)
                {               /* for destination units */
                  for (sp = up->sites; /* find correct site */
                       sp != NULL && strcmp(sp->name,sname);
                       sp = sp->next);
                  if (sp != NULL) /* found ok */
                    for (lp = sp->inputs; lp != NULL; lp = lp->next)
                      {         /* for all links */
                        if (lp->from_unit >= ul && lp->from_unit <= uh &&
                          (!us || (UnitList[lp->from_unit].sets >> usind) & 1))
                         lp->link_f = func; /* change function */
                      }
                  else          /* can't find site */
                    LOGfprintf(stderr,
                               "Can't find site %s on unit %d\n",sname,u);
                }
            }
          else
            LOGfprintf(stderr,"Can only turn linkfunction on or off for all links to a unit\n");
      }
	  
	for (i=0; i<NUM_PROC; i++)							/* synchronize all processes */
		in("finished_command");
	in("command", ? int *);								/* clean up tuple space */
	in("parameters", ? linda char ** : int *);
	
    return 0;

  helpinfo:
  Format = TRUE;                /* print detailed help */
    LOGfprintf(Dispf,"The lfunc command is used to set the link function of links from one or more units arriving at one or all sites on one or more units, or alternatively to turn on or off all the link functions on one or more units.  It expects one or more unit-identifier/unit-identifier/site-name/function-name quadruples.  The unit identifiers may be specified in any of the usual ways (type `help UnitId' for information).\n\n");
    LOGfprintf(Dispf,"The first unit-identifier specifies the units at which the affected links originate.  The second unit-identifier specifies the units at which the affected links arrive.  The site name is that of a site on the destination unit(s), or `all' meaning all sites.\n\n");
    LOGfprintf(Dispf,"The function may be any global user function or a library function, or the keywords `on' or `off'.  If one of the keywords is given, the sitename must be `all' and the source units (the first unit-identifier) must be all the units.  The effect then is to set the NO_LINK_FUNC_FLAG on or off.\n\n");
    
  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: lfunc [<SourceUnitId> <DestinationUnitID> <site> <func|on|off>]+\n\n");
    return 0;
BFLYTHROW("Cmd_lfunc",0)
} /* function */

Cmd_weight(argc,argv)
     int argc;
     char ** argv;

{
    int u, i;
    weight_type val, pert;
    char * sitename;
    short ul,uh,us,usind,typ;

BFLYCATCH
    if (argc == 2 && Lex(argv[1]) == HELP)
      goto helpinfo;
	  
    out("command", L_WEIGHT);							/* broadcast command & parameters */
    out("parameters", (linda char **) argv : argc);
	  
    if (argc == 2 && Lex(argv[1]) == RANDOM)
      {
        RandomiseWeights(500,100);
        return(0);
      }
    if (argc < 5) goto synerror;

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        ul = Ulow; uh = Uhigh; us = Uset; usind = Usetind;
        if(!GetUnits(argc,argv)) return 0;
        if (argc <= Curarg) goto synerror;
        sitename = argv[Curarg];
        u = Lex(argv[Curarg]);  /* read ident */
        if (u == HELP || u == INT || u == FLOAT || u == PLUS ||
            u == MINUS || u == CBRACK || u == OBRACK)
          goto synerror;
        switch(Lex(argv[Curarg]))
          {
          case INT :
            typ = FALSE;
            val = Yyint;
            break;
          case FLOAT :
            typ = FALSE;
            val = Yyfloat;
            break;
          case RANDOM:
            typ = TRUE;
            if (argc <= Curarg)
              val = 500;
            else
              switch(Lex(argv[Curarg]))
                {
                case INT:
                  val = Yyint;
                  break;
                case FLOAT:
                  val = Yyfloat;
                  break;
                default:
                  goto synerror;
                }
            if (argc <= Curarg)
              pert = 200;
            else
              switch(Lex(argv[Curarg]))
                {
                case INT:
                  pert = Yyint;
                  break;
                case FLOAT:
                  pert = Yyfloat;
                  break;
                default:
                  goto synerror;
                }
            break;
          default:
            LOGfprintf(stdout,"Unknown weight value: %s\n",argv[--Curarg]);
            EAT;
            return 0;
          }

        SetWeight(ul,uh,us,usind,Ulow,Uhigh,Uset,Usetind,
                  sitename,typ,val,pert);
      }
	
	for (i=0; i<NUM_PROC; i++)							/* synchronize all processes */
		in("finished_command");
	in("command", ? int *);								/* clean up tuple space */
	in("parameters", ? linda char ** : int *);

    return 0;

  helpinfo:
  Format = TRUE;                                          /* print detailed help */
    LOGfprintf(Dispf,"The weight command sets the value of a weight on a link.");
	
  synerror:
        Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: weight [<From-UnitID> <To-UnitID> <To-site> <value| random [<mean> <deviation>]>]+\n\n");
    return 0;
BFLYTHROW("Cmd_weight",0)
} /* weight */

/***** call *****/

Cmd_call(argc,argv)             /* ????????????????????? */
int argc;
char **argv;
{
	func_ptr func;
	int i;

	if ((argc == 2) && (Lex(argv[1]) == HELP))
	goto helpinfo;
	if(argc < 2)
	goto helpinfo;
	else
	{
	  func = NameToFunc(argv[1]);
	  if(func == NULL)
		{
		  LOGfprintf(stderr,"cannot find function %s\n",argv[1]);
		  EAT;
		}
	  else  {
		if (!strcmp(argv[1], "build"))  {			/* build the net on the Chorus */
			out("command", L_BUILD);
			out("parameters", (linda char **) ++argv : (argc-1));
			func(argc-1,++argv);
			for (i = 0; i < NUM_PROC; i++)			/* synchronize */
				in("finished_command");
			in("command", ? int *);
			in("parameters", ? linda char ** : int *);
		}
		else
			func(argc-1,++argv);
	  }
	}
	return 0;
	
	helpinfo:
	Format = TRUE;
	LOGfprintf(Dispf,"The call command is used to call any function in your code (so long as it was not declared static).  The syntax is given below.  The command simply calls the function with the arguments as given, no checking is done to see that you have given the right number and type of arguments.  Therefore missing arguments will get random values.\n");
	
	synerror:
		Format = FALSE;
	LOGfprintf(Dispf,"\nUsage: call <function name> [arg1 arg2 ...]\n");
	
	return 0;
}

/***** restore *****/

Cmd_restore(argc,argv)
     int argc;
     char ** argv;

{
    FILE *fp;
	int	   i;

    if ((argc == 2) && (Lex(argv[1]) == HELP))
      goto helpinfo;
    if (argc != 2)
      goto synerror;
    if ((fp = fopen(argv[1],"r")) == NULL)  {
      LOGfprintf(stderr,"cannot open %s\n\n",argv[1]);
	  return(0);
	}
    else
      {
        RestoreNetwork(fp);
        close(fp);
      }
	
	printf("\nRestored net OK on the Mac. Doing it on the Chorus now.\n"); fflush(stdout);
	out("command", L_RESTORE);					/* if it got here => there were no problems */
	linda_send(NUM_PROC);						/* out the restored net piece by piece */
	for (i = 0; i < NUM_PROC; i++)				/* synchronize */
		in("finished_command"); 
		
	in("command", L_RESTORE);					/* clean up command */
	
    return(0);

  helpinfo:
  Format = TRUE;
    LOGfprintf(Dispf,"The restore command restores the state of a network from a file made with the checkpoint command.  This command simply restores the state of the network (i.e. the weights, potentials, etc), it does NOT rebuild the network.  That means the network must already have been built.\n");

  synerror:
    Format = FALSE;
    LOGfprintf(Dispf,"\nUsage: restore <file name>\n\n");

    return(0);
} /* restore */

/*----------------------------------------------------------------------------

	The sync, async, fsync, and psync step functions.
	
----------------------------------------------------------------------------*/

linda_sync(no_steps, seed)
int no_steps, seed;
{
    register Link *ip;
    register Site *sp;
    register Unit *up;
    register int which;
    register int ucount;
    int step_count, s, i, no_sites, no_inputs, dummy;
    static char buff[ST_SIZE];						/* to hold each SuperTuple */
	char temp[30];									/* for temporary results */

	for (step_count = 0; step_count < no_steps; step_count++)  {
		rd("outputs_array", ? Outputs : &dummy);
	
											/* always simulate unit #0 even if seed is not 0 */
		if(seed && !TestFlagF((which = (up = UnitList)->flags), NO_UNIT_FUNC_FLAG))  {
			if(!TestFlagF(which,NO_SITE_FUNC_FLAG))
				for(sp = up->sites;sp != NULL;sp = sp->next)  {
					sp->site_f(up,sp);
					if(!TestFlagF(which,NO_LINK_FUNC_FLAG))
						for(ip = sp->inputs; ip != NULL; ip = ip->next)
							ip->link_f(up,sp,ip);
				}
			up->unit_f(up);
		}
		
		for (ucount = seed; ucount < NoUnits; ucount += NUM_PROC)  {
		
			if(!TestFlagF((which = (up = UnitList+ucount)->flags), NO_UNIT_FUNC_FLAG))  {
				if(!TestFlagF(which,NO_SITE_FUNC_FLAG))
					for(sp = up->sites;sp != NULL;sp = sp->next)  {
						sp->site_f(up,sp);
						if(!TestFlagF(which,NO_LINK_FUNC_FLAG))
							for(ip = sp->inputs; ip != NULL; ip = ip->next)
								ip->link_f(up,sp,ip);
					}
				up->unit_f(up);
				out("result", ucount, up->output);
			}
			else
				out("result", ucount, up->output);
		printf("seed = %d, done unit %d\n", seed, ucount);
		}
				
		if (step_count == no_steps-1)							/* last step */
			for (ucount = seed; ucount < NoUnits; ucount += NUM_PROC)  {
				up = UnitList + ucount;
				in("transfer_token");						/* data transfer enable token */
				out("final", ucount, up->flags, up->potential, up->state, up->data, up->sets);
				buff[0] = '\0';								/* 'empty' buffer */
				no_sites = up->no_site;
				sprintf(temp, "%d ", no_sites);
				strcat(buff, temp);
				
				for (s = 0, sp = up->sites; s < no_sites && sp != NULL; s++, sp = sp->next)  {
					no_inputs = sp->no_inputs;
					sprintf(temp, "%f %d ", sp->value, no_inputs);
					strcat(buff, temp);
					
					for (i = 0, ip = sp->inputs; i < no_inputs && ip != NULL; i++, ip = ip->next) {
						sprintf(temp, "%f ", ip->weight);
						strcat(buff, temp);
					}
				}
				out("done_sync", ucount, (linda char *) buff);		/* the SuperTuple */
				out("transfer_token");								/* relinquish control */
			}
	}
}


linda_async(no_steps, seed, Clock)
int no_steps, seed, Clock;
{
    register Link * ip;
    register Site * sp;
    register Unit * up;
    register int which;
    register int ucount;
    int step_count, most, done;

	if (seed != 0) return;							/* only process #0 does this */
	linda_get();									/* get latest copy of the network */
	   
	for (step_count = 0; step_count < no_steps; step_count++)
	  {
		Clock += 1;
		/* do 90% randomly */
		most = (int) (0.8 * NoUnits);
		for(done = 0; done < most;done++)
		  {
			which = NoUnits;						/* get in register */
			while (TestFlagP((up = UnitList+(ucount = rand()%which)),
							 STEP_SIM_FLAG));		/* get unit pointer in reg*/
			if(!TestFlagF((which = up->flags),NO_UNIT_FUNC_FLAG))
			  {										/* get unit flags in reg */
				if(!TestFlagF(which,NO_SITE_FUNC_FLAG))
				  for(sp = up->sites;sp != NULL;sp = sp->next)
					{
					  sp->site_f(up,sp);
					  if(!TestFlagF(which,NO_LINK_FUNC_FLAG))
						for(ip = sp->inputs; ip != NULL; ip = ip->next)
						  ip->link_f(up,sp,ip);
					}
				up->unit_f(up);
	
				Outputs[ucount] = up->output;
			  }
			SetFlagP(up,STEP_SIM_FLAG);
		  }
		/* add a search to last 10% to lower search time */
		most = NoUnits - most;
		for(done = 0;done < most;done++)
		  {
			which = NoUnits;                  /* get in register */
			for(up = UnitList+(ucount = rand()%which);
				TestFlagP(up,STEP_SIM_FLAG);
				up = UnitList+(ucount = ++ucount%which));
			if(!TestFlagF((which = up->flags),NO_UNIT_FUNC_FLAG))
			  {
				if(!TestFlagF(which,NO_SITE_FUNC_FLAG))
				  for(sp = up->sites;sp != NULL;sp = sp->next)
					{
					  sp->site_f(up,sp);
					  if(!TestFlagF(which,NO_LINK_FUNC_FLAG))
						for(ip = sp->inputs; ip != NULL; ip = ip->next)
						  ip->link_f(up,sp,ip);
					}
				up->unit_f(up);
	
				Outputs[ucount] = up->output;
			  }
			SetFlagP(up,STEP_SIM_FLAG);
		  }
	
		for ( which = 0, up = UnitList, ucount = NoUnits;
			 which < ucount;
			 which++,up++)
		  UnsetFlagP(up,STEP_SIM_FLAG);					/*  unset for next step */
	}
	
	linda_spread(1);									/* send the result back to Mac */
}


linda_fsync(no_steps, seed, Clock)
int no_steps, seed, Clock;
{
    register Link * ip;
    register Site * sp;
    register Unit * up;
    register int which;
    register int ucount;
    int step_count, most, done;

	if (seed != 0) return;							/* only process #0 does this */
	linda_get();									/* get latest copy of the network */
	
	for (step_count = 0; step_count < no_steps; step_count++)
	  {
		Clock += 1;
		done = 0;
		if (!(Clock % ExecLimit))             /* make sure all done */
		  for (ucount = 0,up = UnitList; ucount < NoUnits; ucount++, up++)
			if (!TestFlagF((which = up->flags),LIMIT_SIM_FLAG))
			  {                               /* not done, so do now */
				done++;
				if(!TestFlagF(which,NO_UNIT_FUNC_FLAG))
				  {
					if(!TestFlagF(which,NO_SITE_FUNC_FLAG))
					  for(sp = up->sites;sp != NULL;sp = sp->next)
						{
						  sp->site_f(up,sp);
						  if(!TestFlagF(which,NO_LINK_FUNC_FLAG))
							for(ip = sp->inputs; ip != NULL; ip = ip->next)
							  ip->link_f(up,sp,ip);
						}
					up->unit_f(up);
				  }
				SetFlagP(up,STEP_SIM_FLAG);   /* done this step */
			  }
			else
			  UnsetFlagP(up,LIMIT_SIM_FLAG); /* unset for next cycle*/

		most = (NoUnits * ExecFract)/100;     /* total to be done */
		for(; done < most;done++)
		  {
			which = NoUnits;                  /* get in register */
			while (TestFlagP((up = UnitList+(ucount =
											 (rand()%which))),
							 STEP_SIM_FLAG)); /* get unit pointer in reg*/
			if(!TestFlagF((which = up->flags),NO_UNIT_FUNC_FLAG))
			  {                               /* get unit flags in reg */
				if(!TestFlagF(which,NO_SITE_FUNC_FLAG))
				  for(sp = up->sites;sp != NULL;sp = sp->next)
					{
					  sp->site_f(up,sp);
					  if(!TestFlagF(which,NO_LINK_FUNC_FLAG))
						for(ip = sp->inputs; ip != NULL; ip = ip->next)
						  ip->link_f(up,sp,ip);
					}
				up->unit_f(up);

				Outputs[ucount] = up->output;
			  }
			SetFlagP(up,STEP_SIM_FLAG);
			SetFlagP(up,LIMIT_SIM_FLAG);
		  }
		
		for (which = 0,up = UnitList, ucount = NoUnits;
			 which < ucount;
			 which++,up++)
		  UnsetFlagP(up,STEP_SIM_FLAG);
	}
	
	linda_spread(1);									/* send the result back to Mac */	
}

/*-------------------------------------------------------------------------------------

	The rest of this file contains the transputer versions of the commands issued by the 
	user at the command interface. They are stripped down versions of the 'Cmd_...' functions.
	
  --------------------------------------------------------------------------------------*/
  
linda_weight(argc,argv)
int argc;
char ** argv;
{
    int u;
    weight_type val, pert;
    char * sitename;
    short ul,uh,us,usind,typ;

    if (argc == 2 && Lex(argv[1]) == RANDOM)  {
        RandomiseWeights(500,100);
        return(0);
    }

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return(0);
        if (argc <= Curarg) return(0);
        ul = Ulow; uh = Uhigh; us = Uset; usind = Usetind;
        if(!GetUnits(argc,argv)) return(0);
        if (argc <= Curarg) return(0);
        sitename = argv[Curarg];
        u = Lex(argv[Curarg]);  /* read ident */
        if (u == HELP || u == INT || u == FLOAT || u == PLUS ||
            u == MINUS || u == CBRACK || u == OBRACK)
          return(0);
        switch(Lex(argv[Curarg]))
          {
          case INT :
            typ = FALSE;
            val = Yyint;
            break;
          case FLOAT :
            typ = FALSE;
            val = Yyfloat;
            break;
          case RANDOM:
            typ = TRUE;
            if (argc <= Curarg)
              val = 500;
            else
              switch(Lex(argv[Curarg]))
                {
                case INT:
                  val = Yyint;
                  break;
                case FLOAT:
                  val = Yyfloat;
                  break;
                default:
                  return(0);
                }
            if (argc <= Curarg)
              pert = 200;
            else
              switch(Lex(argv[Curarg]))
                {
                case INT:
                  pert = Yyint;
                  break;
                case FLOAT:
                  pert = Yyfloat;
                  break;
                default:
                  return(0);
                }
            break;
          default:
            EAT;
            return(0);
          }
#ifdef BCNTL
        Set_Cmd_Numbs(9,ul,uh,us,usind,Uset,Usetind,typ,val,pert);
        Set_Cmd_Names(1,sitename);
        Send_Range_Cmd(WEIGHT_C,Ulow,Uhigh);
#else
        SetWeight(ul,uh,us,usind,Ulow,Uhigh,Uset,Usetind,
                  sitename,typ,val,pert);
#endif
      }
    return(0);
}

/* ************************************************************************************ */

linda_lfunc(argc,argv)
int argc;
char ** argv;
{
    register int u;
    register func_ptr func;
    register Unit * up;
    char * sname;
    Site * sp;
    register Link * lp;
    int flag,ul,uh,us,usind;

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return(0);
        if (argc <= Curarg) return(0);
        ul = Ulow; uh = Uhigh; us = Uset; usind = Usetind;
        if(!GetUnits(argc,argv)) return(0);
        if (argc <= Curarg) return(0);
        Lex(argv[Curarg]);      /* process it */
        sname = argv[Curarg-1];
        if (argc <= Curarg) return(0);
        switch(Lex(argv[Curarg]))
          {
          case OFF:
            flag = -1;
            break;
          case ON:
            flag = 1;
            break;
          default:
            flag = 0;
            if ((func = NameToFunc(Yyarg)) == NULL)
               {
                 EAT;
                 return(0);
               }
          }

#ifdef BCNTL
        Set_Cmd_Numbs(6,Uset,Usetind,ul,uh,us,usind);
        Set_Cmd_Names(2,sname,Yyarg);
        Send_Range_Cmd(FUNC_C,Ulow,Uhigh); /* see eg state command for impl */
#else
        if (!(strcmp(sname,"all")))
          {
            if (flag == 0)      								/* reset to new link function */
              FOR_UNITS_P(u,up) 								/* for all destination units */
                for (sp = up->sites; 							/* for all sites */
                     sp != NULL;
                     sp = sp->next)
                  for (lp = sp->inputs; lp != NULL; lp = lp->next)
                    {
                      if (lp->from_unit >= ul && lp->from_unit <= uh &&
                          (!us || (UnitList[lp->from_unit].sets >> usind) & 1))
                        lp->link_f = func;
                    }
            else
              if (ul == 0 && uh == NoUnits-1 && !us) 			/* source all units */
                if (flag > 0)
                  FOR_UNITS_P(u,up)     						/* for all destination units */
                    UnsetFlagP(up,NO_LINK_FUNC_FLAG);
                else
                  FOR_UNITS_P(u,up)     						/* for all destination units */
                    SetFlagP(up,NO_LINK_FUNC_FLAG);
              else
                return(0);						/* can turn on/off for all source units only */
          }
        else                    								/* a particular site */
          if (flag == 0)        								/* so should be function change */
            {
              FOR_UNITS_P(u,up)
                {               								/* for destination units */
                  for (sp = up->sites; 							/* find correct site */
                       sp != NULL && strcmp(sp->name,sname);
                       sp = sp->next);
                  if (sp != NULL) 								/* found ok */
                    for (lp = sp->inputs; lp != NULL; lp = lp->next)
                      {         								/* for all links */
                        if (lp->from_unit >= ul && lp->from_unit <= uh &&
                          (!us || (UnitList[lp->from_unit].sets >> usind) & 1))
                         lp->link_f = func; 					/* change function */
                      }
                  else ;         								/* can't find site */
                }
            }
          else
            return(0);
#endif
      }
    return(0);
}

/* ************************************************************************************ */

linda_sfunc(argc,argv)
int argc;
char ** argv;
{
    register int u;
    register func_ptr func;
    register Unit * up;
    char * sname;
    Site * sp;
    int flag;

    Curarg=1;                             			/* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return(0);
        if (argc <= Curarg) return(0);
        Lex(argv[Curarg]);      					/* process it */
        sname = argv[Curarg-1];
        if (argc <= Curarg) return(0);
        switch(Lex(argv[Curarg]))
          {
          case OFF:
            flag = -1;
            break;
          case ON:
            flag = 1;
            break;
          default:
            flag = 0;
            if ((func = NameToFunc(Yyarg)) == NULL)
               {
                 EAT;
                 return(0);
               }
          }

#ifdef BCNTL
        Set_Cmd_Numbs(2,Uset,Usetind);
        Set_Cmd_Names(2,sname,Yyarg);
        Send_Range_Cmd(FUNC_C,Ulow,Uhigh); /* see eg state command for impl */
#else
        if (!(strcmp(sname,"all")))
          {
            FOR_UNITS_P(u,up)
              for (sp = up->sites;
                   sp != NULL;
                   sp = sp->next)
                if (flag == 0)
                  sp->site_f = func;
                else
                  if (flag > 0)
                    UnsetFlagP(up,NO_SITE_FUNC_FLAG);
                  else
                    SetFlagP(up,NO_SITE_FUNC_FLAG);
          }
        else
          if (flag == 0)
            {
              FOR_UNITS_P(u,up)
                {
                  for (sp = up->sites;
                       sp != NULL && strcmp(sp->name,sname);
                       sp = sp->next);
                  if (sp != NULL)
                    sp->site_f = func;
                  else ;
                }
            }
          else
            return(0);
#endif
      }
    return(0);
}

/* ************************************************************************************ */

linda_ufunc(argc,argv)
int argc;
char ** argv;
{
    register int u;
    register func_ptr func;
    register Unit * up;
    int flag;

    Curarg=1;										/* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return(0);
        if (argc <= Curarg) return(0);
        switch(Lex(argv[Curarg]))
          {
          case OFF:
            flag = -1;
            break;
          case ON:
            flag = 1;
            break;
          default:
            flag = 0;
            if((func = NameToFunc(Yyarg)) == NULL)
              {
                EAT;
                return 0;
              }
          }
#ifdef BCNTL
            Set_Cmd_Numbs(2,Uset,Usetind);
            Set_Cmd_Names(1,Yyarg);
            Send_Range_Cmd(UFUNC_C,Ulow,Uhigh);
            /* see eg state command for impl */
#else
        if (flag == 0)
          {
            FOR_UNITS_P(u,up)
              up->unit_f = func; 						/* reset function */
          }
        else
          if (flag > 0)         						/* turn unit on */
            FOR_UNITS_P(u,up)
              UnsetFlagP(up,NO_UNIT_FUNC_FLAG);
          else
            FOR_UNITS_P(u,up)   						/* turn unit off */
              SetFlagP(up,NO_UNIT_FUNC_FLAG);
#endif
      }
    return(0);
}

/* ************************************************************************************ */

linda_state(argc,argv)
int argc;
char ** argv;
{
    register int u;
    register int val;
    register Unit * up;

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return(0);
        if (argc <= Curarg) return(0);
        switch(Lex(argv[Curarg]))
          {
          case INT :
            val = Yyint;
            break;
          default:      
            if((val = NameToState(Yyarg)) == -1)
              {
                EAT;
                return 0;
              }
          }
#ifdef BCNTL
        Set_Cmd_Numbs(3,Uset,Usetind,val);
        Send_Range_Cmd(STATE_C,Ulow,Uhigh);
#else
        FOR_UNITS_P(u,up)
          up->state = val;
#endif
      }
    return(0);
}

/* ************************************************************************************ */

linda_out(argc,argv)
int argc;
char ** argv;
{
  register int u;
  register Output val;
  register Unit * up;

    Curarg=1;                             /* first command argument */
    while(argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return(0);
        if (argc <= Curarg) return(0);
        switch(Lex(argv[Curarg]))
          {
          case INT :
            val = Yyint;
            break;
          case FLOAT :
            val = Yyfloat;
            break;
          default:
            EAT;
            return(0);
          }
#ifdef BCNTL
        Set_Cmd_Numbs(3,Uset,Usetind,val);
        Send_Range_Cmd(OUT_C,Ulow,Uhigh);
#else
        FOR_UNITS_P(u,up)
        {
#ifndef TSIM
            up->output = Outputs[u] = val;
#else
            up->output = *((Outputs[u])+1) = val;
#endif
        }
#endif
      }
    return(0);
}

/* ************************************************************************************ */

linda_pot(argc,argv)
int argc;
char ** argv;
{
  register int u;
  register pot_type val;
  register Unit * up;

    Curarg = 1;
    while (argc > Curarg)
      {
        if(!GetUnits(argc,argv)) return(0);
        if (argc <= Curarg) return(0);
        switch(Lex(argv[Curarg]))
          {
          case INT :
            val = Yyint;
            break;
          case FLOAT :
            val = Yyfloat;
            break;
          default:
            EAT;
            return 0;
          }
#ifdef BCNTL
        Set_Cmd_Numbs(3,Uset,Usetind,val);
        Send_Range_Cmd(POT_C,Ulow,Uhigh);
#else
        FOR_UNITS_P(u,up)
          up->potential = val;
#endif
      }
    return(0);
}
