%{
#include <stdio.h>
#include <setjmp.h>
#include "parse.h"
#include "symtab.h"


/*
** Parser routines which return something other than int.
*/

extern ParseNode *vardef();
extern char *TokenStr();

/*
** Parser global variables
*/

extern jmp_buf	BreakBuf;	/* Used to break out of a loop */
extern jmp_buf	ReturnBuf;	/* Used to return out of a script */

static int	BreakAllowed = 0; /* In a loop control structure */
static int	ReturnIdents = 0; /* 1 ==> lexer doesn't classify IDENTs */
static int	Compiling = 0;	/* Make a statement list rather than execute */
static int	InFunctionDefinition = 0;
static int	NextLocal;	/* Next local symbol offset */
static int	ArgMatch;	/* Matches argument number to name in func */
static Symtab	GlobalSymbols;	/* Symbols defined outside a function */
static Symtab	*LocalSymbols = NULL;	/* Symbols local to a function */

%}

%nonassoc LT LE GT GE EQ NE
%left '!'
%left OR AND
%left '+' '-' '&' '|' '^'
%left '*' '/' '%' '~'
%left UMINUS

%token WHILE IF ELSE FOR FOREACH END INCLUDE BREAK
%token INT FLOAT STR RETURN WHITESPACE FUNCTION
%token <iconst> INTCONST DOLLARARG
%token <fconst> FLOATCONST
%token <str> STRCONST LITERAL IDENT VARREF FUNCREF
  /*
  ** The following tokens are defined only to be used as unique parse tree
  ** node types.  Real tokens are used when applicable.
  */

%token SL COMMAND ARGUMENT ARGLIST LOCREF ICAST FCAST SCAST

%type <pn> script statement statement_list while_stmnt for_stmnt
%type <pn> foreach_stmnt if_stmnt else_clause assign_stmnt cmd_stmnt
%type <pn> arg_list func_call_list arg_component_list arg_component
%type <pn> func_def decl_stmnt init return_stmnt null_stmnt expr
%type <pn> func_args func_arg_list func_hdr include_stmnt func_call
%type <pn> break_stmnt
%type <str> funcref cmd_name include_hdr
%{
/*
** Start of lexical analyzer.  LEX source is in "script.l".
*/

extern YYSTYPE	yylval;

#include "lex.yy.c"
%}

%%

script		: statement_list
		;

statement_list	: /* NOTHING */
		  { 
		    $$ = NULL;
 		  }
		| statement_list statement '\n'
		  {
		    if (InFunctionDefinition || Compiling)
			$$ = PTNew(SL, NULL, $1, $2);
		    else
		      {
		        /* execute statement */
			if (setjmp(BreakBuf) == 0)
			    if (setjmp(ReturnBuf) == 0)
				PTEval($2);
			    else
				EndScript();
			PTFree($2);
		      }
		  }
		;

statement	: while_stmnt
		| for_stmnt
		| foreach_stmnt
		| if_stmnt
		| assign_stmnt
		| include_stmnt
		| cmd_stmnt
		| func_call
		| func_def
		| decl_stmnt
		| break_stmnt
		| return_stmnt
		| null_stmnt
		;

while_stmnt	: WHILE '(' expr ')'
		  {
		    Compiling++;
		    BreakAllowed++;
		  }
		  statement_list END
		  {
		    $$ = PTNew(WHILE, NULL, $3, $6);
		    Compiling--;
		    BreakAllowed--;
		  }
		;

for_stmnt	: FOR '(' assign_stmnt ';' expr ';' assign_stmnt ')'
		    { Compiling++; BreakAllowed++; }
		      statement_list
 		  END
		  {
		    ParseNode	*forbody, *whilepart;

		    forbody = PTNew(SL, NULL, $10, $7);
		    whilepart = PTNew(WHILE, NULL, $5, forbody);
		    $$ = PTNew(SL, NULL, $3, whilepart);
		    Compiling--;
		    BreakAllowed--;
		  }
		;

foreach_stmnt	: FOREACH VARREF '('
		    {
			BEGIN FUNCLIT;
			Compiling++;
			BreakAllowed++;
		    }
		  ws arg_list ws
		    {
			BEGIN 0;
		    }
 		  ')' statement_list END
		  {
		    Result	*rp;
		    ResultValue	v;
		    char        buf[100];

		    rp = (Result *) $2;
		      {
			if (rp->r_type != STR &&
			    (rp->r_type == LOCREF && rp->r.r_loc.l_type != STR))
			  {
			   yyerror("Variable in FOREACH not a string variable");
			    /* No Return */
			  }
		      }

		    v.r_str = (char *) rp;
		    $$ = PTNew(FOREACH, v, $6, $10);
		    Compiling--;
		    BreakAllowed--;
		  }
		;

if_stmnt	: IF '(' expr ')'
		  { Compiling++; }
		    statement_list else_clause END
		  {
		    ParseNode	*stmntlists;

		    stmntlists = PTNew(0, NULL, $6, $7);
		    $$ = PTNew(IF, NULL, $3, stmntlists);
		    Compiling--;
		  }
		;

else_clause	: /* Nothing */
		  {
 		    $$ = NULL;
 		  }
		| ELSE statement_list
		  { $$ = $2; }
		;

assign_stmnt	: VARREF '=' expr
		  {
		    ResultValue	v;
		    Result	*rp;
		    char        buf[100];

		    $$ = NULL;
		    rp = (Result *) $1;
			  {
			    v.r_str = (char *) rp;
		            $$ = PTNew('=', v, $3, NULL);
			  }
		  }
		;

include_hdr	: INCLUDE
		  {
		    BEGIN LIT;
		  }
		  ws
		  {
		    $$ = NULL;
		  }
		;

include_stmnt	: include_hdr LITERAL ws arg_list ws
		  {
		    ResultValue	v;
		    Result	*rp;
		    int		argc;
		    char	*argv[100];
		    char	argbuf[1000];
		    jmp_buf	save;
		    Result	r;

		    BEGIN 0;

		      {
			fprintf(argbuf, "%s", $2);
			argc = 1;
			argv[0] = argbuf;
			do_cmd_args($4, &argc, argv, argbuf+strlen(argbuf)+1);
			argv[argc] = NULL;

			if (!IncludeScript(argc, argv))
			  {
			    sprintf(argbuf, "Script '%s' not found", $2);
			    PTFree($4);
			    free($2);
			    yyerror(argbuf);
			  }

			PTFree($4);
			free($2);
			$$ = NULL;
		      }
		  }
		| include_hdr LITERAL ws
		  {
		    ResultValue	v;
		    Result	*rp;
		    int		argc;
		    char	*argv[100];
		    char	argbuf[1000];
		    jmp_buf	save;
		    Result	r;

		    BEGIN 0;

		      {
			sprintf(argbuf, "%s", $2);
			argc = 1;
			argv[0] = argbuf;
			argv[argc] = NULL;

			if (!IncludeScript(argc, argv))
			  {
			    sprintf(argbuf, "Script '%s' not found", $2);
			    free($1);
			    yyerror(argbuf);
			  }

			free($2);
			$$ = NULL;
		      }
		  }
		;

cmd_name	: IDENT
		  {
		    BEGIN LIT;
		    $$ = $1;
		  }
		;

cmd_stmnt	: cmd_name ws arg_list ws
		  {
		    ResultValue	v;
		    Result	*rp;
		    int		argc;
		    char	*argv[100];
		    char	argbuf[1000];
		    jmp_buf	save;
		    Result	r;

		    BEGIN 0;

		    if (!IsCommand($1))
		      {
			sprintf(argbuf, "%s", $1);
			argc = 1;
			argv[0] = argbuf;
			do_cmd_args($3, &argc, argv, argbuf+strlen(argbuf)+1);
			argv[argc] = NULL;

			if (!IncludeScript(argc, argv))
			  {
			    v.r_str = $1;
			    $$ = PTNew(COMMAND, v, $3, NULL);
			  }
			else
			  {
			    PTFree($3);
			    free($1);
			    $$ = NULL;
			  }
		      }
		    else
		      {
			v.r_str = $1;
			$$ = PTNew(COMMAND, v, $3, NULL);
		      }
		  }
		| cmd_name ws
		  {
		    ResultValue	v;
		    Result	*rp;
		    int		argc;
		    char	*argv[100];
		    char	argbuf[1000];
		    jmp_buf	save;
		    Result	r;

		    BEGIN 0;

		    if (!IsCommand($1))
		      {
			sprintf(argbuf, "%s", $1);
			argc = 1;
			argv[0] = argbuf;
			argv[argc] = NULL;

			if (!IncludeScript(argc, argv))
			  {
			    v.r_str = $1;
			    $$ = PTNew(COMMAND, v, NULL, NULL);
			  }
			else
			  {
			    free($1);
			    $$ = NULL;
			  }
		      }
		    else
		      {
			v.r_str = $1;
			$$ = PTNew(COMMAND, v, NULL, NULL);
		      }
		  }
		;

funcref		: FUNCREF
		    {
			BEGIN FUNCLIT;
		    }
		  ws
		    {
			$$ = $1;
		    }
		;

func_call	: funcref '('
		  func_call_list
		    {
			BEGIN 0;
		    }
		  ')'
		  {
		    ResultValue	v;
		    Result	*rp;

		    rp = (Result *) $1;
		    $$ = PTNew(FUNCTION, rp->r, $3, NULL);
		  }
		| funcref
		    {
			BEGIN LIT;
		    }
		  arg_list ws
		  {
		    ResultValue	v;
		    Result	*rp;

		    BEGIN 0;
		    rp = (Result *) $1;
		    $$ = PTNew(FUNCTION, rp->r, $3, NULL);
		  }
		| funcref
		  {
		    Result	*rp;

		    BEGIN 0;
		    rp = (Result *) $1;
		    $$ = PTNew(FUNCTION, rp->r, NULL, NULL);
		  }
		;

arg_list	: arg_component_list
		  {
		    $$ = PTNew(ARGLIST, NULL, NULL, $1);
		  }
		| arg_list wslist arg_component_list
		  {
		    $$ = PTNew(ARGLIST, NULL, $1, $3);
		  }
		;

wslist		: WHITESPACE
		| wslist WHITESPACE
		;

func_call_list	: arg_component_list
		  {
		    $$ = PTNew(ARGLIST, NULL, NULL, $1);
		  }
		| func_call_list ws ',' ws arg_component_list
		  {
		    $$ = PTNew(ARGLIST, NULL, $1, $5);
		  }
		;

ws		: /* NOTHING */
		| WHITESPACE
		;

arg_component_list	: arg_component
			  {
			    ResultValue	v;

			    $$ = PTNew(ARGUMENT, v, NULL, $1);
			  }
			| arg_component_list arg_component
			  {
			    ResultValue	v;

			    $$ = PTNew(ARGUMENT, v, $1, $2);
			  }
			;

arg_component	: LITERAL
		  {
		    ResultValue	v;

		    v.r_str = $1;
		    $$ = PTNew(LITERAL, v, NULL, NULL);
		  }
		| STRCONST
		  {
		    ResultValue	v;

		    v.r_str = $1;
		    $$ = PTNew(LITERAL, v, NULL, NULL);
		  }
		| DOLLARARG
		  {
		    ResultValue	v;

		    v.r_int = $1;
		    $$ = PTNew(DOLLARARG, v, NULL, NULL);
		  }
		| '{' expr '}'
		  {
		    if ($2->pn_val.r_type == STRCONST)
			$2->pn_val.r_type = LITERAL;

		    $$ = $2;
		  }
		;

func_hdr	: FUNCTION IDENT
		  {
		    ParseNode	*funcpn;
		    ResultValue	v;
		    Result	*rp;

		    if (InFunctionDefinition)
		      {
			fprintf(stderr, "Function definition within another function or\n");
			fprintf(stderr, "within a control structure (FUNCTION %s).\n", $2);
			yyerror("");
			/* No Return */
		      }

		    InFunctionDefinition++;
		    NextLocal = 0;
		    rp = SymtabNew(&GlobalSymbols, $2);
		    if (rp->r_type != 0 && rp->r_type != FUNCTION)
			fprintf(stderr, "WARNING: function name '%s' is redefining a variable!\n", $2);

		    rp->r_type = FUNCTION;

		    LocalSymbols = SymtabCreate();
		    v.r_str = (char *) LocalSymbols;
		    funcpn = PTNew(SL, v, NULL, NULL);
		    rp->r.r_str = (char *) funcpn;

		    $$ = funcpn;
		  }
		| FUNCTION FUNCREF
		  {
		    ParseNode	*funcpn;
		    ResultValue	v;
		    Result	*rp;

		    rp = (Result *) $2;
		    if (InFunctionDefinition)
		      {
			fprintf(stderr, "Function definition within another function or\n");
			fprintf(stderr, "within a control structure (FUNCTION %s).\n", $2);
			yyerror("");
			/* No Return */
		      }

		    /*
		    ** Free old function parse tree and symtab
		    */

		    funcpn = (ParseNode *) rp->r.r_str;
		    SymtabDestroy(funcpn->pn_val.r.r_str);
		    PTFree(funcpn);

		    InFunctionDefinition++;
		    NextLocal = 0;
		    LocalSymbols = SymtabCreate();
		    v.r_str = (char *) LocalSymbols;
		    funcpn = PTNew(SL, v, NULL, NULL);
		    rp->r.r_str = (char *) funcpn;

		    $$ = funcpn;
		  }
		;

func_def	: func_hdr 
		  {
		    ReturnIdents = 1;
		  }
		  func_args
		  {
		    ReturnIdents = 0;
		  }
		  statement_list END
		  {
		    InFunctionDefinition--;

		    $1->pn_left = $3;
		    $1->pn_right = $5;

		    LocalSymbols = NULL;
		    $$ = NULL;
		  }
		;

func_args	: /* NOTHING */
		  { $$ = NULL; }
		| '(' func_arg_list ')'
		  { $$ = $2; }
		;

func_arg_list	: IDENT
		  {
		    ResultValue	v;
		    ParseNode	*init;

		    ArgMatch = 1;
		    v.r_int = ArgMatch++;
		    init = PTNew(DOLLARARG, v, NULL, NULL);
		    $$ = vardef($1, STR, SCAST, init);
		  }
		| func_arg_list ',' IDENT
		  {
		    ResultValue	v;
		    ParseNode	*init;

		    v.r_int = ArgMatch++;
		    init = PTNew(DOLLARARG, v, NULL, NULL);
		    $$ = PTNew(SL, v, $1, vardef($3, STR, SCAST, init));
		  }
		;

decl_stmnt	: INT
		  {
		    ReturnIdents = 1;
		  }
		  IDENT
		  {
		    ReturnIdents = 0;
		  }
		  init
		  {
		    $$ = vardef($3, INT, ICAST, $5);
		  }
		| FLOAT
		  {
		    ReturnIdents = 1;
		  }
		  IDENT
		  {
		    ReturnIdents = 0;
		  }
		  init
		  {
		    $$ = vardef($3, FLOAT, FCAST, $5);
		  }
		| STR
		  {
		    ReturnIdents = 1;
		  }
		  IDENT
		  {
		    ReturnIdents = 0;
		  }
		  init
		  {
		    $$ = vardef($3, STR, SCAST, $5);
		  }
		;

init		: /* NOTHING */
		  { $$ = NULL; }
		| '=' expr
		  { $$ = $2; }
		;

break_stmnt	: BREAK
		  {
		    ResultValue	v;

		    if (BreakAllowed)
			$$ = PTNew(BREAK, v, NULL, NULL);
		    else
			yyerror("BREAK found outside of a loop");
			/* No Return */
		  }
		;

return_stmnt	: RETURN expr
		  {
		    ResultValue	v;

		    $$ = PTNew(RETURN, v, $2, NULL);
		  }
		| RETURN
		  {
		    ResultValue	v;

		    $$ = PTNew(RETURN, v, NULL, NULL);
		  }
		;

null_stmnt	: /* Nothing */
		  { $$ = NULL; }
		;

expr		: expr '|' expr
		  { $$ = PTNew('|', NULL, $1, $3); }
		| expr '&' expr
		  { $$ = PTNew('&', NULL, $1, $3); }
		| expr '^' expr
		  { $$ = PTNew('^', NULL, $1, $3); }
		| '~' expr
		  { $$ = PTNew('~', NULL, $2, NULL); }

		| expr '+' expr
		  { $$ = PTNew('+', NULL, $1, $3); }
		| expr '-' expr
		  { $$ = PTNew('-', NULL, $1, $3); }
		| expr '*' expr
		  { $$ = PTNew('*', NULL, $1, $3); }
		| expr '/' expr
		  { $$ = PTNew('/', NULL, $1, $3); }
		| expr '%' expr
		  { $$ = PTNew('%', NULL, $1, $3); }
		| '-' expr %prec UMINUS
		  { $$ = PTNew(UMINUS, NULL, $2, NULL); }

		| expr OR expr
		  { $$ = PTNew(OR, NULL, $1, $3); }
		| expr AND expr
		  { $$ = PTNew(AND, NULL, $1, $3); }
		| '!' expr
		  { $$ = PTNew('!', NULL, $2, NULL); }

		| expr LT expr
		  { $$ = PTNew(LT, NULL, $1, $3); }
		| expr LE expr
		  { $$ = PTNew(LE, NULL, $1, $3); }
		| expr GT expr
		  { $$ = PTNew(GT, NULL, $1, $3); }
		| expr GE expr
		  { $$ = PTNew(GE, NULL, $1, $3); }
		| expr EQ expr
		  { $$ = PTNew(EQ, NULL, $1, $3); }
		| expr NE expr
		  { $$ = PTNew(NE, NULL, $1, $3); }

		| IDENT '('
		    {
			BEGIN FUNCLIT;
		    }
 		  func_call_list
		    {
			BEGIN 0;
		    }
 		  ')'
		  {
		    ResultValue	v;
		    Result	*rp;

		    v.r_str = $1;
		    $$ = PTNew(COMMAND, v, $4, NULL);
		  }

		| FUNCREF '('
		    {
			BEGIN FUNCLIT;
		    }
 		  func_call_list
		    {
			BEGIN 0;
		    }
 		  ')'
		  {
		    ResultValue	v;
		    Result	*rp;

		    rp = (Result *) $1;
		    if (rp != NULL && rp->r_type == FUNCTION)
		        $$ = PTNew(FUNCTION, rp->r, $4, NULL);
		  }

		| VARREF
		  { 
		    Result	*rp;
		    ResultValue	v;

		    /*
		    ** Variable reference
		    */

		    rp = (Result *) $1;
		      {
			if (rp->r_type == FUNCTION || rp->r_type == LOCREF)
			    v = rp->r;
			else /* Global Variable */
			    v.r_str = (char *) rp;

		        $$ = PTNew(rp->r_type, v, NULL, NULL);
		      }
 		  }

		| FUNCREF
		  { 
		    Result	*rp;
		    ResultValue	v;

		    /*
		    ** Function call
		    */

		    rp = (Result *) $1;
		    $$ = PTNew(FUNCTION, rp->r, NULL, NULL);
 		  }

		| IDENT
		  { 
		    Result	*rp;
		    ResultValue	v;

		    /*
		    ** Command
		    */

		    v.r_str = $1;
		    $$ = PTNew(COMMAND, v, NULL, NULL);
 		  }

		| FLOATCONST
		  { 
		    ResultValue	v;

		    v.r_float = $1;
		    $$ = PTNew(FLOATCONST, v, NULL, NULL);
 		  }
		| INTCONST
		  { 
		    ResultValue	v;

		    v.r_int = $1;
		    $$ = PTNew(INTCONST, v, NULL, NULL);
 		  }
		| STRCONST
		  { 
		    ResultValue	v;

		    v.r_str = $1;
		    $$ = PTNew(STRCONST, v, NULL, NULL);
 		  }

		| DOLLARARG
		  {
		    ResultValue	v;

		    v.r_int = $1;
		    $$ = PTNew(DOLLARARG, v, NULL, NULL);
		  }

		| '(' expr ')'
		  { $$ = $2; }
		;
%%

/*
** TokenStr
**
**	Return the token string for the given token.
*/

char *TokenStr(token)

int	token;

{	/* TokenStr --- Return token string for token */

	static char	buf[100];

	switch (token)
	  {

	  case LT: return("<");
	  case LE: return("<=");
	  case GT: return(">");
	  case GE: return(">=");
	  case EQ: return("==");
	  case NE: return("!=");

	  case OR: return("||");
	  case AND: return("&&");

#define	RET(tok)	case tok: return("tok")

	  RET(UMINUS);

	  RET(WHILE);
	  RET(IF);
	  RET(ELSE);
	  RET(FOR);
	  RET(FOREACH);
	  RET(END);
	  RET(INCLUDE);
	  RET(BREAK);
	  RET(INT);
	  RET(FLOAT);
	  RET(STR);
	  RET(RETURN);
	  RET(WHITESPACE);
	  RET(FUNCTION);
	  RET(INTCONST);
	  RET(DOLLARARG);
	  RET(FLOATCONST);
	  RET(STRCONST);
	  RET(LITERAL);
	  RET(IDENT);
	  RET(VARREF);
	  RET(FUNCREF);
	  RET(SL);
	  RET(COMMAND);
	  RET(ARGUMENT);
	  RET(ARGLIST);
	  RET(LOCREF);
	  RET(ICAST);
	  RET(FCAST);
	  RET(SCAST);

	  }

	if (token < 128)
	    if (token < ' ')
		sprintf(buf, "^%c", token+'@');
	    else
		sprintf(buf, "%c", token);
	else
	    sprintf(buf, "%d", token);

	return(buf);

}	/* TokenStr */



#ifdef COMMENT
Result *vardef(ident, type)

char	*ident;
int	type;

{	/* vardef --- Define a variable */

	Result	*rp, r;
	Symtab	*symtab;

	if (InFunctionDefinition && LocalSymbols != NULL)
	  {
	    rp = SymtabNew(LocalSymbols, ident);
	    if (rp->r_type == 0)
	      {
	        rp->r_type = LOCREF;
		rp->r.r_loc.l_type = type;
		rp->r.r_loc.l_offs = NextLocal++;
	      }
	  }
	else
	  {
	    rp = SymtabNew(&GlobalSymbols, ident);
	    switch(type)
	      {

	      case INT:
	        if (rp->r_type == 0)
	            rp->r.r_int = 0;
	        else
		    CastToInt(rp);
	        break;

	      case FLOAT:
	        if (rp->r_type == 0)
	            rp->r.r_float = 0.0;
	        else
		    CastToFloat(rp);
	        break;

	      case STR:
	        if (rp->r_type == 0)
	            rp->r.r_str = strsave("");
	        else
		    CastToStr(rp);
	        break;

	      }

	    rp->r_type = type;
	  }

	return(rp);

}	/* vardef */
#endif



ParseNode *vardef(ident, type, castop, init)

char		*ident;
int		type;
int		castop;
ParseNode	*init;

{	/* vardef --- Define a variable */

	ParseNode	*pn;
	Result		*rp, r;
	ResultValue	v;

	if (InFunctionDefinition && LocalSymbols != NULL)
	  {
	    rp = SymtabNew(LocalSymbols, ident);
	    if (rp->r_type == 0)
	      {
	        rp->r_type = LOCREF;
		rp->r.r_loc.l_type = type;
		rp->r.r_loc.l_offs = NextLocal++;
	      }

	    v.r_str = (char *) rp;
	    pn = PTNew(castop, v, NULL, NULL);
	    if (init)
		pn = PTNew(SL, v, pn, PTNew('=', v, init, NULL));
	  }
	else
	  {
	    rp = SymtabNew(&GlobalSymbols, ident);
	    switch(type)
	      {

	      case INT:
	        if (rp->r_type == 0)
	            rp->r.r_int = 0;
	        else
		    CastToInt(rp);
	        break;

	      case FLOAT:
	        if (rp->r_type == 0)
	            rp->r.r_float = 0.0;
	        else
		    CastToFloat(rp);
	        break;

	      case STR:
	        if (rp->r_type == 0)
	            rp->r.r_str = (char *) strsave("");
	        else
		    CastToStr(rp);
	        break;

	      }

	    rp->r_type = type;
	    v.r_str = (char *) rp;
	    if (init)
	        pn = PTNew('=', v, init, NULL);
	    else
	        pn = NULL;
	  }

	return(pn);

}	/* vardef */


ParseInit()

{    /* ParseInit --- Initialize parser variables */

        InFunctionDefinition = 0;
	Compiling = 0;
	BreakAllowed = 0;
	LocalSymbols = NULL;
	nextchar(1);	/* Flush lexer input */

}    /* ParseInit */


int NestedLevel()

{    /* NestedLevel --- Return TRUE if in func_def or control structure */

        return(InFunctionDefinition || Compiling);

}    /* NestedLevel */
