package it.sauronsoftware.cron4j;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.StringTokenizer;

/**
 * This class parses and represents as its instances the scheduling patterns
 * expressed with a crontab-like syntax.
 * 
 * @author Carlo Pelliccia
 */
class SchedulingPattern {

	/**
	 * Months aliases.
	 */
	private static String[] MONTHS = new String[] { "", "jan", "feb", "mar",
			"apr", "may", "jun", "jul", "aug", "sep", "oct", "nov", "dec" };

	/**
	 * Days of week aliases.
	 */
	private static String[] DAYS_OF_WEEK = new String[] { "sun", "mon", "tue",
			"wed", "thu", "fri", "sat" };

	/**
	 * The original string from which the object has been built.
	 */
	private String originalString;

	/**
	 * The ValueMatcher list for the "minute" field.
	 */
	protected ArrayList minuteMatchers = new ArrayList();

	/**
	 * The ValueMatcher list for the "hour" field.
	 */
	protected ArrayList hourMatchers = new ArrayList();

	/**
	 * The ValueMatcher list for the "day of month" field.
	 */
	protected ArrayList dayOfMonthMatchers = new ArrayList();

	/**
	 * The ValueMatcher list for the "month" field.
	 */
	protected ArrayList monthMatchers = new ArrayList();

	/**
	 * The ValueMatcher list for the "day of week" field.
	 */
	protected ArrayList dayOfWeekMatchers = new ArrayList();

	/**
	 * How many matcher groups in this pattern?
	 */
	protected int matcherSize = 0;

	/**
	 * Builds a SchedulingPattern parsing it from a string.
	 * 
	 * @param pattern
	 *            The pattern as a crontab-like string.
	 * @throws InvalidPatternException
	 *             If the supplied string is not a valid pattern.
	 */
	public SchedulingPattern(String pattern) throws InvalidPatternException {
		this.originalString = pattern;
		StringTokenizer st1 = new StringTokenizer(pattern, "|");
		if (st1.countTokens() < 1) {
			throw new InvalidPatternException();
		}
		while (st1.hasMoreTokens()) {
			String localPattern = st1.nextToken();
			StringTokenizer st2 = new StringTokenizer(localPattern, " \t");
			if (st2.countTokens() != 5) {
				throw new InvalidPatternException();
			}
			minuteMatchers.add(buildValueMatcher(st2.nextToken(), 0, 59, null));
			hourMatchers.add(buildValueMatcher(st2.nextToken(), 0, 23, null));
			dayOfMonthMatchers.add(buildValueMatcher(st2.nextToken(), 1, 31, null));
			monthMatchers.add(buildValueMatcher(st2.nextToken(), 1, 12, MONTHS));
			dayOfWeekMatchers.add(buildValueMatcher(st2.nextToken(), 0, 6, DAYS_OF_WEEK));
			matcherSize++;
		}
	}

	/**
	 * A ValueMatcher utility builder.
	 * 
	 * @param str
	 *            The pattern part for the ValueMatcher creation.
	 * @param minValue
	 *            The minimum value for the field represented by the
	 *            ValueMatcher.
	 * @param maxValue
	 *            The maximum value for the field represented by the
	 *            ValueMatcher.
	 * @param stringEquivalents
	 *            An array of aliases for the numeric values. It can be null if
	 *            no alias is provided.
	 * @return The requested ValueMatcher.
	 * @throws InvalidPatternException
	 *             If the pattern part supplied is not valid.
	 */
	private ValueMatcher buildValueMatcher(String str, int minValue,
			int maxValue, String[] stringEquivalents)
			throws InvalidPatternException {
		if (str.length() == 1 && str.equals("*")) {
			return new AlwaysTrueValueMatcher();
		}
		StringTokenizer st = new StringTokenizer(str, "/");
		int tokens = st.countTokens();
		if (tokens < 1 || tokens > 2) {
			throw new InvalidPatternException();
		}
		ArrayList list = buildPart1(st.nextToken(), minValue, maxValue,
				stringEquivalents);
		if (tokens == 2) {
			list = buildPart2(list, st.nextToken(), minValue, maxValue);
		}
		return new IntArrayValueMatcher(list);
	}

	private ArrayList buildPart1(String str, int minValue, int maxValue,
			String[] stringEquivalents) throws InvalidPatternException {
		if (str.length() == 1 && str.equals("*")) {
			ArrayList ret = new ArrayList();
			for (int i = minValue; i <= maxValue; i++) {
				ret.add(new Integer(i));
			}
			return ret;
		}
		StringTokenizer st = new StringTokenizer(str, ",");
		if (st.countTokens() < 1) {
			throw new InvalidPatternException();
		}
		ArrayList list = new ArrayList();
		while (st.hasMoreTokens()) {
			ArrayList list2 = buildPart1_1(st.nextToken(), minValue, maxValue,
					stringEquivalents);
			int size = list2.size();
			for (int i = 0; i < size; i++) {
				list.add(list2.get(i));
			}
		}
		return list;
	}

	private ArrayList buildPart1_1(String str, int minValue, int maxValue,
			String[] stringEquivalents) throws InvalidPatternException {
		StringTokenizer st = new StringTokenizer(str, "-");
		int tokens = st.countTokens();
		if (tokens < 1 || tokens > 2) {
			throw new InvalidPatternException();
		}
		String str1 = st.nextToken();
		int value1;
		try {
			value1 = Integer.parseInt(str1);
		} catch (NumberFormatException e) {
			if (stringEquivalents != null) {
				try {
					value1 = getIntValue(str1, stringEquivalents);
				} catch (Exception e1) {
					throw new InvalidPatternException();
				}
			} else {
				throw new InvalidPatternException();
			}
		}
		if (value1 < minValue || value1 > maxValue) {
			throw new InvalidPatternException();
		}
		int value2 = value1;
		if (tokens == 2) {
			String str2 = st.nextToken();
			try {
				value2 = Integer.parseInt(str2);
			} catch (NumberFormatException e) {
				if (stringEquivalents != null) {
					try {
						value2 = getIntValue(str2, stringEquivalents);
					} catch (Exception e1) {
						throw new InvalidPatternException();
					}
				} else {
					throw new InvalidPatternException();
				}
			}
			if (value2 < minValue || value2 > maxValue || value2 <= value1) {
				throw new InvalidPatternException();
			}
		}
		ArrayList list = new ArrayList();
		for (int i = value1; i <= value2; i++) {
			Integer aux = new Integer(i);
			if (!list.contains(aux)) {
				list.add(aux);
			}
		}
		return list;
	}

	private ArrayList buildPart2(ArrayList list, String p2, int minValue,
			int maxValue) throws InvalidPatternException {
		int div = 0;
		try {
			div = Integer.parseInt(p2);
		} catch (NumberFormatException e) {
			;
		}
		if (div <= minValue || div >= maxValue) {
			throw new InvalidPatternException();
		}
		int size = list.size();
		ArrayList list2 = new ArrayList();
		for (int i = 0; i < size; i++) {
			Integer aux = (Integer) list.get(i);
			if (aux.intValue() % div == 0) {
				list2.add(aux);
			}
		}
		return list2;
	}

	/**
	 * This utility method changes an alias to an int value.
	 */
	private int getIntValue(String value, String[] values) throws Exception {
		for (int i = 0; i < values.length; i++) {
			if (values[i].equalsIgnoreCase(value)) {
				return i;
			}
		}
		throw new Exception();
	}

	/**
	 * This methods returns true if the given timestamp (expressed as a UNIX-era
	 * millis value) matches the pattern.
	 * 
	 * @param millis
	 *            The timestamp, as a UNIX-era millis value.
	 * @return true if the given timestamp matches the pattern.
	 */
	public boolean match(long millis) {
		GregorianCalendar gc = new GregorianCalendar();
		gc.setTime(new Date(millis));
		int minute = gc.get(Calendar.MINUTE);
		int hour = gc.get(Calendar.HOUR_OF_DAY);
		int dayOfMonth = gc.get(Calendar.DAY_OF_MONTH);
		int month = gc.get(Calendar.MONTH) + 1;
		int dayOfWeek = gc.get(Calendar.DAY_OF_WEEK) - 1;
		for (int i = 0; i < matcherSize; i++) {
			ValueMatcher minuteMatcher = (ValueMatcher) minuteMatchers.get(i);
			ValueMatcher hourMatcher = (ValueMatcher) hourMatchers.get(i);
			ValueMatcher dayOfMonthMatcher = (ValueMatcher) dayOfMonthMatchers.get(i);
			ValueMatcher monthMatcher = (ValueMatcher) monthMatchers.get(i);
			ValueMatcher dayOfWeekMatcher = (ValueMatcher) dayOfWeekMatchers.get(i);
			boolean eval = minuteMatcher.match(minute)
					&& hourMatcher.match(hour)
					&& dayOfMonthMatcher.match(dayOfMonth)
					&& monthMatcher.match(month)
					&& dayOfWeekMatcher.match(dayOfWeek);
			if (eval) {
				return true;
			}
		}
		return false;
	}

	/**
	 * This method returns the original string from which the object has been
	 * built.
	 * 
	 * @return The original string from which the object has been built.
	 */
	public String getOriginalString() {
		return originalString;
	}

}
